/*
   Pathetic Writer
   Copyright (C) 1997, 1998  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * fileio_html.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>

#include "../common/common.h"
#include "../common/fonts.h"
#include "pw.h"

static void save_line(FILE *fp, rich_char *line)
{
	if (!line) return;
	while (line->c) {
		putc(line->c, fp);
		line++;
	}
}

static int save(char *fn, buffer *buf)
/* Returns: 0 if successful, otherwise 1 */
{
	FILE *fp;
	int sty;
	int i, ll = line_last_used(buf), ls = STY_DEFAULT;
	char *closer = NULL;

	if ((fp = fopen(fn, "w")) == NULL) return 1;

	/* do the header */
	fprintf(fp, "<HTML>\n<HEAD>\n<TITLE>\n");
	/* use first non-blank line as title */
	for (i = 1; i <= ll; i++)
		if (buf->text[i].p) break;

	if (i > ll) printf("No title");
	else save_line(fp, buf->text[i].p);

	fprintf(fp, "\n</TITLE>\n</HEAD>\n\n<BODY>\n");

	for (i = 1; i <= ll; i++) {
		sty = ret_style(buf, i);
		if (sty != ls) {
			if (closer) fprintf(fp, "%s\n", closer);
			closer = NULL;
		}
		switch (sty) {
		case STY_HEADER1:
			if (sty != ls) {
				fprintf(fp, "<H1>\n");
				closer = "</H1>";
			}
			save_line(fp, buf->text[i].p);
			break;
		case STY_HEADER2:
			if (sty != ls) {
				fprintf(fp, "<H2>\n");
				closer = "</H2>";
			}
			save_line(fp, buf->text[i].p);
			break;
		case STY_HEADER3:
			if (sty != ls) {
				fprintf(fp, "<H3>\n");
				closer = "</H3>";
			}
			save_line(fp, buf->text[i].p);
			break;
		case STY_HEADER4:
			if (sty != ls) {
				fprintf(fp, "<H4>\n");
				closer = "</H4>";
			}
			save_line(fp, buf->text[i].p);
			break;
		case STY_HEADER5:
			if (sty != ls) {
				fprintf(fp, "<H5>\n");
				closer = "</H5>";
			}
			save_line(fp, buf->text[i].p);
			break;
		case STY_HEADER6:
			if (sty != ls) {
				fprintf(fp, "<H6>\n");
				closer = "</H6>";
			}
			save_line(fp, buf->text[i].p);
			break;
		case STY_ADDRESS:
			if (sty != ls) {
				fprintf(fp, "<ADDRESS>\n");
				closer = "</ADDRESS>";
			}
			save_line(fp, buf->text[i].p);
			break;
		case STY_PREFORMAT:
			if (sty != ls) {
				fprintf(fp, "<PRE>\n");
				closer = "</PRE>";
			}
			save_line(fp, buf->text[i].p);
			break;
		case STY_EMBED:
			fprintf(fp, "<BR><IMG SRC=\"");
			save_line(fp, buf->text[i].p);
			fprintf(fp, "\" ALT=\"\"><BR>");
			break;
		default:	/* STY_DEFAULT and all STY_USERx styles */
			if (rc_strlen(buf->text[i].p) == 0)
				fprintf(fp, "<P>\n");
			save_line(fp, buf->text[i].p);
			break;
		}
		fprintf(fp, "\n");
		ls = sty;
	}
	if (closer) fprintf(fp, "%s\n", closer);
	fprintf(fp, "</BODY>\n</HTML>\n");
	fclose(fp);
	return 0;
}


static enum {START, INATAG, INACHAR, END} state;
static int errflag;
static char tagbuf[256];	/* for &chars; and <tags> */
static rich_char outbuf[356];	/* to buffer output */
static int tbi, obi, mute, pre;
static unsigned long fmt;
static unsigned long row;
static buffer *buf;

static void store_text(rich_char *p)
{
	int i;
	for (i = 0; p[i].c; i++) {
#if 1
		ins_char(buf, row, line_length(buf, row),
			p[i].c, p[i].fmt);
#else
		alloc_line(buf, row);
		printf("%c", p[i].c);
#endif
	}
}

static void emitchar(int c)
{
	if (mute) return;
	if (c == '\n') {	/* break the current line */
		outbuf[obi].c = '\0';
		store_text(outbuf);
		row++;
		obi = 0;
	} else {		/* try to cut at space */
		if (obi > 70) {
			int i = obi;
			while (i && outbuf[i].c != ' ') i--;
			if (i) {
				outbuf[i].c = '\0';
				store_text(outbuf);
				row++;
				obi -= (i+1);
				memmove(outbuf, outbuf+i+1,
					obi*sizeof(rich_char));
			}
		}
		outbuf[obi].fmt = fmt;
		outbuf[obi++].c = c;	/* append char */
	}
}

static struct {
	char *name;
	int value;
} cchar[] = {
	{"quot", '"'}, {"amp", '&'}, {"lt", '<'}, {"gt", '>'},
	{"nbsp", 160}, {"iexcl", 161}, {"cent", 162}, {"pound", 163},
	{"curren", 164}, {"yen", 165}, {"brvbar", 166}, {"sect", 167},
	{"uml", 168}, {"copy", 169}, {"ordf", 170}, {"laquo", 171},
	{"not", 172}, {"shy", 173}, {"reg", 174}, {"macr", 175},
	{"deg", 176}, {"plusmn", 177}, {"sup2", 178}, {"sup3", 179},
	{"acute", 180}, {"micro", 181}, {"para", 182}, {"middot", 183},
	{"cedil", 184}, {"sup1", 185}, {"ordm", 186}, {"raquo", 187},
	{"frac14", 188}, {"frac12", 189}, {"frac34", 190}, {"iquest", 191},
	{"Agrave", 192}, {"Aacute", 193}, {"Acirc", 194}, {"Atilde", 195},
	{"Auml", 196}, {"Aring", 197}, {"AElig", 198}, {"Ccedil", 199},
	{"Egrave", 200}, {"Eacute", 201}, {"Ecirc", 202}, {"Euml", 203},
	{"Igrave", 204}, {"Iacute", 205}, {"Icirc", 206}, {"Euml", 207},
	{"ETH", 208}, {"Ntilde", 209}, {"Ograve", 210}, {"Oacute", 211},
	{"Ocirc", 212}, {"Otilde", 213}, {"Ouml", 214}, {"times", 215},
	{"Oslash", 216}, {"Ugrave", 217}, {"Uacute", 218}, {"Ucirc", 219},
	{"Uuml", 220}, {"Yacute", 221}, {"THORN", 222}, {"szlig", 223},
	{"agrave", 224}, {"aacute", 225}, {"acirc", 226}, {"atilde", 227},
	{"auml", 228}, {"aring", 229}, {"aelig", 230}, {"ccedil", 231},
	{"egrave", 232}, {"eacute", 233}, {"ecirc", 234}, {"euml", 235},
	{"igrave", 236}, {"iacute", 237}, {"icirc", 238}, {"iuml", 239},
	{"eth", 240}, {"ntilde", 241}, {"ograve", 242}, {"oacute", 243},
	{"ocirc", 244}, {"otilde", 245}, {"ouml", 246}, {"divide", 247},
	{"slash", 248}, {"ugrave", 249}, {"uacute", 250}, {"ucirc", 251},
	{"uuml", 252}, {"yacute", 253}, {"thorn", 254}, {"yuml", 255},
	{NULL, 0}
};

/* &#xxx; => emitchar(atoi(xxx)) */
static void emitcchar(char *p)
{
	int i;
	if (p[0] == '#') {
		i = atoi(p+1);
		if (i >= ' ' && i <= 255) emitchar(atoi(p+1));
	} else {
		for (i = 0; cchar[i].name; i++)
			if (!strcmp(p, cchar[i].name)) break;
		if (cchar[i].name) emitchar(cchar[i].value);
	}
}

static void tag_ignore(void)
{
	;
}

static void tag_newline(void)
{
	emitchar('\n');
}

static void tag_mute(void)
{
	mute = 1;
}

static void tag_unmute(void)
{
	mute = 0;
}

static void tag_pre(void)
{
	fmt = styles[STY_PREFORMAT].format;
	emitchar('\n');
	pre = 1;
	set_style(buf, row, STY_PREFORMAT);
}

static void tag_pre_(void)
{
	fmt = styles[STY_DEFAULT].format;
	emitchar('\n');
	pre = 0;
	set_style(buf, row, STY_DEFAULT);
}

static void tag_h1(void)
{
	fmt = styles[STY_HEADER1].format;
	emitchar('\n');
	set_style(buf, row, STY_HEADER1);
}

static void tag_h2(void)
{
	fmt = styles[STY_HEADER2].format;
	emitchar('\n');
	set_style(buf, row, STY_HEADER2);
}

static void tag_h3(void)
{
	fmt = styles[STY_HEADER3].format;
	emitchar('\n');
	set_style(buf, row, STY_HEADER3);
}

static void tag_h4(void)
{
	fmt = styles[STY_HEADER4].format;
	emitchar('\n');
	set_style(buf, row, STY_HEADER4);
}

static void tag_h5(void)
{
	fmt = styles[STY_HEADER5].format;
	emitchar('\n');
	set_style(buf, row, STY_HEADER5);
}

static void tag_h6(void)
{
	fmt = styles[STY_HEADER6].format;
	emitchar('\n');
	set_style(buf, row, STY_HEADER6);
}

static void tag_address(void)
{
	fmt = styles[STY_ADDRESS].format;
}

static void tag_plain(void)
{
	fmt = styles[STY_DEFAULT].format;
	emitchar('\n');
	set_style(buf, row, STY_DEFAULT);
}

static void tag_i(void)
{
	fmt |= ITALIC;
}

static void tag_i_(void)
{
	fmt &= ~ITALIC;
}

static void tag_b(void)
{
	fmt |= BOLD;
}

static void tag_b_(void)
{
	fmt &= ~BOLD;
}

static struct {
	char *name;
	void (*action)(void);
} tag[] = {
	{"br", tag_newline},
	{"p", tag_newline},
	{"ol", tag_newline},
	{"/ol", tag_newline},
	{"ul", tag_newline},
	{"li", tag_newline},
	{"dl", tag_newline},
	{"dt", tag_newline},
	{"dd", tag_newline},
	{"table", tag_newline},
	{"/table", tag_newline},
	{"tr", tag_newline},
	{"th", tag_newline},
	{"/p", tag_ignore},
	{"h1", tag_h1},
	{"/h1", tag_plain},
	{"h2", tag_h2},
	{"/h2", tag_plain},
	{"h3", tag_h3},
	{"/h3", tag_plain},
	{"h4", tag_h4},
	{"/h4", tag_plain},
	{"h5", tag_h5},
	{"/h5", tag_plain},
	{"h6", tag_h6},
	{"/h6", tag_plain},
	{"hr", tag_newline},
	{"li", tag_newline},
	{"pre", tag_pre},
	{"/pre", tag_pre_},
	{"address", tag_address},
	{"/address", tag_plain},
	{"i", tag_i},
	{"/i", tag_i_},
	{"b", tag_b},
	{"/b", tag_b_},
	{"head", tag_mute},
	{"/head", tag_unmute},
	{NULL, NULL}
};

/* any unrecognized tag is ignored */
static void emittag(char *p)
{
	int i;
	for (i = 0; tag[i].name; i++)
		if (!cstrcasecmp(p, tag[i].name)) break;
	if (tag[i].name) (*tag[i].action)();
}

static void html_char(int c)
{
	switch (state) {
	case START:
		if (isspace(c)) {
			if (pre) emitchar(c);
			else emitchar(' ');
		} else if (c == '<') {
			tbi = 0;
			state = INATAG;
		} else if (c == '&') {
			tbi = 0;
			state = INACHAR;
		} else if (c == EOF) {
			state = END;
		} else {
			emitchar(c);
		}
		break;
	case INATAG:
		if (c == '>') {
			tagbuf[tbi] = '\0';
			emittag(tagbuf);
			state = START;
		} else if (c == EOF) {
			errflag = 1;
			state = END;
		} else {
			tagbuf[tbi++] = c;
		}
		break;
	case INACHAR:
		if (c == ';') {
			tagbuf[tbi] = '\0';
			emitcchar(tagbuf);
			state = START;
		} else if (c == EOF) {
			errflag = 1;
			state = END;
		} else {
			tagbuf[tbi++] = c;
		}
		break;
	default:
		fprintf(stderr, "In html_char(): shouldn't be here!\n");
		errflag = 1;
		state = END;
		break;
	}
}

static int load(char *fn, buffer *b)
{
	FILE *fp;

	fp = fopen(fn, "r");
	if (!fp) return 1;

	state = START;
	errflag = 0;
	mute = 0;
	obi = 0;
	tbi = 0;
	pre = 0;
	row = 1;
	buf = b;
	fmt = styles[STY_DEFAULT].format;
	while (state != END)
		html_char(getc(fp));
	emitchar('\n');		/* flush if anything left */
	fclose(fp);
	return 0;
}


/* conservative file format guessing:
   1. extension .html or .htm
   2. contains the string "<HTML>"
*/
static int myformat(char *fn)
{
        char *ext;
        FILE *fp;
        char b[256];

        ext = strrchr(fn, '.');
        if (!ext) return 0;     /* no extension */
        if (cstrcasecmp(ext, ".html") && cstrcasecmp(ext, ".htm"))
                return 0;       /* wrong extension */
        if ((fp = fopen(fn, "r")) == NULL) return 0;    /* can't open */
        while (fgets(b, sizeof b, fp)) {
                if (strstr(b, "<html>") || strstr(b, "<HTML>")) {
                        fclose(fp);
                        return 1;
                }
        }
        fclose(fp);
        return 0;
}

void fileio_html_init()
{
	register_format(load, save, myformat,
			"Hypertext Markup Language (*.html)");
}

