/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

/* AIX requires this to be the first thing in the file.  */
#include <config.h>
#if __GNUC__
#define alloca __builtin_alloca
#else
#if HAVE_ALLOCA_H
#include <alloca.h>
#else
#ifdef _AIX
#pragma alloca
#else
#ifndef alloca			/* predefined by HP cc +Olibcalls */
char *alloca ();
#endif
#endif
#endif
#endif

#include <assert.h>
#include <ctype.h>
#include <settings.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include "common.h"
#include "error.h"
#include "getline.h"
#include "output.h"
#include "str.h"
#include "var.h"
#include "command.h"

#if HAVE_LIBHISTORY
#if HAVE_READLINE_HISTORY_H
#include <readline/history.h>
#else
int write_history (char *);
#endif
#endif

int error_count;
int warning_count;

int error_already_flagged;

const char *cust_fn;
int cust_ln;

int cust_fc;
int cust_lc;
fmt_spec cust_field;

int verbosity;

/* Set when hcf() is entered, prevents recursion. */
static int terminating;

void
push_cust (const char *fn)
{
  cust_fn = fn;
  cust_ln = 0;
}

void
pop_cust (void)
{
  cust_fn = NULL;
  cust_ln = 0;
}

/* Halt-catch-fire, returning EXIT_CODE to the operating system.
   Despite the name, this is the usual way to finish, successfully or
   not. */
void
hcf (int exit_code)
{
  terminating = 1;

#if HAVE_LIBHISTORY && unix
  if (history_file)
    write_history (history_file);
#endif
  outp_done ();

#if __CHECKER__
  if (exit_code != EXIT_SUCCESS)
    induce_segfault ();
#endif
  exit (exit_code);
}

void
puts_stderr (const char *s)
{
  fputs (s, stderr);
  fputc ('\n', stderr);
}

void
puts_stdout (const char *s)
{
  puts (s);
}

/* Writes a blank line to the error device(s).
   FIXME: currently a no-op. */
void
error_break (void)
{
}

/* Terminate due to fatal error in input. */
void
failure (void)
{
  static const char message[] = N_("Terminating NOW due to a fatal error!");

  fflush (stdout);
  fflush (stderr);
  fprintf (stderr, "%s: %s\n", pgmname, gettext (message));

  hcf (EXIT_FAILURE);
}

static void dump_message (char *errbuf, int indent,
			  void (*func) (const char *), int width);

void
vmsg (int class, const char *format, va_list args, const char *title)
{
  static int garbage;
  static int proc_category[MSG_CLASS_COUNT] =
  {0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0};
  static int *counting_category[MSG_CLASS_COUNT] =
  {
    &garbage,
    &error_count, &warning_count, &garbage,
    &error_count, &error_count,
    &error_count, &warning_count,
    &error_count, &warning_count, &garbage,
  };

  const char *fn;
  int ln;

  char errbuf[1024], *cp;

  fn = cust_fn ? cust_fn : curfn;
  ln = cust_fn ? cust_ln : curln;

  cp = errbuf;
  switch (class)
    {
    case FE:
      cp = stpcpy (cp, _("fatal: "));
      break;

    case DE:
    case SE:
      if (fn)
	cp = spprintf (cp, "%s:%d: ", fn, ln);
      else
	cp = stpcpy (cp, _("error: "));
      break;

    case DW:
    case SW:
      if (fn)
	cp = spprintf (cp, _("%s:%d: warning: "), fn, ln);
      else
	cp = stpcpy (cp, _("warning: "));
      break;

    case SM:
      if (fn)
	cp = spprintf (cp, _("%s:%d: message: "), fn, ln);
      else
	cp = stpcpy (cp, _("message: "));
      break;

    case IS:
      if (fn)
	{
	  cp = spprintf (cp, _("installation error: %s:%d: "), fn, ln);
	  break;
	}
      /* fall through */
    case IE:
      cp = stpcpy (cp, _("installation error: "));
      break;

    case ME:
      cp = stpcpy (cp, _("error: "));
      break;

    case MW:
      cp = stpcpy (cp, _("warning: "));
      break;

    case MM:
      cp = stpcpy (cp, _("note: "));
      break;

    default:
      assert (0);
    }

  (*counting_category[class])++;
  if (cur_proc && proc_category[class])
    cp = spprintf (cp, "%s: ", cur_proc);
  if (title)
    cp = stpcpy (cp, title);
  if (class == DE || class == DW)
    {
      if (cust_fc == cust_lc)
	cp = spprintf (cp, _("(column %d"), cust_fc);
      else
	cp = spprintf (cp, _("(columns %d-%d"), cust_fc, cust_lc);
      cp = spprintf (cp, _(", field type %s) "), fmt_to_string (&cust_field));
    }

  vsprintf (cp, format, args);

  /* FIXME: Check set_messages and set_errors to determine where to
     send errors and messages.

     Please note that this is not trivial.  We have to avoid an
     infinite loop in reporting errors that originate in the output
     section. */
  dump_message (errbuf, 8, puts_stdout, set_viewwidth);

  if (terminating)
    return;

  if (class == FE)
    hcf (EXIT_FAILURE);
}

/* Checks whether we've had so many errors that it's time to quit
   processing this syntax file.  If so, then take appropriate
   action. */
void
check_error_count (void)
{
  int error_class = getl_interactive ? MM : FE;

  if (set_errorbreak && error_count)
    msg (error_class, _("Terminating execution of syntax file due to error."));
  else if (error_count > set_mxerrs)
    msg (error_class, _("Errors (%d) exceeds limit (%d)."),
	 error_count, set_mxerrs);
  else if (error_count + warning_count > set_mxwarns)
    msg (error_class, _("Warnings (%d) exceed limit (%d)."),
	 error_count + warning_count, set_mxwarns);
  else
    return;

  getl_close_all ();
}

int
msg (int class, const char *format,...)
{
  va_list args;

  va_start (args, format);
  vmsg (class, format, args, NULL);
  va_end (args);

  return 0;
}

#if !__GNUC__ || __STRICT_ANSI__
/* Outputs message FORMAT in class MM only if verbosity is at level
   LEVEL or greater. */
void
verbose_msg (int level, const char *format,...)
{
  va_list args;

  if (verbosity < level)
    return;

  va_start (args, format);
  vmsg (MM, format, args, NULL);
  va_end (args);
}
#endif

/* dump_message(). */

/* Returns 1 if C is a `break character', that is, if it is a good
   place to break a message into lines. */
#define isbreak(C)							\
	((quote && (C) == DIR_SEPARATOR)				\
	 || (!quote && (isspace ((unsigned char) (C)) || (C)=='-'	\
			|| (C)=='/')))

/* Returns 1 if C is a break character where the break should be made
   BEFORE the character. */
#define isbefore(C) 				\
	(!quote && isspace ((unsigned char) (C)))

/* If C is a break character, returns 1 if the break should be made
   AFTER the character.  Does not return a meaningful result if C is
   not a break character. */
#define isafter(C) 				\
	(!isbefore (C))

/* If you want very long words that occur at a bad break point to be
   broken into two lines even if they're shorter than a whole line by
   themselves, define as 2/3, or 4/5, or whatever fraction of a whole
   line you think is necessary in order to consider a word long enough
   to break into pieces.  Otherwise, define as 0.  See code to grok
   the details.  Do NOT parenthesize the expression!  */
#if 1
#define BREAK_LONG_WORD 0
#else /* 0 */
#define BREAK_LONG_WORD 2/3
#endif /* 0 */

/* Divides ERRBUF, into lines of WIDTH width for the first line and
   WIDTH-INDENT width for each succeeding line.  Each line is dumped
   through FUNC, which may do with the string what it will.

   This is ugly.  It works.  */
static void
dump_message (char *errbuf, int indent, void (*func) (const char *), int width)
{
  int len, c;
  char *cp, *cp2;

  /* 1 when CP2 points to a position inside double quotes ("). */
  int quote = 0;

  /* Buffer for a single line. */
  char *buf = local_alloc (width + 1);

  if (indent > width / 3)
    indent = width / 3;
  len = strlen (errbuf);
  if (len < width)
    {
      func (errbuf);
      local_free (buf);
      return;
    }

  /* The first line has WIDTH width. */
  cp = errbuf;
  {
    int i;
    
    for (i = 0; i < width - 1; i++)
      quote ^= *cp++ == '"';
  }
  if (!isbefore (*cp))
    {
      while (!isbreak (*cp) && cp > errbuf)
	quote ^= *--cp == '"';
      if (isafter (*cp))
	cp++;
    }
  if (cp <= errbuf + width * BREAK_LONG_WORD)
    while (cp < errbuf + width - 1)
      quote ^= *cp++ == '"';
  c = *cp;
  *cp = 0;
  func (errbuf);
  *cp = c;
  while (isspace ((unsigned char) *cp))
    cp++;
  cp2 = cp;

  /* Succeeding lines have WIDTH-INDENT width. */
  while (*cp)
    {
      int w;
      
      {
	int i;
	
	for (i = 0; i < width - indent && *cp2; i++)
	  quote ^= *cp2++ == '"';
      }
      w = cp2 - cp;
      if (*cp2)
	while (!isbreak (*cp2) && cp2 > cp)
	  quote ^= *--cp2 == '"';
      if (w == width - indent
	  && cp2 <= cp + (width - indent) * BREAK_LONG_WORD)
	{
	  int i;
	  
	  for (i = 0; i < width - indent && *cp2; i++)
	    quote ^= *cp2++ = '"';
	}
      if (indent)
	{
	  memset (buf, ' ', indent);
	  strncpy (&buf[indent], cp, cp2 - cp);
	  buf[indent + cp2 - cp] = 0;
	  func (buf);
	}
      else
	{
	  c = *cp2;
	  *cp2 = 0;
	  func (cp);
	  *cp2 = c;
	}
      cp = cp2;
      while (isspace ((unsigned char) *cp))
	cp++;
    }
  local_free (buf);
}

#if __CHECKER__
/* Causes a segfault in order to force Checker to print a stack
   backtrace. */
void
induce_segfault (void)
{
  printf (_("\t*********************\n"
	  "\t* INDUCING SEGFAULT *\n"
	  "\t*********************\n"));
  fflush (stdout);
  fflush (stderr);
  abort ();
}
#endif

