/* $Id: list.hh,v 1.2 1998/09/07 19:33:04 cthulhu Exp $ */

#ifndef LIST_HH
#define LIST_HH

#define UINT unsigned int

template <class T> class Lista;

template <class ElemType>
class Node {
public:
  int IsFirst();
  /* Checks if current element is first in list */
  int IsLast();
  /* Checks if current element is last in list */
private:
  ElemType *element;
  Node *next;
  Node *prev;
  friend class Lista<ElemType>;
};

template <class ElemType>
class Lista {
private:
  UINT num_elements;
  Node<ElemType> *head;
  Node<ElemType> *tail;

public:
  Lista();
  /* Construct an empty list of void pointers.
     Returns new list. */
  
  ~Lista() { RemoveAll(); }
  /* Removes list and any elements left in it. 
     The elements are only removed from the list, not freed. */
  
  ElemType* GetHead();
  ElemType* GetTail();
  /* Returns first/last element of list. 
     If the list is empty the result is undefined. IsEmpty() is recomended. */

  ElemType* RemoveHead();
  ElemType* RemoveTail();
  /* Removes and returns first/last element from list.
     If the list is empty the result is undefined. IsEmpty() is recomended. */
  
  Node<ElemType>* AddHead(ElemType*);
  Node<ElemType>* AddTail(ElemType*);
  /* Adds an element to head/tail of list.
     Returns the position of the added element. 
     On error returns NULL. */

  void RemoveAll();
  /* Removes all elements from list. The elements are not freed. */

  Node<ElemType>* GetHeadPosition();
  Node<ElemType>* GetTailPosition();
  /* Returns the position of the first/last element of list. */

  ElemType* GetNext(Node<ElemType>**);
  ElemType* GetPrev(Node<ElemType>**);
  /* Returns the element of the given position and updates it
     to the next/previous element. The new position is NULL if
     the position given was the last/first element of list .*/

  ElemType* GetAt(Node<ElemType>*);
  /* Returns the element at given position. */

  ElemType* SetAt(Node<ElemType>*, ElemType*);
  /* Puts an element at the given position and
     Returns the element previous located at that position. */
  
  ElemType* RemoveAt(Node<ElemType>*);
  /* Removes the element at the given position and returns it. */

  Node<ElemType>* InsertBefore(Node<ElemType>*, ElemType*);
  Node<ElemType>* InsertAfter(Node<ElemType>*, ElemType*);
  /* Inserts an element before/after a given position.
     Returns the position of the new element. 
     On error returns NULL. */

  void MergeList (Lista<ElemType> list2);
  /* Merges this list with list2. list2 becomes invalid. */

  Node<ElemType>* Find(ElemType*);
  /* Returns the position of the element given. 
     If not found returns NULL. */

  Node<ElemType>* FindIndex(UINT);
  /* Returns the position of an element specified by a zero-based index. */

  UINT GetCount();
  /* Returns the number of elements in list. */

  int IsEmpty();
  /* Checks if the list is empty. */

  ElemType* GetAtIndex(UINT index) {
    GetAt (FindIndex(index));
  }
  
  Node<ElemType>* InsBefIndex(UINT index, ElemType* elem) {
    InsertBefore (FindIndex(index), elem);
  }

  Node<ElemType>* InsAftIndex(UINT index, ElemType* elem) {
    InsertAfter (FindIndex(index), elem);
  }
};

#endif

/* $Log: list.hh,v $
 * Revision 1.2  1998/09/07 19:33:04  cthulhu
 * Added forward declaration:
 * 	template <class T> class Lista;
 * seems people were having trouble without it with the egcs compiler.
 *
 * Revision 1.1  1998/08/17 20:23:26  cthulhu
 * Initial revision
 * */
