# folder.tcl --
#
# This file contains code which handles a folder window.
#
#
#  TkRat software and its included text is Copyright 1996,1997,1998
#  by Martin Forssn
#
#  The full text of the legal notice is contained in the file called
#  COPYRIGHT, included with this distribution.


# The list of folder windows
set folderWindowList {}

# FolderWindowInit --
#
# Initializes a folder window, that is it populates it with widgets and
# installs all callbacks. It reaturns the handler that should be used for
# this folder window.
#
# Arguments:
# w -		Window which the folder window should be packed into

proc FolderWindowInit w {
    global b t idCnt statusText option defaultFont folderWindowList \
	   valueFont ratDeferred

    # Create the handler
    set handler f[incr idCnt]
    lappend folderWindowList $handler
    upvar #0 $handler fh

    # Initialize variables
    set fh(toplevel) [winfo toplevel $w]
    set fh(w) $w
    set fh(folder_name) {}
    set fh(folder_size) {}
    set fh(folder_messages) {}
    set fh(num_messages) 0
    set fh(groupMessageLists) {}
    set fh(message_scroll) $w.messlist.scroll
    set fh(message_list) $w.messlist.list
    set fh(after_id) {}
    set fh(group) {}
    set fh(text) $w.text.text
    set fh(pgp_enable) $option(pgp_enable)
    set fh(find_ignore_case) 1
    set fh(find_match) exact
    set fh(find_loc) body
    set fh(browse) 0
    upvar #0 $fh(text) texth
    set texth(show_header) $option(show_header)
    set texth(struct_menu) $w.mbar.message.m.structmenu

    # The menu and information line
    frame $w.mbar -relief raised -bd 1
    FindAccelerators a {tkrat folders message group admin help}

    # Tkrat menu
    menubutton $w.mbar.tkrat -menu $w.mbar.tkrat.m -text $t(tkrat) \
			     -underline $a(tkrat)
    set m $w.mbar.tkrat.m
    menu $m
    $m add command -label $t(send_deferred) -command SendDeferred \
	    -state disabled
    trace variable ratDeferred w "TraceDeferred $m [$m index end]"
    set b($m,[$m index end]) send_deferred
    $m add command -label $t(notifications)... -command ShowDSNList
    set b($m,[$m index end]) show_notifications
    $m add checkbutton -label $t(watcher) \
	    -variable option(watcher_enable) -onvalue 1 -offvalue 0 \
	    -command SaveOptions
    set b($m,[$m index end]) watcher_enable
    $m add separator
    $m add command -label $t(reread_aliases) -command AliasRead
    set b($m,[$m index end]) reread_aliases
    $m add command -label $t(reread_userproc) -command ReadUserproc
    set b($m,[$m index end]) reread_userproc
    $m add command -label $t(import_aliases) -command ScanAliases
    set b($m,[$m index end]) import_aliases
    $m add command -label $t(reread_mailcap) -command RatMailcapReload
    set b($m,[$m index end]) reread_mailcap
    $m add command -label $t(take_mail)... -command "MailSteal $handler 0"
    set b($m,[$m index end]) take_mail
    $m add separator
    $m add command -label $t(check_dbase)... -command "DbaseCheck 0"
    set b($m,[$m index end]) check_dbase
    $m add command -label $t(check_fix_dbase)...  -command "DbaseCheck 1"
    set b($m,[$m index end]) check_fix_dbase
    $m add separator
    $m add command -label $t(see_log)... -command SeeLog
    set b($m,[$m index end]) see_log
    $m add separator
    $m add command -label $t(quit) -command "Quit $handler"
    set fh(quit_menu) [list $m [$m index end]]
    set b($m,[$m index end]) quit
    
    # Folder menu
    menubutton $w.mbar.folder -menu $w.mbar.folder.m -text $t(folders) \
			      -underline $a(folders)
    menu $w.mbar.folder.m -postcommand "PostFolder $handler $w.mbar.folder.m"
    set b($w.mbar.folder) folder_menu

    # Message menu
    menubutton $w.mbar.message -menu $w.mbar.message.m -text $t(message) \
			       -underline $a(message)
    set b($w.mbar.message) message_menu
    set m $w.mbar.message.m
    menu $m
    set fh(message_menu_nokeep) {}
    $m add command -label $t(find)... -command "FolderFind $handler"
    set b($m,[$m index end]) find
    set fh(find_menu) [list $m [$m index end]]
    $m add separator
    $m add command -label $t(compose)... -command Compose
    set fh(compose_menu) [list $m [$m index end]]
    set b($m,[$m index end]) compose
    $m add command -label $t(reply_sender)... \
	    -command "FolderReply $handler sender"
    lappend fh(message_menu_nokeep) [$m index end]
    set b($m,[$m index end]) reply_sender
    set fh(replys_menu) [list $m [$m index end]]
    $m add command -label $t(reply_all)... -command "FolderReply $handler all"
    lappend fh(message_menu_nokeep) [$m index end]
    set b($m,[$m index end]) reply_all
    set fh(replya_menu) [list $m [$m index end]]
    $m add command -label $t(forward_inline)... \
	    -command "FolderSomeCompse $handler ComposeForwardInline"
    lappend fh(message_menu_nokeep) [$m index end]
    set b($m,[$m index end]) forward_inline
    set fh(forward_i_menu) [list $m [$m index end]]
    $m add command -label $t(forward_as_attachment)... \
	    -command "FolderSomeCompse $handler ComposeForwardAttachment"
    lappend fh(message_menu_nokeep) [$m index end]
    set b($m,[$m index end]) forward_attached
    set fh(forward_a_menu) [list $m [$m index end]]
    $m add command -label $t(getheld)... -command ComposeHeld
    set b($m,[$m index end]) compose_held
    $m add command -label $t(extract_adr)... -command "AliasExtract $handler"
    lappend fh(message_menu_nokeep) [$m index end]
    set b($m,[$m index end]) extract_adr
    $m add separator
    $m add cascade -label $t(move) -menu $m.move
    lappend fh(message_menu_nokeep) [$m index end]
    set b($m,[$m index end]) move
    menu $m.move -postcommand "PostMove $handler current $m.move"
    $m add command -label $t(delete) \
	    -command "SetFlag $handler deleted 1; FolderNext $handler"
    set fh(delete_menu) [list $m [$m index end]]
    lappend fh(message_menu_nokeep) [$m index end]
    set b($m,[$m index end]) delete
    $m add command -label $t(undelete) \
	    -command "SetFlag $handler deleted 0; FolderNext $handler"
    set fh(undelete_menu) [list $m [$m index end]]
    lappend fh(message_menu_nokeep) [$m index end]
    set b($m,[$m index end]) undelete
    $m add command -label $t(print) -command "Print $handler current"
    lappend fh(message_menu_nokeep) [$m index end]
    set b($m,[$m index end]) print
    $m add cascade -label $t(structure) -menu $m.structmenu
    lappend fh(message_menu_nokeep) [$m index end]
    set b($m,[$m index end]) structure
    $m add separator
    $m add radiobutton -label $t(show_all_headers) \
	    -variable $fh(text)(show_header) -value all \
	    -command "FolderSelect $fh(message_list) $handler \
		      \$${handler}(active) 0; SaveOptions"
    set b($m,[$m index end]) show_all_headers
    $m add radiobutton -label $t(show_selected_headers) \
	    -variable $fh(text)(show_header) -value selected \
	    -command "FolderSelect $fh(message_list) $handler \
		      \$${handler}(active) 0; SaveOptions"
    set b($m,[$m index end]) show_selected_headers
    $m add radiobutton -label $t(show_no_headers) \
	    -variable $fh(text)(show_header) -value no \
	    -command "FolderSelect $fh(message_list) $handler \
		      \$${handler}(active) 0; SaveOptions"
    set b($m,[$m index end]) show_no_headers

    # Group menu
    menubutton $w.mbar.group -menu $w.mbar.group.m -text $t(group) \
			      -underline $a(group)
    set b($w.mbar.group) group_menu
    set m $w.mbar.group.m
    menu $m -postcommand "SetupGroupMenu $m $handler"
    $m add command -label $t(create_in_win)... \
	    -command "GroupMessageList $handler"
    set b($m,[$m index end]) create_in_win
    $m add command -label $t(create_by_expr)... -command "ExpCreate $handler"
    set b($m,[$m index end]) create_by_expr
    $m add cascade -label $t(use_saved_expr) -menu $m.saved
    set b($m,[$m index end]) use_saved_expr
    menu $m.saved -postcommand "ExpBuildMenu $m.saved $handler"
    $m add command -label $t(clear_group) -command "GroupClear $handler"
    set b($m,[$m index end]) clear_group
    $m add separator
    $m add command -label $t(delete) -command "SetFlag \
	    $handler deleted 1 \[\$${handler}(folder_handler) flagged flagged\]"
    set b($m,[$m index end]) delete_group
    $m add command -label $t(undelete) -command "SetFlag \
	    $handler deleted 0 \[\$${handler}(folder_handler) flagged flagged\]"
    set b($m,[$m index end]) undelete_group
    $m add command -label $t(print) -command "Print $handler group"
    set b($m,[$m index end]) print_group
    $m add cascade -label $t(move) -menu $m.move
    set b($m,[$m index end]) move_group
    menu $m.move -postcommand "PostMove $handler group $m.move"

    # Admin menu
    menubutton $w.mbar.admin -menu $w.mbar.admin.m -text $t(admin) \
			     -underline $a(admin)
    set b($w.mbar.admin) admin_menu
    set m $w.mbar.admin.m
    menu $m
    $m add checkbutton -label $t(browse_mode) -variable ${handler}(browse)
    set b($m,[$m index end]) browse_mode
    $m add command -label $t(update_folder) -command "Sync $handler 0"
    set fh(update_menu) [list $m [$m index end]]
    set b($m,[$m index end]) update
    $m add command -label $t(sync_folder) -command "Sync $handler 1"
    set fh(sync_menu) [list $m [$m index end]]
    set b($m,[$m index end]) sync
    $m add separator
    $m add command -label $t(newedit_folder)... -command VFolderDef
    set b($m,[$m index end]) newedit_folder
    $m add command -label $t(aliases)... -command Aliases
    set b($m,[$m index end]) aliases
    $m add command -label $t(preferences)... -command Preferences
    set b($m,[$m index end]) preferences
    $m add command -label $t(define_keys)... -command "KeyDef folder"
    set b($m,[$m index end]) define_keys
    $m add command -label $t(saved_expr)... -command ExpHandleSaved
    set b($m,[$m index end]) saved_expr
    $m add separator
    $m add cascade -label $t(sort_order) -menu $m.sort
    set b($m,[$m index end]) sort_order
    menu $m.sort
    $m.sort add radiobutton -label $t(subject_by_date) \
	    -variable option(folder_sort) -value subject \
	    -command {foreach f $folderWindowList {Sync $f 0}; SaveOptions}
    set b($m.sort,[$m.sort index end]) sort_subjdate
    $m.sort add radiobutton -label $t(subjectonly) \
	    -variable option(folder_sort) -value subjectonly \
	    -command {foreach f $folderWindowList {Sync $f 0}; SaveOptions}
    set b($m.sort,[$m.sort index end]) sort_subj
    $m.sort add radiobutton -label $t(bysender) \
	    -variable option(folder_sort) -value sender \
	    -command {foreach f $folderWindowList {Sync $f 0}; SaveOptions}
    set b($m.sort,[$m.sort index end]) sort_sender
    $m.sort add radiobutton -label $t(natural_order) \
	    -variable option(folder_sort) -value folder \
	    -command {foreach f $folderWindowList {Sync $f 0}; SaveOptions}
    set b($m.sort,[$m.sort index end]) sort_natural
    $m.sort add radiobutton -label $t(reverse_natural_order) \
	    -variable option(folder_sort) -value reverseFolder \
	    -command {foreach f $folderWindowList {Sync $f 0}; SaveOptions}
    set b($m.sort,[$m.sort index end]) sort_reverse_natural
    $m.sort add radiobutton -label $t(by_date) \
	    -variable option(folder_sort) -value date \
	    -command {foreach f $folderWindowList {Sync $f 0}; SaveOptions}
    set b($m.sort,[$m.sort index end]) sort_date
    $m.sort add radiobutton -label $t(reverse_date) \
	    -variable option(folder_sort) -value reverseDate \
	    -command {foreach f $folderWindowList {Sync $f 0}; SaveOptions}
    set b($m.sort,[$m.sort index end]) sort_reversedate

    # Help menu
    menubutton $w.mbar.help -menu $w.mbar.help.m -text $t(help) \
			    -underline $a(help)
    set b($w.mbar.help) help_menu
    set m $w.mbar.help.m
    menu $m
    $m add checkbutton -label $t(balloon_help) \
	    -variable option(show_balhelp) \
	    -command {SaveOptions}
    set b($m,[$m index end]) balloon_help
    $m add separator
    $m add command -label $t(version)... -command Version
    set b($m,[$m index end]) show_version
    $m add command -label Ratatosk... -command Ratatosk
    set b($m,[$m index end]) explain_ratatosk
    $m add command -label $t(help_window)... -command Help
    set b($m,[$m index end]) help_window
    $m add command -label $t(send_bug)... -command SendBugReport
    set b($m,[$m index end]) send_bug

    # The structure menu (constructed by the Show routine)
    menu $texth(struct_menu) -tearoff 0

    # Pack the menus into the menu bar
    pack $w.mbar.tkrat \
	 $w.mbar.folder \
	 $w.mbar.message \
	 $w.mbar.group \
	 $w.mbar.admin -side left -padx 5
    pack $w.mbar.help -side right -padx 5

    # The information part
    frame $w.info -relief raised -bd 1
    label $w.info.flabel -text $t(name):
    label $w.info.fname -textvariable ${handler}(folder_name) \
        -font $valueFont -anchor w
    label $w.info.slabel -text $t(size):
    label $w.info.size -textvariable ${handler}(folder_size) -width 5 \
	-anchor w -font $valueFont
    label $w.info.mlabel -text $t(messages):
    label $w.info.messages -textvariable ${handler}(folder_messages) -width 4 \
        -anchor w -font $valueFont
    pack $w.info.messages \
         $w.info.mlabel \
         $w.info.size \
         $w.info.slabel -side right
    pack $w.info.flabel -side left
    pack $w.info.fname -side left -fill x -expand 1
    set b($w.info.fname) current_folder_name
    set b($w.info.size) current_folder_size
    set b($w.info.messages) current_folder_nummsg

    # The message list
    frame $w.messlist
    scrollbar $fh(message_scroll) \
        -relief sunken \
        -command "$w.messlist.list yview" \
	-highlightthickness 0
    text $fh(message_list) \
        -yscroll "$fh(message_scroll) set" \
        -bd 0 \
        -font $defaultFont \
	-highlightthickness 0 \
	-wrap none \
	-spacing1 1 \
	-spacing3 2 \
	-height $option(message_llength) \
	-cursor {}
    $fh(message_list) tag configure Active \
	    -relief raised \
	    -borderwidth [$fh(message_list) cget -selectborderwidth] \
	    -foreground [$fh(message_list) cget -selectforeground] \
	    -background [$fh(message_list) cget -selectbackground]
    if { 4 < [winfo cells $fh(message_list)]} {
	$fh(message_list) tag configure sel -background #ffff80
	$fh(message_list) tag configure Found -background #ffff80
    } else {
	$fh(message_list) tag configure sel -underline 1
	$fh(message_list) tag configure Found -borderwidth 2 -relief raised
    }
    $fh(message_list) tag raise sel
    pack $fh(message_scroll) -side right -fill y
    pack $fh(message_list) -side left -expand 1 -fill both
    set b($fh(message_list)) list_of_messages

    # The status line
    label $w.statustext -textvariable statusText -relief raised \
	    -bd 1 -font $valueFont -width 80
    set b($w.statustext) status_text
    
    # The command buttons
    frame $w.buttons
    menubutton $w.buttons.move -text $t(move) -bd 1 -relief raised \
	    -menu $w.buttons.move.m -indicatoron 1
    menu $w.buttons.move.m \
	    -postcommand "PostMove $handler current $w.buttons.move.m"
    button $w.buttons.delete -text $t(delete) -bd 1 -highlightthickness 0 \
	    -command "SetFlag $handler deleted 1; FolderNext $handler"
    button $w.buttons.compose -text $t(compose)... -bd 1 -highlightthickness 0 \
	    -command Compose
    button $w.buttons.reply_sender -text $t(reply_sender)... -bd 1 \
	    -highlightthickness 0 -command "FolderReply $handler sender"
    button $w.buttons.reply_all -text $t(reply_all)... -bd 1 \
	    -highlightthickness 0 -command "FolderReply $handler all"
    pack $w.buttons.move \
	 $w.buttons.delete \
	 $w.buttons.compose \
	 $w.buttons.reply_sender \
	 $w.buttons.reply_all -side left -expand 1 -fill x
    if $option(pgp_enable) {
	set fh(sigbut) $w.buttons.signature
	button $fh(sigbut) -text $t(sig): -bd 1 -anchor w -state disabled \
		-highlightthickness 0 -width [expr [string length $t(sig)]+5]
	pack $fh(sigbut) -side right -expand 1 -fill x
	set b($fh(sigbut)) no_sig
    }
    set b($w.buttons.move) move_msg
    set b($w.buttons.delete) delete_msg
    set b($w.buttons.compose) compose_msg
    set b($w.buttons.reply_sender) reply_to_sender
    set b($w.buttons.reply_all) reply_to_all

    # The actual text
    frame $w.text -relief raised -bd 1
    scrollbar $w.text.scroll \
        -relief sunken \
        -command "$w.text.text yview" \
	-highlightthickness 0
    text $fh(text) \
	-yscroll "RatScrollShow $w.text.text $w.text.scroll" \
        -setgrid true \
        -relief raised \
        -wrap char \
        -font $defaultFont \
	-bd 0 \
	-highlightthickness 0
    $fh(text) mark set searched 1.0
    Size $w.text.text folderM
    pack $w.text.scroll -side right -fill y
    pack $w.text.text -side left -expand yes -fill both
    set b($fh(text)) body_of_message

    # Pack all the parts into the window
    pack $w.mbar -side top -fill x
    pack $w.info -side top -fill x
    pack $w.messlist -side top -fill both
    pack $w.statustext -side top -fill x -anchor center
    pack $w.buttons -side top -fill x
    pack $w.text -side top -expand yes -fill both

    # Do bindings
    focus $w
    bind $fh(message_list) <1> \
	    "FolderSelect $w.messlist.list $handler \
		    \[expr int(\[%W index @%x,%y\])-1\] 0"
    bind $fh(message_list) <Double-1> \
	    "FolderSelect $w.messlist.list $handler \
		    \[expr int(\[%W index @%x,%y\])-1\] 1; break"
    bind $fh(message_list) <Triple-1> break
    bind $fh(message_list) <B1-Motion> break
    bind $fh(message_list) <Shift-1> break
    bind $fh(message_list) <Double-Shift-1> break
    bind $fh(message_list) <Triple-Shift-1> break
    bind $fh(message_list) <B1-Leave> break
    bind $fh(message_list) <ButtonPress-3> \
	    "SetFlag $handler flagged toggle \
		    \[expr int(\[%W index @%x,%y\])-1\]"
    bind $fh(w) <Map>	"WatcherSleep $handler"
    FolderBind $handler
    wm protocol $fh(toplevel) WM_DELETE_WINDOW "Quit $handler"

    return $handler
}


# FolderBind --
#
# Bind the key definitions for a folder window
#
# Arguments:
# handler - The handler which identifies the fodler window

proc FolderBind {handler} {
    upvar #0 $handler fh

    RatBind $fh(w) folder_key_compose   "Compose" $fh(compose_menu)
    RatBind $fh(w) folder_key_quit	"Quit $handler" $fh(quit_menu)
    RatBind $fh(w) folder_key_nextu	"FolderSelectUnread $handler; break"
    RatBind $fh(w) folder_key_sync	"Sync $handler 1" $fh(sync_menu)
    RatBind $fh(w) folder_key_update    "Sync $handler 0" $fh(update_menu)
    RatBind $fh(w) folder_key_delete    "SetFlag $handler deleted 1; \
	FolderNext $handler" $fh(delete_menu)
    RatBind $fh(w) folder_key_undelete  "SetFlag $handler deleted 0; \
	FolderNext $handler" $fh(undelete_menu)
    RatBind $fh(w) folder_key_flag	"SetFlag $handler flagged toggle;
	FolderNext $handler"
    RatBind $fh(w) folder_key_next	"FolderNext $handler"
    RatBind $fh(w) folder_key_prev	"FolderPrev $handler"
    RatBind $fh(w) folder_key_home	"ShowHome $fh(text)"
    RatBind $fh(w) folder_key_bottom	"ShowBottom $fh(text)"
    RatBind $fh(w) folder_key_pagedown  "ShowPageDown $fh(text)"
    RatBind $fh(w) folder_key_pageup    "ShowPageUp $fh(text)"
    RatBind $fh(w) folder_key_linedown  "ShowLineDown $fh(text)"
    RatBind $fh(w) folder_key_lineup    "ShowLineUp $fh(text)"
    RatBind $fh(w) folder_key_replya    "FolderReply $handler all" \
	$fh(replya_menu)
    RatBind $fh(w) folder_key_replys    "FolderReply $handler sender" \
	$fh(replys_menu)
    RatBind $fh(w) folder_key_forward_i \
	"FolderSomeCompse $handler ComposeForwardInline" $fh(forward_i_menu)
    RatBind $fh(w) folder_key_forward_a \
	"FolderSomeCompse $handler ComposeForwardAttachment" $fh(forward_a_menu)
    RatBind $fh(w) folder_key_cycle_header "CycleShowHeader $handler"
    RatBind $fh(w) folder_key_find 	"FolderFind $handler" $fh(find_menu)
}


# FolderRead --
#
# Reads and the given folder and calls draw_folder_list to show the content.
# If there already is an active folder it is closed. As arguments it expects
# a folderwindow handler and a command which when run opens the folder.
#
# Arguments:
# handler   -	The handler which identifies the folder window
# foldercmd -	Command which creates the new folder

proc FolderRead {handler foldercmd} {
    global option inbox t
    upvar #0 $handler fh

    # First we expunge the old folder
    if [info exists fh(folder_handler)] {
	$fh(folder_handler) close
	if ![string compare $inbox $fh(folder_handler)] {
	    set inbox ""
	}
	unset fh(folder_handler)
    }

    # Open the new folder
    set id [RatLog 2 $t(opening_folder)... explicit]
    if [catch $foldercmd folder_handler] {
	set fh(folder_name) ""
	set fh(folder_messages) ""
	set fh(folder_size) ""
	ShowNothing $fh(text)
	catch {unset fh(current); unset fh(folder_handler)}
	$fh(message_list) configure -state normal
	$fh(message_list) delete 1.0 end
	$fh(message_list) configure -state disabled
	set fh(active) ""
	RatClearLog $id
	FolderButtons $handler 0
	return
    }
    RatClearLog $id

    # Update our information
    set fh(folder_handler) $folder_handler
    set i [$fh(folder_handler) info]
    set fh(folder_name) [lindex $i 0]
    set fh(folder_messages) [RatMangleNumber [lindex $i 1]]
    set fh(num_messages) [lindex $i 1]
    FolderDrawList $handler
    set i [$fh(folder_handler) info]
    if { -1 == [lindex $i 2]} {
	set fh(folder_size) ???
    } else {
	set fh(folder_size) [RatMangleNumber [lindex $i 2]]
    }
    switch $option(folder_sort) {
    reverseFolder {
	    set dir -1
	    set start 0
	}
    reverseDate {
	    set dir -1
	    set start 0
	}
    default {
	    set dir 1
	    set start [expr $fh(size)-1]
	}
    }
    if { 0 != $fh(size)} {
	switch $option(start_selection) {
	last {
		set index [expr $fh(size)-1]
	    }
	first_new {
		set index [FolderGetNextUnread $handler $start $dir]
	    }
	before_new {
		set index [FolderGetNextUnread $handler $start $dir]
		if { 0 == [$fh(folder_handler) getFlag $index seen]} {
		    incr index -$dir
		    if {$index < 0 || $index >= $fh(size)} {
			incr index $dir
		    }
		}
	    }
	default {	# And first
		set index 0
	    }
	}
    } else {
	set index ""
    }
    FolderSelect $fh(message_list) $handler $index 0

    # Initialize watcher
    if [string length $fh(after_id)] {
	after cancel $fh(after_id)
	set fh(after_id) {}
    }
    set type [$folder_handler type]
    foreach timePair $option(watcher_time) {
	if ![string compare [lindex $timePair 0] $type] {
	    set time [expr 1000*[lindex $timePair 1]]
	    if $time {
		set fh(after_id) [after $time WatcherDoRound $handler $type]
	    }
	}
    }
    return $folder_handler
}

# FolderDrawList --
#
# Constructs the list of messages in the folder and shows this.
#
# Arguments:
# handler -	The handler which identifies the folder window

proc FolderDrawList {handler} {
    upvar #0 $handler fh
    global option

    $fh(message_list) configure -state normal
    $fh(message_list) delete 1.0 end
    set lines [$fh(folder_handler) list $option(list_format)]
    set fh(size) [llength $lines]
    foreach l $lines {
	$fh(message_list) insert end "$l\n"
    }
    $fh(message_list) delete end-1c
    foreach w $fh(groupMessageLists) {
	GroupMessageListUpdate $w $handler
    }
    $fh(message_list) see end
    $fh(message_list) configure -state disabled
}

# FolderListRefreshEntry --
#
# Refresh an entry in the message list
#
# Arguments:
# handler -	The handler which identifies the folder window
# index   -	Index of the entry to refresh

proc FolderListRefreshEntry {handler index} {
    upvar #0 $handler fh
    global option

    set line [expr $index+1]
    $fh(message_list) configure -state normal
    set tags [$fh(message_list) tag names $line.0]
    $fh(message_list) delete $line.0 "$line.0 lineend-1c"
    $fh(message_list) insert $line.0 [format %-128s [lindex \
	    [[$fh(folder_handler) get $index] list $option(list_format)] 0]] \
	    $tags
    $fh(message_list) configure -state disabled
}

# FolderSelect --
#
# Handle the selection of a message
#
# Arguments:
# l       -	The listbox the message was selected from
# handler -	The handler which identifies the folder window
# index   -	Index to select
# force   -	Force showing regarding of browse mode

proc FolderSelect {l handler index force} {
    upvar #0 $handler fh
    global option t b

    set fh(active) $index
    if {0 == [string length $index] || $index >= $fh(num_messages)} {
	catch {unset fh(current)}
	FolderButtons $handler 0
	ShowNothing $fh(text)
	return
    }
    if ![info exists fh(current)] {
	FolderButtons $handler 1
    }
    set line [expr $index+1]
    $fh(message_list) tag remove Active 1.0 end
    $fh(message_list) tag add Active $line.0 "$line.0 lineend+1c"
    $fh(message_list) see $line.0
    update idletasks
    set fh(current) [$fh(folder_handler) get $index]
    set seen [$fh(folder_handler) getFlag $index seen]
    $fh(folder_handler) setFlag $index seen 1
    if $force {
	set mode 0
    } else {
	set mode $fh(browse)
    }
    set result [Show $fh(text) $fh(current) $mode]
    set sigstatus [lindex $result 0]
    set pgpOutput [lindex $result 1]
    if { 0 == $seen } {
	FolderListRefreshEntry $handler $index
    }

    if $fh(pgp_enable) {
	set state normal
	set command {}
	switch $sigstatus {
	pgp_none {
		set state disabled
		set b($fh(sigbut)) no_sig
	    }
	pgp_unchecked {
		set command "FolderCheckSignature $handler"
		set b($fh(sigbut)) unchecked_sig
	    }
	pgp_good {
		set command [list RatText $t(pgp_output) $pgpOutput]
		set b($fh(sigbut)) good_sig
	    }
	pgp_bad {
		set command [list RatText $t(pgp_output) $pgpOutput]
		set b($fh(sigbut)) bad_sig
	    }
	}
	$fh(sigbut) configure -state $state \
			      -text "$t(sig): $t($sigstatus)" \
			      -command $command
    }
}

# FolderNext --
#
# Advance to the next message in the folder
#
# Arguments:
# handler -	The handler which identifies the folder window

proc FolderNext {handler} {
    upvar #0 $handler fh
    set ml $fh(message_list)

    if ![string length $fh(active)] { return }

    set index [expr 1+$fh(active)]
    if { $index >= $fh(size) } {
	return
    }

    if {$index >= [expr round([lindex [$ml yview] 1]*$fh(size))]} {
	$ml yview $index
    }
    FolderSelect $ml $handler $index 0
}

# FolderPrev --
#
# Retreat to the previous message in the folder
#
# Arguments:
# handler -	The handler which identifies the folder window

proc FolderPrev {handler} {
    upvar #0 $handler fh
    set ml $fh(message_list)

    if ![string length $fh(active)] { return }

    set index [expr $fh(active)-1]
    if {$index < 0} {
	return
    }

    if {$index < [expr round([lindex [$ml yview] 0]*$fh(size))]} {
	$ml yview scroll -1 pages
    }
    FolderSelect $ml $handler $index 0
}

# SetFlag --
#
# Set a flag to a specified value for the current message or a set of
# messages given as argument.
#
# Arguments:
# handler  -	The handler which identifies the folder window
# flag	   -	The flag to set
# value    -	The new value of the deletion flag ('1' or '0')
# messages -	Ano optional list of messages to do this to

proc SetFlag {handler flag value {messages {}}} {
    upvar #0 $handler fh
    global option

    if ![string length $fh(active)] { return }

    set sel $fh(active)
    if ![string length $messages] {
	set messages $sel
    }
    foreach i $messages {
	if [string compare toggle $value] {
	    $fh(folder_handler) setFlag $i $flag $value
	} else {
	    if [$fh(folder_handler) getFlag $i $flag] {
		set v 0 
	    } else {
		set v 1
	    }
	    $fh(folder_handler) setFlag $i $flag $v
	}
	FolderListRefreshEntry $handler $i
    }
}

# Quit --
#
# Closes the given folder window
#
# Arguments:
# handler -	The handler which identifies the folder window

proc Quit {handler} {
    global expAfter logAfter ratDeferred ratSenderSending t composeWindowList
    upvar #0 $handler fh

    if [info exists composeWindowList] {
	if {0 < [llength $composeWindowList]} {
	    if {0 == [RatDialog $t(really_quit) $t(compose_sessions) \
		    {} 0 $t(dont_quit) $t(do_quit)]} {
		return
	    }
	}
    }
    if {0 < $ratDeferred} {
	if {1 < $ratDeferred} {
	    set text "$t(you_have) $ratDeferred $t(deferred_messages)"
	} else {
	    set text "$t(you_have) 1 $t(deferred_message)"
	}
	if {0 == [RatDialog $t(send_deferred) $text {} 0 \
		$t(send_now) $t(no)]} {
	    tkwait window [SendDeferred]
	}
    }
    if $ratSenderSending {
	RatLog 2 $t(waiting_on_sender) explicit
	while {1 == $ratSenderSending} {
	    tkwait variable ratSenderSending
	}
	RatLog 2 "" explicit
    }
    if [string length $fh(after_id)] {
	after cancel $fh(after_id)
    }
    if [string length $expAfter] {
	after cancel $expAfter
    }
    if [string length $logAfter] {
	after cancel $logAfter
    }
    if [info exists fh(folder_handler)] {
	$fh(folder_handler) close
    }
    RecordPos [winfo toplevel $fh(w)] folder
    RecordSize $fh(text) folderM
    if [info exists fh(watcher_w)] {
	if [info exists fh(watcher_geom)] {
	    wm geom $fh(watcher_w) $fh(watcher_geom)
	    RecordPos $fh(watcher_w) watcher
	}
	RecordSize $fh(watcher_list) watcher
    }
    SavePos
    unset fh
    destroy .
}

# Sync --
#
# Does an expunge or update on the current folder.
#
# Arguments:
# handler   -	The handler which identifies the folder window
# expunge   -	Boolean value which indicates if we should expunge or not

proc Sync {handler expunge} {
    upvar #0 $handler fh
    global option

    if ![info exists fh(folder_handler)] { return }

    set oldActive $fh(active)
    set listBefore [$fh(folder_handler) list $option(msgfind_format)]
    if [string length $oldActive] {
	set subject [lindex $listBefore $oldActive]
	set topBefore [expr round([lindex [$fh(message_list) yview] 0]*\
			        $fh(size))]
	set msg $fh(current)
    }
    set topBefore [lindex [$fh(message_list) yview] 0]

    $fh(folder_handler) update $expunge
    FolderDrawList $handler

    # Update information
    set i [$fh(folder_handler) info]
    set fh(folder_messages) [RatMangleNumber [lindex $i 1]]
    set fh(num_messages) [lindex $i 1]
    if { -1 == [lindex $i 2]} {
	set fh(folder_size) ???
    } else {
	set fh(folder_size) [RatMangleNumber [lindex $i 2]]
    }
    if { 0 == [lindex $i 1] } {
	FolderSelect $fh(message_list) $handler "" 0
	return
    }

    # Check if our element is still in there
    set fh(active) ""
    if [string length $oldActive] {
	set index [$fh(folder_handler) find $msg]
	if { -1 != $index } {
	    set line [expr $index+1]
	    $fh(message_list) tag add Active $line.0 "$line.0 lineend+1c"
	    $fh(message_list) see $line.0
	    set fh(active) $index
	} else {
	    # Search for adjacent elements
	    set listAfter [$fh(folder_handler) list $option(msgfind_format)]
	    set list [lrange $listBefore [expr $oldActive+1] end]
	    for {set i $oldActive} {$i >= 0} {incr i -1} {
		lappend list [lindex $listBefore $i]
	    }
	    foreach element $list {
		if {-1!=[set index [lsearch -exact $listAfter $element]]} {
		    FolderSelect $fh(message_list) $handler $index 0
		    break
		}
	    }
	}
    }
    if ![string length $fh(active)] {
	FolderSelect $fh(message_list) $handler 0 0
    }
}

# FolderReply --
#
# Construct a reply to a message and update the messages status if the
# reply was sent.
#
# Arguments:
# handler   -	The handler which identifies the folder window
# recipient -	Who the reply should be sent to 'sender' or 'all'

proc FolderReply {handler recipient} {
    upvar #0 $handler fh

    if ![string length $fh(active)] { return }

    set current $fh(current)
    set hd [ComposeReply $current $recipient]
    upvar #0 $hd sendHandler
    trace variable sendHandler w "FolderReplySent $hd $handler $current"
}
proc FolderReplySent {hd handler current name1 name2 op} {
    upvar #0 $handler fh
    upvar #0 $hd sendHandler
    global option

    trace vdelete $name1 $op "FolderReplySent $hd $handler $current"

    if [info exists sendHandler(do)] {
	if ![string compare send $sendHandler(do)] {
	    set index [$fh(folder_handler) find $current]
	    if { -1 != $index } {
		if ![$fh(folder_handler) getFlag $index answered] {
		    SetFlag $handler answered 1 $index
		}
	    }
	}
    }
}

# FolderSomeCompse --
#
# Run a compose function on a message and update the message status if the
# message actually was sent.
#
# Arguments:
# handler   -	The handler which identifies the folder window

proc FolderSomeCompse {handler composeFunc} {
    upvar #0 $handler fh
    global option

    if ![string length $fh(active)] { return }

    $composeFunc $fh(current)
}

# PostFolder --
#
# Populate the folder menu
#
# Arguments:
# handler -	The handler which identifies the folder window
# m       -	The menu which we should populate

proc PostFolder {handler m} {
    global t option

    $m delete 1 end
    VFolderBuildMenu $m 0 "VFolderOpen $handler" 0
    $m add separator
    $m add command -label $t(open_file)... -command "SelectFileFolder $handler"
    $m add command -label $t(open_dbase)... \
	    -command "SelectDbasefolder $handler"
    FixMenu $m
}

# PostMove --
#
# Populate the move menu
#
# Arguments:
# handler -	The handler which identifies the folder window
# which	  -	Which set of messages we should move (current or group)
# m       -	The menu which we should populate

proc PostMove {handler which m} {
    upvar #0 $handler fh
    global t

    if [string compare "group" $which] {
	set a 1
    } else {
	set a 0
    }
    $m delete 1 end
    VFolderBuildMenu $m 0 \
	    "VFolderInsert $handler $a \[GetMsgSet $handler $which\]" 1
    $m add separator
    $m add command -label $t(to_file)... \
	    -command "VFolderInsert $handler $a \[GetMsgSet $handler $which\] \
		      \[InsertIntoFile\]"
    $m add command -label $t(to_dbase)... \
	    -command "VFolderInsert $handler $a \[GetMsgSet $handler $which\] \
		      \[InsertIntoDBase\]"
    FixMenu $m
}

# GetMsgSet --
#
# Get the messages that the current operation should be performed on
#
# Arguments:
# handler -	The handler which identifies the folder window
# which	  -	Which set of messages we should move (current or group)

proc GetMsgSet {handler which} {
    upvar #0 $handler fh

    if [string compare group $which] {
	return $fh(current)
    } else {
	set msgs {}
	foreach i [$fh(folder_handler) flagged flagged] {
	    lappend msgs [$fh(folder_handler) get $i]
	}
	return $msgs
    }
}

# FolderButtons --
#
# Enable or disable the buttons in the folder window which depends on
# an active message.
#
# Arguments:
# handler -	The handler which identifies the folder window
# onoff   -	The new state of the buttons

proc FolderButtons {handler onoff} {
    upvar #0 $handler fh
    global option t b
    set w $fh(w)

    if $onoff {
	set state normal
    } else {
	set state disabled
	if $fh(pgp_enable) {
	    $fh(sigbut) configure -state disabled -text "$t(sig): $t(none)"
	    set b($fh(sigbut)) no_sig
	}
    }
    foreach but [list $w.buttons.move \
		      $w.buttons.delete \
		      $w.buttons.reply_sender \
		      $w.buttons.reply_all] {
	$but configure -state $state
    }
    foreach i $fh(message_menu_nokeep) {
	$w.mbar.message.m entryconfigure $i -state $state
    }
}

# FolderGetNextUnread --
#
# Return the index of the next unread message in folder after index,
# or index if none is found.
#
# Arguments:
# handler -	The handler which identifies the folder window
# index   -	Where in the list to start looking

proc FolderGetNextUnread {handler index dir} {
    upvar #0 $handler fh

    if {0 == $fh(size)} {
	return 0
    }
    for {set i [expr $index+$dir]} {$i != $index} {incr i $dir} {
	if {$i >= $fh(size)} {
	    set i 0
	} elseif {$i < 0} {
	    set i [expr $fh(size)-1]
	}
	if {$i == $index} {
	    return 0
	}
	if { 0 == [$fh(folder_handler) getFlag $i seen]} {
	    return $i
	}
    }
    return $index
}


# FolderSelectUnread --
#
# Selects the next unread message in the folder.
#
# Arguments:
# handler -	The handler which identifies the folder window

proc FolderSelectUnread {handler} {
    upvar #0 $handler fh

    set index $fh(active)
    if {0 == [llength $index]} {
	return
    }
    set i [FolderGetNextUnread $handler $index 1]
    FolderSelect $fh(message_list) $handler $i 0
}

# TraceDeferred --
#
# Trace the ratDeferred variable
#
# Arguments:
# m	- The menu to update
# index - The menu entry to update
# name1	- notused
# name2	- notused
# op	- notused

proc TraceDeferred {m index name1 name2 op} {
     global ratDeferred t

    if $ratDeferred {
	set state normal
    } else {
	set state disabled
    }
    $m entryconfigure $index -label "$t(send_deferred)    \[$ratDeferred\]" \
	    -state $state
}

# GroupMessageList --
#
# Pops a message list that lets the user select messages for a group
#
# Arguments:
# handler -	The handler which identifies the folder window

proc GroupMessageList {handler} {
    global b idCnt defaultFont t option
    upvar #0 $handler fh

    # Create identifier
    set id f[incr idCnt]
    set w .$id

    # Create toplevel
    toplevel $w
    wm title $w $t(edit_group)
    frame $w.f
    scrollbar $w.f.scroll \
        -relief sunken \
        -command "$w.f.list yview" \
	-highlightthickness 0
    listbox $w.f.list \
        -yscroll "$w.f.scroll set" \
        -font $defaultFont \
        -exportselection false \
	-highlightthickness 0 \
	-selectmode multiple \
	-setgrid true
    set b($w.f.list) group_list_editor
    Size $w.f.list gFolderL
    pack $w.f.scroll -side right -fill y
    pack $w.f.list -side left -expand 1 -fill both
    frame $w.buttons
    button $w.buttons.ok -text $t(ok) \
	    -command "GroupMessageListDone $w $handler 1"
    set b($w.buttons.ok) group_window_ok
    button $w.buttons.sel -text $t(select_all) \
	    -command "$w.f.list selection set 0 end"
    set b($w.buttons.sel) group_window_selall
    button $w.buttons.unsel -text $t(deselect_all) \
	    -command "$w.f.list selection clear 0 end"
    set b($w.buttons.unsel) group_window_unselall
    button $w.buttons.cancel -text $t(cancel) \
	    -command "GroupMessageListDone $w $handler 0"
    set b($w.buttons.cancel) cancel
    pack $w.buttons.ok \
	 $w.buttons.sel \
	 $w.buttons.unsel \
	 $w.buttons.cancel -side left -expand 1
    pack $w.buttons -side bottom -fill x -pady 5
    pack $w.f -expand 1 -fill both

    eval "$w.f.list insert 0 [$fh(folder_handler) list $option(list_format)]"
    foreach i [$fh(folder_handler) flagged flagged] {
	$w.f.list selection set $i
    }
    lappend fh(groupMessageLists) $w

    Place $w groupMessageList
}

# GroupMessageListUpdate --
#
# Update the message list since the underlying folder was updated
#
# Arguments:
# w	  -	The group selection window
# handler -	The handler which identifies the folder window

proc GroupMessageListUpdate {w handler} {
    upvar #0 $handler fh
    global option

    $w.f.list delete 0 end
    eval "$w.f.list insert 0 [$fh(folder_handler) list $option(list_format)]"
    foreach i [$fh(folder_handler) flagged flagged] {
	$w.f.list selection set $i
    }
}

# GroupMessageListDone --
#
# Calls when the grouping is done
#
# Arguments:
# w	  -	The group selection window
# handler -	The handler which identifies the folder window
# done    -	The users selection (1=ok, 0=cancel)

proc GroupMessageListDone {w handler done} {
    upvar #0 $handler fh
    global b option

    if $done {
	set toset [$w.f.list curselection]
	set isset [$fh(folder_handler) flagged flagged]
	for {set i 0} {$i < [$w.f.list size]} {incr i} {
	    set s [expr -1 != [lsearch $toset $i]]
	    if {$s != [expr -1 != [lsearch $isset $i]]} {
		$fh(folder_handler) setFlag $i flagged $s
		FolderListRefreshEntry $handler $i
	    }
	}
    }
    RecordPos $w groupMessageList
    RecordSize $w.f.list gFolderL
    set index [lsearch $w $fh(groupMessageLists)]
    set fh(groupMessageLists) [lreplace $fh(groupMessageLists) $index $index]
    destroy $w
    foreach a [array names b $w*] {
	unset b($a)
    }
}

# GroupClear --
#
# Removes the flag from every message
#
# Arguments:
# handler -	The handler which identifies the folder window

proc GroupClear {handler} {
    upvar #0 $handler fh
    global option

    foreach i [$fh(folder_handler) flagged flagged] {
	$fh(folder_handler) setFlag $i flagged 0
	FolderListRefreshEntry $handler $i
    }
}

# SetupGroupMenu --
#
# Setup the entries in the group menu
#
# Arguments:
# m	  -	The menu command name
# handler -	The handler which identifies the folder window

proc SetupGroupMenu {m handler} {
    upvar #0 $handler fh

    if ![info exists fh(folder_handler)] {
	set s disabled
    } elseif {[llength [$fh(folder_handler) flagged flagged]]} {
	set s normal
    } else {
	set s disabled
    }
    $m entryconfigure 4 -state $s
    $m entryconfigure 6 -state $s
    $m entryconfigure 7 -state $s
    $m entryconfigure 8 -state $s
    $m entryconfigure 9 -state $s
}

# CycleShowHeader --
#
# Cycle through the values of the show_header option
#
# Arguments:
# handler -	The handler which identifies the folder window

proc CycleShowHeader {handler} {
    global option
    upvar #0 $handler fh
    upvar #0 $fh(text) texth

    switch $texth(show_header) {
    all		{ set texth(show_header) selected }
    selected	{ set texth(show_header) no }
    no		{ set texth(show_header) all }
    }
    FolderSelect $fh(message_list) $handler $fh(active) 0
    SaveOptions
}

# FolderCheckSignature --
#
# Check the signature(s) of the current message
#
# Arguments:
# handler -	The handler which identifies the folder window

proc FolderCheckSignature {handler} {
    upvar #0 $handler fh
    upvar #0 msgInfo_$fh(current) msgInfo
    global t b

    set tot pgp_good
    set b($fh(sigbut)) good_sig
    set first 1
    set result {}
    foreach bodypart $msgInfo(pgp,signed_parts) {
	regsub -all "\a" [$bodypart checksig] {} part
	if [string length $result] {
	    set result "$totresult\n\n$part"
	} else {
	    set result $part
	}
	set status [$bodypart sigstatus]
	if [string compare pgp_good $status] {
	    set tot pgp_bad
	    set b($fh(sigbut)) bad_sig
	} elseif $first {
	    set first 0
	    if [string compare $bodypart [$fh(current) body]] {
		set tot pgp_part
		set b($fh(sigbut)) part_sig
	    }
	}
    }
    if [string length $result] {
	RatText $t(pgp_output) $result
    }
    $fh(sigbut) configure -state normal \
			  -text "$t(sig): $t($tot)" \
			  -command [list RatText $t(pgp_output) $result]
}

# FolderFind --
#
# Find text in a message or the message list
#
# Arguments:
# handler -	The handler which identifies the folder window

proc FolderFind {handler} {
    global t b idCnt
    upvar #0 $handler fh

    # Create identifier
    set id f[incr idCnt]
    upvar #0 $id hd
    set w .$id

    # Initialize variables
    set hd(ignore_case) $fh(find_ignore_case)
    set hd(match) $fh(find_match)
    set hd(loc) $fh(find_loc)
    set hd(w) $w
    set hd(handler) $handler
    set hd(def_start) 0

    # Create toplevel
    toplevel $w
    wm title $w $t(find)

    # Create window
    entry $w.e -textvariable ${id}(text)
    set b($w.e) enter_search_exp_here

    checkbutton $w.c -text $t(ignore_case) -variable ${id}(ignore_case)
    set b($w.c) toggle_ignore_case

    frame $w.m
    label $w.m.label -text $t(match): -anchor e -width 10
    radiobutton $w.m.exact -text $t(exact) -variable ${id}(match) \
	    -value exact
    radiobutton $w.m.regexp -text $t(regexp) -variable ${id}(match) \
	    -value regexp
    pack $w.m.label \
	 $w.m.exact \
	 $w.m.regexp -side left
    set b($w.m.exact) find_exact
    set b($w.m.regexp) find_regexp

    frame $w.l
    label $w.l.label -text $t(find_in): -anchor e -width 10
    radiobutton $w.l.list -text $t(message_list) -variable ${id}(loc) \
	    -value list
    radiobutton $w.l.body -text $t(message_body) -variable ${id}(loc) \
	    -value body
    pack $w.l.label \
	 $w.l.list \
	 $w.l.body -side left
    set b($w.l.list) find_in_mlist
    set b($w.l.body) find_in_body

    frame $w.b
    frame $w.b.find -relief sunken -bd 2
    button $w.b.find.b -text $t(find) -state disabled \
	    -command "FolderFindDo $id 0"
    pack $w.b.find.b -padx 1 -pady 1
    frame $w.b.find_next -relief flat -bd 2
    button $w.b.find_next.b -text $t(find_next) -state disabled \
	    -command "FolderFindDo $id 1"
    pack $w.b.find_next.b -padx 1 -pady 1
    button $w.b.dismiss -text $t(dismiss) \
	    -command "FolderFindDone $id"
    pack $w.b.find \
	 $w.b.find_next \
	 $w.b.dismiss -side left -padx 5 -pady 5 -expand 1
    set b($w.b.find.b) find_first
    set b($w.b.find_next.b) find_next
    set b($w.b.dismiss) dismiss

    pack $w.e \
	 $w.c \
	 $w.m \
	 $w.l \
	 $w.b -side top -pady 5 -padx 5 -anchor w -fill x -expand 1

    Place $w find
    focus $w.e
    bind $w.e <Return> "FolderFindDo $id \$${id}(def_start); break"

    trace variable hd(text) w "FinderFindTrace $id"
}

# FinderFindTrace --
#
# Trace the find text variable and change the button state accordingly
#
# Arguments:
# id      -	The handler which identifies the find window

proc FinderFindTrace {id args} {
    upvar #0 $id hd

    if [string length $hd(text)] {
	set state normal
    } else {
	set state disabled
	set hd(def_start) 0
    }
    $hd(w).b.find.b configure -state $state
    $hd(w).b.find_next.b configure -state $state
    if $hd(def_start) {
	$hd(w).b.find configure -relief flat
	$hd(w).b.find_next configure -relief sunken
    } else {
	$hd(w).b.find configure -relief sunken
	$hd(w).b.find_next configure -relief flat
    }
}

# FolderFindDo --
#
# Actually do the finding
#
# Arguments:
# id      -	The handler which identifies the find window
# current  -	Start at current location

proc FolderFindDo {id current} {
    upvar #0 $id hd
    upvar #0 $hd(handler) fh

    if {"list" == $hd(loc)} {
	set w $fh(message_list)
    } else {
	set w $fh(text)
    }
    if $current {
	set r [$w tag nextrange Found 1.0]
	if {[llength $r] > 0} {
	    set start [lindex $r 1]
	} else {
	    set start @0,0
	}
    } else {
	set start 1.0
    }
    if $hd(ignore_case) {
	set found [$w search -$hd(match) -nocase -count len \
		$hd(text) $start end]
    } else {
	set found [$w search -$hd(match) -count len $hd(text) $start end]
    }
    if [string length $found] {
	$w tag remove Found 1.0 end
	$w tag add Found $found $found+${len}c
	$w see $found
	set hd(def_start) 1
    } else {
	bell
	set hd(def_start) 0
    }
    FinderFindTrace $id
    set fh(find_ignore_case) $hd(ignore_case)
    set fh(find_match) $hd(match)
    set fh(find_loc) $hd(loc)
}

# FolderFindDone --
#
# Close find window
#
# Arguments:
# id      -	The handler which identifies the find window

proc FolderFindDone {id} {
    upvar #0 $id hd
    global b

    RecordPos $hd(w) find
    destroy $hd(w)
    foreach a [array names b $hd(w)*] {
	unset b($a)
    }
    unset hd
}
