/*
 * ratFrMessage.c --
 *
 *	This file contains code which implements free messages.
 *
 * TkRat software and its included text is Copyright 1996,1997,1998
 * by Martin Forssn
 *
 * Postilion software and its included text and images
 * Copyright (C) 1998 Nic Bernstein
 *
 * The full text of the legal notices is contained in the files called
 * COPYING and COPYRIGHT.TkRat, included with this distribution.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "ratFolder.h"

/*
 * The ClientData for each message entity
 */
typedef struct FrMessageInfo {
    MESSAGE *messagePtr;
    char *headers;
    char *bodyData;
} FrMessageInfo;

/*
 * The ClientData for each bodypart entity
 */
typedef struct FrBodyInfo {
    char *mark;
    char *text;
} FrBodyInfo;

/*
 * The number of message entities created. This is used to create new
 * unique command names.
 */
static int numFrMessages = 0;

static RatGetHeadersProc Fr_GetHeadersProc;
static RatGetEnvelopeProc Fr_GetEnvelopeProc;
static RatFetchTextProc Fr_FetchTextProc;
static RatEnvelopeProc Fr_EnvelopeProc;
static RatSetIndexProc Fr_SetIndexProc;
static RatMsgDeleteProc Fr_MsgDeleteProc;
static RatMakeChildrenProc Fr_MakeChildrenProc;
static RatFetchBodyProc Fr_FetchBodyProc;
static RatBodyDeleteProc Fr_BodyDeleteProc;
static RatInfoProc Fr_GetInfoProc;


/*
 *----------------------------------------------------------------------
 *
 * RatFrMessagesInit --
 *
 *      Initializes the given MessageProcInfo entry for a free message
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The given MessageProcInfo is initialized.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatFrMessagesInit(MessageProcInfo *messageProcInfoPtr)
{
    messageProcInfoPtr->getHeadersProc = Fr_GetHeadersProc;
    messageProcInfoPtr->getEnvelopeProc = Fr_GetEnvelopeProc;
    messageProcInfoPtr->getInfoProc = Fr_GetInfoProc;
    messageProcInfoPtr->createBodyProc = Fr_CreateBodyProc;
    messageProcInfoPtr->fetchTextProc = Fr_FetchTextProc;
    messageProcInfoPtr->envelopeProc = Fr_EnvelopeProc;
    messageProcInfoPtr->setIndexProc = Fr_SetIndexProc;
    messageProcInfoPtr->msgDeleteProc = Fr_MsgDeleteProc;
    messageProcInfoPtr->makeChildrenProc = Fr_MakeChildrenProc;
    messageProcInfoPtr->fetchBodyProc = Fr_FetchBodyProc;
    messageProcInfoPtr->bodyDeleteProc = Fr_BodyDeleteProc;
}


/*
 *----------------------------------------------------------------------
 *
 * RatFrMessageCreate --
 *
 *      Creates a free message entity
 *
 * Results:
 *	The name of the new message entity.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

char*
RatFrMessageCreate(Tcl_Interp *interp, char *data, int length,
		   MessageInfo **msgPtrPtr)
{
    FrMessageInfo *msgFrPtr=(FrMessageInfo*)ckalloc(sizeof(FrMessageInfo));
    MessageInfo *msgPtr=(MessageInfo*)ckalloc(sizeof(MessageInfo));
    char *name = (char*)malloc(16);
    char *bodyData;
    int headerLength;

    for (headerLength = 0; data[headerLength]; headerLength++) {
	if (data[headerLength] == '\n' && data[headerLength+1] == '\n') {
	    headerLength++;
	    break;
	}
	if (data[headerLength]=='\r' && data[headerLength+1]=='\n'
		&& data[headerLength+2]=='\r' && data[headerLength+3]=='\n') {
	    headerLength += 2;
	    break;
	}
    }

    bodyData = (char*)malloc(length+1);
    memcpy(bodyData, data, length);
    bodyData[length] = '\0';

    msgPtr->folderInfoPtr = NULL;
    msgPtr->type = RAT_FREE_MESSAGE;
    msgPtr->bodyInfoPtr = NULL;
    msgPtr->name = name;
    msgPtr->msgNo = 0;
    msgPtr->fromMe = RAT_ISME_UNKOWN;
    msgPtr->toMe = RAT_ISME_UNKOWN;
    msgPtr->clientData = (ClientData)msgFrPtr;
    msgFrPtr->bodyData = bodyData;
    msgFrPtr->messagePtr = RatParseMsg(interp, bodyData);
    msgFrPtr->headers = (char*)ckalloc(headerLength+1);
    strncpy(msgFrPtr->headers, data, headerLength);
    msgFrPtr->headers[headerLength] = '\0';

    if (msgPtrPtr) {
	*msgPtrPtr = msgPtr;
    }

    sprintf(name, "RatFrMsg%d", numFrMessages++);
    Tcl_CreateCommand(interp, name, RatMessageCmd, (ClientData) msgPtr, NULL);
    return name;
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_GetHeadersProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static char*
Fr_GetHeadersProc(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    FrMessageInfo *frMsgPtr = (FrMessageInfo*)msgPtr->clientData;
    return frMsgPtr->headers;
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_GetEnvelopeProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static char*
Fr_GetEnvelopeProc(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    FrMessageInfo *frMsgPtr = (FrMessageInfo*)msgPtr->clientData;
    static char buf[1024];
    MESSAGECACHE elt;
    ADDRESS *adrPtr;
    time_t date;
    struct tm tm, *tmPtr;

    if (frMsgPtr->messagePtr->env->return_path) {
	adrPtr = frMsgPtr->messagePtr->env->sender;
    } else if (frMsgPtr->messagePtr->env->sender) {
	adrPtr = frMsgPtr->messagePtr->env->sender;
    } else {
	adrPtr = frMsgPtr->messagePtr->env->from;
    }
    if (!strcmp(currentHost, adrPtr->host)) {
	sprintf(buf, "From %s", adrPtr->mailbox);
    } else {
	sprintf(buf, "From ");
	rfc822_address(buf+5, adrPtr);
    }
    if (T == mail_parse_date(&elt, frMsgPtr->messagePtr->env->date)) {
	tm.tm_sec = elt.seconds;
	tm.tm_min = elt.minutes;
	tm.tm_hour = elt.hours;
	tm.tm_mday = elt.day;
	tm.tm_mon = elt.month - 1;
	tm.tm_year = elt.year+69;
	tm.tm_wday = 0;
	tm.tm_yday = 0;
	tm.tm_isdst = -1;
	date = (int)mktime(&tm);
    } else {
	date = 0;
    }
    tmPtr = gmtime(&date);
    sprintf(buf + strlen(buf), " %s %s %2d %02d:%02d GMT %04d\n",
	    dayName[tmPtr->tm_wday], monthName[tmPtr->tm_mon],
	    tmPtr->tm_mday, tmPtr->tm_hour, tmPtr->tm_min, 
	    1900+tmPtr->tm_year);
    return buf;
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_CreateBodyProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

BodyInfo*
Fr_CreateBodyProc(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    FrMessageInfo *frMsgPtr = (FrMessageInfo*)msgPtr->clientData;
    FrBodyInfo *frBodyInfoPtr = (FrBodyInfo*)malloc(sizeof(FrBodyInfo));
    msgPtr->bodyInfoPtr = CreateBodyInfo(msgPtr);

    msgPtr->bodyInfoPtr->bodyPtr = frMsgPtr->messagePtr->body;
    msgPtr->bodyInfoPtr->clientData = (ClientData)frBodyInfoPtr;
    frBodyInfoPtr->text = frMsgPtr->messagePtr->text.text.data +
			  frMsgPtr->messagePtr->text.offset;
    frBodyInfoPtr->mark = frBodyInfoPtr->text;
    return msgPtr->bodyInfoPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_FetchTextProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static char*
Fr_FetchTextProc(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    FrMessageInfo *frMsgPtr = (FrMessageInfo*)msgPtr->clientData;
    return frMsgPtr->messagePtr->text.text.data +
	   frMsgPtr->messagePtr->text.offset;
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_EnvelopeProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static ENVELOPE*
Fr_EnvelopeProc(MessageInfo *msgPtr)
{
    return ((FrMessageInfo*)msgPtr->clientData)->messagePtr->env;
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_SetIndexProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static void
Fr_SetIndexProc(MessageInfo *msgPtr, int index)
{
     return;
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_MsgDeleteProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static void
Fr_MsgDeleteProc(MessageInfo *msgPtr)
{
    FrMessageInfo *frMsgPtr = (FrMessageInfo*)msgPtr->clientData;
    mail_free_envelope(&frMsgPtr->messagePtr->env);
    mail_free_body(&frMsgPtr->messagePtr->body);
    ckfree(frMsgPtr->messagePtr);
    ckfree(frMsgPtr->headers);
    ckfree(frMsgPtr->bodyData);
    ckfree(frMsgPtr);
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_MakeChildrenProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static void
Fr_MakeChildrenProc(Tcl_Interp *interp, BodyInfo *bodyInfoPtr)
{
    FrBodyInfo *frBodyInfoPtr = (FrBodyInfo*)bodyInfoPtr->clientData;
    BODY *bodyPtr = bodyInfoPtr->bodyPtr;
    BodyInfo *partInfoPtr, **partInfoPtrPtr;
    FrBodyInfo *frPartInfoPtr;
    PART *partPtr;

    if (!bodyInfoPtr->firstbornPtr) {
	partInfoPtrPtr = &bodyInfoPtr->firstbornPtr;
	for (partPtr = bodyPtr->nested.part; partPtr;
		partPtr = partPtr->next) {
	    frPartInfoPtr = (FrBodyInfo*)ckalloc(sizeof(FrBodyInfo));
	    partInfoPtr = CreateBodyInfo(bodyInfoPtr->msgPtr);
	    *partInfoPtrPtr = partInfoPtr;
	    partInfoPtr->bodyPtr = &partPtr->body;
	    partInfoPtrPtr = &partInfoPtr->nextPtr;
	    partInfoPtr->clientData = (ClientData)frPartInfoPtr;
	    frPartInfoPtr->text = frBodyInfoPtr->mark +
				  partPtr->body.contents.offset;
	    if (TYPEMULTIPART == partPtr->body.type) {
		frPartInfoPtr->mark = frBodyInfoPtr->text;
	    } else {
		frPartInfoPtr->mark = frPartInfoPtr->text;
	    }
	}
    }
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_FetchBodyProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static char*
Fr_FetchBodyProc(BodyInfo *bodyInfoPtr, unsigned long *lengthPtr)
{
    FrBodyInfo *frBodyInfoPtr = (FrBodyInfo*)bodyInfoPtr->clientData;

    if (bodyInfoPtr->decodedTextPtr) {
	*lengthPtr = Tcl_DStringLength(bodyInfoPtr->decodedTextPtr);
	return Tcl_DStringValue(bodyInfoPtr->decodedTextPtr);
    }
    *lengthPtr = bodyInfoPtr->bodyPtr->contents.text.size;
    return frBodyInfoPtr->text;
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_BodyDeleteProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static void
Fr_BodyDeleteProc(BodyInfo *bodyInfoPtr)
{
    FrBodyInfo *frBodyInfoPtr = (FrBodyInfo*)bodyInfoPtr->clientData;
    ckfree(frBodyInfoPtr);
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_GetInfoProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static char*
Fr_GetInfoProc(Tcl_Interp *interp, ClientData clientData,
	RatFolderInfoType type, int index)
{
    static char buf[128];
    MessageInfo *msgPtr = (MessageInfo*)clientData;
    FrMessageInfo *frMsgPtr = (FrMessageInfo*)msgPtr->clientData;
    MESSAGECACHE elt;
    char *cPtr;
    int i;

    switch (type) {
	case RAT_FOLDER_SUBJECT:	/* fallthrough */
	case RAT_FOLDER_NAME:		/* fallthrough */
	case RAT_FOLDER_MAIL_REAL:	/* fallthrough */
	case RAT_FOLDER_MAIL:		/* fallthrough */
	case RAT_FOLDER_NAME_RECIPIENT:	/* fallthrough */
	case RAT_FOLDER_MAIL_RECIPIENT:	/* fallthrough */
	case RAT_FOLDER_TYPE:		/* fallthrough */
	case RAT_FOLDER_TO:		/* fallthrough */
	case RAT_FOLDER_FROM:		/* fallthrough */
	case RAT_FOLDER_SENDER:		/* fallthrough */
	case RAT_FOLDER_CC:		/* fallthrough */
	case RAT_FOLDER_REPLY_TO:	/* fallthrough */
	case RAT_FOLDER_PARAMETERS:
	    return RatGetMsgInfo(interp, type, msgPtr,frMsgPtr->messagePtr->env,
		    frMsgPtr->messagePtr->body, NULL, 0);
	case RAT_FOLDER_SIZE:		/* fallthrough */
	case RAT_FOLDER_SIZE_F:
	    return RatGetMsgInfo(interp, type, msgPtr, NULL, NULL, NULL,
		    frMsgPtr->messagePtr->header.text.size +
		    frMsgPtr->messagePtr->text.text.size);
	case RAT_FOLDER_DATE_F:	/* fallthrough */
	case RAT_FOLDER_DATE_N:	/* fallthrough */
	case RAT_FOLDER_DATE_IMAP4:
	    if (T != mail_parse_date(&elt, frMsgPtr->messagePtr->env->date)) {
		rfc822_date(buf);
		mail_parse_date(&elt, buf);
	    }
	    return RatGetMsgInfo(interp, type, msgPtr,
		    frMsgPtr->messagePtr->env, NULL, &elt, 0);
	case RAT_FOLDER_STATUS:
	    cPtr = frMsgPtr->headers;
	    do {
		if (!strncasecmp(cPtr, "status:", 7)) {
		    int seen, deleted, marked, answered;
		    ADDRESS *addressPtr;

		    seen = deleted = marked = answered = 0;
		    for (i=7; cPtr[i]; i++) {
			switch (cPtr[i]) {
			case 'R': seen = 1;	break;
			case 'D': deleted = 1;	break;
			case 'F': marked = 1;	break;
			case 'A': answered = 1;	break;
			}
		    }
		    if (RAT_ISME_UNKOWN == msgPtr->toMe) {
			msgPtr->toMe = RAT_ISME_NO;
			for (addressPtr = frMsgPtr->messagePtr->env->to;
				addressPtr;
				addressPtr = addressPtr->next) {
			    if (RatAddressIsMe(interp, addressPtr, 1)) {
				msgPtr->toMe = RAT_ISME_YES;
				break;
			    }
			}
		    }
		    i = 0;
		    if (!seen) {
			buf[i++] = 'N';
		    }
		    if (deleted) {
			buf[i++] = 'D';
		    }
		    if (marked) {
			buf[i++] = 'F';
		    }
		    if (answered) {
			buf[i++] = 'A';
		    }
		    if (RAT_ISME_YES == msgPtr->toMe) {
			buf[i++] = '+';
		    } else {
			buf[i++] = ' ';
		    }
		    buf[i] = '\0';
		    return buf;
		}
	    } while ((cPtr = strchr(cPtr, '\n')) && cPtr++ && *cPtr);
	    return "";
	case RAT_FOLDER_FLAGS:
	    cPtr = frMsgPtr->headers;
	    buf[0] = '\0';
	    do {
		if (!strncasecmp(cPtr, "status:", 7)) {
		    for (i=7; cPtr[i]; i++) {
			switch (cPtr[i]) {
			case 'R': strcat(buf, " " RAT_SEEN_STR); break;
			case 'D': strcat(buf, " " RAT_DELETED_STR); break;
			case 'F': strcat(buf, " " RAT_FLAGGED_STR); break;
			case 'A': strcat(buf, " " RAT_ANSWERED_STR); break;
			}
		    }
		    return buf;
		}
	    } while ((cPtr = strchr(cPtr, '\n')) && cPtr++ && *cPtr);
	    return "";
	case RAT_FOLDER_INDEX:
	    return "1";
	case RAT_FOLDER_END:
	    break;
    }
    return NULL;
}
