/*
 * GWT - General Windowing Toolkit
 *
 * Copyright (C) 1998 MenTaLguY - mentalg@geocities.com
 *                    Rodolphe Ortalo - ortalo@laas.fr
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Log: zorder.c,v $
 * Revision 1.6  1998/10/22 02:59:13  ajapted
 * Added `#include <errno.h>' to make it compile.
 *
 * Revision 1.5  1998/09/02 02:21:50  tristan
 * Loadsa great stuff done. Visible/invalid regions,
 *  copy on move - fixed tt_region.c too!
 *
 * Revision 1.4  1998/08/23 18:57:16  ortalo
 * Minor editing (mainly comments).
 *
 * Revision 1.3  1998/08/16 18:40:59  ortalo
 * A new version, STILL IN VERY ALPHA STATE... (Be patient.)
 * (However, this version is a little better... :-)
 *
 * Revision 1.2  1998/08/11 22:10:12  ortalo
 * Incorporated the changes made by Tristan Wibberley. The code is
 * still in alpha development state.
 *
 * Revision 1.1  1998/07/09 18:38:39  ortalo
 * Initial integration of libgwt in the GGI repository.
 * This is experimental code, for your eyes only !
 * (Don't execute... ;-)
 *
 */

#include "internal.h"
#include <errno.h>

/************************************************************************
 *                         STATIC FUNCTIONS                             *
 ************************************************************************/
/*
 * Insert a window after the specified window.
 * REQUIRES: after must not be NULL.
 *           win must not be in a zorder
 */
static int gwt_zinsert_after(gwt_window_t win, gwt_window_t after) {
  /* We don't allow inserting in a different parent */
  if (win->wparent!=after->wparent) return -EINVAL;
  /* Update windows siblings links */
  win->zprev = after;
  win->znext = after->znext;
  /* Update the bottom link of parent */
  if (after->znext == NULL)
    after->wparent->zorder.zbottom = win;
  else
    after->znext->zprev=win;
  after->znext = win;
  return 0;
}
/*
 * Insert a window before the specified window.
 * REQUIRES: after before not be NULL.
 *           win must not be in a zorder
 */
static int gwt_zinsert_before(gwt_window_t win, gwt_window_t before) {
  /* We don't allow inserting in a different parent */
  if (win->wparent!=before->wparent) return -EINVAL;
  /* Update windows siblings links */
  win->znext = before;
  win->zprev = before->zprev;
  /* Update the top link of parent */
  if (before->zprev == NULL)
    before->wparent->zorder.ztop = win;
  else
    before->zprev->znext=win;
  before->zprev = win;
  return 0;
}
/*
 * Make a window the top children
 * REQUIRES: win must not be in the zorder (ie extracted)
 */
static int gwt_zmake_top(gwt_window_t win) {
  if (win->wparent->zorder.ztop != NULL) {
    /* Parent has childs */
    gwt_zinsert_before(win, win->wparent->zorder.ztop);
  } else {
    /* Parent has no child... */
    win->wparent->zorder.ztop = win->wparent->zorder.zbottom = win;
  }    
  return 0;
}
/*
 * Make a window the bottom children
 * REQUIRES: win must not be in a zorder (ie extracted)
 */
static int gwt_zmake_bottom(gwt_window_t win) {
  if (win->wparent->zorder.zbottom != NULL) {
    /* Parent has childs */
    gwt_zinsert_after(win, win->wparent->zorder.zbottom);
  } else {
    /* Parent has no child... */
    win->wparent->zorder.ztop = win->wparent->zorder.zbottom = win;
  }    
  return 0;
}
/**********************************************************************
 *                   GWT PRIVATE EXPORTED FUNCTIONS                   *
 **********************************************************************/

/*
 * Initialise a window for zorder management
 * (child list, and prev/next pointers)
 */
int gwt_zinit_window(gwt_window_t win) {
  win->zorder.ztop = NULL;
  win->zorder.zbottom = NULL;
  win->zprev = NULL;
  win->znext = NULL;
  return 0; /* Always succeed :-) */
}
/*
 * Extract a window from its zorder, reconnect
 * siblings, update parent window.
 */
int gwt_zextract(gwt_window_t win) {
  if (win->wparent != NULL) {
    /* If the window already has a parent, extract it */
    /* Update the zorder */
    if (win->zprev != NULL) {
      /* Reconnect previous window */
      win->zprev->znext=win->znext;
    } else {
      /* else adjust parent top */
      win->wparent->zorder.ztop = win->znext;
    }
    if (win->znext != NULL) {
      /* Reconnect following window */
      win->znext->zprev=win->zprev;
    } else {
      /* else adjust parent bottom */
      win->wparent->zorder.zbottom = win->zprev;
    }
  }
  /* Nullify the window links */
  win->zprev=win->znext=NULL;
  return 0;
}

/*
 * Gives a new parent to the window
 */
int gwt_reparent(gwt_window_t win, gwt_window_t newparent) {
  gwt_zextract(win);
  win->wparent = newparent;
  gwt_zmake_top(win);
  return 0;
}

/**********************************************************************
 *                    GWT PUBLIC EXPORTED FUNCTIONS                   *
 **********************************************************************/

/*
 * Sub-Windows Z-Order access (top,bottom)
 */
int gwtGetBottomChild(gwt_window_t win, gwt_window_t *bottom) {
  *bottom = win->zorder.zbottom;
  return 0;
}
int gwtGetTopChild(gwt_window_t win, gwt_window_t *top) {
  *top = win->zorder.ztop;
  return 0;
}

/*
 * Sibling windows Z-ordered access
 */
int gwtGetNextChild(gwt_window_t win, gwt_window_t *next) {
  *next = win->znext;
  return 0;
}
int gwtGetPrevChild(gwt_window_t win, gwt_window_t *prev) {
  *prev = win->zprev;
  return 0;
}

/*
 * Windows Z-Order management functions
 */
/*
 * Put a window above the given window.
 * REQUIRES: above != NULL || win->wparent != NULL
 */
int gwtRaise(gwt_window_t win, gwt_window_t above) {
  gwt_window_t old_z;

  /* No zorder management for root windows */
  if ( win->level == 0 ) return -EINVAL;

  old_z = GWT_Z_UP(win);

  if ( above != NULL ) {
    /* We allow such operations only between sibling windows */
    if (above->wparent != win->wparent) return -EINVAL;
    gwt_zextract(win);
    gwt_zinsert_before(win, above);
  } else {
    gwt_zextract(win);
    gwt_zmake_top(win);
  }

  gwt_visible_raise(old_z, win, win);
  gwt_visible_setvisinv_family(win);

  return 0;
}
int gwtLower(gwt_window_t win, gwt_window_t below) {
  gwt_window_t old_z;

  /* No zorder management for root windows */
  if ( win->level == 0 ) return -EINVAL;

  old_z = GWT_Z_DOWN(win);

  if ( below != NULL ) {
    /* We allow such operations only between sibling windows */
    if (below->wparent != win->wparent) return -EINVAL;
    gwt_zextract(win);
    gwt_zinsert_after(win, below);
  } else {
    gwt_zextract(win);
    gwt_zmake_bottom(win);
  }

  gwt_visible_lower(old_z, win, win);
  gwt_visible_setvisinv_family(win);

  return 0;
}

/*
 * Windows hierarchy manipulation
 */
int gwtGetParent(gwt_window_t win, gwt_window_t *parent)
{
  (*parent) = win->wparent;
  return 0;
}
int gwtReparent(gwt_window_t win, gwt_window_t newparent)
{
  /*
   * This call always invalidates the whole of the new visible region (for now)
   */
  gwt_visible_hide(win);
  gwt_reparent(win, newparent);
  gwt_visible_show(win);
  gwt_visible_setvisinv_family(win);
  return 0;
}

