/* 
 *  Internal declarations for loadable libraries
 *
 *  Copyright (C) 1998 by Thomas Tanner. See CREDITS for details.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
 
#ifndef _GGILIB2D_H_
#define _GGILIB2D_H_

#define _BUILDING_LIBGGI2D_

#include <ggi/internal/internal.h>

typedef struct ggi2d_image_t	*ggi2d_image;

#include <ggi/ggi2d.h>
#include <ggi/ggi_ext.h>

/*#include "debug.h"*/

extern int ggi2d_offset;

typedef	void	(*updatefunc)(ggi_visual_t vis, ggi_uint change);

/* mode changes: functions should be updated */
#define	GGI2D_UPD_ARCMODE	1	
#define	GGI2D_UPD_POLYMODE	2
#define	GGI2D_UPD_LINEMODE	4
#define	GGI2D_UPD_APPENDMODE	8	
#define	GGI2D_UPD_TEXTUREMODE	16
#define	GGI2D_UPD_OPERATOR	32

/* value changes: accelerator should be updated */
#define	GGI2D_UPD_LINE		256
#define	GGI2D_UPD_CLIP		512	
#define	GGI2D_UPD_DRAWCOL	1024	
#define	GGI2D_UPD_FILLCOL	2048	
#define	GGI2D_UPD_TEXTURE	4096

#define	GGI2D_UPD_ALL		(2*GGI2D_UPD_TEXTURE-1)

#define ISVALUE(change)	(change > GGI_UPD_OPERATOR)

typedef struct {
	ggi2d_arcmode	arcmode;
	ggi2d_polymode	polymode;
	ggi_uint	*dash;
	ggi_uint	dashsize;
	int		append;
	ggi_coord	cliptl;
	ggi_coord	clipbr;
	ggi_pixel	drawcolor;
	ggi_pixel	fillcolor;
	int		textured;
	ggi2d_image	texture;
	ggi2d_coord	texref;
	ggi2d_operator	op;
	int		accel_changes;	
} ggi2d_gc;

typedef struct {
	updatefunc update;
	ggi_uint	change_mask;
} ggi2d_module; 

typedef struct {
/*------- Drawing functions (33) -------------*/
void	(*putPixel)(ggi_visual_t vis, ggi_sint x, ggi_sint y, ggi_pixel color);
void	(*drawPixel)(ggi_visual_t vis, ggi_sint x, ggi_sint y);
void	(*fillPixel)(ggi_visual_t vis, ggi_sint x, ggi_sint y);
void	(*drawPixels)(ggi_visual_t vis, ggi2d_coord coords[], ggi_uint count);
void	(*scanline)(ggi_visual_t vis, ggi_sint x1, ggi_sint x2, ggi_sint y);
void	(*scanlines)(ggi_visual_t vis, ggi2d_scanline scanlines[], ggi_uint count);
void	(*hLine)(ggi_visual_t vis, ggi_sint x1, ggi_sint x2, ggi_sint y);
void	(*vLine)(ggi_visual_t vis, ggi_sint x, ggi_sint y1, ggi_sint y2);
void	(*drawRect)(ggi_visual_t vis, ggi_sint x1, ggi_sint y1, ggi_sint x2, ggi_sint y2);
void	(*fillRect)(ggi_visual_t vis, ggi_sint x1, ggi_sint y1, ggi_sint x2, ggi_sint y2);
void	(*line)(ggi_visual_t vis, ggi_sint x1, ggi_sint y1, ggi_sint x2, ggi_sint y2);
void	(*drawLines)(ggi_visual_t vis, ggi2d_line lines[], ggi_uint count);
void	(*drawCircle)(ggi_visual_t vis, ggi_sint x, ggi_sint y, ggi_uint r);
void	(*fillCircle)(ggi_visual_t vis, ggi_sint x, ggi_sint y, ggi_uint r);
void	(*drawEllipse)(ggi_visual_t vis, ggi_sint x, ggi_sint y, ggi_uint rx, ggi_uint ry);
void	(*fillEllipse)(ggi_visual_t vis, ggi_sint x, ggi_sint y, ggi_uint rx, ggi_uint ry);
void	(*drawArc)(ggi_visual_t vis, ggi_sint x, ggi_sint y, ggi_uint rx, ggi_uint ry,
	    	   ggi_float startangle, ggi_float endangle, int close);
void	(*fillArc)(ggi_visual_t vis, ggi_sint x, ggi_sint y, ggi_uint rx, ggi_uint ry,
		   ggi_float startangle, ggi_float endangle);
void	(*bezier)(ggi_visual_t vis, ggi_sint x1, ggi_sint y1, ggi_sint x2, ggi_sint y2,
		     ggi_sint x3, ggi_sint y3, ggi_sint x4, ggi_sint y4);
void	(*trapezoid)(ggi_visual_t vis, 
		    ggi_sint xl1, ggi_sint xr1, ggi_sint y1,
		    ggi_sint xl2, ggi_sint xr2, ggi_sint y2);
void	(*triangle)(ggi_visual_t vis, 
		    ggi_sint x1, ggi_sint y1, ggi_sint x2, ggi_sint y2, ggi_sint x3, ggi_sint y3);
void	(*drawPoly)(ggi_visual_t vis, ggi2d_coord coords[], ggi_uint count);
void	(*fillPoly)(ggi_visual_t vis, ggi2d_coord coords[], ggi_uint count);
void	(*fillPolys)(ggi_visual_t vis, ggi2d_coord coords[], 
		ggi_uint counts[], ggi_uint count);
void	(*blit)(
		ggi_visual_t vis, ggi_sint dx, ggi_sint dy, 
		ggi2d_image src, ggi_sint sx, ggi_sint sy, 
		ggi_sint width, ggi_sint height);	
void	(*blitOp)(
		ggi_visual_t vis, ggi_sint dx, ggi_sint dy, 
		ggi2d_image src1, ggi_sint s1x, ggi_sint s1y,
		ggi2d_image src2, ggi_sint s2x, ggi_sint s2y, 
		ggi_sint width, ggi_sint height, ggi2d_operator op);
void	(*stretchBlit)(
		ggi_visual_t vis, ggi_sint dx, ggi_sint dy, 
		ggi_sint dwidth, ggi_sint dheight, 
		ggi2d_image src, ggi_sint sx, ggi_sint sy, 
		ggi_sint swidth, ggi_sint sheight);
void	(*stretchBlitOp)(
		ggi_visual_t vis, ggi_sint dx, ggi_sint dy, 
		ggi_sint dwidth, ggi_sint dheight, 
		ggi2d_image src1, ggi_sint s1x, ggi_sint s1y,
		ggi2d_image src2, ggi_sint s2x, ggi_sint s2y, 
		ggi_sint swidth, ggi_sint sheight, ggi2d_operator op);
} ggi2d_funcs;

#define	MAX_MODULES	4

typedef struct {
	ggi_uint	version;		
	ggi_uint	modules;	/* number of loaded modules */
	ggi_uint	change_mask;	/* general mask */
	void		*stubs_private;
	void		*fb_private;
	void		*drv_private;
	void		*misc_private;
	ggi2d_module	module[MAX_MODULES];
	ggi2d_gc	gc;
	ggi2d_funcs	func;
} ggi2d_data;

extern ggi_extid ggi2dID;

#define GGI2D_DATA(vis)		((ggi2d_data *)LIBGGI_EXT(vis,ggi2dID))
#define GGI2D_FUNC(vis)		(&(GGI2D_DATA(vis)->func))

#define GGI2D_PRIV_STUBS(vis)	(GGI2D_DATA(vis)->stubs_private)
#define GGI2D_PRIV_FB(vis)	(GGI2D_DATA(vis)->fb_private)
#define GGI2D_PRIV_DRV(vis)	(GGI2D_DATA(vis)->drv_private)
#define GGI2D_PRIV_MISC(vis)	(GGI2D_DATA(vis)->misc_private)

#define GGI2D_GC(vis)		(GGI2D_DATA(vis)->gc)

#define GGI2D_GC_ARCMODE(vis) 	(GGI2D_GC(vis).arcmode)
#define GGI2D_GC_POLYMODE(vis)	(GGI2D_GC(vis).polymode)
#define GGI2D_GC_DASH(vis) 	(GGI2D_GC(vis).dash)
#define GGI2D_GC_DASHSIZE(vis)	(GGI2D_GC(vis).dashsize)
#define GGI2D_GC_APPEND(vis)	(GGI2D_GC(vis).append)
#define GGI2D_GC_CLIP_X1(vis)	(GGI2D_GC(vis).cliptl.x)
#define GGI2D_GC_CLIP_Y1(vis)	(GGI2D_GC(vis).cliptl.y)
#define GGI2D_GC_CLIP_X2(vis)	(GGI2D_GC(vis).clipbr.x)
#define GGI2D_GC_CLIP_Y2(vis)	(GGI2D_GC(vis).clipbr.y)
#define GGI2D_GC_DRAWCOL(vis)	(GGI2D_GC(vis).drawcolor)
#define GGI2D_GC_FILLCOL(vis)	(GGI2D_GC(vis).fillcolor)
#define GGI2D_GC_TEXTURED(vis)	(GGI2D_GC(vis).textured)
#define GGI2D_GC_TEXTURE(vis) 	(GGI2D_GC(vis).texture)
#define GGI2D_GC_TEXREF(vis)	(GGI2D_GC(vis).texref)
#define GGI2D_GC_OP(vis) 	(GGI2D_GC(vis).op)
#define GGI2D_ACCELCHANGES(vis)	(GGI2D_GC(vis).accel_changes)

void	GGI2DregisterUpdateFunc(ggi_visual_t vis, updatefunc f, ggi_uint mask);

#define	GGIEXP(name)		GGI2D__##name
#define	GGILIB_INIT		GGIdlinit
#define	GGILIB_CLEANUP		GGIdlcleanup

#endif /* _GGILIB2D_H_ */
