/* $Id: init.c,v 1.31 1998/10/30 05:08:10 ajapted Exp $
***************************************************************************

   LibGGI initialization.

   Copyright (C) 1997 Jason McMullan	[jmcc@ggi-project.org]
   Copyright (C) 1998 Marcus Sundberg	[marcus@ggi-project.org]
  
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
 
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
 
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>

#include <ggi/internal/internal.h>
#include <ggi/gg.h>

/* Global variables */
uint32                _ggiDebugState = 0;
int                   _ggiDebugSync = 0;
void                 *_ggiConfigHandle;

/* Static variables */
static struct {
	ggi_mutex	mutex;	/* Lock when changing.. */
	int visuals;		/* # of visuals active */
	ggi_visual *visual;	/* Linked list of all visuals */
} _ggiVisuals;	/* This is for remembering visuals. */

static int 	      _ggiLibIsUp=0;
static char           ggiconfstub[512]=GGICONFFILE;
static char          *ggiconffile=ggiconfstub+4;
static int            numextensions=0;
static ggi_extension *_ggiExtension;

#if defined(DEBUG) && __GLIBC__ >= 2 && defined(HAVE_MALLOC_HOOK)
#define HAVE_GGIMEMDEBUG
#include <signal.h>
#include <malloc.h>
#define MEMTAG1 0x12345678
#define MEMTAG2 0x09876543
#define MEMTAG3 0x4321ffe4
#define set32idx8(ptr,i,val) *((uint32*)(((uint8*)(ptr))+(i)))=val
#define get32idx8(ptr,i) *((uint32*)(((uint8*)(ptr))+(i)))

static void *(*_ggi_old_realloc) (void *ptr, size_t size);
static void *(*_ggi_old_malloc)  (size_t);
static void  (*_ggi_old_free)     (void *ptr);

static unsigned long _ggi_alloced = 0;
static int _ggi_memdebug_init = 0;


unsigned long
_get_ggi_alloced(void)
{
	return _ggi_alloced;
}

static void *
_ggi_malloc_hook(size_t size)
{
	uint32 *mem;
	__malloc_hook = _ggi_old_malloc;
	if ((mem = malloc(size+16)) == NULL) {
		ggiPanic("_ggi_malloc_hook failed!\n");
	}
	__malloc_hook = _ggi_malloc_hook;
	_ggi_alloced += size;
	
	set32idx8(mem,size+8,MEMTAG2);
	set32idx8(mem,size+12,MEMTAG3);
	*mem = size;
	mem++;
	*mem = MEMTAG1;
	mem++;

	return (void*)mem;
}

static void *
_ggi_realloc_hook(void *ptr, size_t size)
{
	uint32 *ret = ptr;

	if (ret != NULL) {
		ret--;
		if (*ret != MEMTAG1) {
			ggiPanic("Inconsistency 1 detected in realloc!\n");
		}
		ret--;
		if (get32idx8(ret,*ret+8) != MEMTAG2) {
			ggiPanic("Inconsistency 2 detected in realloc!\n");
		}
		if (get32idx8(ret,*ret+12) != MEMTAG3) {
			ggiPanic("Inconsistency 3 detected in realloc!\n");
		}
		
		_ggi_alloced -= *ret;
	}
	__realloc_hook = _ggi_old_realloc;
	if ((ret = realloc(ret, size+16)) == NULL) {
		ggiPanic("_ggi_realloc_hook failed!\n");
	}
	__realloc_hook = _ggi_realloc_hook;
	_ggi_alloced += size;

	set32idx8(ret,size+8,MEMTAG2);
	set32idx8(ret,size+12,MEMTAG3);
	*ret = size;
	ret++;
	*ret = MEMTAG1;
	ret++;

	return ret;
}

static int outofbounds;

static void (*_ggi_oldsighandler)(int);

static void 
_ggi_freesighandler(int unused)
{
	outofbounds++;
	signal(SIGSEGV, _ggi_freesighandler);
}

static void
_ggi_free_hook(void *ptr)
{
	uint32 *ptr32 = ptr;

	if (ptr32 == NULL) {
		ggiPanic("free() called with NULL argument!\n");
	}

	outofbounds = 0;
	_ggi_oldsighandler = signal(SIGSEGV, _ggi_freesighandler);
	ptr32--;
	if (*ptr32 != MEMTAG1 && !outofbounds) {
		ggiPanic("Inconsistency 1 detected in free!\n");
	}
	ptr32--;
	if (get32idx8(ptr32,*ptr32+8) != MEMTAG2 && !outofbounds) {
		ggiPanic("Inconsistency 2 detected in free!\n");
	}
	if (get32idx8(ptr32,*ptr32+12) != MEMTAG3 && !outofbounds) {
		ggiPanic("Inconsistency 3 detected in free!\n");
	}
	if (outofbounds) {
		puts("free() argument caused SIGSEGV!\n");
		exit(42);
	}
	signal(SIGSEGV, _ggi_oldsighandler);
	_ggi_alloced -= *ptr32;
	__free_hook = _ggi_old_free;
	free(ptr32);
	__free_hook = _ggi_free_hook;
}
#endif

/*
 * Initalize the strutures for the library
 */

int ggiInit(void)
{
	int err;
	const char *str;
	
	_ggiLibIsUp++;
	if (_ggiLibIsUp>1) return 0;	/* Initialize only at first call. */

#ifdef HAVE_GGIMEMDEBUG
	if (!_ggi_memdebug_init && getenv("GGI_MEMDEBUG")) {
		_ggi_old_realloc = __realloc_hook;
		__realloc_hook = _ggi_realloc_hook;
		_ggi_old_free = __free_hook;
		__free_hook = _ggi_free_hook;
		_ggi_old_malloc = __malloc_hook;
		__malloc_hook = _ggi_malloc_hook;
		_ggi_memdebug_init = 1;
	}
#endif
	err=giiInit();
	if (err) {
		fprintf(stderr,"LibGGI: couldn't initialize LibGII.\n");
		return err;
	}

	_ggiLockInit(_ggiVisuals.mutex);
	_ggiVisuals.visuals=0;
	_ggiVisuals.visual=NULL;

	str=getenv("GGI_DEBUG");
	if (str!=NULL) {
		_ggiDebugState=atoi(str);
		DPRINT_CORE("Debugging=%d\n",_ggiDebugState);
	}

	str=getenv("GGI_DEBUGSYNC");
	if (str!=NULL) {
		_ggiDebugSync=1;
	}
		
	str=getenv("GGI_DEFMODE");
	if (str!=NULL) {
		_ggiSetDefaultMode(str);
	}

	err = ggLoadConfig(ggiconffile, &_ggiConfigHandle);
	if (err == GGI_OK)
		return GGI_OK;

	fprintf(stderr,"LibGGI: couldn't open %s", ggiconffile );

	_ggiLockDestroy(_ggiVisuals.mutex);
	_ggiLibIsUp--;
	return err;
}

int ggiExit(void)
{
	ggi_extension *tmp,*next;

	DPRINT_CORE("ggiExit called\n");
	if (!_ggiLibIsUp)
		return -1;

	if (_ggiLibIsUp > 1) {
		_ggiLibIsUp--;
		return _ggiLibIsUp;
	}

	DPRINT_CORE("ggiExit: really destroying.\n");
	while (_ggiVisuals.visual!=NULL) 
		ggiClose(_ggiVisuals.visual);

	_ggiLockDestroy(_ggiVisuals.mutex);
	
	for (tmp=_ggiExtension;tmp;tmp=next)
	{
		next=tmp->next;
		free(tmp);
	}

	ggFreeConfig(_ggiConfigHandle);
	giiExit();
	_ggiLibIsUp=0;

#ifdef HAVE_GGIMEMDEBUG
	if (_ggi_memdebug_init) {
		__realloc_hook = _ggi_old_realloc;
		__free_hook    = _ggi_old_free;
		__malloc_hook  = _ggi_old_malloc;
		_ggi_memdebug_init = 0;
	}
#endif
	DPRINT_CORE("ggiExit: done!\n");
	return 0;
}

void ggiPanic(const char *format,...)
{
	DPRINT_CORE("ggiPanic called\n");
	fprintf(stderr,"%s",format);
	fflush(stderr);
	while(ggiExit()>0);	/* kill all instances ! */
	exit(1);
}


/* Opens a visual.
 */
ggi_visual *ggiOpen(const char *driver,...)
{
	va_list drivers;
	ggi_visual *vis;
	char *cp, *inplist;
	char str[1024];
	char target[1024];
	int  success=0;
	void *argptr;

	if (!_ggiLibIsUp)
		return NULL;

	if (driver==NULL) {
		static int recursion_level=0;
		void *ret;
		
		/* GGI_DISPLAY will _always_ override, unless
		 * recursion_level is > 1, which means that the previous
		 * target has called ggiOpen(NULL) wanting the
		 * `auto-detect' feature.
		 */

		recursion_level++;
		if (recursion_level == 1) {
			cp=getenv("GGI_DISPLAY");
			if (cp!=NULL) {
				ret = ggiOpen(cp,NULL);
				recursion_level--;
				return ret;
			}
		}
		recursion_level--;

		/* Try the X display.. */
		cp=getenv("DISPLAY");
		if (cp!=NULL) {
			strcpy(str,"display-x:");
			strcat(str,cp);
			ret = ggiOpen(str,NULL);
			if (ret != NULL)
				return ret;
		}

#if 0
		/* Try the KGI console.. */
		ret=ggiOpen("display-KGI:/dev/graphic",NULL);
		if (ret != NULL)
			return ret;
#endif
		
		/* Try the framebuffer console.. */
		ret = ggiOpen("display-fbdev", NULL);
		if (ret != NULL)
			return ret;

		/* Try svgalib target.. */
		ret = ggiOpen("display-svga",NULL);
		if (ret != NULL)
			return ret;

		/* Try AAlib target.. */
		return ggiOpen("display-aa",NULL);
	}

	DPRINT_CORE("ggiOpen(\"%s\") called\n", driver);
	
	vis=_ggiNewVisual();

	va_start(drivers,driver);

	argptr=va_arg(drivers,void *);
	va_end(drivers);

	DPRINT_CORE("Loading driver %s\n",driver);

	do {
		if (ggParseTarget((char *)driver,target,1024) == NULL) {
			break;
		}

		if (strlen(target) == 0) {
			fprintf(stderr, "libggi: Missing target descriptor !\n");
			break;
		}
		
		cp=strchr(target, ':');

		if (cp != NULL) {
			*cp++ = 0;
		}
		if (_ggiOpenDL(vis,target,cp,argptr) != NULL) {
			success=1;
		}

	} while (0);
	
	_ggiLock(_ggiVisuals.mutex);

	if (success) {
		vis->next=_ggiVisuals.visual;
		_ggiVisuals.visual=vis;
		_ggiVisuals.visuals++;
		DPRINT_CORE("ggiOpen: returning %p\n", vis);
	} else {
		_ggiDestroyVisual(vis);
		vis=NULL;
		DPRINT_CORE("ggiOpen: failure\n");
	}
	
	_ggiUnlock(_ggiVisuals.mutex);

	va_end(drivers);

	DPRINT_CORE("Loading extra inputs %s\n",driver);

	inplist = getenv("GGI_INPUT");

	if (inplist) for (;;) {

		gii_input *inp;

		inplist = ggParseTarget(inplist, target, 1024);

		if (inplist == NULL) {
			break;
		}

		if (strlen(target) == 0) {
			fprintf(stderr, "libggi: Missing input descriptor !\n");
			break;
		}
		
		inp = giiOpen(target, NULL);

		if (inp == NULL) {
			fprintf(stderr, "libggi: failed to load input: %s\n",
				target);
		} else {
			vis->input = giiJoinInputs(inp, vis->input);
		}

		while (*inplist && isspace((int) *inplist)) {
			inplist++;
		}

		if (*inplist != ':') {
			break;
		}

		inplist++;   /* skip ':' */
	}
	
	return vis;
}
	
/* ggiClose
 *	Attempt to close the visual associated with a vt
 *      returns 0 on success, nonzero on error
 */
int ggiClose(ggi_visual *visual)
{
	ggi_visual *vis,*pvis=NULL;

	DPRINT_CORE("ggiClose(\"%p\") called\n", visual);

	if (!_ggiLibIsUp)
		return -1;

	_ggiLock(_ggiVisuals.mutex);

	DPRINT_CORE("ggiClose: closing\n");

	for (vis=_ggiVisuals.visual;vis!=NULL;pvis=vis,vis=vis->next)
		if (vis==visual)
			break;

	if (vis==NULL) {
		_ggiUnlock(_ggiVisuals.mutex);
		return -1;
	}

	if (pvis==NULL)
		_ggiVisuals.visual=vis->next;
	else
		pvis->next=vis->next;

	_ggiVisuals.visuals--;

	vis->next=NULL;
	
	_ggiUnlock(_ggiVisuals.mutex);
	_ggiDestroyVisual(vis);

	DPRINT_CORE("ggiClose: done!\n");
	return 0;
}

/*
 * Extension handling.
 */

/*
 * Register an Extension for usage. It will return an extension ID 
 * (ggi_extid) that can be used to address this extension.
 */
ggi_extid ggiExtensionRegister(char *name,int size,
			int (*change)(ggi_visual_t,int whatchanged))
{
	ggi_extension *tmp,*ext;

	DPRINT_CORE("ggiExtensionRegister(\"%s\", %d, %p) called\n",
		    name, size, change);
	if (_ggiExtension)
		for (tmp=_ggiExtension;tmp;tmp=tmp->next)
			if (strcmp(tmp->name,name)==0)
			{
				tmp->initcount++;
				DPRINT_CORE("ggiExtensionRegister: accepting copy #%d of extension %s\n",
				       tmp->initcount,tmp->name);
				return tmp->id;
			}

	ext=_ggi_malloc(sizeof(ggi_extension));

	ext->size=size;
	ext->paramchange=change;
	ext->next=NULL;
	ext->initcount=1;
	strncpy(ext->name,name,sizeof(ext->name));
	ext->name[sizeof(ext->name)-1]='\0';	/* Make sure it terminates */
	
	if (_ggiExtension) {
		for (tmp=_ggiExtension;tmp->next;tmp=tmp->next) ;
		tmp->next=ext;
		ext->prev=tmp;			
	} else {
		_ggiExtension=ext;
		ext->prev=NULL;
	}

	DPRINT_CORE("ggiExtensionRegister: installing first copy of extension %s\n", name);
	return ext->id=numextensions++;
}

/*
 * Unregister an Extension.It takes the ggi_extid gotten from
 * ggiExtensionRegister. It disallows further calls to 
 * ggiExtensionAttach (for that extid) and frees memory for the
 * extension registry. 
 * It dos NOT automatically ggiExtensionDetach it from all visuals.
 */
int ggiExtensionUnregister(ggi_extid id)
{
	ggi_extension *tmp;

	DPRINT_CORE("ggiExtensionUnregister(%d) called\n", id);
	if (_ggiExtension)
		for (tmp=_ggiExtension;tmp;tmp=tmp->next)
			if (tmp->id==id)
			{
				if (--tmp->initcount) {
					DPRINT_CORE("ggiExtensionUnregister: removing #%d copy of extension %s\n",tmp->initcount+1,tmp->name);
					return 0;	/* Removed copy */
				}

				if (tmp->prev==NULL) 
					_ggiExtension=tmp->next;
				else 
					tmp->prev->next=tmp->next;

				if (tmp->next!=NULL) 
					tmp->next->prev=tmp->prev;

				DPRINT_CORE("ggiExtensionUnregister: removing last copy of extension %s\n",tmp->name);

				free(tmp);
				return 0;
			}
	return -1;
}

/*
 * Make an extension available for a given visual.
 * The extension has to be registered for that. 
 * RC: -1 for trying to attach an unkown extension
 *      x for the number of times this extension had already been
 *        registered to that visual. So
 *      0 means you installed the extension as the first one. Note that
 *    >=0 should be regarded as "success". It is legal to attach an 
 *        extension multiple times. You might want to set up private
 *        data if RC==0.
 */
int ggiExtensionAttach(ggi_visual *vis,ggi_extid id)
{
	ggi_extension *tmp=NULL;

	DPRINT_CORE("ggiExtensionAttach(%p, %d) called\n", vis, id);
	if (_ggiExtension)
		for (tmp=_ggiExtension;tmp;tmp=tmp->next)
			if (tmp->id==id)
				break;
	if ( ! tmp )
		return -1;

	if ( vis->numknownext<=id)
	{
		vis->extlist=_ggi_realloc(vis->extlist,sizeof(vis->extlist[0])*(id+1));
		memset(&vis->extlist[vis->numknownext],0,
			sizeof(vis->extlist[0])*(id+1-vis->numknownext));
		vis->numknownext=id+1;
		DPRINT_CORE("ggiExtensionAttach: ExtList now at %p (%d)\n",vis->extlist,vis->numknownext);
	}

	if ( 0==vis->extlist[id].attachcount ) 
		vis->extlist[id].private=_ggi_malloc(tmp->size);

	return vis->extlist[id].attachcount++;
}

/*
 * Destroy an extension for a given visual.
 * The extension need not be registered for that anymore, though the extid
 * has to be valid. 
 * RC: -1 for trying to detach an unkown extension
 *      x for the number of times this extension remains attached. So
 *      0 means you removed the last copy of the extension. Note that
 *    >=0 should be regarded as "success". It is legal to attach an 
 *        extension multiple times.
 */
int ggiExtensionDetach(ggi_visual *vis,ggi_extid id)
{
	DPRINT_CORE("ggiExtensionDetach(%p, %d) called\n", vis, id);
	if ( vis->numknownext<=id ||
	     0==vis->extlist[id].attachcount ) 
	     	return -1;

	if (--vis->extlist[id].attachcount) 
		return vis->extlist[id].attachcount;
	
	free(vis->extlist[id].private);
	vis->extlist[id].private=NULL;	/* Make sure ... */
	return 0;
}

int ggiIndicateChange(ggi_visual_t vis, int whatchanged)
{
	ggi_extension *tmp=NULL;

	DPRINT_CORE("ggiIndicateChange(%p, 0x%x) called\n", vis, whatchanged);

	/* tell all attached extensions on this visual */

	DPRINT_CORE("ggiIndicateChange: %i changed for %p.\n",whatchanged,vis);

	if (_ggiExtension)
		for (tmp=_ggiExtension;tmp;tmp=tmp->next)
			if (tmp->id<vis->numknownext && vis->extlist[tmp->id].attachcount)
				tmp->paramchange(vis,whatchanged);
	return 0;
}

#include <ggi/internal/ggilibinit.h>
