/* $Id: trueemu.h,v 1.12 1998/10/10 13:21:37 ajapted Exp $
***************************************************************************

   Display-trueemu: truecolor emulation target for GGI

   Copyright (C) 1998 Andrew Apted    [andrew@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#ifndef _DISPLAY_TRUEEMU_H
#define _DISPLAY_TRUEEMU_H

#include <ggi/internal/ggi-dl.h>

#include "../mansync/mansync.h"


/**************************************************
 **
 **  Trueemu private defines
 **
 **************************************************/


#define TrueColor	sint32

#define TC_RED(tc)	(((tc) >> 16) & 0xff)
#define TC_GREEN(tc)	(((tc) >>  8) & 0xff)
#define TC_BLUE(tc)	(((tc)      ) & 0xff)

#define T2GGI(tc)  { TC_RED(tc) << 8, TC_GREEN(tc) << 8, TC_BLUE(tc) << 8 }


#define BLITFUNC(def)	void def(struct trueemu_hook *th,  \
				void *dest_raw, uint8 *src, int width)

#ifndef NULL
#define NULL  ((void *) 0L)
#endif

#ifndef MIN
#define MIN(a, b)  ((a) < (b) ? (a) : (b))
#endif

#ifndef MAX
#define MAX(a, b)  ((a) > (b) ? (a) : (b))
#endif

#ifndef ABS
#define ABS(n)  (((n) < 0) ? -(n) : (n))
#endif


#define UPDATE_MOD(vis, x1, y1, w, h)  \
	do {                                                   \
		TrueemuHook *th = TRUEEMU_PRIV(vis);           \
		int x2=(x1)+(w);                               \
		int y2=(y1)+(h);                               \
                                                               \
		if ((x1) < th->dirty_tl.x) th->dirty_tl.x =    \
			MAX((x1), (vis)->gc->cliptl.x);        \
		if ((y1) < th->dirty_tl.y) th->dirty_tl.y =    \
			MAX((y1), (vis)->gc->cliptl.y);        \
		if ((x2) > th->dirty_br.x) th->dirty_br.x =    \
			MIN((x2), (vis)->gc->clipbr.x);        \
		if ((y2) > th->dirty_br.y) th->dirty_br.y =    \
			MIN((y2), (vis)->gc->clipbr.y);        \
	} while (0)


/**************************************************** 
 **
 **  TrueEmu private data
 **
 ****************************************************/


struct trueemu_hook;  /* declare forward */

typedef struct trueemu_blits
{
	BLITFUNC( (* blitter_b32_d0) );

	BLITFUNC( (* blitter_b24_d0) );
	
	BLITFUNC( (* blitter_b16_d0) );
	BLITFUNC( (* blitter_b16_d2_ev) );
	BLITFUNC( (* blitter_b16_d2_od) );
	BLITFUNC( (* blitter_b16_d4_ev) );
	BLITFUNC( (* blitter_b16_d4_od) );
	
	BLITFUNC( (* blitter_b8_d0) );
	BLITFUNC( (* blitter_b8_d2_ev) );
	BLITFUNC( (* blitter_b8_d2_od) );
	BLITFUNC( (* blitter_b8_d4_ev) );
	BLITFUNC( (* blitter_b8_d4_od) );
	
	BLITFUNC( (* blitter_b4_d0) );
	BLITFUNC( (* blitter_b4_d2_ev) );
	BLITFUNC( (* blitter_b4_d2_od) );
	BLITFUNC( (* blitter_b4_d4_ev) );
	BLITFUNC( (* blitter_b4_d4_od) );

} TrueemuBlits;


typedef struct trueemu_hook
{
	int flags;

	ggi_visual_t parent;
	ggi_mode mode;

	/* framebuffer */

	void *fb_ptr;
	long  fb_size;
	long  frame_size;
	
	/* 2D operations on memory buffer */

	struct ggi_visual_opdraw *mem_opdraw;

	ggi_coord dirty_tl;	/* dirty region */
	ggi_coord dirty_br;
	
	/* blit information */
	
	BLITFUNC( (* blitter_even) );
	BLITFUNC( (* blitter_odd) );

	void *src_buf;
	void *dest_buf;
	
	/* hicolor dither tables */

	uint16 (* R)[4];	/* actually [256][4] */
	uint16 (* G)[4];
	uint16 (* B)[4];

	/* palette dither table */

	uint8 (* T)[4];		/* actually [32768][4] */
	
	/* mansync info */

	ggi_mutex flush_lock;
	_ggi_opmansync *opmansync;

} TrueemuHook;

#define TRUEEMU_PRIV(vis)  ((TrueemuHook *) LIBGGI_PRIVATE(vis))


/* flags */

#define TRUEEMU_F_DITHER_0	0x0001
#define TRUEEMU_F_DITHER_2	0x0002
#define TRUEEMU_F_DITHER_4	0x0004

#define TRUEEMU_F_RGB		0x0010
#define TRUEEMU_F_CUBE		0x0020
#define TRUEEMU_F_PASTEL	0x0040

#define TE_DITHER_MASK  \
	(TRUEEMU_F_DITHER_0 | TRUEEMU_F_DITHER_2 | TRUEEMU_F_DITHER_4)

#define TE_PALETTE_MASK  \
	(TRUEEMU_F_RGB | TRUEEMU_F_CUBE | TRUEEMU_F_PASTEL)

/* The meaning of the various modes:
 *
 *	Dithering:
 *
 *	TRUEEMU_M_DITHER_0 : Don't dither if possible.
 *	TRUEEMU_M_DITHER_2 : Use a two pixel dither if possible.
 *	TRUEEMU_M_DITHER_4 : Use a four pixel dither if possible.
 *
 *	two pixel:	four pixel:
 *
 *	--  o-  oo	--  o-  o-  oo  oo
 *	--  -o  oo	--  --  -o  -o  oo
 *
 *	Palette:
 *
 *	TRUEEMU_M_RGB    : Use a whole number of bits for red, green and
 *			   blue.  For an 8 bit mode, this is 3:3:2.  For
 *			   a 4 bit mode, this is 1:2:1.
 *	TRUEEMU_M_CUBE   : Use the same number of reds, greens and
 *			   blues.  For an 8 bit mode, this is 6x6x6.
 *	TRUEEMU_M_PASTEL : Use a shaded pastel scheme.
 */


/* blitter tables */

extern TrueemuBlits _ggi_trueemu_blit24_table;
extern TrueemuBlits _ggi_trueemu_blit32_table;


/****************************************************
 **
 **  TrueEmu private functions
 **
 ****************************************************/
 
  
extern int _ggi_trueemu_Open(ggi_visual *vis);
extern int _ggi_trueemu_Close(ggi_visual *vis);
extern int _ggi_trueemu_Transfer(ggi_visual*vis, int x, int y, int w, int h);
extern int _ggi_trueemu_Flush(ggi_visual*vis);
extern int _ggi_trueemu_NewMode(ggi_visual *vis);


/****************************************************
 **
 **  TrueEmu internal interfaces
 **
 ****************************************************/
 

ggifunc_getmode		GGI_trueemu_getmode;
ggifunc_setmode		GGI_trueemu_setmode;
ggifunc_checkmode	GGI_trueemu_checkmode;
ggifunc_resetmode	GGI_trueemu_resetmode;
ggifunc_getapi		GGI_trueemu_getapi;
ggifunc_flush		GGI_trueemu_flush;
ggifunc_setflags	GGI_trueemu_setflags;

ggifunc_eventpoll	GGI_trueemu_eventpoll;
ggifunc_eventread	GGI_trueemu_eventread;
ggifunc_seteventmask	GGIseteventmask;

ggifunc_drawpixel_nc	GGI_trueemu_drawpixel_nc;
ggifunc_drawpixel	GGI_trueemu_drawpixel;
ggifunc_drawhline_nc	GGI_trueemu_drawhline_nc;
ggifunc_drawhline	GGI_trueemu_drawhline;
ggifunc_drawvline_nc	GGI_trueemu_drawvline_nc;
ggifunc_drawvline	GGI_trueemu_drawvline;
ggifunc_drawline	GGI_trueemu_drawline;

ggifunc_putc		GGI_trueemu_putc;
ggifunc_putpixel_nc	GGI_trueemu_putpixel_nc;
ggifunc_putpixel	GGI_trueemu_putpixel;
ggifunc_puthline	GGI_trueemu_puthline;
ggifunc_putvline	GGI_trueemu_putvline;
ggifunc_putbox		GGI_trueemu_putbox;
ggifunc_drawbox		GGI_trueemu_drawbox;
ggifunc_copybox		GGI_trueemu_copybox;
ggifunc_crossblit	GGI_trueemu_crossblit;
ggifunc_fillscreen	GGI_trueemu_fillscreen;
ggifunc_setorigin	GGI_trueemu_setorigin;

ggifunc_setdisplayframe	GGI_trueemu_setdisplayframe;
ggifunc_setreadframe	GGI_trueemu_setreadframe;
ggifunc_setwriteframe	GGI_trueemu_setwriteframe;


#define MANSYNC_init(vis)    TRUEEMU_PRIV(vis)->opmansync->init(vis)
#define MANSYNC_deinit(vis)  TRUEEMU_PRIV(vis)->opmansync->deinit(vis)
#define MANSYNC_start(vis)   TRUEEMU_PRIV(vis)->opmansync->start(vis)
#define MANSYNC_stop(vis)    TRUEEMU_PRIV(vis)->opmansync->stop(vis)
#define MANSYNC_ignore(vis)  TRUEEMU_PRIV(vis)->opmansync->ignore(vis)
#define MANSYNC_cont(vis)    TRUEEMU_PRIV(vis)->opmansync->cont(vis)


#endif  /* _DISPLAY_TRUEEMU_H */
