/* $Id: mode.c,v 1.37 1998/10/18 15:53:07 becka Exp $
***************************************************************************

   Tile target: setting modes

   Copyright (C) 1998 Steve Cheng    [steve@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "tilevisual.h"

int GGI_tile_flush_db(ggi_visual *vis, int tryflag)
{
	struct TileHooks *tilehook = LIBGGI_PRIVATE(vis);
	int i, width, height;
	ggi_visual_t currvis;

#if 0
	for(i = 0; i<tilehook->numvis; i++) {
		currvis = tilehook->vislist[i];
		width = tilehook->vis_sizes[i].x;
		height = tilehook->vis_sizes[i].y;
		
		ggiGetBox(vis, 
			tilehook->vis_origins[i].x+vis->origin_x, 
			tilehook->vis_origins[i].y+vis->origin_y,
			width, height, tilehook->buf);
		ggiPutBox(currvis, 0, 0, width, height, tilehook->buf);

#else

	/* This is a better CrossBlit. 
	*/

	int rowadd, stride;
	uint8 *buf;
		
	if(tilehook->use_db) {
		MANSYNC_ignore(vis);
	}
                        
	rowadd = (LIBGGI_PIXFMT(vis)->size+7)/8/sizeof(uint8);
	stride = tilehook->d_frame->buffer.plb.stride;
	
	for(i = 0; i<tilehook->numvis; i++) {
		currvis = tilehook->vislist[i];
		width = tilehook->vis_sizes[i].x;
		height = tilehook->vis_sizes[i].y - 1;

		buf = (uint8*)tilehook->d_frame->read +
				stride * (tilehook->vis_origins[i].y + vis->origin_y + height) +
				rowadd * (tilehook->vis_origins[i].x + vis->origin_x);

		do {
			ggiPutHLine(currvis, 0, height, width, buf);
			buf -= stride;
		} while(height--);
#endif

		_ggiInternFlush(currvis, tryflag);
	}
	
	if(tilehook->use_db) {
		MANSYNC_cont(vis);
	}

	return 0;
}

int GGI_tile_getapi(ggi_visual *vis, int num, char *apiname, char *arguments)
{
	int depth;

	switch(num) {
		case 0:
			strcpy(apiname, "display-tile");
			strcpy(arguments, "");
			return 0;
		case 1:
			strcpy(apiname, "generic-stubs");
			strcpy(arguments, "");
			return 0;
		case 2:
			if(!TILE_PRIV(vis)->use_db)
				return -1;

			switch(LIBGGI_MODE(vis)->graphtype)
			{
			  case GT_8BIT : depth=8;break;
			  case GT_15BIT: depth=15;break;
			  case GT_16BIT: depth=16;break;
			  case GT_24BIT: depth=24;break;
			  case GT_32BIT: depth=32;break;
			  case GT_TEXT16: depth=16;break;
			  case GT_TEXT32: depth=32;break;
			  default: return -1;
			}

			if(LIBGGI_MODE(vis)->graphtype==GT_TEXT16 || LIBGGI_MODE(vis)->graphtype==GT_TEXT32)
				sprintf(apiname, "generic-text-%d", depth);
			else
				sprintf(apiname, "generic-linear-%d", depth);

			strcpy(arguments, "");
			return 0;
	}

	return -1;
}


static int _GGIdomode(ggi_visual *vis)
{
	int err, id;
	char sugname[256];
	char args[256];

	_ggiZapMode(vis,~GGI_DL_OPDISPLAY);

	for(id=1;0==GGI_tile_getapi(vis,id,sugname,args);id++) {
		err=(_ggiOpenDL(vis, sugname, args, NULL)==NULL);

		if (err) {
			fprintf(stderr,"display-tile: Can't open the %s (%s) library.\n",
				sugname, args);
			return err;
		} else {
			DPRINT("Success in loading %s (%s)\n", sugname, args);
		}
	}

	if(!TILE_PRIV(vis)->use_db) {
		vis->opdraw->drawpixel_nc=GGI_tile_drawpixel_nc;
		vis->opdraw->drawpixel=GGI_tile_drawpixel;
		vis->opdraw->putpixel_nc=GGI_tile_putpixel_nc;
		vis->opdraw->putpixel=GGI_tile_putpixel;
		vis->opdraw->getpixel=GGI_tile_getpixel;

		vis->opdraw->drawhline_nc=GGI_tile_drawhline_nc;
		vis->opdraw->drawhline=GGI_tile_drawhline;
		vis->opdraw->puthline=GGI_tile_puthline;
		vis->opdraw->gethline=GGI_tile_gethline;

		vis->opdraw->drawvline_nc=GGI_tile_drawvline_nc;
		vis->opdraw->drawvline=GGI_tile_drawvline;
		vis->opdraw->putvline=GGI_tile_putvline;
		vis->opdraw->getvline=GGI_tile_getvline;

		vis->opdraw->drawbox=GGI_tile_drawbox;
		vis->opdraw->putbox=GGI_tile_putbox;
		vis->opdraw->getbox=GGI_tile_getbox;

		vis->opdraw->copybox=GGI_tile_copybox;
		vis->opdraw->fillscreen=GGI_tile_fillscreen;

		vis->opdraw->setdisplayframe=GGI_tile_setdisplayframe;
		vis->opdraw->setreadframe=GGI_tile_setreadframe;
		vis->opdraw->setwriteframe=GGI_tile_setwriteframe;

		vis->opdraw->drawline=GGI_tile_drawline;
#if 0	/* Not implemented */
		vis->opdraw->putc=GGI_tile_putc;
		vis->opdraw->puts=GGI_tile_puts;
#endif

		vis->opgc->gcchanged=GGI_tile_gcchanged;
	} 
	else {
		vis->opdraw->setdisplayframe=GGI_tile_setdisplayframe_db;
		vis->opdraw->setorigin=GGI_tile_setorigin;
	}

	vis->opcolor->mapcolor=GGI_tile_mapcolor;
	vis->opcolor->unmappixel=GGI_tile_unmappixel;
	vis->opcolor->setpalvec=GGI_tile_setpalvec;
	vis->opcolor->getpalvec=GGI_tile_getpalvec;

	ggiIndicateChange(vis, GGI_CHG_APILIST);

	return 0;
}

int GGI_tile_setmode(ggi_visual *vis,ggi_mode *tm)
{ 
	struct TileHooks *tilehook;
	int depth, i;
	/*int, currbuf, maxbuf=0;*/
	ggi_visual_t currvis;
	ggi_mode sugmode;

	if(GGI_tile_checkmode(vis, tm))
		return -1;
		
	tilehook=LIBGGI_PRIVATE(vis);

	depth = GT_SIZE(tm->graphtype);

	if(tilehook->use_db) {
		char *fbaddr;

		MANSYNC_ignore(vis);

		_GGI_tile_freedbs(vis);

		/* Set up DirectBuffer(s) */

		for(i = 0; i<tm->frames; i++) {
			fbaddr = malloc(LIBGGI_FB_SIZE(tm));

			if(!fbaddr) {
				fprintf(stderr, "display-tile: Out of memory for framebuffer!\n");
				return GGI_DL_ERROR;
			}

			_ggi_db_add_buffer(LIBGGI_APPLIST(vis), _ggi_db_get_new());

			LIBGGI_APPBUFS(vis)[i]->frame = i;
			LIBGGI_APPBUFS(vis)[i]->type = GGI_DB_NORMAL | GGI_DB_SIMPLE_PLB;
			LIBGGI_APPBUFS(vis)[i]->read = LIBGGI_APPBUFS(vis)[i]->write = fbaddr;
			LIBGGI_APPBUFS(vis)[i]->layout = blPixelLinearBuffer;
			LIBGGI_APPBUFS(vis)[i]->buffer.plb.stride
				= (GT_SIZE(tm->graphtype)+7)/8 * tm->virt.x;
		}
	}
	
	for(i = 0; i<tilehook->numvis; i++) {
		currvis = tilehook->vislist[i];
		sugmode = *tm;
		sugmode.visible.x = tilehook->vis_sizes[i].x;
		sugmode.visible.y = tilehook->vis_sizes[i].y;
		sugmode.virt.x=sugmode.virt.y=GGI_AUTO;

		/* Multiple buffering is handled by us in DB mode */
		if(tilehook->use_db)
			sugmode.frames = 1;

		DPRINT("Setting mode for visual #%d...\n", i);

		/* Set mode mantra from lib/libggi/mode.c.  Be careful here.
		   See GGIcheckmode() for why we do this. */

		_ggiLock(currvis->mutex);

		if(currvis->opdisplay->setmode(currvis, &sugmode)) {
			fprintf(stderr, "display-tile: Error setting mode on visual #%d!\n", i);
			_ggiUnlock(currvis->mutex);
			return GGI_DL_ERROR;
		}

		ggiSetDisplayFrame(currvis, 0);
		ggiSetReadFrame(currvis, 0);
		ggiSetWriteFrame(currvis, 0);
		ggiSetOrigin(currvis, 0, 0);
		ggiSetGCForeground(currvis, 0);
		ggiSetGCBackground(currvis, 0);
		ggiSetGCClipping(currvis, 0, 0, sugmode.virt.x, sugmode.virt.y);
		ggiFillscreen(currvis);

		_ggiUnlock(currvis->mutex);

		DPRINT("Success setting mode for visual #%d\n", i);

		if(!tilehook->use_db) {
			/* Adjust clipping rectangle for mode dimensions. */
		
			tilehook->vis_clipbr[i].x = tilehook->vis_origins[i].x + tilehook->vis_sizes[i].x;
			if(tilehook->vis_clipbr[i].x > tm->virt.x)
				tilehook->vis_clipbr[i].x = tm->virt.x;

			tilehook->vis_clipbr[i].y = tilehook->vis_origins[i].y + tilehook->vis_sizes[i].y;
			if(tilehook->vis_clipbr[i].y > tm->virt.y)
				tilehook->vis_clipbr[i].y = tm->virt.y;
		}

#if 0
		/* This is to determine the largest buffer size needed for copybox */
		currbuf = tilehook->vis_sizes[i].x * tilehook->vis_sizes[i].y;
		if(currbuf > maxbuf) maxbuf=currbuf;
	}

	if (tilehook->buf) {
		free(tilehook->buf);
		tilehook->buf=NULL;
	}
	
	tilehook->buf = _ggi_malloc(maxbuf*depth/8);
	if(!tilehook->buf) {
		fprintf(stderr, "display-tile: Out of memory for copybox buffer\n");
		return GGI_DL_ERROR;
#endif
	}

	/* Assume first visual's pixelformat properties */
	memcpy(LIBGGI_PIXFMT(vis), LIBGGI_PIXFMT(tilehook->vislist[0]), 
		sizeof(ggi_pixelformat));

	memcpy(LIBGGI_MODE(vis),tm,sizeof(ggi_mode));

	if(_GGIdomode(vis))
		return -1;

	if(tilehook->use_db) {
		for(i = 0; i<tm->frames; i++)
			LIBGGI_APPBUFS(vis)[i]->buffer.plb.pixelformat = LIBGGI_PIXFMT(vis);

		tilehook->d_frame = LIBGGI_APPBUFS(vis)[0];

		MANSYNC_cont(vis);
	}
	
	return 0;
}

/**********************************/
/* check any mode (text/graphics) */
/**********************************/
int GGI_tile_checkmode(ggi_visual *vis,ggi_mode *tm)
{
	int i;
	struct TileHooks *tilehook;
	ggi_mode sugmode;
	
	if (vis==NULL) {
		DPRINT("Visual==NULL\n");
		return -1;
	}
	
	tilehook=LIBGGI_PRIVATE(vis);
							
	/* Find out the "bounding rectangle" for GGI_AUTO values */
	if(tm->virt.x==GGI_AUTO) {
		int x;
		tm->virt.x=0;
		for(i = 0; i<tilehook->numvis; i++) {
			x = tilehook->vis_origins[i].x+tilehook->vis_sizes[i].x;
			if(x > tm->virt.x)
				tm->virt.x = x;
		}
	}
	if(tm->virt.y==GGI_AUTO) {
		int y;
		tm->virt.y=0;
		for(i = 0; i<tilehook->numvis; i++) {
			y = tilehook->vis_origins[i].y+tilehook->vis_sizes[i].y;
			if(y > tm->virt.y)
				tm->virt.y = y;
		}
	}

	/* We ignore visible size */
	if(tm->visible.x==GGI_AUTO)
		tm->visible.x=tm->virt.x;
	if(tm->visible.y==GGI_AUTO)
		tm->visible.y=tm->virt.y;

	if(tm->frames==GGI_AUTO)
		tm->frames = 1;

	for(i = 0; i<tilehook->numvis; i++) {
		/* Note that we don't use ggiCheckGraphMode() since we don't want 
		   GGI_AUTO to be substituted by GGI_DEFMODE values */
		   
		sugmode.frames = tilehook->use_db ? 1 : tm->frames;
		sugmode.visible.x=tilehook->vis_sizes[i].x;
		sugmode.visible.y=tilehook->vis_sizes[i].y;
		sugmode.virt.x=sugmode.virt.y=GGI_AUTO;
		sugmode.size.x=sugmode.size.y=GGI_AUTO;
		sugmode.graphtype=tm->graphtype;
		sugmode.dpp=tm->dpp;

		if(tilehook->vislist[i]->opdisplay->checkmode(tilehook->vislist[i], &sugmode)) {
			/* Forget searching all visuals for the source of error,
			   it's way too complicated. Just say fail with impossible mode */

			memset(tm, 0, sizeof(ggi_mode));

			fprintf(stderr, "display-tile: ggiCheckMode() on visual #%d error -- "
				"please explicitly specify correct mode instead.\n", i);
			return GGI_DL_ERROR;
		}
		
		/* Just hope this works :( */
		if(tm->graphtype==GGI_AUTO)
			tm->graphtype = sugmode.graphtype;
	}

	return 0;
}

/************************/
/* get the current mode */
/************************/
int GGI_tile_getmode(ggi_visual *vis,ggi_mode *tm)
{
	DPRINT("In GGIgetmode(%p,%p)\n",vis,tm);
	if (vis==NULL)
		return -1;

	memcpy(tm,LIBGGI_MODE(vis),sizeof(ggi_mode));
	return 0;
}

/*************************/
/* set the current flags */
/*************************/
int GGI_tile_setflags(ggi_visual *vis,ggi_flags flags)
{
	struct TileHooks *tilehook = TILE_PRIV(vis);

	LIBGGI_FLAGS(vis)=flags;

	if(tilehook->use_db)
		MANSYNC_SETFLAGS(vis,flags)
	else {
		int i;
		for(i = 0; i<tilehook->numvis; i++)
			ggiSetFlags(tilehook->vislist[i], flags);
	}	

	return 0;
}

int GGI_tile_setorigin(ggi_visual *vis,int x,int y)
{
	ggi_mode *mode=LIBGGI_MODE(vis);

	if ( x<0 || x> mode->virt.x-mode->visible.x ||
	     y<0 || y> mode->virt.y-mode->visible.y )
	     return -1;

	vis->origin_x=x;
	vis->origin_y=y;
	
	ggiFlush(vis);

	return 0;
}
