/* $Id: mode.c,v 1.44 1998/10/30 05:09:51 ajapted Exp $
***************************************************************************

   Display-FBDEV

   Copyright (C) 1998 Andrew Apted    [andrew@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <dlfcn.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>

#include <sys/ioctl.h>
#include <sys/mman.h>

#include <linux/fb.h>
#include <linux/vt.h>
#include <linux/kd.h>

#include <ggi/internal/ggi-dl.h>

#include "fbdev.h"

#include "../common/ggi-auto.inc"
#include "../common/gt-auto.inc"

#include "../Linux_common/vtswitch.inc"

#define FB_KLUDGE_FONTX  8
#define FB_KLUDGE_FONTY  16
#define FB_KLUDGE_TEXTMODE  13


static void _GGI_free_dbs(ggi_visual *vis) 
{
	int i;

	for (i=LIBGGI_APPLIST(vis)->num-1; i >= 0; i--) {
		_ggi_db_free(LIBGGI_APPBUFS(vis)[i]);
		_ggi_db_del_buffer(LIBGGI_APPLIST(vis), i);
	}
}

int GGI_fbdev_getapi(ggi_visual *vis, int num, char *apiname, char *arguments)
{
	fbdev_hook *ff = LIBGGI_PRIVATE(vis);

	strcpy(arguments, "");

	switch(num) {

	case 0: strcpy(apiname, "display-fbdev");
		return 0;

	case 1: strcpy(apiname, "generic-stubs");
		return 0;

	case 2: strcpy(apiname, "generic-color");
		return 0;

	case 3: if (GT_SCHEME(LIBGGI_GT(vis)) == GT_TEXT) {
			sprintf(apiname, "generic-text-%d",
				GT_SIZE(LIBGGI_GT(vis)));
			return 0;
		} 
		
		if (ff->fix.type == FB_TYPE_PLANES) {
			strcpy(apiname, "generic-planar");
			return 0;
		}
		
		if (ff->fix.type == FB_TYPE_INTERLEAVED_PLANES) {
			sprintf(apiname, "generic-%s",
				(ff->fix.type_aux == 2) ? 
				"iplanar-2p" : "ilbm");
			return 0;
		}
		
		sprintf(apiname, "generic-linear-%d", 
			GT_SIZE(LIBGGI_GT(vis)));
		return 0;
		
	case 4:
		if (ff->have_kgi) {
			strcpy(apiname, "generic-kgi");
			return 0;
		}
		break;
	}

	return -1;
}


static int do_change_mode(ggi_visual *vis)
{
	fbdev_hook *ff = LIBGGI_PRIVATE(vis);

	ggi_mode *mode = LIBGGI_MODE(vis);
	ggi_graphtype gt = mode->graphtype;

	int err;


	/* setup var structure */

	ff->var = ff->orig_var;

	ff->var.xres = mode->visible.x * mode->dpp.x;
	ff->var.yres = mode->visible.y * mode->dpp.y;
	ff->var.xres_virtual = mode->virt.x * mode->dpp.x;
	ff->var.yres_virtual = mode->virt.y * mode->dpp.y * mode->frames;
	
	ff->var.activate = FB_ACTIVATE_NOW;
	ff->var.nonstd = 0;
	ff->var.grayscale = (GT_SCHEME(gt) == GT_GREYSCALE) ? 1 : 0;

	switch (GT_SCHEME(gt)) {

		case GT_GREYSCALE:
		case GT_PALETTE:
			ff->var.bits_per_pixel = GT_DEPTH(gt);
			break;
			
		case GT_TRUECOLOR:
			if (GT_DEPTH(gt) <= 16) {
				ff->var.bits_per_pixel = GT_DEPTH(gt);
			} else {
				ff->var.bits_per_pixel = GT_SIZE(gt);
			}
			break;

		case GT_TEXT: /* Oh this is such a hack */
			ff->var.bits_per_pixel = FB_KLUDGE_TEXTMODE;
			break;
	}
	

	/* tell framebuffer to change mode */

	if (ioctl(LIBGGI_FD(vis), FBIOPUT_VSCREENINFO, &ff->var) < 0) {
		perror("display-fbdev: FBIOPUT_VSCREENINFO");
		return -1;
	}

	if (ioctl(LIBGGI_FD(vis), FBIOGET_FSCREENINFO, &ff->fix) < 0) {
		perror("display-fbdev: FBIOGET_FSCREENINFO");
		return -1;
	}


	/* check whether we got what we asked for */

	err = 0;

	switch (GT_SCHEME(gt)) {

		case GT_GREYSCALE:
			if (! ff->var.grayscale ||
			    GT_DEPTH(gt) != ff->var.bits_per_pixel ||
#ifdef FB_TYPE_TEXT
			    ff->fix.type == FB_TYPE_TEXT ||
#endif
			    ff->fix.visual == FB_VISUAL_TRUECOLOR ||
			    ff->fix.visual == FB_VISUAL_DIRECTCOLOR)
				err--;
			break;
	
		case GT_PALETTE:
			if (GT_DEPTH(gt) != ff->var.bits_per_pixel ||
#ifdef FB_TYPE_TEXT
			    ff->fix.type == FB_TYPE_TEXT ||
#endif
			    ff->fix.visual == FB_VISUAL_TRUECOLOR ||
			    ff->fix.visual == FB_VISUAL_DIRECTCOLOR)
				err--;
			break;

		case GT_TRUECOLOR:
			if ((GT_SIZE(gt)  != ff->var.bits_per_pixel &&
			     GT_DEPTH(gt) != ff->var.bits_per_pixel) ||
#ifdef FB_TYPE_TEXT
			    ff->fix.type  == FB_TYPE_TEXT ||
#endif
			    (ff->fix.visual != FB_VISUAL_TRUECOLOR &&
			     ff->fix.visual != FB_VISUAL_DIRECTCOLOR))
				err--;
			break;
		
#ifdef FB_TYPE_TEXT
		case GT_TEXT:
			if (ff->fix.type != FB_TYPE_TEXT)
#endif
				err--;
	}
	
	if (err) {
		DPRINT_MODE("display-fbdev: Santa passed us by :-(\n");
	} else {
		DPRINT_MODE("display-fbdev: Change mode OK.\n");
	}
	

	return err;
}


static int do_mmap(ggi_visual *vis)
{
	fbdev_hook *ff = LIBGGI_PRIVATE(vis);

	ggi_graphtype gt;

	int i;


        /* calculate framebuffer size */
	
	gt = LIBGGI_MODE(vis)->graphtype;

	ff->frame_size = LIBGGI_VIRTX(vis) * LIBGGI_VIRTY(vis);
	ff->frame_size = (ff->frame_size * GT_SIZE(gt) + 7) / 8;

	ff->fb_size = ff->frame_size * LIBGGI_MODE(vis)->frames;
	ff->mmap_size = (ff->fb_size + 0x0fff) & ~0x0fff;
	
	DPRINT_MODE("display-fbdev: frame_size=0x%x fb_size=0x%x "
		    "mmap_size=0x%x\n", ff->frame_size,
		    ff->fb_size, ff->mmap_size);

	ff->fb_ptr = mmap(NULL, ff->mmap_size, PROT_READ | PROT_WRITE, 
			  MAP_SHARED, LIBGGI_FD(vis), 0);

	DPRINT_MODE("display-fbdev: FB_PTR=%p\n", ff->fb_ptr);

	if (ff->fb_ptr == (void *)(-1)) {
		ff->fb_ptr = NULL;
		return -1;
	}

	
	/* clear all frames */

	memset(ff->fb_ptr, 0, ff->fb_size);
	

	/* set up pixel format */

	LIBGGI_PIXFMT(vis)->size  = GT_SIZE(gt);
	LIBGGI_PIXFMT(vis)->depth = GT_DEPTH(gt);

	switch (GT_SCHEME(gt)) {

	case GT_PALETTE:
	case GT_GREYSCALE:
		LIBGGI_PIXFMT(vis)->clut_mask = (1 << GT_DEPTH(gt)) - 1;
		break;

	case GT_TRUECOLOR:
		DPRINT_MODE("fbdev: RGB %d:%d:%d offsets %d:%d:%d\n",
			ff->var.red.length, ff->var.green.length,
			ff->var.blue.length, ff->var.red.offset,
			ff->var.green.offset, ff->var.blue.offset);

		LIBGGI_PIXFMT(vis)->red_mask =
		((1 << ff->var.red.length) - 1) << ff->var.red.offset;
			
		LIBGGI_PIXFMT(vis)->green_mask =
		((1 << ff->var.green.length) - 1) << ff->var.green.offset;
			
		LIBGGI_PIXFMT(vis)->blue_mask =
		((1 << ff->var.blue.length) - 1) << ff->var.blue.offset;
		break;

	case GT_TEXT:
		/* Assumes VGA text */
		LIBGGI_PIXFMT(vis)->texture_mask = 0x00ff;
		LIBGGI_PIXFMT(vis)->fg_mask = 0x0f00;
		LIBGGI_PIXFMT(vis)->bg_mask = 0xf000;
		break;
	}


	/* set up direct buffers */

	for (i=0; i < LIBGGI_MODE(vis)->frames; i++) {

		ggi_directbuffer *buf;

		_ggi_db_add_buffer(LIBGGI_APPLIST(vis), _ggi_db_get_new());

		buf = LIBGGI_APPBUFS(vis)[i];

		buf->frame = i;
		buf->type  = GGI_DB_NORMAL;
		buf->read  = (uint8 *) ff->fb_ptr + i * ff->frame_size;
		buf->write = buf->read;

		if (ff->fix.type == FB_TYPE_PLANES) {
		
			buf->layout = blPixelPlanarBuffer;

			if (ff->fix.line_length) {
				buf->buffer.plan.next_line = 
					ff->fix.line_length;
			} else {
				buf->buffer.plan.next_line = 
					ff->var.xres_virtual/8;
			}
			buf->buffer.plan.next_plane = 
				ff->fix.line_length * LIBGGI_VIRTY(vis);
			buf->buffer.plan.pixelformat = LIBGGI_PIXFMT(vis);

		} else if (ff->fix.type == FB_TYPE_INTERLEAVED_PLANES) {

			buf->layout = blPixelPlanarBuffer;

			if (ff->fix.line_length) {
				buf->buffer.plan.next_line = 
					ff->fix.line_length *
					ff->var.bits_per_pixel;
			} else {
				buf->buffer.plan.next_line = 
					ff->var.xres_virtual *
					ff->var.bits_per_pixel/8;
			}
			buf->buffer.plan.next_plane = ff->fix.type_aux;
			buf->buffer.plan.pixelformat = LIBGGI_PIXFMT(vis);

		} else {

			buf->type  |= GGI_DB_SIMPLE_PLB;
			buf->layout = blPixelLinearBuffer;
			buf->buffer.plb.stride = ff->fix.line_length;
			buf->buffer.plb.pixelformat = LIBGGI_PIXFMT(vis);
		}
	}

	return 0;
}


static int do_setmode(ggi_visual *vis)
{
	fbdev_hook *ff = LIBGGI_PRIVATE(vis);
	int err, id;

	char libname[256], libargs[256];

	ggi_graphtype gt;


	err = do_change_mode(vis);

	if (err) {
		return err;
	}

	err = do_mmap(vis); 
	
	if (err) {
		return err;
	}

	for(id=1; GGI_fbdev_getapi(vis, id, libname, libargs) == 0; id++) {

		if (_ggiOpenDL(vis, libname, libargs, NULL) == NULL) {

			fprintf(stderr,"display-fbdev: Error opening the "
				"%s (%s) library.\n", libname, libargs);
			return -1;
		}

		DPRINT_LIBS("Success in loading %s (%s)\n", 
			    libname, libargs);
	}

	if (_ggiOpenDL(vis, "generic-kgi", NULL, NULL) != NULL) {
		ff->have_kgi = 1;
	} else {
		LIBGGI_GC(vis) = ff->normalgc;
		ff->have_kgi = 0;
	}

	/* set up palette */

	gt = LIBGGI_GT(vis);

	if ((GT_SCHEME(gt) == GT_PALETTE) || (GT_SCHEME(gt) == GT_TEXT)) {

	    	int nocols = 1 << GT_DEPTH(gt);

		vis->palette = _ggi_malloc(nocols * sizeof(ggi_color));

		vis->opcolor->setpalvec = GGI_fbdev_setpalvec;
	}
	
	vis->opdraw->setorigin = GGI_fbdev_setorigin;
	vis->opdraw->setdisplayframe = GGI_fbdev_setdisplayframe;

	ggiIndicateChange(vis, GGI_CHG_APILIST);

	DPRINT_MODE("display-fbdev: do_setmode SUCCESS\n");

	return 0;
}


int GGI_fbdev_setmode(ggi_visual *vis, ggi_mode *mode)
{ 
	fbdev_hook *ff = LIBGGI_PRIVATE(vis);

	int err;


        if ((err = ggiCheckMode(vis, mode)) != 0) {
		return err;
	}

	_GGI_free_dbs(vis);

	if (ff->fb_ptr != NULL) {
		munmap(ff->fb_ptr, ff->mmap_size);
		ff->fb_ptr = NULL;
	}
	

	DPRINT_MODE("display-fbdev: setmode %dx%d#%dx%dF%d[0x%02x]\n",
			mode->visible.x, mode->visible.y,
			mode->virt.x, mode->virt.y, 
			mode->frames, mode->graphtype);

	memcpy(LIBGGI_MODE(vis), mode, sizeof(ggi_mode));

	_ggiZapMode(vis, ~GGI_DL_OPDISPLAY);

	
	/* setup VT, framebuffer, and mmapping */

	err = vtswitch_open(vis); 
	
	if (err < 0) {
		return err;
	}

	_GGI_fbdev_save_palette(vis);
	_GGI_fbdev_save_panning(vis);


	/* now actually set the mode */

	if (do_setmode(vis) != 0) {
		vtswitch_close(vis);
		return -1;
	}

	DPRINT_MODE("display-fbdev: setmode Success.\n");

	return 0;
}


int GGI_fbdev_resetmode(ggi_visual *vis)
{
	fbdev_hook *ff = LIBGGI_PRIVATE(vis);

	if (ff->fb_ptr != NULL) {
	
		_GGI_free_dbs(vis);

		munmap(ff->fb_ptr, ff->mmap_size);
		ff->fb_ptr = NULL;

		vtswitch_close(vis);
	}

	ioctl(LIBGGI_FD(vis), FBIOPUT_VSCREENINFO, &ff->orig_var);

	_GGI_fbdev_restore_palette(vis);
	_GGI_fbdev_restore_panning(vis);

	return 0;
}


int GGI_fbdev_checkmode(ggi_visual *vis, ggi_mode *mode)
{
	fbdev_hook *ff = LIBGGI_PRIVATE(vis);

	ggi_graphtype gt = mode->graphtype;

	int err;


	DPRINT_MODE("display-fbdev: checkmode %dx%d#%dx%dF%d[0x%02x]\n",
			mode->visible.x, mode->visible.y,
			mode->virt.x, mode->virt.y, 
			mode->frames, mode->graphtype);

	
	/* handle GT_AUTO in graphtype */

	if (GT_SCHEME(gt) == GT_AUTO) {

#ifdef FB_TYPE_TEXT
		if (ff->orig_fix.type == FB_TYPE_TEXT)
			GT_SETSCHEME(gt, GT_TEXT);
		else
#endif
		switch (ff->orig_fix.visual) {

		case FB_VISUAL_MONO01:
		case FB_VISUAL_MONO10:
			GT_SETSCHEME(gt, GT_GREYSCALE);
			break;

		case FB_VISUAL_PSEUDOCOLOR:
		case FB_VISUAL_STATIC_PSEUDOCOLOR:
			GT_SETSCHEME(gt, ff->orig_var.grayscale ? 
				GT_GREYSCALE : GT_PALETTE);
			break;

		case FB_VISUAL_TRUECOLOR:
		case FB_VISUAL_DIRECTCOLOR:
			GT_SETSCHEME(gt, GT_TRUECOLOR);
			break;

		default:
			fprintf(stderr, "display-fbdev: WARNING: unknown "
				"visual (0x%02x) of framebuffer.\n",
				ff->orig_fix.visual);
			break;
		}
	}

	if (GT_DEPTH(gt) == GT_AUTO) {

		if ((GT_SCHEME(gt) == GT_TRUECOLOR) &&
		    (ff->orig_fix.visual == FB_VISUAL_TRUECOLOR ||
		     ff->orig_fix.visual == FB_VISUAL_TRUECOLOR)) {

			GT_SETDEPTH(gt, ff->orig_var.red.length   +
					ff->orig_var.green.length +
					ff->orig_var.blue.length);
		} else {
			GT_SETDEPTH(gt, ff->orig_var.bits_per_pixel);
		}
	}

	mode->graphtype = _GGIhandle_gtauto(gt);


	/* handle GGI_AUTO in ggi_mode */

	if (mode->dpp.x == GGI_AUTO) {
		mode->dpp.x = (GT_SCHEME(mode->graphtype) == GT_TEXT) ?
		     FB_KLUDGE_FONTX : 1;
	}

	if (mode->dpp.y == GGI_AUTO) {
		mode->dpp.y = (GT_SCHEME(mode->graphtype) == GT_TEXT) ?
		     FB_KLUDGE_FONTY : 1;
	}

	_GGIhandle_ggiauto(mode, ff->orig_var.xres / mode->dpp.x,
				 ff->orig_var.yres / mode->dpp.y);


	/* now check stuff */

	err = 0;

	if ((mode->visible.x <= 0) || (mode->visible.y <= 0) ||
	    (mode->virt.x    <= 0) || (mode->virt.y    <= 0) ||
	    (GT_SIZE(mode->graphtype) < GT_DEPTH(mode->graphtype))) {

		DPRINT("display-fbdev: checkmode: Bad Geometry.\n");
		return -1;
	}

	if (mode->virt.x < mode->visible.x) {
		mode->virt.x = mode->visible.x;
		err--;
	}
		
	if (mode->virt.y < mode->visible.y) {
		mode->virt.y = mode->visible.y;
		err--;
	}
	
#ifndef VESAFB_IS_NO_LONGER_BROKEN
	if (mode->frames > 1) {
		mode->frames = 1;
		err--;
	}
#endif

	/* Now use FB_ACTIVATE_TEST to let the framebuffer driver check
	 * & upgrade the mode.  !!! IMPLEMENT ME
	 */

	DPRINT_MODE("display-fbdev: result %d %dx%d#%dx%dF%d[0x%02x]\n",
			err, mode->visible.x, mode->visible.y,
			mode->virt.x, mode->virt.y, 
			mode->frames, mode->graphtype);

	return err;
}


int GGI_fbdev_getmode(ggi_visual *vis, ggi_mode *mode)
{
	DPRINT_MODE("display-fbdev: getmode\n");
	
	memcpy(mode, LIBGGI_MODE(vis), sizeof(ggi_mode));

	return 0;
}


int GGI_fbdev_setflags(ggi_visual *vis, ggi_flags flags)
{
	LIBGGI_FLAGS(vis) = flags;

	return 0;
}


/* ----- Routines needed by Linux_common/vtswitch.inc ----- */


void handle_switched_away(ggi_visual *vis)
{
	/* Nothing to do.  
	 * Fbcon will take care of the hardware for us
	 */
}


void handle_switched_back(ggi_visual *vis)
{
	fbdev_hook *ff = LIBGGI_PRIVATE(vis);

	/* Nothing to do.  
	 * Fbcon will take care of the hardware for us
	 */

	ff->need_redraw = 1;
}
