// Fl_Menu_.C

// This is a base class for all items that have a menu:
//	Fl_Menu_Bar, Fl_Menu_Button, Fl_Choice
// This provides storage for a menu item, functions to add/modify/delete
// items, and a call for when the user picks a menu item.

// More code in Fl_Menu_add.C

#include <FL/Fl.H>
#include <FL/Fl_Menu_.H>
#include <stdlib.h>

int Fl_Menu_::value(const Fl_Menu *m) {
  clear_changed();
  if (value_ != m) {value_ = m; return 1;}
  return 0;
}

// When user picks a menu item, call this.  It will do the callback.
// Unfortunatly this also casts away const for the checkboxes, but this
// was necessary so non-checkbox menus can really be declared const...
void Fl_Menu_::picked(const Fl_Menu *v) {
  if (!v) return;
  if (v->radio()) {
    if (!v->value()) { // they are turning on a radio item
      set_changed();
      ((Fl_Menu*)v)->setonly();
    }
  } else if (v->checkbox()) { // toggling check box
    set_changed();
    ((Fl_Menu*)v)->flags ^= FL_MENU_CHECK;
  } else if (v != value_) { // normal item
    set_changed();
  }
  value_ = v;
  if (when()&(FL_WHEN_CHANGED|FL_WHEN_RELEASE)) {
    if (changed() || when()&FL_WHEN_NOT_CHANGED) {
      clear_changed();
      if (value_ && value_->callback_) value_->do_callback((Fl_Widget*)this);
      else do_callback();
    }
  }
}

// turn on one of a set of radio buttons
void Fl_Menu::setonly() {
  flags |= FL_MENU_RADIO | FL_MENU_CHECK;
  Fl_Menu* j;
  for (j = this; ; ) {	// go down
    if (j->flags & FL_MENU_DIVIDER) break; // stop on divider lines
    j++;
    if (!j->text || !j->radio()) break; // stop after group
    j->clear();
  }
  for (j = this-1; ; j--) { // go up
    if (!j->text || (j->flags&FL_MENU_DIVIDER) || !j->radio()) break;
    j->clear();
  }
}

Fl_Menu_::Fl_Menu_(int X,int Y,int W,int H,const char *l)
: Fl_Widget(X,Y,W,H,l) {
  set_flag(SHORTCUT_LABEL);
  box(FL_UP_BOX);
  when(FL_WHEN_RELEASE|FL_WHEN_NOT_CHANGED);
  value_ = menu_ = 0;
  alloc = 0;
  textfont(FL_HELVETICA+FL_BOLD+FL_ITALIC);
  textsize(FL_NORMAL_SIZE);
  textcolor(FL_BLACK);
}

void Fl_Menu_::menu(const Fl_Menu *m) {
  clear();
  if (alloc) delete[] menu_;
  alloc = 0;
  value_ = menu_ = (Fl_Menu *)m;
}

void Fl_Menu_::clear() {
  if (alloc) {
    for (int i = size(); i--;)
      if (menu_[i].text) free((void *)menu_[i].text);
    menu_[0].text = 0;
    value_ = menu_;
  }
}

int Fl_Menu_::size() const {
  if (!menu_) return 0;
  return menu_->size();
}

Fl_Menu_::~Fl_Menu_() {
  clear();
  if (alloc) delete[] menu_;
}

// end of Fl_Menu_.C
