// Fl_Group.C

// The Fl_Group is the only defined container type in fltk.

// Fl_Window itself is a subclass of this, and most of the event
// handling is designed so windows themselves work correctly.

#include <FL/Fl.H>
#include <FL/Fl_Group.H>
#include <FL/Fl_Window.H>
#include <FL/fl_draw.H>
#include <stdlib.h>

Fl_Group* Fl_Group::current_;

// Hack: A single child is stored in the pointer to the array, while
// multiple children are stored in an allocated array:
Fl_Widget*const* Fl_Group::array() const {
  return children_ <= 1 ? (Fl_Widget**)(&array_) : array_;
}

int Fl_Group::find(const Fl_Widget* o) const {
  Fl_Widget*const* a = array();
  int i; for (i=0; i < children_; i++) if (*a++ == o) break;
  return i;
}

int Fl_Group::handle(int event) {

  Fl_Widget*const* a = array();
  int i;
  Fl_Widget* o;

  switch (event) {

  case FL_FOCUS:
    if (savedfocus_ && savedfocus_->take_focus()) return 1;
    for (i = children(); i--;) if ((*a++)->take_focus()) return 1;
    return 0;

  case FL_UNFOCUS:
    o = Fl::focus();
    while (o && o->parent() != this) o = o->parent();
    savedfocus_ = o;
    return 0;

  case FL_KEYBOARD:
    return navigation();

  case FL_SHORTCUT:
    for (i = children(); i--;) {
      o = a[i];
      if (o->activevisible() && Fl::event_inside(o) && o->handle(FL_SHORTCUT))
	return 1;
    }
    for (i = children(); i--;) {
      o = a[i];
      if (o->activevisible() && !Fl::event_inside(o) && o->handle(FL_SHORTCUT))
	return 1;
    }
    if (Fl::event_key() == FL_Enter) return navigation(FL_Down);
    return 0;

  case FL_ENTER:
  case FL_MOVE:
    for (i = children(); i--;) {
      o = a[i];
      if (o->activevisible() && Fl::event_inside(o)) {
	if (o->contains(Fl::belowmouse())) {
	  return o->handle(FL_MOVE);
	} else if (o->handle(FL_ENTER)) {
	  if (!o->contains(Fl::belowmouse())) Fl::belowmouse(o);
	  return 1;
	}
      }
    }
    Fl::belowmouse(this);
    return 1;

  case FL_PUSH:
    for (i = children(); i--;) {
      o = a[i];
      if (o->activevisible() && Fl::event_inside(o)) {
	if (o->handle(FL_PUSH)) {
	  if (Fl::pushed() && !o->contains(Fl::pushed())) Fl::pushed(o);
	  return 1;
	}
      }
    }
    return 0;

  case FL_DEACTIVATE:
  case FL_ACTIVATE:
    for (i = children(); i--;) {
      o = *a++;
      if (o->active()) o->handle(event);
    }
    return 1;

  case FL_HIDE:
  case FL_SHOW:
    for (i = children(); i--;) {
      o = *a++;
      if (o->visible()) o->handle(event);
    }
    return 1;

  default:
    return 0;

  }
}

// translate the current keystroke into up/down/left/right for navigation:
#define ctrl(x) (x^0x40)
int navkey() {
  switch (Fl::event_key()) {
  case FL_Tab:
    return (Fl::event_state(FL_SHIFT) ? FL_Left : FL_Right);
  case FL_Right:
    return FL_Right;
  case FL_Left:
    return FL_Left;
  case FL_Up:
    return FL_Up;
  case FL_Down:
    return FL_Down;
  default:
    switch (Fl::event_text()[0]) {
    case ctrl('N') : return FL_Down;
    case ctrl('P') : return FL_Up;
    case ctrl('F') : return FL_Right;
    case ctrl('B') : return FL_Left;
    }
  }
  return 0;
}

//void Fl_Group::focus(Fl_Widget *o) {Fl::focus(o); o->handle(FL_FOCUS);}

#if 0
const char *nameof(Fl_Widget *o) {
  if (!o) return "NULL";
  if (!o->label()) return "<no label>";
  return o->label();
}
#include <stdio.h>
#endif

// try to move the focus in response to a keystroke:
int Fl_Group::navigation(int key) {
  if (children() <= 1) return 0;
  if (!key) {key = navkey(); if (!key) return 0;}
  Fl_Widget *focus_ = Fl::focus();
  int old_i;
  for (old_i=0;;old_i++) {
    if (old_i >= children_) return 0;
    if (array_[old_i]->contains(focus_)) break;
  }
  int i = old_i;

  for (;;) {
    switch (key) {
    case FL_Right:
    case FL_Down:
      i++; if (i >= children_) i = 0;
      break;
    case FL_Left:
    case FL_Up:
      if (i) i--; else i = children_-1;
      break;
    default:
      return 0;
    }
    if (i == old_i) return 0;
    Fl_Widget* o = array_[i];
    switch (key) {
    case FL_Down:
    case FL_Up:
      if (o->x() >= focus_->x()+focus_->w() ||
	  o->x()+o->w() <= focus_->x()) continue;
    }
    if (o->take_focus()) return 1;
  }
}

////////////////////////////////////////////////////////////////

Fl_Group::Fl_Group(int X,int Y,int W,int H,const char *l)
: Fl_Widget(X,Y,W,H,l) {
  type(FL_GROUP);
  align(FL_ALIGN_TOP);
  children_ = 0;
  array_ = 0;
  savedfocus_ = 0;
  resizable_ = this;
  gix = X; giy = Y; giw = W; gih = H;
  // Subclasses may want to construct child objects as part of their
  // constructor, so make sure they are add()'d to this object.
  // But you must end() the object!
  begin();
}

Fl_Group::~Fl_Group() {
  Fl_Widget*const* a = array();
  for (int i=children(); i--;) {
    Fl_Widget* o = *a++;
    // test the parent to see if child already destructed:
    if (o->parent() == this) delete o;
  }
  if (children() > 1) free((void*)array_);
}

void Fl_Group::insert(Fl_Widget &o, int i) {
  if (o.parent()) return;	// ignore double adds
  o.parent_ = this;
  if (children_ == 0) { // use array pointer to point at single child
    array_ = (Fl_Widget**)&o;
  } else if (children_ == 1) { // go from 1 to 2 children
    Fl_Widget* t = (Fl_Widget*)array_;
    array_ = (Fl_Widget**)malloc(2*sizeof(Fl_Widget*));
    array_[!i] = t; array_[i] = &o;
  } else {
    if (!(children_ & (children_-1))) // double number of children
      array_ = (Fl_Widget**)realloc((void*)array_,
				    2*children_*sizeof(Fl_Widget*));
    for (int j = children_; j > i; j--) array_[j] = array_[j-1];
    array_[i] = &o;
  }
  children_++;
}

void Fl_Group::add(Fl_Widget &o) {insert(o, children_);}

void Fl_Group::remove(Fl_Widget &o) {
  int i = find(o);
  if (i >= children_) return;
  if (&o == savedfocus_) savedfocus_ = 0;
  o.parent_ = 0;
  children_--;
  if (children_ == 1) { // go from 2 to 1 child
    Fl_Widget *t = array_[!i];
    free((void*)array_);
    array_ = (Fl_Widget**)t;
  } else if (children_ > 1) { // delete from array
    for (; i < children_; i++) array_[i] = array_[i+1];
  }
}

////////////////////////////////////////////////////////////////

// Rather lame kludge here, I need to detect windows and ignore the
// changes to X,Y, since all children are relative to X,Y.  That
// is why I check type():

void Fl_Group::resize(int X, int Y, int W, int H) {

  Fl_Widget::resize(X,Y,W,H);
  if (!children_) return;

  int dx = X - gix;
  int dy = Y - giy;
  if (type() >= FL_WINDOW) dx = dy = 0;

  if (!resizable()) {
    Fl_Widget*const* a = array();
    for (int i=children_; i--;) {
      Fl_Widget* o = *a++;
      o->resize(o->ix()+dx, o->iy()+dy, o->iw(), o->ih());
    }
    return;
  }

  int dw = W - giw;
  int dh = H - gih;

  // get the initial size of resizebox:
  int ox, oy, ow, oh;
  if (resizable() == this) {
    if (type() >= FL_WINDOW) ox = oy = 0;
    else {ox = gix; oy = giy;}
    ow = giw;
    oh = gih;
  } else {
    ox = resizable()->ix();
    oy = resizable()->iy();
    ow = resizable()->iw();
    oh = resizable()->ih();
  }

  Fl_Widget*const* a = array();
  for (int i=children_; i--;) {
    Fl_Widget* o = *a++;

    // most of these extra if's are to make resizable()'s that
    // start out with zero dimensions expand and have the other
    // objects expand "across" them, rather than being pushed
    // away by them:

    int X = o->ix();
    if (X > ox+ow || X == ox+ow && (ow || o->iw()>0)) X += dw;
    else if (X > ox) X = ox+((X-ox)*(ow+dw)+ow/2)/ow;

    int R = o->ix() + o->iw();
    if (R >= ox+ow) R += dw;
    else if (R > ox) R = ox+((R-ox)*(ow+dw)+ow/2)/ow;

    int Y = o->iy();
    if (Y > oy+oh || Y == oy+oh && (oh || o->ih()>0)) Y += dh;
    else if (Y > oy) Y = oy+((Y-oy)*(oh+dh)+oh/2)/oh;

    int B = o->iy() + o->ih();
    if (B >= oy+oh) B += dh;
    else if (B > oy) B = oy+((B-oy)*(oh+dh)+oh/2)/oh;

    o->resize(X+dx, Y+dy, R-X, B-Y);
  }
}
  
void Fl_Group::draw() {
  Fl_Widget*const* a = array();
  if (damage() & ~1) { // redraw the entire thing:
    draw_box();
    draw_label();
    for (int i=children_; i--;) {
      Fl_Widget& o = **a++;
      draw_child(o);
      draw_outside_label(o);
    }
  } else {	// only redraw the children that need it:
    for (int i=children_; i--;) update_child(**a++);
  }
}

// Draw a child only if it needs it:
void Fl_Group::update_child(Fl_Widget& w) const {
  if (w.damage() && w.visible() && w.type() < FL_WINDOW &&
      fl_not_clipped(w.x(), w.y(), w.w(), w.h())) {
    w.draw();	
    w.clear_damage();
  }
}

// Force a child to redraw:
void Fl_Group::draw_child(Fl_Widget& w) const {
  if (w.visible() && w.type() < FL_WINDOW &&
      fl_not_clipped(w.x(), w.y(), w.w(), w.h())) {
    w.clear_damage(~0);
    w.draw();
    w.clear_damage();
  }
}

extern char fl_draw_shortcut;

// Parents normally call this to draw outside labels:
void Fl_Group::draw_outside_label(const Fl_Widget& w) const {
  if (!w.visible()) return;
  // skip any labels that are inside the widget:
  if (!(w.align()&15) || (w.align() & FL_ALIGN_INSIDE)) return;
  // invent a box that is outside the widget:
  uchar align = w.align();
  int X = w.x();
  int Y = w.y();
  int W = w.w();
  int H = w.h();
  if (align & FL_ALIGN_TOP) {
    align ^= (FL_ALIGN_BOTTOM|FL_ALIGN_TOP);
    Y = y();
    H = w.y()-Y;
  } else if (align & FL_ALIGN_BOTTOM) {
    align ^= (FL_ALIGN_BOTTOM|FL_ALIGN_TOP);
    Y = Y+H;
    H = y()+h()-Y;
  } else if (align & FL_ALIGN_LEFT) {
    align ^= (FL_ALIGN_LEFT|FL_ALIGN_RIGHT);
    X = x();
    W = w.x()-X-3;
  } else if (align & FL_ALIGN_RIGHT) {
    align ^= (FL_ALIGN_LEFT|FL_ALIGN_RIGHT);
    X = X+W+3;
    W = x()+this->w()-X;
  }
  w.draw_label(X,Y,W,H,align);
}

