// Fl_Menuitem_Type.C

// Menu items are kludged by making a phony Fl_Box widget so the normal
// widget panel can be used to control them.

// This file also contains code to make Fl_Menu_Button, Fl_Menu_Bar,
// etc widgets.

#include <FL/Fl.H>
#include "Fl_Widget_Type.H"
#include <FL/fl_message.H>
#include <FL/Fl_Menu_.H>
#include <FL/Fl_Button.H>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

static Fl_Menu menu_item_type_menu[] = {
  {"Normal",0,0,(void*)0},
  {"Toggle",0,0,(void*)FL_MENU_BOX},
  {"Radio",0,0,(void*)FL_MENU_RADIO},
  {0}};

class Fl_Menuitem_Type : public Fl_Widget_Type {
public:
  Fl_Menu *subtypes() {return menu_item_type_menu;}
  const char* type_name() {return "menuitem";}
  Fl_Type* make();
  int is_menu_item() {return 1;}
  int is_button() {return 1;} // this gets shortcut to work
  Fl_Widget* widget(int,int,int,int) {return 0;}
  Fl_Widget_Type* _make() {return 0;}
  void write_declare();
  const char* menu_name(int& i);
  int flags();
  void write_static();
  void write_item();
  void write_code1();
  void write_code2();
};

class Fl_Submenu_Type : public Fl_Menuitem_Type {
public:
  Fl_Menu *subtypes() {return 0;}
  const char* type_name() {return "submenu";}
  int is_parent() {return 1;}
  int is_button() {return 0;} // disable shortcut
  Fl_Type* make();
  // changes to submenu must propagate up so build_menu is called
  // on the parent Fl_Menu_Type:
  void add_child(Fl_Type*a, Fl_Type*b) {parent->add_child(a,b);}
  void move_child(Fl_Type*a, Fl_Type*b) {parent->move_child(a,b);}
  void remove_child(Fl_Type*a) {parent->remove_child(a);}
};

extern int reading_file;
extern int force_parent;

static char submenuflag;

Fl_Type *Fl_Menuitem_Type::make() {
  // Find the current menu item:
  Fl_Type* q = Fl_Type::current;
  Fl_Type* p = q;
  if (p) {
    if (force_parent && q->is_menu_item() || !q->is_parent()) p = p->parent;
  }
  force_parent = 0;
  if (!p || !(p->is_menu_button() || p->is_menu_item() && p->is_parent())) {
    fl_message("Please select a menu to add to");
    return 0;
  }
  if (!o) {
    o = new Fl_Button(0,0,100,20); // create template widget
    o->labelfont(FL_BOLD|FL_ITALIC);
  }

  Fl_Menuitem_Type* t = submenuflag ? new Fl_Submenu_Type() : new Fl_Menuitem_Type();
  t->o = new Fl_Button(0,0,100,20);
  t->o->labelfont(FL_BOLD|FL_ITALIC);
  t->factory = this;
  t->add(p);
  if (!reading_file) t->label(submenuflag ? "submenu" : "item");
  return t;
}

Fl_Type *Fl_Submenu_Type::make() {
  submenuflag = 1;
  Fl_Type* t = Fl_Menuitem_Type::make();
  submenuflag = 0;
  return t;
}

Fl_Menuitem_Type Fl_Menuitem_type;
Fl_Submenu_Type Fl_Submenu_type;

////////////////////////////////////////////////////////////////
// Writing the C code:

#include <ctype.h>

extern FILE *code_file;
extern FILE *header_file;

// test functions in Fl_Widget_Type.C:
int is_name(const char *c);
const char *array_name(Fl_Widget_Type *o);
int isdeclare(const char *c);

void Fl_Menuitem_Type::write_declare() {
  if (!prev->is_menu_item()) {
    int i; ::write_declare("extern Fl_Menu %s[];", menu_name(i));}
  if (callback() && is_name(callback()))
    ::write_declare("extern void %s(Fl_Menu_*, %s);", callback(),
		    user_data_type() ? user_data_type() : "void*");
  for (int n=0; n < NUM_EXTRA_CODE; n++) {
    if (extra_code(n) && isdeclare(extra_code(n)))
      ::write_declare("%s", extra_code(n));
  }
}

const char* Fl_Menuitem_Type::menu_name(int& i) {
  i = 0;
  Fl_Type* t = prev;
  while (t && t->is_menu_item()) {
    // be sure to count the {0} that ends a submenu:
    if (t->level > t->next->level ||
	// detect empty submenu:
	t->level==t->next->level &&t->is_parent())
      i++;
    t = t->prev; i++;
  }
  static char buffer[256];
  sprintf(buffer, "menu_%lx", long(t));
  return buffer;
}

#include "Fluid_Image.H"

void Fl_Menuitem_Type::write_static() {
  const char *c = array_name(this);
  int i; const char* n = menu_name(i);
  if (c)
    fprintf(header_file, "#define %s (%s+%d)\n", c, n, i);
  if (callback() && !is_name(callback())) {
    // see if 'o' or 'v' used, to prevent unused argument warnings:
    int use_o = 0;
    int use_v = 0;
    const char *d;
    for (d = callback(); *d;) {
      if (*d == 'o' && !isalnum(d[1]) && d[1] != '_') use_o = 1;
      if (*d == 'v' && !isalnum(d[1]) && d[1] != '_') use_v = 1;
      do d++; while (isalnum(*d) || *d=='_');
      while (*d && !isalnum(*d)) d++;
    }
    fprintf(code_file, "\nstatic void %s(Fl_Menu*", callback_name());
    if (use_o) fprintf(code_file, " o");
    fprintf(code_file, ", %s", user_data_type() ? user_data_type() : "void*");
    if (use_v) fprintf(code_file, " v");
    fprintf(code_file, ") {\n%s", callback());
    if (*(d-1) != ';') fprintf(code_file, ";");
    fprintf(code_file, "\n}\n");
  }
  if (image) {
    if (image->written != write_number) {
      image->write_static();
      image->written = write_number;
    }
  }
  if (next && next->is_menu_item()) return;
  // okay, when we hit last item in the menu we have to write the
  // entire array out:
  int level;
  fprintf(code_file, "Fl_Menu %s[] = {\n", menu_name(level));
  Fl_Type* t = prev; while (t && t->is_menu_item()) t = t->prev;
  level = t->level+1;
  for (Fl_Type* q = t->next; q && q->is_menu_item(); q = q->next) {
    ((Fl_Menuitem_Type*)q)->write_item();
    if (q->is_parent()) level++;
    int l1 =
      (q->next && q->next->is_menu_item()) ? q->next->level : t->next->level;
    while (level > l1) {fprintf(code_file," {0},\n"); level--;}
    level = l1;
  }
  fprintf(code_file," {0}\n};\n");
}

int Fl_Menuitem_Type::flags() {
  int i = o->type();
  if (!o->active()) i |= FL_MENU_INACTIVE;
  if (!o->visible()) i |= FL_MENU_INVISIBLE;
  if (is_parent()) i |= FL_SUBMENU;
  if (resizable()) i |= FL_MENU_DIVIDER;
  return i;
}

void Fl_Menuitem_Type::write_item() {
  fprintf(code_file," {");
  if (image) fprintf(code_file,"0");
  else if (label()) write_cstring(label());
  else fprintf(code_file,"\"\"");
  if (((Fl_Button*)o)->shortcut())
    fprintf(code_file,", 0x%x, ", ((Fl_Button*)o)->shortcut());
  else
    fprintf(code_file,", 0, ");
  if (callback())
    fprintf(code_file," (Fl_Callback*)%s,", callback_name());
  else
    fprintf(code_file," 0,");
  if (user_data())
    fprintf(code_file, " (void*)(%s),", user_data());
  else
    fprintf(code_file," 0,");
  fprintf(code_file," %d, %d, %d, %d, %d", flags(),
	  o->labeltype(), o->labelfont(), o->labelsize(), o->labelcolor());
  fprintf(code_file,"},\n");
}

void Fl_Menuitem_Type::write_code1() {
  int init = 0;
  if (image) {
    int i; const char* n = menu_name(i);
    fprintf(code_file, " {Fl_Menu* o = &%s[%d];\n", n, i);
    init = 1;
    image->write_code();
  }
  for (int n=0; n < NUM_EXTRA_CODE; n++)
    if (extra_code(n) && !isdeclare(extra_code(n))) {
      if (!init) {
	init = 1;
	int i; const char* n = menu_name(i);
	fprintf(code_file, " {Fl_Menu* o = &%s[%d];\n", n, i);
      }
      fprintf(code_file, "  %s\n", extra_code(n));
    }
  if (init) fprintf(code_file, " }\n");
}

void Fl_Menuitem_Type::write_code2() {}

////////////////////////////////////////////////////////////////
// This is the base class for widgets that contain a menu (ie
// subclasses of Fl_Menu_.
// This is a parent widget and menu items can be added as
// children.  An actual array of Fl_Menu items is kept parallel
// with the child objects and updated as they change.

class Fl_Menu_Type : public Fl_Widget_Type {
public:
  int is_menu_button() {return 1;}
  int is_parent() {return 1;}
  int menusize;
  void build_menu();
  Fl_Menu_Type() : Fl_Widget_Type() {menusize = 0;}
  ~Fl_Menu_Type() {
    if (menusize) delete[] (Fl_Menu*)(((Fl_Menu_*)o)->menu());
  }
  void add_child(Fl_Type*, Fl_Type*) {build_menu();}
  void move_child(Fl_Type*, Fl_Type*) {build_menu();}
  void remove_child(Fl_Type*) {build_menu();}
  Fl_Type* click_test(int x, int y);
  void write_code2();
};

void Fl_Menu_Type::build_menu() {
  Fl_Menu_* w = (Fl_Menu_*)o;
  // count how many Fl_Menu structures needed:
  int n = 0;
  Fl_Type* q;
  for (q = next; q && q->level > level; q = q->next) {
    if (q->is_parent()) n++; // space for null at end of submenu
    n++;
  }
  if (!n) {
    if (menusize) delete[] (Fl_Menu*)(w->menu());
    w->menu(0);
  } else {
    if (menusize<n) {
      if (menusize) delete[] (Fl_Menu*)(w->menu());
      menusize = n+10;
      w->menu(new Fl_Menu[menusize]);
    }
    // fill them all in:
    Fl_Menu* m = (Fl_Menu*)(w->menu());
    int lvl = level+1;
    for (q = next; q && q->level > level; q = q->next) {
      Fl_Menuitem_Type* i = (Fl_Menuitem_Type*)q;
      m->label(i->o->label());
      m->shortcut(((Fl_Button*)(i->o))->shortcut());
      m->callback(0,(void*)i);
      m->flags = i->flags();
      m->labeltype(i->o->labeltype());
      m->labelfont(i->o->labelfont());
      m->labelsize(i->o->labelsize());
      m->labelcolor(i->o->labelcolor());
      m++;
      if (q->is_parent()) lvl++;
      int l1 =
	(q->next && q->next->is_menu_item()) ? q->next->level : level;
      while (lvl > l1) {m->label(0); m++; lvl--;}
      lvl = l1;
    }
  }
  o->redraw();
}

Fl_Type* Fl_Menu_Type::click_test(int, int) {
  if (selected) return 0; // let user move the widget
  Fl_Menu_* w = (Fl_Menu_*)o;
  if (!menusize) return 0;
  const Fl_Menu* save = w->mvalue();
  w->value((Fl_Menu*)0);
  Fl::pushed(w);
  w->handle(FL_PUSH);
  if (w->mvalue()) return (Fl_Type*)(w->mvalue()->user_data());
  w->value(save);
  return this;
}

void Fl_Menu_Type::write_code2() {
  if (next && next->is_menu_item())
    fprintf(code_file, "  o->menu(menu_%lx);\n", long(this));
  Fl_Widget_Type::write_code2();
}

////////////////////////////////////////////////////////////////

#include <FL/Fl_Menu_Button.H>
class Fl_Menu_Button_Type : public Fl_Menu_Type {
public:
  virtual const char *type_name() {return "Fl_Menu_Button";}
  Fl_Widget *widget(int x,int y,int w,int h) {
    return new Fl_Menu_Button(x,y,w,h,"menu");}
  Fl_Widget_Type *_make() {return new Fl_Menu_Button_Type();}
};
Fl_Menu_Button_Type Fl_Menu_Button_type;

////////////////////////////////////////////////////////////////

#include <FL/Fl_Choice.H>

static Fl_Menu dummymenu[] = {{"CHOICE"},{0}};

class Fl_Choice_Type : public Fl_Menu_Type {
  int textstuff(int w, uchar&f, uchar&s, uchar&c) {
    Fl_Choice *o = (Fl_Choice*)(w==4 ? ((Fl_Widget_Type*)this->factory)->o : this->o);
    switch (w) {
    case 4:
    case 0: f = o->textfont(); s = o->textsize(); c = o->textcolor(); break;
    case 1: o->textfont(f); break;
    case 2: o->textsize(s); break;
    case 3: o->textcolor(c); break;
    }
    return 1;
  }
public:
  virtual const char *type_name() {return "Fl_Choice";}
  Fl_Widget *widget(int x,int y,int w,int h) {
    Fl_Choice *o = new Fl_Choice(x,y,w,h,"choice:");
    o->menu(dummymenu);
    return o;
  }
  Fl_Widget_Type *_make() {return new Fl_Choice_Type();}
};
Fl_Choice_Type Fl_Choice_type;

////////////////////////////////////////////////////////////////

#include <FL/Fl_Menu_Bar.H>
class Fl_Menu_Bar_Type : public Fl_Menu_Type {
public:
  virtual const char *type_name() {return "Fl_Menu_Bar";}
  Fl_Widget *widget(int x,int y,int w,int h) {
    return new Fl_Menu_Bar(x,y,w,h);}
  Fl_Widget_Type *_make() {return new Fl_Menu_Bar_Type();}
};
Fl_Menu_Bar_Type Fl_Menu_Bar_type;

////////////////////////////////////////////////////////////////
// Shortcut entry item in panel:

#include <FL/Fl_Output.H>
#include "Shortcut_Button.H"
#include <FL/fl_draw.H>

void Shortcut_Button::draw() {
  if (value()) draw_box(FL_THIN_DOWN_BOX, 9);
  else draw_box(FL_THIN_UP_BOX, FL_WHITE);
  fl_font(0,14); fl_color(0);
  fl_draw(fl_shortcut_label(svalue),x()+8,y(),w(),h(),FL_ALIGN_LEFT);
}

int Shortcut_Button::handle(int e) {
  when(0); type(FL_TOGGLE_BUTTON);
  if (e == FL_KEYBOARD) {
    if (!value()) return 0;
    int v = Fl::event_text()[0];
    if (v > 32 && v < 0x7f || v > 0xa0 && v <= 0xff) {
      v = v | Fl::event_state()&(FL_META|FL_ALT|FL_CTRL);
    } else {
      v = Fl::event_state()&(FL_META|FL_ALT|FL_CTRL|FL_SHIFT) | Fl::event_key();
      if (v == FL_BackSpace && svalue) v = 0;
    }
    if (v != svalue) {svalue = v; set_changed(); redraw();}
    return 1;
  } else if (e == FL_UNFOCUS) {
    value(0);
    return 1;
  } else if (e == FL_FOCUS) {
    return value();
  } else {
    int r = Fl_Button::handle(e);
    if (e == FL_RELEASE && value() && Fl::focus() != this) take_focus();
    return r;
  }
}
  
void shortcut_in_cb(Shortcut_Button* i, void* v) {
  if (v == LOAD) {
    if (!current_widget->is_button()) {i->hide(); return;}
    i->show();
    i->svalue = ((Fl_Button*)(current_widget->o))->shortcut();
    i->redraw();
  } else {
    for (Fl_Type *o = Fl_Type::first; o; o = o->next)
      if (o->selected && o->is_button()) {
	Fl_Button* b = (Fl_Button*)(((Fl_Widget_Type*)o)->o);
	b->shortcut(i->svalue);
	if (o->is_menu_item()) ((Fl_Widget_Type*)o)->redraw();
      }
  }
}
