// Fl_Menu.H

// Definition of the structure to define a popup menu and
// used to construct an Fl_Menubar.

// A menu is an *array* of these structures.  The structure
// is designed so it is relatively easy to define an entire
// hierarchy of menus with a C initialization constant.

#ifndef Fl_Menu_H
#define Fl_Menu_H

#ifndef Fl_Widget_H
// used to get the Fl_Callback typedefs:
#include "Fl_Widget.H"
#endif

enum { // values for flags:
  FL_MENU_INACTIVE = 1,
  FL_MENU_BOX	= 2,
  FL_MENU_CHECK = 4,
  FL_MENU_RADIO = 8,
  FL_MENU_INVISIBLE = 0x10,
  FL_SUBMENU_POINTER = 0x20,
  FL_SUBMENU = 0x40,
  FL_MENU_DIVIDER = 0x80,
  FL_MENU_HORIZONTAL = 0x100
};

extern int fl_old_shortcut(const char*);

// An Fl_Menu is an *array* of these:
struct Fl_Menu {
  const char *text;	// label()
  int shortcut_;
  Fl_Callback *callback_;
  void *user_data_;
  int flags;
  uchar labeltype_;
  uchar labelfont_;
  uchar labelsize_;
  uchar labelcolor_;

  // methods on menu arrays:
  const Fl_Menu *popup(int x, int y, const char *title = 0) const;
  const Fl_Menu *popup(int x, int y, int w, int h,
		       const Fl_Menu* = 0, const char* title = 0) const;
  const Fl_Menu *test_shortcut() const;

  // insert new items into existing Fl_Menu array:
  int add(const char*, int shortcut, Fl_Callback*, void* =0, int = 0);
  int add(const char*a, const char* b, Fl_Callback* c,
	  void* d = 0, int e = 0) {
    return add(a,fl_old_shortcut(b),c,d,e);}
  int size() const ;

  // advance N items, skipping submenus:
  const Fl_Menu *next(int=1) const;
  Fl_Menu *next(int i=1) {return (Fl_Menu*)(((const Fl_Menu*)this)->next(i));}

  // methods on menu items:
  const char* label() const {return text;}
  void label(const char* a) {text=a;}
  void label(uchar a,const char* b) {labeltype_ = a; text = b;}
  uchar labeltype() const {return labeltype_;}
  void labeltype(uchar a) {labeltype_ = a;}
  uchar labelcolor() const {return labelcolor_;}
  void labelcolor(uchar a) {labelcolor_ = a;}
  uchar labelfont() const {return labelfont_;}
  void labelfont(uchar a) {labelfont_ = a;}
  uchar labelsize() const {return labelsize_;}
  void labelsize(uchar a) {labelsize_ = a;}
  Fl_Callback* callback() const {return callback_;}
  void callback(Fl_Callback* c, void* p) {callback_=c; user_data_=p;}
  void callback(Fl_Callback* c) {callback_=c;}
  void callback(Fl_Callback0*c) {callback_=(Fl_Callback*)c;}
  void callback(Fl_Callback1*c, long p=0) {callback_=(Fl_Callback*)c; user_data_=(void*)p;}
  void* user_data() const {return user_data_;}
  void user_data(void* v) {user_data_ = v;}
  long argument() const {return (long)user_data_;}
  void argument(long v) {user_data_ = (void*)v;}
  int shortcut() const {return shortcut_;}
  void shortcut(int s) {shortcut_ = s;}

  int submenu() const {return flags&(FL_SUBMENU|FL_SUBMENU_POINTER);}
  int checkbox() const {return flags&0xe;} // true = draws a checkbox
  int radio() const {return flags&FL_MENU_RADIO;}
  int value() const {return flags&FL_MENU_CHECK;}
  void set() {flags |= FL_MENU_CHECK;}
  void clear() {flags &= ~FL_MENU_CHECK;}
  void setonly();

  int visible() const {return !(flags&FL_MENU_INVISIBLE);}
  void show() {flags &= ~FL_MENU_INVISIBLE;}
  void hide() {flags |= FL_MENU_INVISIBLE;}
  int active() const {return !(flags&FL_MENU_INACTIVE);}
  void activate() {flags &= ~FL_MENU_INACTIVE;}
  void deactivate() {flags |= FL_MENU_INACTIVE;}
  int activevisible() const {return !(flags&0x11);}

  void do_callback(Fl_Widget* o) const {callback_(o, user_data_);}
  void do_callback(Fl_Widget* o,void* arg) const {callback_(o, arg);}
  void do_callback(Fl_Widget* o,long arg) const {callback_(o, (void*)arg);}

  // back-compatability:
  int checked() const {return flags&FL_MENU_CHECK;}
  void check() {flags |= FL_MENU_CHECK;}
  void uncheck() {flags &= ~FL_MENU_CHECK;}
  // fallback font, used if labelsize() == 0:
  static uchar textsize_, textfont_;
  static void textfont(uchar x) {textfont_ = x;}
  static void textsize(uchar x) {textsize_ = x;}
  static uchar textfont() {return textfont_;}
  static uchar textsize() {return textsize_;}
};

enum {	// back-compatability enum:
  FL_PUP_NONE	= 0,
  FL_PUP_GREY	= FL_MENU_INACTIVE,
  FL_PUP_GRAY	= FL_MENU_INACTIVE,
  FL_PUP_BOX	= FL_MENU_BOX,
  FL_PUP_CHECK	= FL_MENU_CHECK,
  FL_PUP_RADIO	= FL_MENU_RADIO,
  FL_PUP_INVISIBLE = FL_MENU_INVISIBLE,
  FL_PUP_SUBMENU = FL_SUBMENU_POINTER
};

#endif
