#include <sys/types.h>
#include <stdio.h>
#include <assert.h>
#include "afinternal.h"

/*
	blockReadFrames is used for reading uncompressed sampled sound
	data from RIFF WAVE or AIFF files.
*/

int blockReadFrames (AFfilehandle file, int track, void *samples,
	int frameCount)
{
	size_t		done = 0, sampleCount;
	u_int8_t	*buffer8 = (u_int8_t *) samples;
	u_int16_t	*buffer16 = (u_int16_t *) samples;
	u_int32_t	*buffer32 = (u_int32_t *) samples;
	int			frameSize, channelCount;

	assert(file);
	assert(track == AF_DEFAULT_TRACK);
	assert(samples);
	assert(frameCount >= 0);

	frameSize = (file->sampleWidth + 7) / 8 * file->channelCount;

	channelCount = file->channelCount;

	sampleCount = frameCount * file->channelCount;
	fseek(file->fp, file->dataStart + file->currentFrame * frameSize, SEEK_SET);

	file->currentFrame += frameCount;

	if (file->sampleWidth <= 8)
	{
		u_int8_t	datum;

		switch (file->sampleFormat)
		{
			case AF_SAMPFMT_TWOSCOMP:
				while (done < sampleCount)
				{
					if (fread(&datum, 1, 1, file->fp) < 1)
						break;
					*buffer8++ = datum;
					done++;
				}
				break;

			case AF_SAMPFMT_UNSIGNED:
				while (done < sampleCount)
				{
					if (fread(&datum, 1, 1, file->fp) < 1)
						break;
					/* Convert datum to a signed integer. */
					datum ^= 128;
					*buffer8++ = datum;
					done++;
				}
				break;

			default:
				error(AF_BAD_SAMPFMT);
				break;
		}
	}
	else if (file->sampleWidth <= 16)
	{
		u_int16_t	datum;

		switch (file->sampleFormat)
		{
			case AF_SAMPFMT_TWOSCOMP:
				while (done < sampleCount)
				{
					if (fread(&datum, 2, 1, file->fp) < 1)
						break;
					if (file->byteOrder != file->virtualByteOrder)
						datum = _byteswapint16(datum);
					*buffer16++ = datum;
					done++;
				}
				break;

			case AF_SAMPFMT_UNSIGNED:
				while (done < sampleCount)
				{
					if (fread(&datum, 2, 1, file->fp) < 1)
						break;
					if (file->byteOrder != file->virtualByteOrder)
						datum = _byteswapint16(datum);
					/* Convert datum to a signed integer. */
					datum ^= 0x8000;
					*buffer16++ = datum;
					done++;
				}
				break;

			default:
				error(AF_BAD_SAMPFMT);
				break;
		}
	}
	else if (file->sampleWidth <= 24)
	{
		u_int8_t	threeBytes[3];
		u_int32_t	datum;

		switch (file->sampleFormat)
		{
			case AF_SAMPFMT_TWOSCOMP:
				while (done < sampleCount)
				{
					if (fread(threeBytes, 3, 1, file->fp) < 1)
						break;

					datum = threeBytes[2] |
						(threeBytes[1] << 8) |
						(threeBytes[0] << 16);
					if (file->byteOrder != file->virtualByteOrder)
					{
						datum = (threeBytes[2] << 24) |
							(threeBytes[1] << 16) |
							(threeBytes[0] << 8);
					}

					datum &= 0xffffff;
					if (datum & 0x800000)
					{
						datum |= 0xff000000;
					}

					*buffer32++ = datum;
					done++;
				}
				break;

			default:
				error(AF_BAD_SAMPFMT);
				break;
		}
	}
	else if (file->sampleWidth <= 32)
	{
		u_int32_t	datum;

		switch (file->sampleFormat)
		{
			case AF_SAMPFMT_TWOSCOMP:
				while (done < sampleCount)
				{
					if (fread(&datum, 4, 1, file->fp) < 1)
						break;
					if (file->byteOrder != file->virtualByteOrder)
						datum = _byteswapint32(datum);
					*buffer32++ = datum;
					done++;
				}
				break;

			case AF_SAMPFMT_UNSIGNED:
				while (done < sampleCount)
				{
					if (fread(&datum, 4, 1, file->fp) < 1)
						break;
					if (file->byteOrder != file->virtualByteOrder)
						datum = _byteswapint32(datum);
					/* Convert datum to a signed integer. */
					datum ^= 0x80000000;
					*buffer32++ = datum;
					done++;
				}
				break;

			default:
				error(AF_BAD_SAMPFMT);
				break;
		}
	}

	done /= channelCount;
	return done;
}
