/*
	Audio File Library

	Copyright 1998, Michael Pruett <michael@68k.org>

	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License as
	published by the Free Software Foundation; either version 2 of
	the License, or (at your option) any later version.

	This library is distributed in the hope that it will be
	useful, but WITHOUT ANY WARRANTY; without even the implied
	warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
	PURPOSE.  See the GNU General Public License for more details.

	You should have received a copy of the GNU General Public
	License along with this program; if not, write to the Free
	Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
	MA 02111-1307, USA.
*/

/*
	audiofile.c

	This file implements many of the main interface routines of the
	Audio File Library.
*/

#include <assert.h>
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "audiofile.h"
#include "util.h"
#include "afinternal.h"

int parseaiff (AFfilehandle filehandle);
int parsewave (AFfilehandle filehandle);
int parseau (AFfilehandle filehandle);

int aiffSyncFile (AFfilehandle);
int waveSyncFile (AFfilehandle);
int auSyncFile (AFfilehandle);

int blockReadFrames (const AFfilehandle file, int track, void *samples, const int count);

int afSetVirtualByteOrder (AFfilehandle file, int track, int byteorder)
{
	assert(file);
	assert(track == AF_DEFAULT_TRACK);
	assert(byteorder == AF_BYTEORDER_BIGENDIAN ||
		byteorder == AF_BYTEORDER_LITTLEENDIAN);

	file->virtualByteOrder = byteorder;
}

int afGetByteOrder (AFfilehandle file, int track)
{
	assert(file);
	assert(track == AF_DEFAULT_TRACK);
	assert(file->byteOrder == AF_BYTEORDER_BIGENDIAN ||
		file->byteOrder == AF_BYTEORDER_LITTLEENDIAN);

	return file->byteOrder;
}

int afGetVirtualByteOrder(AFfilehandle file, int track)
{
	assert(file);
	assert(track == AF_DEFAULT_TRACK);
	assert(file->virtualByteOrder == AF_BYTEORDER_BIGENDIAN ||
		file->virtualByteOrder == AF_BYTEORDER_LITTLEENDIAN);

	return file->virtualByteOrder;
}

int afGetChannels(AFfilehandle file, int track)
{
	assert(file);
	assert(track == AF_DEFAULT_TRACK);

	if (file == NULL)
	{
		error(AF_BAD_FILEHANDLE);
		return -1;
	}

	assert(file->channelCount >= 1);

	return file->channelCount;
}

void afGetSampleFormat(AFfilehandle file, int track, int *sampfmt, int *sampwidth)
{
	assert(file);
	assert(track == AF_DEFAULT_TRACK);

	if (sampfmt != NULL)
		*sampfmt = file->sampleFormat;
	if (sampwidth != NULL)
		*sampwidth = file->sampleWidth;
}

int afReadFrames(const AFfilehandle file, int track, void *samples, const int count)
{
	assert(file);
	assert(track == AF_DEFAULT_TRACK);
	assert(samples);
	assert(count >= 0);

	return blockReadFrames(file, track, samples, count);
}

/*
int afReadFrames(const AFfilehandle file, int track, void *samples, const int count)
{
	int		i, frameSize, channelCount;
	ssize_t	bytesRead;

	if (file == NULL)
		return -1;

	frameSize = (file->sampleWidth + 7) / 8 * file->channelCount;

	printf("frameSize = %d\n", frameSize);

	channelCount = file->channelCount;

	fseek(file->fp, file->dataStart + file->currentFrame * frameSize, SEEK_SET);
	bytesRead = fread(samples, 1, count * frameSize, file->fp);

	if (file->byteOrder != file->virtualByteOrder)
	{
		if (file->sampleWidth == 16)
		{
			u_int16_t	*s = (u_int16_t *) samples;

			for (i=0; i < count * channelCount; i++)
			{
				s[i] = _byteswapint16(s[i]);
			}
		}
		else if (file->sampleWidth == 32)
		{
			u_int32_t	*l = (u_int32_t *) samples;

			for (i=0; i < count * channelCount; i++)
			{
				l[i] = _byteswapint32(l[i]);
			}
		}
	}

	if (bytesRead == -1)
		return -1;
	else if (bytesRead == 0)
		return 0;

	else if (bytesRead != count * frameSize)
	{
		file->currentFrame += bytesRead / frameSize;
		return bytesRead / frameSize;
	}

	file->currentFrame += count;
	return count;
}
*/

AFframecount afGetFrameCount(AFfilehandle file, int track)
{
	if (file != NULL)
	{
		return file->frameCount;
	}
}

/* afSyncFile returns 0 on success and -1 on failure */
int afSyncFile (AFfilehandle file)
{
	assert(file != NULL);

	/* It is an error to sync a null file. */
	if (file == NULL)
		return -1;

	if (file->fileFormat == AF_FILE_AIFFC)
		return aiffSyncFile(file);
	else if (file->fileFormat == AF_FILE_AIFF)
		return aiffSyncFile(file);
	else if (file->fileFormat == AF_FILE_WAVE)
		return waveSyncFile(file);
	else if (file->fileFormat == AF_FILE_NEXTSND)
		return auSyncFile(file);
}

/* afCloseFile returns 0 on success and -1 on failure */
int afCloseFile (AFfilehandle file)
{
	int	result;

	assert(file != NULL);

	/* It is an error to close a null file. */
	if (file == NULL)
	{
		/* error(); */
		return -1;
	}

	result = afSyncFile(file);

	if (result != 0)
		return result;

	result = fclose(file->fp);

	/* close returns -1 on error */
	if (result != 0)
		return result;

	/* Free memory associated with loops. */
	if (file->loops != NULL)
		free(file->loops);

	/* Free memory associated with markers. */
	if (file->markers != NULL)
	{
		int	i;

		for (i=0; i<file->markerCount; i++)
			free(file->markers[i].name);

		free(file->markers);
	}

	/* Free memory associated with instruments. */
	if (file->instruments != NULL)
		free(file->instruments);

	if (file->miscellaneous != NULL)
	{
		free(file->miscellaneous);
	}

	free(file);

	return 0;
}

AFfilehandle _afOpenFileWrite (const char *filename, const char *mode, AFfilesetup setup)
{
	FILE			*fp;
	AFfilehandle	filehandle;
	char			data[12];

	fp = fopen(filename, mode);
	if (fp == NULL)
		return AF_NULL_FILEHANDLE;

	filehandle = malloc(sizeof (struct _AFfilehandle));
	if (filehandle == NULL)
		return NULL;

	filehandle->fileFormat = setup->fileFormat;
	filehandle->channelCount = setup->channelCount;
	filehandle->sampleWidth = setup->sampleWidth;
	filehandle->sampleFormat = setup->sampleFormat;
	filehandle->byteOrder = setup->byteOrder;
	filehandle->virtualByteOrder = setup->byteOrder;
	filehandle->sampleRate = setup->sampleRate;
	filehandle->dataStart = 0;
	filehandle->frameCount = 0;
	filehandle->currentFrame = 0;

	filehandle->aesDataPresent = setup->aesDataPresent;

	filehandle->fp = fp;
	filehandle->markerCount = 0;
	filehandle->markers = NULL;
	filehandle->loopCount = 0;
	filehandle->loops = NULL;
	filehandle->instrumentCount = 0;
	filehandle->instruments = NULL;

	if (setup->miscellaneousCount == 0)
	{
		filehandle->miscellaneousCount = 0;
		filehandle->miscellaneous = NULL;
	}
	else
	{
		filehandle->miscellaneousCount = setup->miscellaneousCount;
		filehandle->miscellaneous = malloc(filehandle->miscellaneousCount *
			sizeof (struct _Miscellaneous));
		memcpy(filehandle->miscellaneous, setup->miscellaneous,
			filehandle->miscellaneousCount * sizeof (struct _Miscellaneous));
	}

	if (filehandle->fileFormat == AF_FILE_AIFF ||
		filehandle->fileFormat == AF_FILE_AIFFC)
	{
		int	i;

		filehandle->markerCount = 4;
		filehandle->markers = malloc(4 * sizeof (struct _Marker));
		memcpy(filehandle->markers, setup->markers,
			4 * sizeof (struct _Marker));

		for (i=0; i<4; i++)
		{
			filehandle->markers[i].name = strdup(setup->markers[i].name);
		}

		filehandle->loopCount = 2;
		filehandle->loops = malloc(2 * sizeof (struct _Loop));
		memcpy(filehandle->loops, setup->loops, 2 * sizeof (struct _Loop));

		filehandle->instrumentCount = 1;
		filehandle->instruments = malloc(2 * sizeof (struct _Instrument));
		memcpy(filehandle->instruments, setup->instruments,
			1 * sizeof (struct _Instrument));
	}

#ifdef DEBUG
	printfilehandle(filehandle);
#endif

	return filehandle;
}

AFfilehandle afOpenFile(const char *filename, const char *mode, AFfilesetup setup)
{
	FILE			*fp;
	AFfilehandle	filehandle;
	char			data[12];

	if (!strcmp(mode, "w"))
		return _afOpenFileWrite(filename, mode, setup);

	fp = fopen(filename, mode);
	if (fp == NULL)
		return AF_NULL_FILEHANDLE;

	filehandle = malloc(sizeof (struct _AFfilehandle));
	if (filehandle == NULL)
		return NULL;

	fread(data, 1, 12, fp);
	fseek(fp, 0, SEEK_SET);

	filehandle->fp = fp;
	filehandle->currentFrame = 0;
	filehandle->markerCount = 0;
	filehandle->markers = NULL;
	filehandle->loopCount = 0;
	filehandle->loops = NULL;
	filehandle->instrumentCount = 0;
	filehandle->instruments = NULL;
	filehandle->miscellaneousCount = 0;
	filehandle->miscellaneous = NULL;
	filehandle->aesDataPresent = 0;

	if (!memcmp(data, "FORM", 4) && !memcmp(data + 8, "AIFC", 4))
	{
#ifdef DEBUG
		printf("detected aifc file\n");
#endif

		filehandle->fileFormat = AF_FILE_AIFFC;
		filehandle->sampleFormat = AF_SAMPFMT_TWOSCOMP;
		filehandle->byteOrder = AF_BYTEORDER_BIGENDIAN;
		filehandle->virtualByteOrder = AF_BYTEORDER_BIGENDIAN;

		parseaiff(filehandle);
	}
	else if (!memcmp(data, "FORM", 4) && !memcmp(data + 8, "AIFF", 4))
	{
#ifdef DEBUG
		printf("detected aiff file\n");
#endif

		filehandle->fileFormat = AF_FILE_AIFF;
		filehandle->sampleFormat = AF_SAMPFMT_TWOSCOMP;
		filehandle->byteOrder = AF_BYTEORDER_BIGENDIAN;
		filehandle->virtualByteOrder = AF_BYTEORDER_BIGENDIAN;

		parseaiff(filehandle);
	}
	else if (!memcmp(data, "RIFF", 4) && !memcmp(data + 8, "WAVE", 4))
	{
#ifdef DEBUG
		printf("detected wave file\n");
#endif

		filehandle->fileFormat = AF_FILE_WAVE;
		filehandle->sampleFormat = AF_SAMPFMT_TWOSCOMP;
		filehandle->byteOrder = AF_BYTEORDER_LITTLEENDIAN;
		filehandle->virtualByteOrder = AF_BYTEORDER_LITTLEENDIAN;

		parsewave(filehandle);
	}
	else if (!memcmp(data, ".snd", 4))
	{
#ifdef DEBUG
		printf("detected next/sun sound file\n");
#endif

		filehandle->fileFormat = AF_FILE_NEXTSND;
		filehandle->byteOrder = AF_BYTEORDER_BIGENDIAN;
		filehandle->virtualByteOrder = AF_BYTEORDER_BIGENDIAN;

		parseau(filehandle);
	}
	else
		return AF_NULL_FILEHANDLE;

#ifdef DEBUG
	printfilehandle(filehandle);
#endif

	return filehandle;
}

int afGetFileFormat (AFfilehandle file, int *version)
{
	assert(file);

	return file->fileFormat;
}

double afGetRate (AFfilehandle file, int track)
{
	assert(file);
	assert(track == AF_DEFAULT_TRACK);

	return file->sampleRate;
}

AFfileoffset afGetDataOffset (AFfilehandle file, int track)
{
	assert(file);

	return file->dataStart;
}

AFfileoffset afGetTrackBytes (AFfilehandle file, int track)
{
	assert(file);

	return file->trackBytes;
}

/* This routine is not quite implemented properly. */
float afGetFrameSize (AFfilehandle file, int track, int stretch3to4)
{
	float	frameSize;

	assert(file);

	if (file->sampleWidth <= 8)
		return 1 * file->channelCount;
	else if (file->sampleWidth <= 16)
		return 2 * file->channelCount;
	else if (file->sampleWidth <= 24)
	{
		if (stretch3to4 == 0)
			return 3 * file->channelCount;
		else
			return 4 * file->channelCount;
	}
	else if (file->sampleWidth <= 32)
		return 4 * file->channelCount;

	return (file->channelCount * ((file->sampleWidth + 7) / 8));
}

AFframecount afSeekFrame (AFfilehandle file, int track, AFframecount frame)
{
	int	frameSize;

	assert(file);

	/* frameSize represents the size of a frame in bytes. */
	frameSize = file->channelCount * (file->sampleWidth / 8);

	fseek(file->fp, file->dataStart + frameSize * frame, SEEK_SET);
}

AFfileoffset afTellFrame (AFfilehandle file, int track)
{
	assert(file);

	return file->currentFrame;
}
