
/*
 *	GW4PTS Morse tutor for Linux.
 *
 *	This software is placed under the GNU software license revision 2 (see file
 *	COPYING). 
 *
 *	1.00	Alan Cox	21/09/93	Created it. Quite neat for an hours work
 *
 *  Compile with:
 *    gcc morse.c -o morse
 *
 *	TO DO:
 *
 *	Light option (keyboard led) for the deaf.
 *	Default group options (6 5 letter groups).
 *  Shortcut group of 1 letter 1 group for beginner mode.
 *	Calibrate to wpm.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <sys/types.h>
#include <linux/kd.h>
#include <signal.h>

/* Configuration globals */

int morse_tone=800;	/* Set with -t tone */
int morse_speed=10;	/* Set with -s speed */
int morse_gap=9;	/* Set with -d speed */

typedef struct
{
	char code;
	enum
	{
		NIL,
		DIH,
		DAH,
	} data[7];
} Morse;


Morse MorseTable[]=
{
	' ',NIL,NIL,NIL,NIL,NIL,NIL,NIL,
	'A',DIH,DAH,NIL,NIL,NIL,NIL,NIL,
	'B',DAH,DIH,DIH,DIH,NIL,NIL,NIL,
	'C',DAH,DIH,DAH,DIH,NIL,NIL,NIL,
	'D',DAH,DIH,DIH,NIL,NIL,NIL,NIL,
	'E',DIH,NIL,NIL,NIL,NIL,NIL,NIL,
	'F',DIH,DIH,DAH,DIH,NIL,NIL,NIL,
	'G',DAH,DAH,DIH,NIL,NIL,NIL,NIL,
	'H',DIH,DIH,DIH,DIH,NIL,NIL,NIL,
	'I',DIH,DIH,NIL,NIL,NIL,NIL,NIL,
	'J',DIH,DAH,DAH,DAH,NIL,NIL,NIL,
	'K',DAH,DIH,DAH,NIL,NIL,NIL,NIL,
	'L',DIH,DAH,DIH,DIH,NIL,NIL,NIL,
	'M',DAH,DAH,NIL,NIL,NIL,NIL,NIL,
	'N',DAH,DIH,NIL,NIL,NIL,NIL,NIL,
	'O',DAH,DAH,DAH,NIL,NIL,NIL,NIL,
	'P',DIH,DAH,DAH,DIH,NIL,NIL,NIL,
	'Q',DAH,DAH,DIH,DAH,NIL,NIL,NIL,
	'R',DIH,DAH,DIH,NIL,NIL,NIL,NIL,
	'S',DIH,DIH,DIH,NIL,NIL,NIL,NIL,
	'T',DAH,NIL,NIL,NIL,NIL,NIL,NIL,
	'U',DIH,DIH,DAH,NIL,NIL,NIL,NIL,
	'V',DIH,DIH,DIH,DAH,NIL,NIL,NIL,
	'W',DIH,DAH,DAH,NIL,NIL,NIL,NIL,
	'X',DAH,DIH,DIH,DAH,NIL,NIL,NIL,
	'Y',DAH,DIH,DAH,DAH,NIL,NIL,NIL,
	'Z',DAH,DAH,DIH,DIH,NIL,NIL,NIL,
	'1',DIH,DAH,DAH,DAH,DAH,NIL,NIL,
	'2',DIH,DIH,DAH,DAH,DAH,NIL,NIL,
	'3',DIH,DIH,DIH,DAH,DAH,NIL,NIL,
	'4',DIH,DIH,DIH,DIH,DAH,NIL,NIL,
	'5',DIH,DIH,DIH,DIH,DIH,NIL,NIL,
	'6',DAH,DIH,DIH,DIH,DIH,NIL,NIL,
	'7',DAH,DAH,DIH,DIH,DIH,NIL,NIL,
	'8',DAH,DAH,DAH,DIH,DIH,NIL,NIL,
	'9',DAH,DAH,DAH,DAH,DIH,NIL,NIL,
	'0',DAH,DAH,DAH,DAH,DAH,NIL,NIL,
	'?',DIH,DIH,DAH,DAH,DIH,DIH,NIL,
	'/',DAH,DIH,DIH,DAH,DIH,NIL,NIL,
	0,  NIL,NIL,NIL,NIL,NIL,NIL,NIL	/* END MARKER */
};


int InputToMorse(char *src,Morse *out)
{
	int d=0;
	out->data[0]=NIL;
	out->data[1]=NIL;
	out->data[2]=NIL;
	out->data[3]=NIL;
	out->data[4]=NIL;
	out->data[5]=NIL;
	out->data[6]=NIL;

	while(*src && d<6)
	{
		if(isspace(*src))
		{
			src++;
			continue;
		}
		if(*src=='.')
			out->data[d++]=DIH;
		else if(*src=='-')
			out->data[d++]=DAH;
		else
			return(-1);
		src++;
	}
	return(d);
}

Morse *CharToMorse(char c)
{
	int ct=0;
	while(MorseTable[ct].code)
	{
		if(MorseTable[ct].code==c)
			return(&MorseTable[ct]);
		ct++;
	}
	return(NULL);
}

char MorseToChar(Morse *m)
{
	int ct=0;
	while(MorseTable[ct].code)
	{
		if(memcmp(&MorseTable[ct].data[0],&m->data[0],sizeof(m->data))==0)
			return(MorseTable[ct].code);
		ct++;
	}
	return(0);
}


void PrintMorse(Morse *m)
{
	int d=0;
	int doneone=0;
	while(d<6)
	{
		if(m->data[d]==NIL)
			break;
		if(doneone)
			printf("-");
		doneone=1;
		if(m->data[d]==DAH)
			printf("dah");
		else if(m->data[d]==DIH)
		{
			printf("di");
			if(m->data[d+1]==NIL)
				printf("t");
		}
		else
			break;
		d++;
	}
	printf(" ");
}


void PlayBeep(long duration)
{
	signal(SIGINT,SIG_IGN);
	signal(SIGQUIT,SIG_IGN);
	if(ioctl(0,KIOCSOUND,morse_tone)==-1)
		perror("kiocsound");
	usleep(duration/morse_speed);
	if(ioctl(0,KIOCSOUND,0)==-1)
		perror("kiocsound");
	signal(SIGINT,SIG_DFL);
	signal(SIGQUIT,SIG_DFL);
}

void PlayMorse(Morse *m)
{
	int d=0;
	while(d<6 && m->data[d]!=NIL)
	{
		if(m->data[d]==DIH)
			PlayBeep(500000);
		if(m->data[d]==DAH)
			PlayBeep(1500000);
		d++;
		usleep(500000/morse_speed);
	}
	usleep(1500000/morse_speed+100000*morse_gap);
}

void PlayMorseString(char *x)
{
	while(*x)
	{
		char c=islower(*x)?toupper(*x):*x;
		Morse *m=CharToMorse(c);
		if(m!=NULL)
		{
			printf("\r%75s\r","");
			PrintMorse(m);
			fflush(stdout);
			PlayMorse(m);
		}
		x++;
	}
	printf("\r%75s\r","");
}

void PlayFile(char *x)
{
	FILE *f=fopen(x,"r");
	char buf[512];
	if(f==NULL)
	{
		perror(x);
		return;
	}
	while(fgets(buf,511,f)!=NULL)
	{
		printf("%s",buf);
		buf[strlen(buf)-1]=0;
		PlayMorseString(buf);
	}
	fclose(f);
}

char PickCharacter()
{
	static char Options[]="ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789?/";

	int v=(rand()<<4)/239;

	return(Options[v%strlen(Options)]);
}

void MakeUpSequence(char *buf, int gs, int ng)
{
	int ct=0;
	while(ct<ng)
	{
		int cc=0;
		while(cc<gs)
		{
			*buf++=PickCharacter();
			cc++;
		}
		*buf++=' ';
		ct++;
	}
	buf[-1]=0;
}

void GroupTest(int groupsize, int numgroup)
{
	char *buf;
	buf=(char *)malloc((groupsize+1)*numgroup);
	if(buf==NULL)
	{
		fprintf(stderr,"Out of memory.\n");
		return;
	}
	printf("Random group practice mode.\n\n\
The computer will generate %d %d character combination%s and send them\n\
You should write the answer down and when ready hit return to see the\n\
coreect sequence.\n",numgroup,groupsize,numgroup==1?"":"s");

	while(1)
	{
		MakeUpSequence(buf,groupsize,numgroup);
		printf("Hit newline when ready for groups.  ");
		while(getchar()!='\n'&&!feof(stdin));
		if(feof(stdin))
		{
			free(buf);
			return;
		}
		PlayMorseString(buf);
		printf("Hit newline to see the groups.  ");
		while(getchar()!='\n'&&!feof(stdin));
		if(feof(stdin))
		{	
			free(buf);
			return;
		}
		printf("Sequence:\n%s\n",buf);
	}
}		
		

void main(argc,argv)
int argc;
char *argv[];
{
	char **argp;
	int ct=0;
	int num1,num2;
	long t;
#ifdef DEBUG
	while(MorseTable[ct].code)
	{
		printf("%c = ",MorseTable[ct].code);
		PrintMorse(&MorseTable[ct]);
		printf("\n");
		ct++;
	}
#endif
	if(argc==1 || argv[1][0]!='-' || argv[1][1]=='h')
	{
		fprintf(stderr,"\
Send message: 	%s -p \"message\"\n\
Send file: 	%s -f file\n\
Group practice: %s -g groupsize [numgroups]\n\
Show version: 	%s -v\n\
\n\
Additional options (place first)\n\
Set tone:	%s -t tonevalue\n\
Relative speed: %s -s speed (1-30)\n\
Extra delay:	%s -d delay (default is 9, correct morse is 0)\n\
",
		argv[0],argv[0],argv[0],argv[0],argv[0],argv[0],argv[0]);
		exit(1);
	}
	argp=argv;

	while(argc>2)
	{
		if(strcmp(argp[1],"-t")==0)
		{
			if(sscanf(argp[2],"%d",&num1)==0)
			{
				fprintf(stderr,"Tone value must be a number.\n");
				exit(1);
			}
			if(num1<100||num1>5000)
			{
				fprintf(stderr,"Tone range is 100-5000.\n");
				exit(1);
			}
			morse_tone=num1;
			argc-=2;
			argp+=2;
		}
		else if(strcmp(argp[1],"-s")==0)
		{
			if(sscanf(argp[2],"%d",&num1)==0)
			{
				fprintf(stderr,"Speed value must be a number.\n");
				exit(1);
			}
			if(num1<5||num1>30)
			{
				fprintf(stderr,"Speed range is 5-30.\n");
				exit(1);
			}
			morse_speed=num1;
			argc-=2;
			argp+=2;
		}
		else if(strcmp(argp[1],"-d")==0)
		{
			if(sscanf(argp[2],"%d",&num1)==0)
			{
				fprintf(stderr,"Delay value must be a number.\n");
				exit(1);
			}
			if(num1<0||num1>30)
			{
				fprintf(stderr,"Delay range is 0-30.\n");
				exit(1);
			}
			morse_gap=num1;
			argc-=2;
			argp+=2;
		}
		else
			break;
	}

	if(argc<2)
	{
		fprintf(stderr,"One of -p, -f, -v, or -g must be specified.\n");
		exit(1);
	}

	/* What do we do */
	switch(argp[1][1])
	{
		case 'p':
			if(argc!=3)
			{	
				fprintf(stderr,"%s -p \"text\".\n",argv[0]);
				exit(1);
			}
			PlayMorseString(argp[2]);
			break;
		case 'v':
			printf("GW4PTS Morse tutor for Linux: Revision 1.00\n");
			break;
		case 'f':
			if(argc!=3)
			{
				fprintf(stderr,"%s -f filename.\n",argv[0]);
				exit(1);
			}
			PlayFile(argp[2]);
			break;
		case 'g':
			if(argc>4||argc<3)
			{
				fprintf(stderr,"%s -g groupsize [numgroups]\n",argv[0]);
				exit(1);
			}
			if(argc==4)
			{
				if(sscanf(argp[3],"%d",&num2)==0)
				{
					fprintf(stderr,"Invalid number of groups '%s'.\n",argp[3]);
					exit(1);
				}
			}
			else
				num2=1;
			if(sscanf(argp[2],"%d",&num1)==0)
			{
				fprintf(stderr,"Invalid group size '%s'.\n",argp[2]);
				exit(1);
			}

			if(num1==0 || num2==0 || (num1+1)*num2>77)
			{
				fprintf(stderr,"Unsuitable numbers.\n");
				exit(1);
			}

			time(&t);
			srand(t);

			GroupTest(num1,num2);
			
			break;
		default:
			fprintf(stderr,"Unknown option '%c', %s -h for help.\n",argp[1][1],argv[0]);
			exit(1);	
	}
}



