#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <math.h>
#include <signal.h>

#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/XawInit.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Simple.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Text.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/Scrollbar.h>

#include "config.h"

#ifdef HAVE_MITSHM
#include <X11/extensions/XShm.h>
#endif

#ifdef HAVE_LIBJPEG
#include "jpeglib.h"
#endif

#ifdef HAVE_LIBTIFF
#include "tiff.h"
#include "tiffio.h"
#endif

#include "toolbox.h"
#include "complete.h"
#include "pcd.h"
#include "x11.h"
#include "dither.h"
#include "shmalloc.h"
#include "xpcd.h"
#include "ipc.h"

/* ----------------------------------------------------------------------- */

#define LOWER 32
#define UPPER 224

#define PLOT_GRAY    1
#define PLOT_RED     2
#define PLOT_GREEN   4
#define PLOT_BLUE    8
#define APPLY_RED   16
#define APPLY_GREEN 32
#define APPLY_BLUE  64

#define SAVE_PPM     1
#define SAVE_JPEG    2
#define SAVE_TIFF    3

struct VIEWER {
    int             width, height, have, pid, busy, fd;
    Widget          shell, label;
    char           *title;
    char           *filename;
    unsigned char  *rgb_data;
    unsigned char  *ximage_line;
    XtInputId       iproc;
    XtWorkProcId    wproc;
    GC              wgc;
    XImage         *ximage;
    void           *ximage_shm;

    /* icon */
    unsigned int    ix, iy, is, dx, dy;		/* icon width, height, scale */
    unsigned char  *icon_rgb_data;
    Pixmap          icon_pixmap;

    /* save */
    int             line, format;
    FILE           *fp;
#ifdef HAVE_LIBJPEG
    struct jpeg_compress_struct cinfo;
    struct jpeg_error_mgr jerr;
#endif
#ifdef HAVE_LIBTIFF
    TIFF           *TiffHndl;
#endif

    /* hist */
    unsigned int    hist_red[256];
    unsigned int    hist_green[256];
    unsigned int    hist_blue[256];
    unsigned int    hist_gray[256];
    unsigned int    hist_max;
    unsigned int    hist_plot;
    Widget          hist_shell, hist_label, plot_area;
    GC              hist_gc;

    unsigned int    hist_map[256];
    int             map_left, map_right, map_gamma, map_what;	/* gamma: *100 */

    int             filter, preview, use_grays, load_grays;	/* flags */
};

/* ----------------------------------------------------------------------- */

Boolean
viewer_save_WP(XtPointer client_data)
{
    struct VIEWER  *v;
    unsigned char  *line;

    v = client_data;
    if (v->line < v->height) {
	line = v->rgb_data + v->width * v->line * (v->load_grays ? 1 : 3);
	switch (v->format) {
#if HAVE_LIBJPEG
	case SAVE_JPEG:
	    jpeg_write_scanlines(&(v->cinfo), &line, 1);
	    break;
#endif
#ifdef HAVE_LIBTIFF
	case SAVE_TIFF:
	    {
		char *doublebuffer = malloc(v->width * (v->load_grays?1:3));
		memcpy(doublebuffer,line,v->width * (v->load_grays?1:3));
		TIFFWriteScanline(v->TiffHndl, doublebuffer, v->line, 0);
		free(doublebuffer);
	    }
	    break;
#endif
	case SAVE_PPM:
	    fwrite(line, v->width, (v->load_grays ? 1 : 3), v->fp);
	    break;
	}
	v->line++;
	return FALSE;
    }
    switch (v->format) {
#if HAVE_LIBJPEG
    case SAVE_JPEG:
	jpeg_finish_compress(&(v->cinfo));
	jpeg_destroy_compress(&(v->cinfo));
	fclose(v->fp);
	v->fp = NULL;
	break;
#endif
#ifdef HAVE_LIBTIFF
    case SAVE_TIFF:
	TIFFClose(v->TiffHndl);
	v->TiffHndl = NULL;
	break;
#endif
    case SAVE_PPM:
	fclose(v->fp);
	v->fp = NULL;
	break;
    }
    XtVaSetValues(v->shell, XtNtitle, v->title, NULL);
    v->busy = 0;
    v->wproc = 0;
    return TRUE;
}

void
viewer_save_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    struct VIEWER  *v;
    char           *value, *filename;
    Widget          dialog = XtParent(widget);

    v = client_data;
    value = XawDialogGetValueString(dialog);
    filename = tilde_expand(value);

    switch (v->format) {
#ifdef HAVE_LIBJPEG
    case SAVE_JPEG:
	if (NULL == (v->fp = fopen(filename, "w"))) {
	    xperror(app_shell, filename);
	    return;
	}
	v->cinfo.err = jpeg_std_error(&(v->jerr));
	jpeg_create_compress(&(v->cinfo));
	jpeg_stdio_dest(&(v->cinfo), v->fp);
	v->cinfo.image_width = v->width;
	v->cinfo.image_height = v->height;
	v->cinfo.input_components = v->load_grays ? 1 : 3;
	v->cinfo.in_color_space = v->load_grays ? JCS_GRAYSCALE : JCS_RGB;
	jpeg_set_defaults(&(v->cinfo));
	jpeg_set_quality(&(v->cinfo), jpeg_quality, TRUE);
	jpeg_start_compress(&(v->cinfo), TRUE);
	break;
#endif
#ifdef HAVE_LIBTIFF
    case SAVE_TIFF:
	v->TiffHndl = TIFFOpen(filename, "w");
	if (v->TiffHndl == NULL) {
	    xperror(app_shell, filename);
	    return;
	}
	TIFFSetField(v->TiffHndl, TIFFTAG_IMAGEWIDTH, v->width);
	TIFFSetField(v->TiffHndl, TIFFTAG_IMAGELENGTH, v->height);
	TIFFSetField(v->TiffHndl, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
	TIFFSetField(v->TiffHndl, TIFFTAG_PHOTOMETRIC,
	       v->load_grays ? PHOTOMETRIC_MINISBLACK : PHOTOMETRIC_RGB);
	TIFFSetField(v->TiffHndl, TIFFTAG_BITSPERSAMPLE, 8);
	TIFFSetField(v->TiffHndl, TIFFTAG_SAMPLESPERPIXEL,
		     v->load_grays ? 1 : 3);
	TIFFSetField(v->TiffHndl, TIFFTAG_ROWSPERSTRIP, 2);
	TIFFSetField(v->TiffHndl, TIFFTAG_ORIENTATION, ORIENTATION_TOPLEFT);
	if (tiff_compress) {
	    TIFFSetField(v->TiffHndl, TIFFTAG_COMPRESSION, COMPRESSION_LZW);
	    TIFFSetField(v->TiffHndl, TIFFTAG_PREDICTOR, 2);
	}
	break;
#endif
    case SAVE_PPM:
	if (NULL == (v->fp = fopen(filename, "w"))) {
	    xperror(app_shell, filename);
	    return;
	}
	fprintf(v->fp, "P%c\n%d %d\n255\n",
		v->load_grays ? '5' : '6', v->width, v->height);
	break;
    }

    free(filename);
    XtDestroyWidget(XtParent(dialog));
    XtVaSetValues(v->shell,
		  XtNtitle, get_string_resource(app_shell, "str_save"),
		  NULL);
    v->line = 0;
    v->wproc = XtAppAddWorkProc
	(app_context, viewer_save_WP, (XtPointer) v);
}

/* ----------------------------------------------------------------------- */

void
viewer_filter_map(struct VIEWER *v, unsigned char *data, int len)
{
    register int    i;

    if (v->load_grays) {
	for (i = 0; i < len; i++)
	    data[i] = v->hist_map[data[i]];
    } else {
	if (v->hist_plot & APPLY_RED)
	    for (i = 0; i < 3 * len; i += 3)
		data[i] = v->hist_map[data[i]];
	if (v->hist_plot & APPLY_GREEN)
	    for (i = 1; i < 3 * len; i += 3)
		data[i] = v->hist_map[data[i]];
	if (v->hist_plot & APPLY_BLUE)
	    for (i = 2; i < 3 * len; i += 3)
		data[i] = v->hist_map[data[i]];
    }
}

Boolean
view_mk_ximage_WP(XtPointer client_data)
{
    struct VIEWER  *v;
    Pixmap          ip;
    unsigned char  *icon_ximage;
    unsigned char  *rgb_data, *blk = NULL;
    unsigned long  *map;

    v = client_data;
    if (v->line < v->height) {
	rgb_data = v->rgb_data + (v->load_grays ? 1 : 3) * v->width * v->line;
	if (v->filter) {
	    if (v->preview) {
		blk = malloc((v->load_grays ? 1 : 3) * v->width);
		rgb_data = memcpy(blk, rgb_data, (v->load_grays ? 1 : 3) * v->width);
	    }
	    viewer_filter_map(v, rgb_data, v->width);
	}
	if (v->use_grays && !v->load_grays) {
	    int             i;
	    unsigned char  *hlp;

	    hlp = malloc(v->width);
	    for (i = 0; i < v->width; i++)
		hlp[i] = (rgb_data[3 * i] * 3 +
			  rgb_data[3 * i + 1] * 6 +
			  rgb_data[3 * i + 2]) / 10;
	    if (blk)
		free(blk);
	    rgb_data = blk = hlp;
	}
	if (display_type == PSEUDOCOLOR) {
	    register int    x;
	    register unsigned char *d;

	    if (v->use_grays) {
		map = x11_map_gray;
		dither_line_gray(rgb_data, v->ximage_line, v->line, v->width);
	    } else {
		map = x11_map;
		dither_line(rgb_data, v->ximage_line, v->line, v->width);
	    }
	    d = v->ximage_line;
	    for (x = 0; x < v->width; x++, d++)
		XPutPixel(v->ximage, x, v->line, map[*d]);
	} else {
	    register int    x;
	    register unsigned long d;

	    if (v->use_grays) {
		for (x = 0; x < v->width; x++) {
		    d = x11_lut_gray[rgb_data[x]];
		    XPutPixel(v->ximage, x, v->line, d);
		}
	    } else {
		for (x = 0; x < v->width; x++) {
		    d = x11_lut_red[rgb_data[3 * x]] |
			x11_lut_green[rgb_data[3 * x + 1]] |
			x11_lut_blue[rgb_data[3 * x + 2]];
		    XPutPixel(v->ximage, x, v->line, d);
		}
	    }
	}

	if (0 == ((v->line - v->dy) % v->is) &&
	    ((v->line - v->dy) / v->is) < v->iy) {
	    int             il, i;
	    unsigned char  *src, *dest;

	    /* copy rgb data for icon */
	    il = (v->line - v->dy) / v->is;
	    src = rgb_data + (v->use_grays ? 1 : 3) * v->dx;
	    dest = v->icon_rgb_data + (v->use_grays ? 1 : 3) * v->ix * il;
	    if (v->use_grays) {
		for (i = 0; i < v->ix; i++) {
		    dest[0] = src[0];
		    src += v->is;
		    dest++;
		}
	    } else {
		for (i = 0; i < v->ix; i++) {
		    dest[0] = src[0];
		    dest[1] = src[1];
		    dest[2] = src[2];
		    src += 3 * v->is;
		    dest += 3;
		}
	    }
	}
	if (31 == (v->line & 31)) {
	    XClearArea(dpy, XtWindow(v->label),
		       0, (v->line & ~31), v->width, 32, True);
	}
	v->line++;
	if (blk)
	    free(blk);
	return FALSE;
    }
    ip = v->icon_pixmap;

    icon_ximage = malloc(v->ix * v->iy * ((display_type == PSEUDOCOLOR) ? 1 : 4));
    x11_data_to_ximage(v->icon_rgb_data, icon_ximage,
		       v->ix, v->iy, 0, v->use_grays);
    v->icon_pixmap =
	x11_create_pixmap(app_shell, icon_ximage, v->ix, v->iy, v->use_grays);
    free(icon_ximage);

    XtVaSetValues(v->shell,
		  XtNiconPixmap, v->icon_pixmap,
		  XtNtitle, v->title,
		  NULL);
    XClearArea(dpy, XtWindow(v->label),
	0, (v->line & ~31), v->width, v->height - (v->line & ~31), True);

    if (ip)
	XFreePixmap(dpy, ip);
    v->filter = 0;
    v->busy = 0;
    v->wproc = 0;
    return TRUE;
}

static void
viewer_input(XtPointer data, int *fd, XtInputId * iproc)
{
    struct VIEWER  *v;

    v = data;

    XtRemoveInput(v->iproc);
    v->iproc = 0;
    close(v->fd);
    v->fd = 0;
    v->pid = 0;

    if (display_type == PSEUDOCOLOR)
	v->ximage_line = malloc(v->width);

    XtVaSetValues(v->shell,
		XtNtitle, get_string_resource(app_shell, "str_mkximage"),
		  NULL);

    v->line = 0;
    v->wproc = XtAppAddWorkProc
	(app_context, view_mk_ximage_WP, (XtPointer) v);
}

/* ----------------------------------------------------------------------- */

void
viewer_calc_hist(struct VIEWER *v)
{
    int             i, n, gray;

    memset(v->hist_red, 0, 256 * sizeof(unsigned int));
    memset(v->hist_green, 0, 256 * sizeof(unsigned int));
    memset(v->hist_blue, 0, 256 * sizeof(unsigned int));
    memset(v->hist_gray, 0, 256 * sizeof(unsigned int));

    v->hist_max = 0;
    n = v->height * v->width;

    for (i = 0; i < 256; i++)
	v->hist_map[i] = i;
    v->map_left = 0;
    v->map_right = 255;
    v->map_gamma = 100;

    if (v->load_grays) {
	for (i = 0; i < n; i++)
	    v->hist_gray[v->rgb_data[i]]++;
	for (i = 4; i < 252; i++)
	    if (v->hist_gray[i] > v->hist_max)
		v->hist_max = v->hist_gray[i];
    } else {
	for (i = 0; i < n; i++) {
	    v->hist_red[v->rgb_data[3 * i]]++;
	    v->hist_green[v->rgb_data[3 * i + 1]]++;
	    v->hist_blue[v->rgb_data[3 * i + 2]]++;
	    gray = (v->rgb_data[3 * i] * 3 + v->rgb_data[3 * i + 1] * 6 + v->rgb_data[3 * i + 2] * 1) / 10;
	    v->hist_gray[gray]++;
	}

	/* look for max value, kick out black & white */
	for (i = 4; i < 252; i++) {
	    if (v->hist_gray[i] > v->hist_max)
		v->hist_max = v->hist_gray[i];
	    if (v->hist_red[i] > v->hist_max)
		v->hist_max = v->hist_red[i];
	    if (v->hist_green[i] > v->hist_max)
		v->hist_max = v->hist_green[i];
	    if (v->hist_blue[i] > v->hist_max)
		v->hist_max = v->hist_blue[i];
	}
    }
    v->hist_plot = PLOT_GRAY | PLOT_RED | PLOT_GREEN | PLOT_BLUE |
	APPLY_RED | APPLY_GREEN | APPLY_BLUE;
}

void
viewer_plot(Widget widget, struct VIEWER *v,
	    unsigned long color, unsigned int *data, int flags)
{
    int             i, y1, y2;
    XGCValues       values;

    values.foreground = color;
    XChangeGC(dpy, v->hist_gc, GCForeground, &values);

    for (i = 0; i < 256; i++) {
	if (flags & 1)
	    y1 = 0;
	else
	    y1 = ((i > 0) ? data[i - 1] : data[i]);
	y2 = data[i];

	if (flags & 2) {
	    y1 = 255 - y1;
	    y2 = 255 - y2;
	} else {
	    y1 = 255 - y1 * 255 / v->hist_max;
	    y2 = 255 - y2 * 255 / v->hist_max;
	}
	XDrawLine(dpy, XtWindow(widget), v->hist_gc,
		  ((i > 0) && !(flags & 1)) ? i - 1 : i, y1,
		  i, y2);
    }
}

void
viewer_expose_hist(Widget widget, XtPointer client_data, XExposeEvent * event)
{
    struct VIEWER  *v;
    XGCValues       values;

    v = client_data;
    if (v->hist_gc == 0) {
	v->hist_gc = XCreateGC(dpy, XtWindow(widget),
		       0, NULL /* GCForeground|GCBackground,&values */ );
    }
    values.foreground = x11_lightgray;
    XChangeGC(dpy, v->hist_gc, GCForeground, &values);
    XDrawLine(dpy, XtWindow(widget), v->hist_gc, 0, LOWER, 255, LOWER);
    XDrawLine(dpy, XtWindow(widget), v->hist_gc, 0, UPPER, 255, UPPER);

    if (v->hist_plot & PLOT_GRAY)
	viewer_plot(widget, v, x11_gray, v->hist_gray, 1);
    if (!v->load_grays) {
	if (v->hist_plot & PLOT_RED)
	    viewer_plot(widget, v, x11_red, v->hist_red, 0);
	if (v->hist_plot & PLOT_GREEN)
	    viewer_plot(widget, v, x11_green, v->hist_green, 0);
	if (v->hist_plot & PLOT_BLUE)
	    viewer_plot(widget, v, x11_blue, v->hist_blue, 0);
    }
    viewer_plot(widget, v, x11_black, v->hist_map, 2);
}

void
viewer_mouse_hist(Widget widget, XtPointer client_data, XEvent * event)
{
    struct VIEWER  *v;
    int             i, n;
    float           g;
    char            title[64];

    v = client_data;
    if ((event->type == ButtonPress && event->xbutton.button == Button1) ||
	event->type == MotionNotify) {
	if (event->type == ButtonPress) {
	    if (event->xbutton.y < LOWER)
		/* top -- upper limit */
		v->map_what = 1;
	    else if (event->xbutton.y > UPPER)
		/* bottom -- lower limit */
		v->map_what = 2;
	    else
		/* middle: gamma */
		v->map_what = 3;
	}
	switch (v->map_what) {
	case 1:
	    /* top -- upper limit */
	    if (event->xbutton.x > v->map_left && event->xbutton.x < 256)
		v->map_right = event->xbutton.x;
	    break;
	case 2:
	    /* bottom -- lower limit */
	    if (event->xbutton.x < v->map_right && event->xbutton.x >= 0)
		v->map_left = event->xbutton.x;
	    break;
	case 3:
	    /* middle: gamma */
	    if (event->xbutton.x < v->map_right &&
		event->xbutton.x > v->map_left) {
		n = (event->xbutton.x - v->map_left)
		    * 255 / (v->map_right - v->map_left);
		v->map_gamma =
		    (int) ((log(n / 255.0) * 100 /
			    log((255 - event->xbutton.y) / 255.0)) + 0.5);
	    }
	    break;
	}
	for (i = 0; i < v->map_left; i++)
	    v->hist_map[i] = 0;
	g = 100.0 / (float) v->map_gamma;
	for (; i < v->map_right; i++) {
	    n = (i - v->map_left) * 255 / (v->map_right - v->map_left);
	    v->hist_map[i] = ((int) (0.5 + 255 * pow((float) n / 255.0, g)));
	}
	for (; i < 256; i++)
	    v->hist_map[i] = 255;
	XClearArea(dpy, XtWindow(v->plot_area),
		   0, 0, 0, 0, True);
	sprintf(title, "[ %d ; %d ], gamma = %5.2f", v->map_left, v->map_right,
		(float) v->map_gamma / 100.0);
	XtVaSetValues(v->hist_label, XtNlabel, title, NULL);

	v->busy = 1;
	v->filter = 1;
	v->preview = 1;
	v->line = 0;
	if (!v->wproc)
	    v->wproc = XtAppAddWorkProc
		(app_context, view_mk_ximage_WP, (XtPointer) v);
	XtVaSetValues(v->shell,
		 XtNtitle, get_string_resource(app_shell, "str_preview"),
		      NULL);
    }
}

static void
viewer_hist_toggle_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    int             state, mask = 0;
    char           *name;
    struct VIEWER  *v;

    v = client_data;
    XtVaGetValues(widget, XtNstate, &state, NULL);
    name = XtName(widget);
    if (0 == strcmp(name, "gray"))
	mask = PLOT_GRAY;
    if (0 == strcmp(name, "red"))
	mask = PLOT_RED;
    if (0 == strcmp(name, "green"))
	mask = PLOT_GREEN;
    if (0 == strcmp(name, "blue"))
	mask = PLOT_BLUE;
    if (0 == strcmp(name, "red2"))
	mask = APPLY_RED;
    if (0 == strcmp(name, "green2"))
	mask = APPLY_GREEN;
    if (0 == strcmp(name, "blue2"))
	mask = APPLY_BLUE;
    if (state & 1)
	v->hist_plot |= mask;
    else
	v->hist_plot &= ~mask;
    if (mask & (PLOT_GRAY | PLOT_RED | PLOT_GREEN | PLOT_BLUE))
	XClearArea(dpy, XtWindow(v->plot_area),
		   0, 0, 0, 0, True);
    if (mask & (APPLY_RED | APPLY_GREEN | APPLY_BLUE)) {
	v->busy = 1;
	v->filter = 1;
	v->preview = 1;
	v->line = 0;
	if (!v->wproc)
	    v->wproc = XtAppAddWorkProc
		(app_context, view_mk_ximage_WP, (XtPointer) v);
	XtVaSetValues(v->shell,
		 XtNtitle, get_string_resource(app_shell, "str_preview"),
		      NULL);
    }
}

static void
viewer_hist_ok_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    struct VIEWER  *v;

    v = client_data;
    v->busy = 1;
    v->filter = 1;
    v->preview = 0;
    v->line = 0;
    if (!v->wproc)
	v->wproc = XtAppAddWorkProc
	    (app_context, view_mk_ximage_WP, (XtPointer) v);

    XtVaSetValues(v->shell,
		  XtNtitle, get_string_resource(app_shell, "str_busy"),
		  NULL);
    XtDestroyWidget(v->hist_shell);
}

static void
viewer_hist_cancel_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    struct VIEWER  *v;

    v = client_data;
    if (v->preview) {
	v->busy = 1;
	v->filter = 0;
	v->preview = 0;
	v->line = 0;
	if (!v->wproc)
	    v->wproc = XtAppAddWorkProc
		(app_context, view_mk_ximage_WP, (XtPointer) v);

	XtVaSetValues(v->shell,
		    XtNtitle, get_string_resource(app_shell, "str_busy"),
		      NULL);
    }
    XtDestroyWidget(v->hist_shell);
}

#define FIX_LEFT_TOP       \
    XtNleft,XawChainLeft,  \
    XtNright,XawChainLeft, \
    XtNtop,XawChainTop,    \
    XtNbottom,XawChainTop

void
viewer_hist(Widget parent, struct VIEWER *v)
{
    Widget          shell, form, label, simple, l1, gray, r, g, b, l2,
                    r2, g2, b2, ok, cancel;

    v->hist_shell = shell =
	XtVaCreatePopupShell("hist", transientShellWidgetClass, parent, NULL);
    form = XtVaCreateManagedWidget("form", formWidgetClass, shell,
				   NULL);
    v->hist_label = label =
	XtVaCreateManagedWidget("label", labelWidgetClass, form,
				FIX_LEFT_TOP,
				XtNwidth, 256,
				NULL);
    v->plot_area = simple =
	XtVaCreateManagedWidget("simple", simpleWidgetClass, form,
				FIX_LEFT_TOP,
				XtNwidth, 256,
				XtNheight, 256,
				XtNfromVert, label,
				NULL);

    if (!v->load_grays) {
	l1 = XtVaCreateManagedWidget("l1", labelWidgetClass, form,
				     FIX_LEFT_TOP,
				     XtNfromVert, simple,
				     NULL);
	gray = XtVaCreateManagedWidget("gray", toggleWidgetClass, form,
				       FIX_LEFT_TOP,
				       XtNstate, True,
				       XtNfromVert, simple,
				       XtNfromHoriz, l1,
				       NULL);
	r = XtVaCreateManagedWidget("red", toggleWidgetClass, form,
				    FIX_LEFT_TOP,
				    XtNstate, True,
				    XtNfromVert, simple,
				    XtNfromHoriz, gray,
				    NULL);
	g = XtVaCreateManagedWidget("green", toggleWidgetClass, form,
				    FIX_LEFT_TOP,
				    XtNstate, True,
				    XtNfromVert, simple,
				    XtNfromHoriz, r,
				    NULL);
	b = XtVaCreateManagedWidget("blue", toggleWidgetClass, form,
				    FIX_LEFT_TOP,
				    XtNstate, True,
				    XtNfromVert, simple,
				    XtNfromHoriz, g,
				    NULL);

	l2 = XtVaCreateManagedWidget("l2", labelWidgetClass, form,
				     FIX_LEFT_TOP,
				     XtNfromVert, l1,
				     NULL);
	r2 = XtVaCreateManagedWidget("red2", toggleWidgetClass, form,
				     FIX_LEFT_TOP,
				     XtNstate, True,
				     XtNfromVert, l1,
				     XtNfromHoriz, gray,
				     NULL);
	g2 = XtVaCreateManagedWidget("green2", toggleWidgetClass, form,
				     FIX_LEFT_TOP,
				     XtNstate, True,
				     XtNfromVert, l1,
				     XtNfromHoriz, r2,
				     NULL);
	b2 = XtVaCreateManagedWidget("blue2", toggleWidgetClass, form,
				     FIX_LEFT_TOP,
				     XtNstate, True,
				     XtNfromVert, l1,
				     XtNfromHoriz, g2,
				     NULL);
	XtAddCallback(gray, XtNcallback, viewer_hist_toggle_CB, v);
	XtAddCallback(r, XtNcallback, viewer_hist_toggle_CB, v);
	XtAddCallback(g, XtNcallback, viewer_hist_toggle_CB, v);
	XtAddCallback(b, XtNcallback, viewer_hist_toggle_CB, v);
	XtAddCallback(r2, XtNcallback, viewer_hist_toggle_CB, v);
	XtAddCallback(g2, XtNcallback, viewer_hist_toggle_CB, v);
	XtAddCallback(b2, XtNcallback, viewer_hist_toggle_CB, v);
    }
    ok = XtVaCreateManagedWidget("ok", commandWidgetClass, form,
				 FIX_LEFT_TOP,
				 XtNfromVert, v->load_grays ? simple : l2,
				 NULL);
    cancel = XtVaCreateManagedWidget("cancel", commandWidgetClass, form,
				     FIX_LEFT_TOP,
				XtNfromVert, v->load_grays ? simple : l2,
				     XtNfromHoriz, ok,
				     NULL);
    XtAddCallback(ok, XtNcallback, viewer_hist_ok_CB, v);
    XtAddCallback(cancel, XtNcallback, viewer_hist_cancel_CB, v);
    XtAddEventHandler(simple, ExposureMask, False,
		      (XtEventHandler) viewer_expose_hist,
		      (XtPointer) v);
    XtAddEventHandler(simple, ButtonPressMask | Button1MotionMask, False,
		      (XtEventHandler) viewer_mouse_hist,
		      (XtPointer) v);
    viewer_calc_hist(v);

    center_under_mouse(shell, 256, 300);
    XtPopup(shell, XtGrabNonexclusive);
    if (shell)
	XDefineCursor(dpy, XtWindow(shell), left_ptr);
    XtInstallAllAccelerators
	(form, v->hist_shell);
}

/* ----------------------------------------------------------------------- */

static struct MENU viewer_menu[] =
{
    {1, "hist", NULL, 0},
    {2, "color", NULL, 0},
    {0, "", NULL, 0},
    {3, "ppm", NULL, 0},
#ifdef HAVE_LIBJPEG
    {4, "jpeg", NULL, 0},
#endif
#ifdef HAVE_LIBTIFF
    {5, "tiff", NULL, 0},
#endif
    {0, NULL, NULL, 0}
};

void
viewer_mouse_event(Widget widget,
		   XtPointer client_data,
		   XEvent * event)
{
    struct VIEWER  *v;
    int             sel;

    v = client_data;

    switch (event->type) {
    case ButtonRelease:
	if (event->xbutton.button == Button3 && !v->busy) {
	    viewer_menu[1].disabled = (force_grays || v->load_grays);
	    sel = popup_menu(widget, viewer_menu);
	    switch (sel) {
	    case 1:
		viewer_hist(widget, v);
		break;
	    case 2:
		if (!force_grays && !v->load_grays) {
		    v->use_grays = !v->use_grays;
		    v->busy = 1;
		    v->line = 0;
		    v->wproc = XtAppAddWorkProc(app_context,
						view_mk_ximage_WP,
						(XtPointer) v);
		    XtVaSetValues(v->shell, XtNtitle,
			      get_string_resource(app_shell, "str_busy"),
				  NULL);
		}
		break;
	    case 3:
		v->format = SAVE_PPM;
		get_user_string(app_shell,
				"str_save_ppm_title",
				"str_save_ppm_label",
				"",
				viewer_save_CB, v);
		break;
#ifdef HAVE_LIBJPEG
	    case 4:
		v->format = SAVE_JPEG;
		get_user_string(app_shell,
				"str_save_jpeg_title",
				"str_save_jpeg_label",
				"",
				viewer_save_CB, v);
		break;
#endif
#ifdef HAVE_LIBTIFF
	    case 5:
		v->format = SAVE_TIFF;
		get_user_string(app_shell,
				"str_save_tiff_title",
				"str_save_tiff_label",
				"",
				viewer_save_CB, v);
		break;
#endif
	    default:
		break;
	    }
	}
	break;
    }
}

/* ----------------------------------------------------------------------- */

static void
viewer_expose_event(Widget widget,
		    XtPointer client_data,
		    XExposeEvent * event)
{
    struct VIEWER  *v;

    v = (struct VIEWER *) client_data;
    XPUTIMAGE(dpy, XtWindow(widget), v->wgc, v->ximage,
	      event->x, event->y, event->x, event->y,
	      event->width, event->height);
}

/* ----------------------------------------------------------------------- */

void
destroy_viewer_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    struct VIEWER  *v;

    v = client_data;
    if (v->iproc)
	XtRemoveInput(v->iproc);
    if (v->wproc)
	XtRemoveWorkProc(v->wproc);
    if (v->fd)
	close(v->fd);
    if (v->fp)
	fclose(v->fp);
    if (v->pid)
	kill(v->pid, SIGTERM);

    if (v->rgb_data)
	sh_free(v->rgb_data);
    if (v->ximage_line)
	free(v->ximage_line);
    if (v->ximage)
	x11_destroy_ximage(widget, v->ximage, v->ximage_shm);

    if (v->icon_pixmap)
	XFreePixmap(dpy, v->icon_pixmap);
    if (v->icon_rgb_data)
	free(v->icon_rgb_data);

    if (v->title)
	free(v->title);
    free(v);
}

/* ----------------------------------------------------------------------- */

void
new_viewer(int fd, unsigned char *addr, int pid, char *filename,
	   int width, int height)
{
    Widget          viewport;
    struct VIEWER  *v = NULL;

    v = malloc(sizeof(struct VIEWER));
    memset(v, 0, sizeof(struct VIEWER));

    v->fd = fd;
    v->pid = pid;
    v->rgb_data = addr;
    v->width = width;
    v->height = height;
    v->load_grays = load_grays;
    v->use_grays = load_grays ? 1 : use_grays;

    v->title = malloc(strlen(filename) + 8);
    strcpy(v->title, v->load_grays ? "gray: " : "RGB: ");
    strcat(v->title, filename);
    v->filename = strrchr(v->title, '/');
    v->iproc = XtAppAddInput(app_context, fd, (XtPointer) XtInputReadMask,
			     viewer_input, (XtPointer) v);
    v->busy = 1;

    v->shell = XtVaAppCreateShell("viewer", "Xpcd-2",
				  applicationShellWidgetClass,
				  dpy,
				  XtNtitle,
			      get_string_resource(app_shell, "str_load"),
				  XtNmaxWidth, v->width,
				  XtNmaxHeight, v->height,
				  NULL);

    XtOverrideTranslations(v->shell, XtParseTranslationTable
			   ("<Message>WM_PROTOCOLS: CloseFile()"));

    viewport = XtVaCreateManagedWidget("viewport",
				       viewportWidgetClass, v->shell,
				       XtNallowHoriz, True,
				       XtNallowVert, True,
				       NULL);
    v->label = XtVaCreateManagedWidget("label",
				       simpleWidgetClass, viewport,
				       XtNheight, v->height,
				       XtNwidth, v->width,
				       XtNbackgroundPixmap, None,
				       NULL);
    v->ximage = x11_create_ximage(v->shell, v->width, v->height,
				  &(v->ximage_shm));
    if (NULL == v->ximage) {
	/* user already has a error dialog box */
	destroy_viewer_CB(NULL, (XtPointer) v, NULL);
	return;
    }
    XtRealizeWidget(v->shell);
    add_window_to_list(v->shell, "viewer");
    XDefineCursor(dpy, XtWindow(v->shell), left_ptr);
    v->wgc = XCreateGC(dpy, XtWindow(v->label), 0, NULL);

    XSetWMProtocols(dpy, XtWindow(v->shell),
		    &wm_close, 1);
    XtAddCallback(v->shell, XtNdestroyCallback,
		  (XtCallbackProc) destroy_viewer_CB, (XtPointer) v);
    XtAddEventHandler(v->label,
		      ButtonPressMask | ButtonReleaseMask,
		      False, (XtEventHandler) viewer_mouse_event,
		      (XtPointer) v);
    XtAddEventHandler(v->label,
		      ExposureMask,
		      False, (XtEventHandler) viewer_expose_event,
		      (XtPointer) v);

    /* icon */
    v->ix = atoi(get_string_resource(v->shell, "ix"));
    v->iy = atoi(get_string_resource(v->shell, "iy"));
    if (v->ix > v->width)
	v->ix = v->width;
    if (v->iy > v->height)
	v->iy = v->height;
    v->is = min(v->height / v->iy, v->width / v->ix);
    v->dx = (v->width - v->ix * v->is) / 2;
    v->dy = (v->height - v->iy * v->is) / 2;
    v->icon_rgb_data = malloc(v->ix * v->iy * 3);
    memset(v->icon_rgb_data, 0, v->ix * v->iy * 3);

    /* kbd scroll */
    XtInstallAllAccelerators(v->label, v->shell);
    XtAddCallback(viewport, XtNreportCallback,
	      (XtCallbackProc) report_viewport_CB, (XtPointer) v->label);
    return;
}
