
/*
 * simple PhotoCD to ppm/jpeg converter
 *
 *   (c) 1996 Gerd Knorr <kraxel@cs.tu-berlin.de>
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>

#include "config.h"
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#else
extern int      optind;
extern char    *optarg;

#endif

#include "pcd.h"

#ifdef HAVE_LIBJPEG
#include "jpeglib.h"
#endif

#define TRUE  1
#define FALSE 0

#ifdef HAVE_GETOPT_LONG
static struct option long_opts[] =
{
    {"help", no_argument, NULL, 'h'},
    {"verbose", no_argument, NULL, 'v'},
    {"gray", no_argument, NULL, '1'},
    {"resolution", required_argument, NULL, 'r'},
    {"orientation", required_argument, NULL, 'o'},
    {"geometry", required_argument, NULL, 'g'},
    {"jpeg", optional_argument, NULL, 'j'},
    {NULL, 0, NULL, 0}
};

#endif

int             rot = -1, res = 3, left = 0, top = 0, width = 0, height = 0,
                bytes, jpeg, quality = 75;

void
usage(char *name)
{
    char           *h;

    h = strrchr(name, '/');
    fprintf(stderr,
	    "This program converts PhotoCD images to ppm or jpeg\n"
	    "\n"
	    "usage: %s [ options ] infile outfile\n"
	    "\n"
	    "options:\n"
#ifdef HAVE_GETOPT_LONG
	    "  --help             print this text\n"
	    "  --verbose          prints a rotor while decoding\n"
	    "  --gray             write grayscaled image\n"
	    "  --resolution=n     select resolution [1..5]\n"
	    "  --orientation=n    orientation [0..3]\n"
	    "  --geometry=WxH+L+T decode only a part of the image, all 4 "
	    "values are mandatory\n"
	  "                     W=width, H=height, L/T=left/top offset\n"
#ifdef HAVE_LIBJPEG
	    "  --jpeg[=quality]   write jpeg, optional set jpeg quality\n"
#endif
	    "\n"
	    "The short option is the first letter "
	    "(except '--gray', this one has '-1').\n"
#else				/* HAVE_GETOPT_LONG */
	    "  -h            print this text\n"
	    "  -v            prints a rotor while decoding\n"
	    "  -1            write grayscaled image\n"
	    "  -r n          select resolution [1..5]\n"
	    "  -o n          orientation [0..3]\n"
	    "  -g WxH+L+T    decode only a part of the image, all 4 "
	    "values are mandatory\n"
	    "                W=width, H=height, L/T=left/top offset\n"
#ifdef HAVE_LIBJPEG
	    "  -j [quality]  write jpeg, optional set jpeg quality\n"
#endif
	    "\n"
#endif				/* HAVE_GETOPT_LONG */
	    "Using '-' as outfile writes to stdout.\n"
	    "\n"
	    ,h ? h + 1 : name);
    exit(1);
}

int
main(int argc, char *argv[])
{
    int             gray = FALSE, verbose = FALSE, c, type, y;
    FILE           *fp = stdout;
    unsigned char  *image;
    struct PCD_IMAGE img;

#ifdef HAVE_LIBJPEG
    struct jpeg_compress_struct cinfo;
    struct jpeg_error_mgr jerr;

#endif

    /* getopt */
    for (;;) {
#ifdef HAVE_GETOPT_LONG
	c = getopt_long(argc, argv, "hv1r:o:g:j::", long_opts, NULL);
#else
	c = getopt(argc, argv, "hv1r:o:g:j::");
#endif
	if (c == -1)
	    break;
	switch (c) {
	case 'h':
	    usage(argv[0]);
	    break;
	case 'v':
	    verbose = TRUE;
	    break;
	case '1':
	    gray = TRUE;
	    break;
	case 'r':
	    res = atoi(optarg);
	    break;
	case 'o':
	    rot = atoi(optarg);
	    break;
	case 'g':
	    if (4 != (sscanf(optarg, "%ix%i+%i+%i",
			     &width, &height, &left, &top))) {
		fprintf(stderr, "invalid geometry specification\n");
		exit(1);
	    }
	    break;
	case 'j':
#ifdef HAVE_LIBJPEG
	    jpeg = TRUE;
	    if (optarg)
		quality = atoi(optarg);
	    break;
#else
	    fprintf(stderr, "Compiled without jpeg support, sorry.\n");
	    exit(1);
#endif
	default:
	    exit(1);
	}
    }

    if (optind + 2 != argc)
	usage(argv[0]);

    bytes = gray ? 1 : 3;
    type = gray ? PCD_TYPE_GRAY : PCD_TYPE_RGB;

    if (0 != pcd_open(&img, argv[optind])) {
	fprintf(stderr, "libpcd: %s\n", pcd_errmsg);
	exit(1);
    }
    if (-1 == rot)
	rot = pcd_get_rot(&img, 0);
    if (-1 == pcd_select(&img, res, 0, gray, verbose, rot,
			 &left, &top, &width, &height)) {
	fprintf(stderr, "libpcd: %s\n", pcd_errmsg);
	exit(1);
    }
    if (strcmp(argv[optind + 1], "-"))
	if (NULL == (fp = fopen(argv[optind + 1], "w"))) {
	    perror(argv[optind + 1]);
	    exit(1);
	}
#ifdef HAVE_LIBJPEG
    if (jpeg) {
	cinfo.err = jpeg_std_error(&jerr);
	jpeg_create_compress(&cinfo);
	jpeg_stdio_dest(&cinfo, fp);
	cinfo.image_width = width;
	cinfo.image_height = height;
	cinfo.input_components = bytes;
	cinfo.in_color_space = gray ? JCS_GRAYSCALE : JCS_RGB;
	jpeg_set_defaults(&cinfo);
	jpeg_set_quality(&cinfo, quality, TRUE);
	jpeg_start_compress(&cinfo, TRUE);
    } else
#endif
	fprintf(fp, "P%c\n%i %i\n255\n", gray ? '5' : '6', width, height);

    if (-1 == pcd_decode(&img)) {
	fprintf(stderr, "libpcd: %s\n", pcd_errmsg);
	exit(1);
    }
    image = malloc(width * bytes);
    for (y = 0; y < height; y++) {
	if (verbose)
	    fprintf(stderr, "%c\010", pcd_rotor[y & 3]);
	if (-1 == pcd_get_image_line(&img, y, image, type, 0)) {
	    fprintf(stderr, "libpcd: %s\n", pcd_errmsg);
	    exit(1);
	}
#ifdef HAVE_LIBJPEG
	if (jpeg) {
	    jpeg_write_scanlines(&cinfo, &image, 1);
	} else
#endif
	if (width != fwrite(image, bytes, width, fp)) {
	    perror("fwrite");
	    exit(1);
	}
    }
    if (verbose)
	fputc('*', stderr);
    pcd_close(&img);
    free(image);
#ifdef HAVE_LIBJPEG
    if (jpeg) {
	jpeg_finish_compress(&cinfo);
	jpeg_destroy_compress(&cinfo);
    }
#endif
    fclose(fp);
    exit(0);
}
