/*****************************************************************************
 *                                                                           *
 * Programm:  paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Modul:     filter.c                                                       *
 *            Bilder filtern                                                 *
 * Autor:     Andreas Tille                                                  *
 * Datum:     20.06.1998                                                     *
 *                                                                           *
 *****************************************************************************/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "paul.h" 

#ifdef __DMALLOC__
#include <dmalloc.h>
#endif

void MedianFilter3(unsigned char *buf, unsigned char *fil, int size, int ws, int hs)
/* linearer Medianfilter der Ordnung 3
 * --- Parameter: ---
 * unsigned char *buf : zu filternder Speicherbereich
 * unsigned char *fil : Speicherbereich in den das gefilterte Bild gespeichert wird
 * int           size : Anzahl der Bytes
 * int           ws   : horizontaler Abstand der zu filternden Bytes
 * int           hs   : vertikaler Abstand der zu filternden Bytes
 * --- Rueckgabe: ---
 * unsigned char *fil : gefilterter Speicherbereich
 */
{
   register int           s = ws;
   register unsigned char *ap, *bp, *cp, *filp, *fip;
   
   cp   = (bp = (ap = buf) + hs) + hs;
   filp = fil + hs;
   for ( fip = ap + size; cp < fip; ap += s, bp += s, cp += s, filp += s ) {
      if ( *ap > *cp ) {
         if ( *ap > *bp ) {  /* wenn *ap maximaler Wert, dann *bp = MAX(*bp,*cp) */
            if ( *cp > *bp ) *filp = *cp;  
            else             *filp = *bp;
	 } else /* *bp >= *ap > *cp   => *bp = *ap; Test auf == einfach weggelassen */
            *filp = *ap;
      } else {
         if ( *ap < *bp ) {  /* wenn *ap minimaler Wert, dann *bp = MIN(*bp,*cp) */
            if ( *cp < *bp ) *filp = *cp;
            else             *filp = *bp;
	 } else /* *cp >= *ap >= *bp  => *bp = *ap; Test auf == einfach weggelassen */ 
	    *filp = *ap;
      }
               /* ap = bp; bp = cp; ... Das ginge fr rein horizontalen Filter!!! */
   }
}


int cmp(const void *a, const void *b)
/* Vergleicher-Funktion fuer das Sortieren von Pixeln
 * --- Parameter: ---
 * const void *a : Zeiger auf ein Pixel
 * const void *b : Zeiger auf ein Pixel
 * --- Rueckgabe: ---
 * int  cmp()    :  <0 fuer *a  < *b
 *                 ==0 fuer *a == *b
 *                  >0 fuer *a  > *b      (analog zu strcmp())
 */
{
   return *(unsigned char *)a - *(unsigned char *)b;
}
	      
void MedianFilter(unsigned char *buf, unsigned char *fil, int size, int w, int h, int ws, int hs)
/* allgemeiner Medianfilter 
 * --- Parameter: ---
 * unsigned char *buf : zu filternder Speicherbereich
 * unsigned char *fil : Speicherbereich in den das gefilterte Bild gespeichert wird
 * int           size : Anzahl der Bytes
 * int           w    : Breite des Filters
 * int           h    : Hhe des Filters
 * int           ws   : horizontaler Abstand der zu filternden Bytes
 * int           hs   : vertikaler Abstand der zu filternden Bytes
 * --- Rueckgabe: ---
 * unsigned char *fil : gefilterter Speicherbereich
 */
{
   unsigned char          *fbuf, *fip, *fcp;
   register unsigned char *bp, *cp, *ep, *fp, *ap, 
                          *filp = fil + (w>>1)*ws + (h>>1)*hs;
   register int            wh   = w*h,
                           hbuf = wh>>1,
		           wend = w*ws;
			  
   
   assert( (fbuf = malloc(wh)) );
   
   ep   = (ap = buf) + wend + (h-1)*hs;
   for ( fip = ap + size; ep < fip; ap += ws, ep += ws, filp += ws ) {
      for ( fp = fbuf, bp = ap; bp < ep; bp += hs ) 
	 for ( fcp = (cp = bp) + wend; cp < fcp; cp += ws, fp++ ) 
            *fp = *cp;
      qsort(fbuf, wh, 1, cmp);
      *filp = *(fbuf + hbuf);
   }
   free(fbuf);
}

#define SwapBuf(a, b) {register unsigned char *zw = a; a = b; b = zw;}

int FilterBilder(PAUL *p)
/* median filter
 * --- parameter: ---
 * PAUL *p              : list of images, options
 *                      : used options:
 *                        fordn  : order of filter
 *                        filter : typ of filter
 * --- return: ---
 * int   FilterBilder() : -1 in case of error
 */
{
   PICTURE       *bild;
   GList         *pl;
   char           buf[256];
   unsigned char *fil;
   int            fordn = p->opt->fordn, filter = p->opt->filter;

   if ( !(p->piclist) ) return 0;

   for ( bild = BILD(pl = p->piclist); pl; bild = BILD(pl = pl->next) ) {
      assert ( (fil = calloc(bild->size, bild->storepix)) );
      if ( filter != 'M' && fordn == 3 ) {
         /* median filter of order 3 first horizontal than vertikal */
         if ( filter == 'h' || filter =='m' ) {
            MedianFilter3(bild->DATA+(bild->storepix==1?0:1), fil+(bild->storepix==1?0:1), 
                          bild->storepix*bild->size, bild->storepix, bild->storepix);
            if ( !IsMonochrom(bild) ) {
	       MedianFilter3(bild->DATA,   fil,   3*bild->size, 3, 3);
               MedianFilter3(bild->DATA+2, fil+2, 3*bild->size, 3, 3);
	    }
            SwapBuf(bild->DATA, fil);
	 }
         if ( filter == 'v' || filter =='m' ) {
            MedianFilter3(bild->DATA+(bild->storepix==1?0:1), fil+(bild->storepix==1?0:1), 
                          bild->storepix*bild->size, bild->storepix, bild->storepix*bild->W);
            if ( !IsMonochrom(bild) ) {
               MedianFilter3(bild->DATA,   fil,   3*bild->size, 3, 3*bild->W);
               MedianFilter3(bild->DATA+2, fil+2, 3*bild->size, 3, 3*bild->W);
	    }
            SwapBuf(bild->DATA, fil);
         }
      } else {
         if ( filter != 'M' ) {
            if ( filter == 'h' || filter =='m' ) {
               MedianFilter(bild->DATA+(bild->storepix==1?0:1), fil+(bild->storepix==1?0:1), 
                            bild->storepix*bild->size, fordn, 1, bild->storepix, 
                            bild->storepix*bild->W);
               if ( !IsMonochrom(bild) ) {
                  MedianFilter(bild->DATA,   fil,   3*bild->size, fordn, 1, 3, 3*bild->W);
                  MedianFilter(bild->DATA+2, fil+2, 3*bild->size, fordn, 1, 3, 3*bild->W);
	       }
               SwapBuf(bild->DATA, fil);
	    }
            if ( filter == 'v' || filter =='m' ) {
               MedianFilter(bild->DATA+(bild->storepix==1?0:1), fil+(bild->storepix==1?0:1),
                            bild->storepix*bild->size, 1, fordn, bild->storepix, 
                            bild->storepix*bild->W);
               if ( !IsMonochrom(bild) ) {
                  MedianFilter(bild->DATA,   fil,   3*bild->size, 1, fordn, 3, 3*bild->W);
                  MedianFilter(bild->DATA+2, fil+2, 3*bild->size, 1, fordn, 3, 3*bild->W);
               }
               SwapBuf(bild->DATA, fil);
	    }
	 } else {
            MedianFilter(bild->DATA+(bild->storepix==1?0:1), fil+(bild->storepix==1?0:1),
                         bild->storepix*bild->size, fordn, fordn, bild->storepix, 
                         bild->storepix*bild->W);
            if ( !IsMonochrom(bild) ) {
               MedianFilter(bild->DATA,   fil,   3*bild->size, fordn, fordn, 3, 3*bild->W);
               MedianFilter(bild->DATA+2, fil+2, 3*bild->size, fordn, fordn, 3, 3*bild->W);
            }
            SwapBuf(bild->DATA, fil);
	 }
      }
      free(fil);
      if ( bild->im ) bild->im->rgb_data = bild->DATA;
      sprintf(buf, "%s: Median filter of order %i (", TypFilter, fordn);
      switch (filter) {
         case 'h': strcat(buf, "horizontal)"); break;
         case 'v': strcat(buf, "vertical)"); break;
         case 'm': strcat(buf, "one dimensional)"); break;
         case 'M': strcat(buf, "two dimensional)"); break;
         default:  g_warning("Unknown filter type: %c", filter);
                   *buf = 0;
      }
      CopySpec(bild->spec, ChunkNameTyp, buf);
      sprintf(buf, "%s%c%i", APPFILTER, filter, fordn);
      NewFileName(bild, buf);
      bild->flag |= FILTER;
   }
   return 0;
}

