/* GGlyph - a graphical utility for managing one's collection of Type
   1 fonts (and eventually, other sorts of fonts) under X11.

   Copyright (c) 1998 David Huggins-Daines
   <bn711@freenet.carleton.ca>.

   You are permitted to copy, distribute, and modify this file under the
   terms of the GNU General Public License, version 2, or any later
   version.  See the file COPYING for more details. */

/* gglyph.c - contains global methods and data that are common to the
   various GUI components */

#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <X11/Xlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <getopt.h>

#include "gglyph.h"
#include "gdk_t1lib.h"
#include "mover_view.h"
#include "viewer_window.h"
#include "fonts.h"
#include "external.h"

/* */
/* Forward declarations */
/* */
static void get_Xscaledfonts (void);
static void load_fonts (void);
static void usage (void);
static void make_mainwindow (void);

static void cb_gglyph_quit (void);
#ifdef GTK_ITEM_FACTORY
static void cb_show_help (gchar *text);
#else /* C'est quoi a, GTK_ITEM_FACTORY? */
static void cb_show_help (GtkWidget *w, gchar *text);
#endif
static void cb_menu_preview (void);
static void cb_menu_sample_sheet (GtkWidget *w, gint action);

/* */
/* global data */
/* */
gchar *t1dir;
gint debug_level;
/* directories that have been modified, for rescan function */
GSList *dirty_list = NULL;
GSList *inst_fontdirs, *avail_fontdirs;
GHashTable *avail_fontfiles;
gint rescan_interval = ALWAYS_RESCAN;
gchar *viewer_title = NULL;
uid_t uid;
/* FIXME: drop types should probably be "text/plain" if we want to
   cooperate with GNOME Midnight Commander... */
gchar *file_drag_types[] = {"text/plain", "file:application/x-font",
			    "file:ALL"};
gchar *dir_drag_types[] = {"text/plain", "file:directory", "file:ALL"};
gchar *wdir_drop_types[] = {"file:application/x-font"}; 
gchar *tree_drop_types[] = {"file:directory"};

#define ABOUT_TEXT "About:GGlyph " VERSION "\n(c) 1998 \
David Huggins-Daines

Previewer/Installer for Type 1 fonts.

This program is free software; you can
redistribute it and/or modify it under the
terms of the GNU General Public License as
published by the Free Software Foundation;
either version 2 of the License, or (at your
option) any later version.

This program is distributed in the hope that
it will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details."

#ifdef GTK_ITEM_FACTORY
#warning Compiling for GTK+ 1.1, some stuff is gonna be broken
GtkItemFactory *item_factory;
GtkItemFactoryEntry menu_entries[] = {
  {"/File", NULL, NULL, 0, "<Branch>"},
  {"/File/Exit", "<control>Q", (GtkItemFactoryCallback)
   cb_gglyph_quit, 0, NULL},
  
  {"/Font", NULL, NULL, 0, "<Branch>"},
  {"/Font/Preview", "<control>I", (GtkItemFactoryCallback) cb_menu_preview, 0,
   NULL},
  {"/Font/View Sample", "<control>V", (GtkItemFactoryCallback1)
   cb_menu_sample_sheet, VIEW_SAMPLE, NULL},
  {"/Font/Print Sample", "<control>P", (GtkItemFactoryCallback1)
   cb_menu_sample_sheet, PRINT_SAMPLE, NULL},

  {"/Help", NULL, NULL, 0, "<LastBranch>"},
  {"/Help/About...", NULL, (GtkMenuCallback) cb_show_help, 0, NULL}
};
#else /* GTK_ITEM_FACTORY nyet */
GtkMenuFactory *menu_factory;
GtkMenuEntry menu_entries[] = {
  {"File/Exit", "<control>Q", (GtkMenuCallback) cb_gglyph_quit, NULL},
  
  {"Font/Preview", "<control>I", (GtkMenuCallback) cb_menu_preview, NULL},
  {"Font/View Sample", "<control>V", (GtkMenuCallback) cb_menu_sample_sheet,
   GINT_TO_POINTER (VIEW_SAMPLE)},
  {"Font/Print Sample", "<control>P", (GtkMenuCallback) cb_menu_sample_sheet,
   GINT_TO_POINTER (PRINT_SAMPLE)},

  {"Help/About...", NULL, (GtkMenuCallback) cb_show_help, ABOUT_TEXT}
};
#endif /* GTK_ITEM_FACTORY */

static int nmenu_entries = sizeof(menu_entries) / sizeof(menu_entries[0]);
  
/* */
/* configuration parameters */
/* */
gchar *viewer_cmd, *lp_arg;

/* callbacks */

#ifdef GTK_ITEM_FACTORY
static void cb_show_help (gchar *text)
#else /* no GTK_ITEM_FACTORY */
static void cb_show_help (GtkWidget *w, gchar *text)
#endif
{
  GtkWidget *dialog, *label, *button, *box;
  gchar *title, *help;

  g_return_if_fail (text != NULL);
  
  help = strchr (text, ':');
  
  if (help) {
    gint len = help - text;
    
    title = g_new (gchar, len + 1);
    strncpy (title, text, len);
    title[len] = '\0';
    
    help++;
  }
  else {
    title = "";
    help = text;
  }
  
  dialog = gtk_dialog_new();
  gtk_container_border_width (GTK_CONTAINER(dialog), 5);
  gtk_window_set_title (GTK_WINDOW(dialog), title);
  
  label = gtk_label_new (help);
  /* since GTK_JUSTIFY_FILL is non-functional */
  gtk_label_set_justify (GTK_LABEL(label), GTK_JUSTIFY_CENTER);
  gtk_widget_show (label);
  
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), label, FALSE,
		      FALSE, 0);

  box = GTK_DIALOG(dialog)->action_area;
  
  button = gtk_button_new_with_label ("OK");
  gtk_signal_connect_object_after
    (GTK_OBJECT(button), "clicked",
     GTK_SIGNAL_FUNC(gtk_widget_destroy),
     GTK_OBJECT (dialog));
  
  gtk_box_pack_start (GTK_BOX(box), button, TRUE, TRUE, 5);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  
  gtk_widget_show (button);

  gtk_widget_show (dialog);
}


/* a couple of utterly useless callbacks */
static void cb_menu_sample_sheet (GtkWidget *w, gint action)
{
  sample_selection (action);
}

static void cb_menu_preview (void)
{
  /* do it asynchronously */
  preview_selection (TRUE);
}

static void cb_gglyph_quit (void)
{
  /* does nothing for now */
  gtk_main_quit();
}


/* this is better done with libgnomeui, but I'd like to be
   GNOME-independent for the time being */
void question_box (gchar *question, GtkSignalFunc cb_yes,
		   GtkSignalFunc cb_no, gpointer yes_data, gpointer
		   no_data, gchar *help_text)
{
  GtkWidget *dialog, *label, *button, *box;
  
  /* put up dialog and all that tiresome crap */
  /* there will be an option to do this with libgnomeui */
  dialog = gtk_dialog_new();
  gtk_container_border_width (GTK_CONTAINER(dialog), 5);
  gtk_window_set_title (GTK_WINDOW(dialog), "Question");
  
  label = gtk_label_new (question);
  /* since GTK_JUSTIFY_FILL is non-functional */
  gtk_label_set_justify (GTK_LABEL(label), GTK_JUSTIFY_CENTER);
  gtk_widget_show (label);
  
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), label, FALSE,
		      FALSE, 0);

  box = GTK_DIALOG(dialog)->action_area;
  
  button = gtk_button_new_with_label ("Yes");
  gtk_signal_connect (GTK_OBJECT(button), "clicked",
		      GTK_SIGNAL_FUNC(cb_yes),
		      yes_data);
  gtk_signal_connect_object_after
    (GTK_OBJECT(button), "clicked",
     GTK_SIGNAL_FUNC(gtk_widget_destroy),
     GTK_OBJECT (dialog));
  
  gtk_widget_show (button);
  gtk_box_pack_start (GTK_BOX(box), button, TRUE, TRUE, 5);
    
  button = gtk_button_new_with_label ("No");
  if (cb_no)
    gtk_signal_connect (GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(cb_no),
			no_data);
  gtk_signal_connect_object_after
    (GTK_OBJECT(button), "clicked",
     GTK_SIGNAL_FUNC(gtk_widget_destroy),
     GTK_OBJECT (dialog));

  gtk_widget_show (button);
  gtk_box_pack_start (GTK_BOX(box), button, TRUE, TRUE, 5);
    
  button = gtk_button_new_with_label ("Help");
  if (help_text) {
    gtk_signal_connect (GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(cb_show_help),
			help_text);
  }
  else {
    gtk_widget_set_sensitive (button, FALSE);
  }

  gtk_widget_show (button);
  gtk_box_pack_start (GTK_BOX(box), button, TRUE, TRUE, 5);

  gtk_widget_show (dialog);
}

FontFileRec *find_font (gchar *pathname)
{
  gchar *basename = strrchr (pathname, '/');
  FontFileRec *ffile = NULL;
  GSList *i;

  if (basename == NULL)
    return NULL;

  basename++;

  /* should maybe use g_slist_foreach for this sort of stuff */
  for (i = inst_fontdirs; i; i = i->next){
    FontDirRec *fdir = (FontDirRec *) i->data;

    if (strncmp (fdir->path, pathname, basename - pathname - 1) == 0){
      ffile = (FontFileRec *) g_hash_table_lookup (fdir->fonts,
						   basename);
      break;
    }
  }
  /* if not found, look in non-installed dirs */
  if (ffile == NULL){
    for (i = avail_fontdirs; i; i = i->next){
      FontDirRec *fdir = (FontDirRec *) i->data;

      if (strncmp (fdir->path, pathname, basename - pathname - 1) == 0){
	ffile = (FontFileRec *) g_hash_table_lookup (fdir->fonts,
						     basename);
	break;
      }
    }
  }
  /* if still not found, look in avaliable fonts */
  if (ffile == NULL)
    ffile = (FontFileRec *) g_hash_table_lookup (avail_fontfiles,
						 basename);

  return ffile;
}

static void make_mainwindow (void)
{
  GtkWidget *window, *vbox;
#ifdef GTK_ITEM_FACTORY
  GtkWidget *item;
#else
  GtkMenuPath *path;
#endif /* GTK_ITEM_FACTORY */
  
  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  /* allow_shrink, allow_grow, auto_shrink */
  gtk_window_set_policy (GTK_WINDOW (window), FALSE, TRUE, FALSE);
  gtk_container_border_width (GTK_CONTAINER(window), 5);
  gtk_window_set_title (GTK_WINDOW (window), "GGlyph");
  
  /* allow us to exit gracefully */
  gtk_signal_connect (GTK_OBJECT (window), "delete_event",
		      GTK_SIGNAL_FUNC (gtk_main_quit), NULL);
  gtk_signal_connect (GTK_OBJECT (window), "destroy_event",
		      GTK_SIGNAL_FUNC (gtk_main_quit), NULL);

  vbox = gtk_vbox_new (FALSE, 0);

  /* reverse of usual order is necessary to make dnd work correctly */
  gtk_container_add (GTK_CONTAINER(window), vbox);
  gtk_widget_show (vbox);

  /* assuming that GTK_ITEM_FACTORY <-> GTK_ACCEL_GROUP */
#ifdef GTK_ITEM_FACTORY
  item_factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<Menubar>",
				       NULL);
  gtk_item_factory_create_items (item_factory, nmenu_entries - 1,
				 menu_entries, NULL);
  /* why exactly are we not able to specify callback data on an
     item-by-item basis in the entry struct array anymore? */
  gtk_item_factory_create_item (item_factory,
				&menu_entries[nmenu_entries - 1],
				ABOUT_TEXT, 1);
  
  if (debug_level)
    gtk_item_factory_dump_items (NULL, FALSE,
				 gtk_item_factory_print_func, stdout);

  /* make it insensitive by default */
  /* nope, not gonna do it (until GtkItemFactory is fixed) */
/*   item = gtk_item_factory_get_widget (item_factory, "<Menubar>/Font/Font"); */
/*   gtk_widget_set_sensitive (item, FALSE); */

  item = gtk_item_factory_get_widget (item_factory, "<Menubar>");
  gtk_box_pack_start (GTK_BOX (vbox), item, FALSE, FALSE, 0);
  gtk_widget_show (item);
  gtk_window_add_accel_group (GTK_WINDOW(window),
			      item_factory->accel_group);
#else /* GTK_ITEM_FACTORYkh nyet */
  menu_factory = gtk_menu_factory_new (GTK_MENU_FACTORY_MENU_BAR);
  gtk_menu_factory_add_entries (menu_factory, menu_entries,
				nmenu_entries);
  gtk_box_pack_start (GTK_BOX (vbox), menu_factory->widget, FALSE,
		      FALSE, 0);
  path = gtk_menu_factory_find (menu_factory, "Help");
  gtk_menu_item_right_justify (GTK_MENU_ITEM (path->widget));

  /* make it insensitive by default */
  path = gtk_menu_factory_find (menu_factory, "Font");
  gtk_widget_set_sensitive (path->widget, FALSE);
  
  gtk_widget_show (menu_factory->widget);
  gtk_window_add_accelerator_table (GTK_WINDOW(window),
				    menu_factory->table);
#endif /* GTK_ITEM_FACTORY */
  
  make_mover_page (vbox);
  gtk_widget_show (window);
}

/* FIXME: this should *really* be in gdk_t1lib.h */
void add_font_to_db (gchar *key, FontFileRec *value, gpointer
		     use_basename)
{
  int result;
  gchar *name;

  if (GPOINTER_TO_INT(use_basename))
    name = value->basename;
  else
    name = value->pathname;

  if (debug_level)
    g_print ("Adding font %s to database\n", name);

  if ((result = T1_AddFont(name)) < 0)
    fprintf(stderr, "Could not add font file %s (%d)\n",
	    name, result);
  else {
    value->t1_fontid = result;
  }
}

static void load_fonts (void)
{
  GSList *i;

  T1_SetLogLevel (T1LOG_DEBUG);
  if (gdk_t1lib_init(NO_LOGFILE | IGNORE_CONFIGFILE |
		     IGNORE_FONTDATABASE)==NULL) {
    fprintf(stderr, "Initialization of t1lib failed\n");
    exit (1);
  }
  
  for (i = inst_fontdirs; i; i = i->next){
    int result;
    FontDirRec *fdir = (FontDirRec *) i->data;
    
    if ((result = T1_AddToFileSearchPath
	 (T1_PFAB_PATH | T1_AFM_PATH, T1_APPEND_PATH,
	  fdir->path)) < 0){
      fprintf(stderr, "Could not add pathname %s (%d)\n",
	      fdir->path, result);
      break;
    }

    /* (lambda ()), sub {}, sigh.... */
    g_hash_table_foreach (fdir->fonts, (GHFunc) add_font_to_db,
			  GINT_TO_POINTER(TRUE));
  }

  /* foreach (@inst_fontdirs, @avail_fontdirs)... */
  for (i = avail_fontdirs; i; i = i->next){
    int result;
    FontDirRec *fdir = (FontDirRec *) i->data;
    
    if ((result = T1_AddToFileSearchPath
	 (T1_PFAB_PATH | T1_AFM_PATH, T1_APPEND_PATH,
	  fdir->path)) < 0){
      fprintf(stderr, "Could not add pathname %s (%d)\n",
	      fdir->path, result);
      break;
    }
    g_hash_table_foreach (fdir->fonts, (GHFunc) add_font_to_db,
			  GINT_TO_POINTER(TRUE));
  }

  g_hash_table_foreach (avail_fontfiles, (GHFunc) add_font_to_db,
			GINT_TO_POINTER(FALSE));
}

void get_Xscaledfonts (void)
{
  gint i, n;
  char **paths;
  
  /* get the X font path */
  paths = XGetFontPath (GDK_DISPLAY(), &n);
  
  for (i = 0; i < n ; i++){
    add_fontdir (&inst_fontdirs, paths[i]);
  }

  XFreeFontPath (paths);
}

static void gglyph_init (gchar *fdir)
{
  if (fdir){
    if (debug_level)
      g_print ("Attempting to add %s to font path\n", fdir);
    /* this won't add it if it's already there (unlike "xset fp+") */
    add_to_XFontPath (fdir);
  }

  get_Xscaledfonts();
  load_fonts();
  make_mainwindow();
}

#define INDEX_TYPE1DIR 1
#define ASKSTR "\
In order to install fonts as a regular\n \
user, you should place them in dedicated\n \
directories.  Do you want to create these\n \
directories now?"
#define NOTIFYSTR "\n\
Type 1 fonts will be installed in:\n"
#define ASKSTR_EDIT "\n\
In order to make them available to future\n \
X sessions, you must add the command\n \
\"xset fp+ "
#define ASKSTR_EDIT2 "\" to your X session\n \
files. Would you like me to do this now?"
#define COMMENT "!!! next line added by Gglyph\n"

static gchar *ufontdirs[3] = {"Fonts", "Fonts/type1", "Fonts/afm"};
static gchar *home;

static void add_cmd_to_xsession (gchar *xsession, gchar *fdir)
{
  FILE *file = fopen (xsession, "r+");
  gchar line[256];

  if (file == NULL){
    perror (xsession);
    return;
  }

  if (debug_level)
    g_print ("add_cmd_to_xsession: editing %s\n", xsession);

  while (fgets (line, 255, file))
    if (strcmp (line, COMMENT) == 0){
      /* FIXME: should re-edit the file */
      fprintf (stderr, "File %s has already been edited!\n",
	       xsession);
      fclose (file);
      break;
    }
  

  fputs (COMMENT, file);
  fputs ("xset fp+ ", file);
  fputs (fdir, file);
  fputc ('\n', file);
  fclose (file);
}

static void cb_yes_edit_xsession (GtkWidget *w, gchar *fdir)
{
  struct stat buf;
  gchar *xsession, *index;

  xsession = g_new (gchar, strlen(home) + 11);

  if (chdir (home)){
    perror ("Can't chdir to $HOME");
    abort();
  }
  
  strcpy (xsession, home);
  strcat (xsession, "/");
    
  strcat (xsession, ".xsession");
  if (stat (xsession, &buf) == 0)
    add_cmd_to_xsession (xsession, fdir);

  index = strrchr (xsession, '/') + sizeof (gchar);
  *index = '\0';
  strcat (xsession, ".xinitrc");
  if (stat (xsession, &buf) == 0)
    add_cmd_to_xsession (xsession, fdir);
    
  index = strrchr (xsession, '/') + sizeof (gchar);
  *index = '\0';
  strcat (xsession, ".Xclients");
  if (stat (xsession, &buf) == 0)
    add_cmd_to_xsession (xsession, fdir);
  
  /* FIXME: should determine which one to create, and create it,
     instead of failing silently. */
  gglyph_init (fdir);
}

static void cb_dont_makedirs (GtkWidget *w, gchar *fdir)
{
  if (debug_level)
    g_print ("Your loss!\n");

  gglyph_init(fdir);
}

/* makes dirs, creates dummy fonts.scale and fonts.dir, and adds
   Fonts/type1 to X font path */
static void cb_yes_makedirs (GtkWidget *w, gint missing)
{
  gint i;
  gchar *notifystr;
  
  notifystr = g_strdup ("The following directories were created:\n\n");
  
  for (i = 0; i < 3; i++){
    if (missing & (1 << (i + 1))){
      gchar fdir[strlen(home) + strlen(ufontdirs[i]) + 14];
      FILE *dummy;

      strcpy (fdir, home);
      strcat (fdir, "/");
      strcat (fdir, ufontdirs[i]);

      if (debug_level)
	g_print ("Creating %s\n", fdir);

      if (mkdir (fdir, 0777) == -1){
	perror ("Can't make directory");
      }
      else {
	notifystr = g_realloc (notifystr, (strlen (notifystr) + strlen
					   (fdir) + 2)
			       * sizeof (gchar));

	strcat (notifystr, fdir);
	strcat (notifystr, "\n");
      }

      if (i == INDEX_TYPE1DIR) {
	/* Fonts/type1 */
	gchar *end;

	strcat (fdir, "/fonts.dir");
	dummy = fopen (fdir, "w");
	fprintf (dummy, "0\n");
	fclose (dummy);

	end = strrchr (fdir, '/');
	
	strcpy (end, "/fonts.scale");
	dummy = fopen (fdir, "w");
	fprintf (dummy, "0\n");
	fclose (dummy);
      }
    }
  }

  /* We can say "~/Fonts/type1" here because we've just created it */

  /* FIXME: this is probably really, ridiculously inefficient */
  notifystr = g_realloc
    (notifystr, (strlen (notifystr) + strlen (NOTIFYSTR) +
		 strlen (t1dir) + strlen (ASKSTR_EDIT) + strlen
		 ("~/Fonts/type1/") + strlen (ASKSTR_EDIT2) + 1) *
     sizeof (gchar)); 
  strcat (notifystr, NOTIFYSTR);
  strcat (notifystr, t1dir);
  strcat (notifystr, ASKSTR_EDIT);
  strcat (notifystr, "~/Fonts/type1/");
  strcat (notifystr, ASKSTR_EDIT2);

  question_box (notifystr, GTK_SIGNAL_FUNC(cb_yes_edit_xsession),
		GTK_SIGNAL_FUNC(cb_dont_makedirs), t1dir, t1dir,
		NULL);
  g_free (notifystr);
}

static void init_app (void)
{
  uid = getuid();

  if (uid){
    /* running as normal user */
    gint missing;
    gint i;

    home = getenv ("HOME");
    missing = 0;

    /* check for dirs */
    for (i = 0; i < 3; i++){
      struct stat buf;
      gint rv;
      gchar fdir[strlen(home) + strlen(ufontdirs[i]) + 2];

      strcpy (fdir, home);
      strcat (fdir, "/");
      strcat (fdir, ufontdirs[i]);

      if (i == INDEX_TYPE1DIR)
	t1dir = g_strdup (fdir);

      rv = stat (fdir, &buf);
      if (rv == -1){
	if (errno == EACCES){
	  fprintf (stderr, "Can't stat %s: permission denied\n",
		   fdir);
	  exit (1);
	}
	else if (errno == ENOENT){
	  missing |= (1 << (i + 1));
	}
	else {
	  perror ("Can't stat font directory");
	  exit (1);
	}
      }
    }

    if (missing){
      /* walk through a few dialog boxes first */
      question_box (ASKSTR, GTK_SIGNAL_FUNC(cb_yes_makedirs),
		    GTK_SIGNAL_FUNC(cb_dont_makedirs), GINT_TO_POINTER 
		    (missing), NULL, NULL);
      gtk_main();
    }
    else {
      /* normal startup */
      gglyph_init(t1dir);
      gtk_main ();
    }
  }
  else {
    /* evidently, running as root. */
    /* start up normally (special root-stuff handled later) */
    gglyph_init(NULL);  
    gtk_main ();
  }
}

gint main (gint argc, gchar *argv[])
{
  int c;
  gchar *cwd;
  
  gtk_init (&argc, &argv);

  inst_fontdirs = avail_fontdirs = NULL;

  init_external();

  /* defaults */
  lp_arg = NULL;
  viewer_cmd = "gv";
  
  /* process arguments */
  /* taken from GNU libc info documentation */
  while (1)
    {
      static struct option long_options[] =
      {
	{"debug", 1, 0, 0},
	{"printer", 1, 0, 0},
	{"viewer", 1, 0, 0},
	{"help", 0, 0, 0},
	{"version", 0, 0, 0},
	{0, 0, 0, 0}
      };
      /* `getopt_long' stores the option index here. */
      gint option_index = 0;
     
      c = getopt_long (argc, argv, "d:P:v:V?",
		       long_options, &option_index);
     
      /* Detect the end of the options. */
      if (c == -1)
	break;
     
      switch (c) {
      case 0:
	/* If this option set a flag, do nothing else now. */
	if (long_options[option_index].flag != 0)
	  break;

	switch (option_index) {
	case 0:
	  /* --debug */
	  debug_level = atoi (optarg);
	  break;
	  
	case 1:
	  /* --printer */
	  lp_arg = g_new (gchar, (strlen (optarg) + 3) * sizeof
			  (gchar));
	  strcpy (lp_arg, "-P");
	  strcat (lp_arg, optarg);
	  break;
	  
	case 2:
	  /* --viewer */
	  viewer_cmd = g_strdup (optarg);
	  break;
	  
	case 3:
	  /* --help */
	  usage();
	  exit (0);
	  break;
	  
	case 4:
	  /* --version */
	  printf ("gglyph %s\n", VERSION);
	  exit (0);
	  break;
	}
	break;
     
      case 'd':
	/* --debug */
	debug_level = atoi (optarg);
	break;
     
      case 'P':
	/* --printer */
	lp_arg = g_new (gchar, (strlen (optarg) + 3) * sizeof
			(gchar));
	strcpy (lp_arg, "-P");
	strcat (lp_arg, optarg);
	break;

      case 'v':
	/* --viewer */
	viewer_cmd = g_strdup (optarg);
	break;

      case 'V':
	/* --version */
	printf ("gglyph %s\n", VERSION);
	exit (0);
	break;
          
      case '?':
	/* --help */
	usage();
	exit (0);
	break;
     
      default:
	puts ("I shouldn't be here");
	abort();
      }
    }
  
  {
    /* need to do this now, since add_fontdir does some chdir'ing */
    gint pathsize = 256;
	  
    cwd = g_new (gchar, pathsize + 1);
	  
    if (getcwd (cwd, pathsize) == NULL){
      /* highly unlikely */
      do {
	pathsize += 256;
	cwd = g_realloc (cwd, pathsize * sizeof (gchar));
      } while (getcwd (cwd, pathsize) == NULL);
    }
  }
     
  /* Any remaining arguments must be files or directories */
  if (optind < argc) {
    while (optind < argc) {
      struct stat buf;

      chdir (cwd);
      if (stat (argv[optind], &buf) == -1){
	perror (argv[optind]);
	exit (1);
      }

      if (S_ISDIR (buf.st_mode)){
	/* get full path */
	gchar *path;
	
	if (argv[optind][0] == '/'){
	  path = g_strdup (argv[optind]);
	}
	else {
	  path = g_new (gchar, strlen (cwd) + strlen
			(argv[optind]) + 2);
	  strcpy (path, cwd);
	  strcat (path, "/");
	  strcat (path, argv[optind]);
	}
	
	if (debug_level)
	  g_print ("Adding dir %s to avaliable font dirs\n",
		   path);
	
	add_fontdir (&avail_fontdirs, path);
	g_free (path);
      }
      else if (S_ISLNK (buf.st_mode) | S_ISREG (buf.st_mode)){
	FontFileRec *ffile = g_new (FontFileRec, 1);
	gchar *pathname, *basename;

	if (avail_fontfiles == NULL)
	  avail_fontfiles = g_hash_table_new (g_str_hash,
					      g_str_equal);

	if (argv[optind][0] == '/'){
	  pathname = g_strdup (argv[optind]);
	  basename = strrchr (pathname, '/') + sizeof(gchar);
	}
	else {
	  pathname = g_new (gchar, strlen (cwd) + strlen
			    (argv[optind]) + 2);
	  strcpy (pathname, cwd);
	  strcat (pathname, "/");
	  strcat (pathname, argv[optind]);
	  basename = strrchr (pathname, '/') + sizeof(gchar);
	}
	
	ffile->pathname = strdup (pathname);
	ffile->basename = ffile->pathname + (basename - pathname);
	ffile->xlfd = NULL;
	ffile->t1_fontid = NOT_LOADED;
	g_hash_table_insert (avail_fontfiles, ffile->basename,
			     ffile);
	if (debug_level)
	  g_print ("Added %s to avaliable fonts list\n", pathname);
	
	g_free (pathname);
      }
      else {
	fprintf (stderr, "%s isn't a dir or file\n", argv[optind]);
	exit (1);
      }
      optind++;
    }
  }

  g_free (cwd);
  init_app();

  return 0;
}

static void usage (void)
{
  /* sort of borrowed from GNOME, until this program gets (optional!)
     GNOMification */
  puts ("Usage: gglyph [OPTION] [DIRECTORY...] [FILE...]
Preview and install Type-1 fonts in PFA or PFB format, and manage
the user's X11 font path.

  -d DEBUGLEVEL, --debug=DEBUGLEVEL   Show debugging information
  -P PRINTER, --printer=PRINTER       Set name of printer (for printing
                                      sample sheets)
  -v VIEWER, --viewer=VIEWER          Command for viewing sample sheets

Gtk toolkit options
      --class CLASS          set X window class property
      --display DISPLAY      set X display to use
      --gxid_host HOST
      --gxid_port PORT
      --name NAME            set X window name property
      --no-xshm              Don't use X shared memory extension
      --xim-preedit STYLE
      --xim-status STYLE
  
Generic options
  -?, --help                 Give this help list
  -V, --version              Print program version
");
}
