/* XQF - Quake server browser and launcher
 * Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <sys/types.h>
#include <stdio.h>	/* sprintf */
#include <stdlib.h>	/* qsort */

#include <gtk/gtk.h>

#include "xqf.h"
#include "stat.h"
#include "skin.h"
#include "filter.h"
#include "utils.h"
#include "dns.h"
#include "pref.h"
#include "psearch.h"
#include "server.h"
#include "pixmaps.h"
#include "srv-list.h"
#include "srv-info.h"


static last_clicked_row = 0;


static int server_clist_refresh_row (struct server *s, int row) {
  char *text[7];
  char buf1[64], buf2[32], buf3[32], buf4[32];
  int col;

  text[0] = NULL;

  sprintf (buf1, "%s:%d", 
         (show_hostnames && s->host->name)? s->host->name : s->host->address, 
                                                                     s->port);
  text[1] = buf1;

  if (s->ping >= 0) {
    sprintf (buf2, "%d", (s->ping > MAX_PING)? MAX_PING : s->ping);
    text[2] = buf2;
  }
  else {
    text[2] = "n/a";
  }

  if (s->retries >= 0) {
    if (s->ping == MAX_PING + 1) {
      text[3] = "D";	/* DOWN */
    }
    else if (s->ping == MAX_PING) {
      text[3] = "T";	/* TIMEOUT */
    }
    else {
      sprintf (buf3, "%d", s->retries);
      text[3] = buf3;
    }
  }
  else {
    text[3] = "n/a";
  }

  sprintf (buf4, "%d/%d", s->curplayers, s->maxplayers);
  text[4] = buf4;

  text[5] = (s->map)?  s->map : NULL;
  text[6] = (s->game)? s->game : NULL;

  if (row < 0)
    row = gtk_clist_append (GTK_CLIST (server_clist), text);
  else {
    GTK_CLIST_SET_FLAGS (GTK_CLIST (server_clist), CLIST_FROZEN);
    for (col = 1; col < 7; col++) {
      gtk_clist_set_text (GTK_CLIST (server_clist), row, col, text[col]);
    }
    GTK_CLIST_UNSET_FLAGS (GTK_CLIST (server_clist), CLIST_FROZEN);
  }

  switch (s->type) {
  case Q2_SERVER:
    gtk_clist_set_pixtext (GTK_CLIST (server_clist), row, 0, 
                                    (s->name)? s->name : "", 2, q2_pix, NULL);
    break;

  case QW_SERVER:
  default:
    gtk_clist_set_pixtext (GTK_CLIST (server_clist), row, 0, 
                                     (s->name)? s->name : "", 2, q_pix, NULL);
    break;
  }

  return row;
}


static void server_clist_select_servers (GSList *servers) {
  struct server *s;
  int row;

  if (!servers || !cur_filter || !cur_filter->servers)
    return;

  while (servers) {
    s = (struct server *) servers->data;
    if (!cur_filter->func || (*cur_filter->func) (s)) {
      row = slist_index (cur_filter->servers, s);
      if (row >= 0)
	gtk_clist_select_row (GTK_CLIST (server_clist), row, 0);
    }
    servers = servers->next;
  }
}


void populate_server_clist (GSList **servers, GSList *selection) {
  GSList *s;

  if (!servers) {
    gtk_clist_clear (GTK_CLIST (server_clist));
    return;
  }

  s = *servers = qsort_slist (*servers, qsort_servers);

  gtk_clist_freeze (GTK_CLIST (server_clist));

  /* 
   *  It's basically gtk_clist_clear (GTK_CLIST (server_clist))
   *  without touching vertical offset and scrollbars
   */

  while (GTK_CLIST (server_clist)->rows)
    gtk_clist_remove (GTK_CLIST (server_clist), 0);

  while (s) {
    server_clist_refresh_row ((struct server *) s->data, -1);
    s = s->next;
  }

  if (selection)
    server_clist_select_servers (selection);

  gtk_clist_thaw (GTK_CLIST (server_clist));
}


void populate_player_clist (struct server *s) {
  GdkPixmap *pm = NULL;
  GSList *pm_cache = NULL;
  char *text[6];
  char buf1[32], buf2[32], buf3[32], buf4[32];
  int i;
  struct player *p;

  if (s->type == QW_SERVER)
    allocate_quake_player_colors (main_window->window);

  if (s->players == NULL) {
    gtk_clist_clear (GTK_CLIST (player_clist));
    return;
  }

  gtk_clist_freeze (GTK_CLIST (player_clist));
  gtk_clist_clear (GTK_CLIST (player_clist));

  qsort (s->players, s->curplayers, sizeof (gpointer), qsort_players);

  for (i = 0; i < s->curplayers; i++) {
    p = s->players[i];

    text[0] = (p->name)? p->name : NULL;

    sprintf (buf1, "%d", p->frags);
    text[1] = buf1;

    sprintf (buf3, "%d", p->ping);
    text[4] = buf3;

    if (s->type == QW_SERVER) {
      sprintf (buf2, "%d:%d", p->shirt, p->pants);
      text[2] = buf2;

      text[3] = (p->skin)? p->skin : NULL;

      sprintf (buf4, "%02d:%02d", p->time / 60 / 60, p->time / 60 % 60);
      text[5] = buf4;
    }
    else {
      text[2] = NULL;
      text[3] = NULL;
      text[5] = NULL;
    }

    gtk_clist_append (GTK_CLIST (player_clist), text);

    if (s->type == QW_SERVER) {
      pm = create_pm (main_window, fix_qw_player_color (p->shirt), 
                                   fix_qw_player_color (p->pants), &pm_cache);
      gtk_clist_set_pixtext (GTK_CLIST (player_clist), i, 2, text[2], 2, pm, 
                                                                        NULL);
    }

    if (cur_filter == &filters[3] && test_player (p)) {
      gtk_clist_set_background (GTK_CLIST (player_clist), i,
                                &player_clist->style->bg[GTK_STATE_SELECTED]);
      gtk_clist_set_foreground (GTK_CLIST (player_clist), i,
                                &player_clist->style->fg[GTK_STATE_SELECTED]);
    }
  }

  gtk_clist_thaw (GTK_CLIST (player_clist));

  if (pm_cache)
    clear_pm_cache (pm_cache);
}


void sync_selection (void) {
  GList *selection;
  GSList *tmp;

  selection = GTK_CLIST (server_clist)->selection;

  if (selection && !selection->next) {
    tmp = g_slist_nth (cur_filter->servers, (int) selection->data);
    cur_server = (struct server *) tmp->data;
    populate_player_clist (cur_server);
    populate_serverinfo_clist (cur_server);
  }
  else {
    cur_server = NULL;
    gtk_clist_clear (GTK_CLIST (player_clist));
    gtk_clist_clear (GTK_CLIST (serverinfo_clist));
  }

  set_widgets_sensitivity ();
}


int server_clist_refresh_server (struct server *s) {
  int row;

  row = slist_index (cur_filter->servers, s);
  if (row >= 0) {
    server_clist_refresh_row (s, row);
    if (s == cur_server) {
      populate_player_clist (cur_server);
      populate_serverinfo_clist (cur_server);
    }
    return TRUE;
  }
  else {
    if (filter_add_server (cur_filter, s)) {
#ifdef DEBUG
      fprintf (stderr, "New server %s:%d\n", 
                  (s->host->name)? s->host->name : s->host->address, s->port);
#endif
      server_clist_refresh_row (s, -1);
      return TRUE;
    }
  }
  return FALSE;
}


void server_clist_select_row (int row, int state) {
  GList *list = GTK_CLIST (server_clist)->selection;
  int i;
  int low, high;

  if (state & GDK_CONTROL_MASK) {	/* add to selection */
    if (g_list_find (list, (gpointer) row))
      gtk_clist_unselect_row (GTK_CLIST (server_clist), row, 0);
    else
      gtk_clist_select_row (GTK_CLIST (server_clist), row, 0);

    last_clicked_row = row;
  }
  else if (state & GDK_SHIFT_MASK) {	/* extend selection */
    if (row >= last_clicked_row) {
      low = last_clicked_row;
      high = row;
    }
    else {
      low = row;
      high = last_clicked_row;
    }

    while (list) {
      i = (int) list->data;
      list = list->next;
      if (i < low || i > high)
	gtk_clist_unselect_row (GTK_CLIST (server_clist), i, 0);
    }

    list = GTK_CLIST (server_clist)->selection;

    for (i = low; i <= high; i++) {
      if (!g_list_find (list, (gpointer) i))
	gtk_clist_select_row (GTK_CLIST (server_clist), i, 0);
    }
  }
  else {		/* select one server */
    while (list) {
      i = (int) list->data;
      list = list->next;
      if (i != row)
	gtk_clist_unselect_row (GTK_CLIST (server_clist), i, 0);
    }

    if (!GTK_CLIST (server_clist)->selection)
      gtk_clist_select_row (GTK_CLIST (server_clist), row, 0);

    last_clicked_row = row;
  }

  sync_selection ();
}


void server_clist_clear_selection (void) {
  GList *list = GTK_CLIST (server_clist)->selection;
  int i;

  while (list) {
    i = (int) list->data;
    list = list->next;
    gtk_clist_unselect_row (GTK_CLIST (server_clist), i, 0);
  }

  last_clicked_row = 0;
  sync_selection ();
}


static int qsort_integers (const void *a, const void *b) {
  int ia = * (int *) a;
  int ib = * (int *) b;
  
  return ia - ib;
}


void server_clist_remove_selected (void) {
  GSList **cursrv;
  GSList *victim;
  GSList *indexes;
  GSList *index;
  int i;

  indexes = list_replicate (GTK_CLIST (server_clist)->selection);

  if (!indexes)
    return;

  indexes = qsort_slist (indexes, qsort_integers);

  cursrv = &cur_filter->servers;
  index = indexes;

  for (i = 0; *cursrv && index; i++) {
    if (i == (int) index->data) {
      index = index->next;

      victim = *cursrv;
      *cursrv = (*cursrv)->next;

      server_unref ((struct server *) victim->data);
      g_slist_free_1 (victim);
    }
    else {
      cursrv = &(*cursrv)->next;
    }
  }

  indexes = g_slist_reverse (indexes);

  gtk_clist_freeze (GTK_CLIST (server_clist));

  for (index = indexes; index; index = index->next) {
    gtk_clist_remove (GTK_CLIST (server_clist), (int) index->data);
  }

  g_slist_free (indexes);

  gtk_clist_thaw (GTK_CLIST (server_clist));
  sync_selection ();
}


void server_clist_select_all (int invert) {
  GSList *cursrv;
  GSList *indexes;
  GSList *index;
  int i;

  if (!cur_filter->servers)
    return;

  indexes = list_replicate (GTK_CLIST (server_clist)->selection);
  indexes = qsort_slist (indexes, qsort_integers);

  cursrv = cur_filter->servers;
  index = indexes;

  for (i = 0; cursrv; i++) {
    if (!index) {
      gtk_clist_select_row (GTK_CLIST (server_clist), i, 0);
    }
    else {
      if ((int) index->data == i) {
	if (invert)
	  gtk_clist_unselect_row (GTK_CLIST (server_clist), i, 0);
        index = index->next;
      }
      else {
	gtk_clist_select_row (GTK_CLIST (server_clist), i, 0);
      }
    }
    cursrv = cursrv->next;
  }

  g_slist_free (indexes);
  sync_selection ();
}


GSList *server_clist_selected_servers (void) {
  GList *rows = GTK_CLIST (server_clist)->selection;
  GSList *list = NULL;
  GSList *tmp;
  struct server *s;

  if (!cur_filter || !cur_filter->servers)
    return NULL;
    
  while (rows) {
    tmp = g_slist_nth (cur_filter->servers, (int) rows->data);
    s = (struct server *) tmp->data;
    list = g_slist_prepend (list, s);
    s->ref_count++;
    rows = rows->next;
  }

  return list;
}


void server_clist_selection_visible (void) {
  GList *rows = GTK_CLIST (server_clist)->selection;
  GList *row;
  GtkVisibility vis;
  int min;

  if (!rows)
    return;

  for (row = rows; row; row = row->next) {
    vis = gtk_clist_row_is_visible (GTK_CLIST (server_clist), (int) row->data);
    if (vis == GTK_VISIBILITY_FULL)
      return;
  }

  min = (int) rows->data;
  for (row = rows->next; row; row = row->next) {
    if (min > (int) row->data)
      min = (int) row->data;
  }

  if (rows->next)	/* if (g_list_length (rows) > 1) */
    gtk_clist_moveto (GTK_CLIST (server_clist), min, 0, 0.2, 0.0);
  else 
    gtk_clist_moveto (GTK_CLIST (server_clist), min, 0, 0.5, 0.0);
}


void server_clist_show_hostname (struct host *h) {
  struct server *s;
  GSList *tmp;
  char buf[256];
  int row = 0;

  if (!h->name)
    return;

  for (tmp = cur_filter->servers; tmp; tmp = tmp->next) {
    s = (struct server *) tmp->data;
    if (s->host == h) {
      sprintf (buf, "%s:%d", s->host->name, s->port);
      gtk_clist_set_text (GTK_CLIST (server_clist), row, 1, buf);
    }
    row++;
  }
}


void server_clist_show_hostnames (void) {
  struct server *s;
  GSList *tmp;
  char buf[256];
  int row = 0;

  for (tmp = cur_filter->servers; tmp; tmp = tmp->next) {
    s = (struct server *) tmp->data;
    if (s->host->name) {
      sprintf (buf, "%s:%d", 
         (show_hostnames && s->host->name)? s->host->name : s->host->address, 
                                                                     s->port);
      gtk_clist_set_text (GTK_CLIST (server_clist), row, 1, buf);
    }
    row++;
  }
}

