/* XQF - Quake server browser and launcher
 * Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>	/* sprintf, FILE, fprintf, etc... */
#include <string.h>	/* strchr, strcmp, strlen, ... */
#include <sys/stat.h>	/* stat */
#include <unistd.h>	/* unlink, rename, stat */
#include <sys/types.h>	/* readdir, dirent */
#include <dirent.h>	/* readdir, dirent */

#include <gtk/gtk.h>

#include "xqf.h"
#include "source.h"
#include "utils.h"
#include "server.h"
#include "pref.h"
#include "stat.h"
#include "dns.h"


GSList *qw_masters = NULL;
GSList *q2_masters = NULL;
struct master *unbound_qw = NULL;
struct master *unbound_q2 = NULL;
struct master *qw_sources = NULL;
struct master *q2_sources = NULL;
struct master *allsources = NULL;
struct master *favorites = NULL;


static char *sp2under (char *str) {
  char *ptr;

  if (str) {
    for (ptr = str; *ptr; ptr++) {
      if (*ptr == ' ') 
	*ptr = '_';
    }
  }
  return str;
}


static char *build_server_list_file_name (char *dir, struct master *m) {
  char *fn;
  char *dfn;

  if (!m->host) {
    fn = g_strdup (m->name);
  }
  else {
    fn = g_malloc (strlen (m->host->address) + 6 + 1 + strlen (m->name) + 1);
    sprintf (fn, "%s:%d-%s", m->host->address, m->port, m->name);
  }
  sp2under (fn);
  dfn = file_in_dir (dir, fn);
  g_free (fn);
  return dfn;
}


static void correct_saved_list_file_name (char *oldname, char *newname) {
  struct stat st_buf;
  char *newfn;
  char *oldfn;

  oldfn = file_in_dir (user.rcdir, oldname);
  sp2under (oldfn);

  newfn = file_in_dir (user.rcdir, newname);
  sp2under (newfn);

  if (stat (newfn, &st_buf) != 0)
    rename (oldfn, newfn);	/* can fail, but it's not interesting */

  g_free (oldfn);
  g_free (newfn);
}


static char *find_obsolete (char *dir, struct master *m) {
  DIR *directory;
  struct dirent *dirent_ptr;
  char *namestr;
  char *address;
  int dlen, alen, nlen;
  char *address_match = NULL;
  char *name_match = NULL;
  char *exact_name_match = NULL;
  char *res = NULL;

  directory = opendir (dir);
  if (directory == NULL)
    return NULL;

  namestr = g_strdup (m->name);
  sp2under (namestr);
  nlen = strlen (namestr);

  address = g_malloc (strlen (m->host->address) + 6 + 1);
  sprintf (address, "%s:%d", m->host->address, m->port);
  alen = strlen (address);

  while ((dirent_ptr = readdir (directory)) != NULL) {
    dlen = strlen (dirent_ptr->d_name);

    if (dlen >= alen && !strncmp (address, dirent_ptr->d_name, alen)) {
      if (!address_match)
	address_match = g_strdup (dirent_ptr->d_name);
    }

    if (dlen >= nlen && 
                    !strncmp (namestr, dirent_ptr->d_name+dlen - nlen, nlen)) {
      if (dlen == nlen) {
	if (!exact_name_match)
	  exact_name_match = g_strdup (dirent_ptr->d_name);
      }
      else {
	if (!name_match)
	  name_match = g_strdup (dirent_ptr->d_name);
      }
    }
  }

  closedir (directory);
  g_free (namestr);
  g_free (address);

  if (address_match)
    res = file_in_dir (dir, address_match);
  else if (name_match)
    res = file_in_dir (dir, name_match);
  else if (exact_name_match)
    res = file_in_dir (dir, exact_name_match);

  if (address_match)    g_free (address_match);
  if (name_match)       g_free (name_match);
  if (exact_name_match) g_free (exact_name_match);

  return res;
}


static FILE *open_saved_server_list (char *dir, struct master *m) {
  FILE *f;
  char *fn;
  char *obsolete;

  if (!m || !m->name)
    return NULL;

  fn = build_server_list_file_name (dir, m);
  f = fopen (fn, "r");

  if (!f && m->host) {
    obsolete = find_obsolete (dir, m);
    if (obsolete) {
      f = fopen (obsolete, "r");
      if (f)
	rename (obsolete, fn);
      g_free (obsolete);
    }
  }

  g_free (fn);
  return f;
}


static void read_saved_server_list (struct master *m) {
  FILE *f;
  char *nl;
  struct host *h;
  struct server *s;
  char *addr;
  int port;
  enum server_type type;
  char buf[1024];
  char *s_addr, *s_name;

  m->servers = NULL;

  f = open_saved_server_list (user.rcdir, m);
  if (!f)
    return;

  while (!feof (f)) {
    fgets (buf, 1024, f);

    nl = buf + strlen (buf) - 1;
    if (*nl == '\n')
      *nl = '\0';

    s_addr = strchr (buf, '|');
    if (!s_addr)
      continue;
    *s_addr++ = '\0';
    
    s_name = strchr (s_addr, '|');
    if (!s_name)
      continue;
    *s_name++ = '\0';

    if (strcmp (buf, "QW") == 0)
      type = QW_SERVER;
    else if (strcmp (buf, "Q2") == 0)
      type = Q2_SERVER;
    else
      continue;

    if (!parse_address (s_addr, &addr, &port))
      continue;

    h = host_add (addr);
    g_free (addr);
    if (!h)
      continue;

    s = server_add (h, port, type);

    if (!s->name && s_name[0] != '\0')
      s->name = g_strdup (s_name);

    add_server_to_master (m, s);
  }

  fclose (f);
}


void save_server_list (struct master *m) {
  FILE *f;
  char *fn;
  GSList *list;
  struct server *s;

  fn = build_server_list_file_name (user.rcdir, m);

  if (m->servers) {
    f = fopen (fn, "w");
    g_free (fn);
    if (!f)
      return;
  }
  else {
    unlink (fn);
    g_free (fn);
    return;
  }

  m->servers = qsort_slist (m->servers, qsort_servers);

  for (list = m->servers; list; list = list->next) {
    s = (struct server *) list->data;
    fprintf (f, "%s|%s:%d|%s\n", (s->type == QW_SERVER) ? "QW" : "Q2",
                          s->host->address, s->port, (s->name)? s->name : "");
  }

  fclose (f);
}


static GSList *read_master_file (char *filename) {
  FILE *f;
  char buf[1024];
  char *str;
  GSList *list = NULL;
  struct master *m;
  struct host *h;
  char *addr;
  int port;

  f = fopen (filename, "r");
  if (!f)
    return NULL;

  while (!feof (f)) {
    fgets (buf, 1024, f);

    str = strchr (buf, '\n');
    if (str != NULL)
      *str = '\0';

    str = strchr (buf, '|');
    if (str == NULL)
      break; 		/* error */
    *str++ = '\0';

    if (!parse_address (buf, &addr, &port))
      continue;

    h = host_add (addr);
    g_free (addr);
    if (!h)
      continue;

    m = g_malloc (sizeof (struct master));

    m->host = h;
    m->host->ref_count++;

    m->port = (port > 0)? port : QWM_DEFAULT_PORT;
    m->name = strdup_strip (str);
    m->servers = NULL;
    m->sources = NULL;
    m->type = QW_SERVER;
    m->protocol = QW_SERVER;
    if (default_save_lists)
      read_saved_server_list (m);
    list = g_slist_prepend (list, m);
  }
  
  fclose (f);

  return list;
}


static char *master_dat[] = {

#include "master-dat.h"

  NULL, NULL
};


static GSList *builtin_master_file (void) {
  GSList *list = NULL;
  struct master *m;
  struct host *h;
  char *addr;
  int port;
  int i;

  for (i = 0; master_dat[i]; i += 2) {
    if (!parse_address (master_dat[i], &addr, &port))
      continue;

    h = host_add (addr);
    g_free (addr);
    if (!h)
      continue;

    m = g_malloc (sizeof (struct master));

    m->host = h;
    m->host->ref_count++;

    m->port = (port > 0)? port : QWM_DEFAULT_PORT;
    m->name = strdup_strip (master_dat[i + 1]);
    m->servers = NULL;
    m->sources = NULL;
    m->type = QW_SERVER;
    m->protocol = QW_SERVER;
    if (default_save_lists)
      read_saved_server_list (m);
    list = g_slist_prepend (list, m);
  }

  return list;
}


static GSList *read_master_dat (char *dir) {
  const char master_dat[] = "master.dat";
  GSList *list;
  char *fn;

  fn = file_in_dir (dir, master_dat);
  list = read_master_file (fn);
  g_free (fn);
  return list;
}


static struct {
  char *address;
  char *name;
  enum server_type protocol;
} quake2_masters[] = {

  /* satan.idsoftware.com:27900 */

  { "192.246.40.37:27900",	"id Q2",		Q2_SERVER }, 

  /* q2master.planetquake.com:27900 */

  { "204.182.161.3:27900",	"PlanetQuake Q2",	QW_SERVER }, 

  /* telefragged.com:27900 */

  { "206.222.86.244:27900",	"TeleFragged Q2", 	QW_SERVER },

  /* q2master.minos.co.uk:27900 */

  { "195.153.47.60:27900",	"Minos (UK) Q2",	QW_SERVER },

  /* master.quake.inet.fi:27900 */

  { "194.251.249.32:27900",	"iNET (Finland) Q2",	QW_SERVER },

  /* q2master.mondial.net.au:27900 */

  { "203.142.228.131:27900",	"Australia Q2",		QW_SERVER },

  { NULL, NULL, 0 }
};


static GSList *builtin_q2_master_list (void) {
  GSList *list = NULL;
  struct master *m;
  struct host *h;
  char *addr;
  int port;
  int i;

  for (i = 0; quake2_masters[i].address; i++) {
    if (!parse_address (quake2_masters[i].address, &addr, &port))
      continue;

    h = host_add (addr);
    g_free (addr);
    if (!h)
      continue;

    m = g_malloc (sizeof (struct master));

    m->host = h;
    m->host->ref_count++;

    m->port = (port > 0)? port : Q2M_DEFAULT_PORT;
    m->name = strdup_strip (quake2_masters[i].name);
    m->servers = NULL;
    m->sources = NULL;
    m->type = Q2_SERVER;
    m->protocol = quake2_masters[i].protocol;
    if (default_save_lists)
      read_saved_server_list (m);
    list = g_slist_prepend (list, m);
  }

  return list;
}


void init_masters (void) {
  GSList *tmp;

  qw_masters = NULL;

  /* search for master.dat in ${rcdir}, ${quakedir}, /usr/local/lib, /usr/lib */

  if (!(qw_masters = read_master_dat (user.rcdir)) &&
      !(qw_masters = read_master_dat (real_quake_dir)) &&
      !(qw_masters = read_master_dat ("/usr/local/lib/")) &&
      !(qw_masters = read_master_dat ("/usr/lib/"))) {
    qw_masters = builtin_master_file ();
  }

  qw_masters = g_slist_reverse (qw_masters);

  unbound_qw = g_malloc (sizeof (struct master));
  unbound_qw->name = g_strdup ("Unbound QW");
  unbound_qw->host = NULL;
  unbound_qw->servers = NULL;
  unbound_qw->sources = NULL;
  unbound_qw->type = QW_SERVER;
  unbound_qw->protocol = -1;
  read_saved_server_list (unbound_qw);

  q2_masters = g_slist_reverse (builtin_q2_master_list ());

  unbound_q2 = g_malloc (sizeof (struct master));
  unbound_q2->name = g_strdup ("Unbound Q2");
  unbound_q2->host = NULL;
  unbound_q2->servers = NULL;
  unbound_q2->sources = NULL;
  unbound_q2->type = Q2_SERVER;
  unbound_q2->protocol = -1;
  correct_saved_list_file_name ("Quake II", unbound_q2->name);
  read_saved_server_list (unbound_q2);

  favorites = g_malloc (sizeof (struct master));
  favorites->name = g_strdup ("Favorites");
  favorites->host = NULL;
  favorites->servers = NULL;
  favorites->sources = NULL;
  favorites->type = ANY_SERVER;
  favorites->protocol = -1;
  read_saved_server_list (favorites);

  qw_sources = g_malloc (sizeof (struct master));
  qw_sources->name = g_strdup ("All QW Sources");
  qw_sources->host = NULL;
  qw_sources->servers = NULL;
  qw_sources->sources = NULL;
  qw_sources->type = QW_SERVER;
  qw_sources->protocol = -1;

  for (tmp = qw_masters; tmp; tmp = tmp->next)
    qw_sources->sources = g_slist_append (qw_sources->sources, tmp->data);

  qw_sources->sources = g_slist_append (qw_sources->sources, unbound_qw);

  q2_sources = g_malloc (sizeof (struct master));
  q2_sources->name = g_strdup ("All Q2 Sources");
  q2_sources->host = NULL;
  q2_sources->servers = NULL;
  q2_sources->sources = NULL;
  q2_sources->type = Q2_SERVER;
  q2_sources->protocol = -1;

  for (tmp = q2_masters; tmp; tmp = tmp->next)
    q2_sources->sources = g_slist_append (q2_sources->sources, tmp->data);

  q2_sources->sources = g_slist_append (q2_sources->sources, unbound_q2);

  allsources = g_malloc (sizeof (struct master));
  allsources->name = g_strdup ("All Sources");
  allsources->host = NULL;
  allsources->servers = NULL;
  allsources->sources = NULL;
  allsources->type = ANY_SERVER;
  allsources->protocol = -1;

  for (tmp = qw_sources->sources; tmp; tmp = tmp->next)
    allsources->sources = g_slist_append (allsources->sources, tmp->data);

  for (tmp = q2_sources->sources; tmp; tmp = tmp->next)
    allsources->sources = g_slist_append (allsources->sources, tmp->data);

  qw_sources->sources = g_slist_append (qw_sources->sources, favorites);
  q2_sources->sources = g_slist_append (q2_sources->sources, favorites);
  allsources->sources = g_slist_append (allsources->sources, favorites);
}


static void free_master (struct master *m) {
  if (m) {
    if ((m->host && default_save_lists) || (!m->host && !m->sources))
      save_server_list (m);

    if (m->host)    host_unref (m->host);
    if (m->name)    g_free (m->name);
    if (m->sources) g_slist_free (m->sources);
    free_servers (m->servers);
    g_free (m);
  }
}


void free_masters (void) {
  GSList *ptr;

  for (ptr = qw_masters; ptr; ptr = ptr->next)
    free_master ((struct master *) ptr->data);
  g_slist_free (qw_masters);
  qw_masters = NULL;

  free_master (unbound_qw);
  unbound_qw = NULL;

  for (ptr = q2_masters; ptr; ptr = ptr->next)
    free_master ((struct master *) ptr->data);
  g_slist_free (q2_masters);
  q2_masters = NULL;

  free_master (unbound_q2);
  unbound_q2 = NULL;

  free_master (favorites);
  favorites = NULL;

  free_master (qw_sources);
  qw_sources = NULL;	 
			 
  free_master (q2_sources);
  q2_sources = NULL;	 
			 
  free_master (allsources);
  allsources = NULL;
}


void add_server_to_master (struct master *m, struct server *s) {
  if (g_slist_find (m->servers, s))
    return;

  m->servers = g_slist_append (m->servers, s);
  s->ref_count++;
}


void remove_server_from_master (struct master *m, struct server *s) {
  if (g_slist_find (m->servers, s)) {
    m->servers = g_slist_remove (m->servers, s);
    server_unref (s);
  }
}


GSList *collate_server_lists (GSList *servers, GSList *sources, 
                                                     enum server_type type) {
  struct master *source;
  struct server *s;
  GSList *list;

  while (sources) {
    source = (struct master *) sources->data;

    for (list = source->servers; list; list = list->next) {
      s = (struct server *) list->data;
      if (type == ANY_SERVER || type == s->type) {
	if (!g_slist_find (servers, s)) {
	  servers = g_slist_prepend (servers, s);
	  s->ref_count++;
	}
      }
    }

    sources = sources->next;
  }

  return servers;
}


GSList *server_sources (struct server *s) {
  GSList *list;
  GSList *res = NULL;
  struct master *m;

  for (list = qw_sources->sources; list; list = list->next) {
    m = (struct master *) list->data;
    if (m != favorites && g_slist_find (m->servers, s))
      res = g_slist_append (res, m);
  }

  for (list = q2_sources->sources; list; list = list->next) {
    m = (struct master *) list->data;
    if (m != favorites && g_slist_find (m->servers, s))
      res = g_slist_append (res, m);
  }

  if (g_slist_find (favorites->servers, s))
    res = g_slist_append (res, favorites);

  return res;
}


char *game_type_str (enum server_type st) {
  switch (st) {

  case QW_SERVER:
    return "QuakeWorld";

  case Q2_SERVER:
    return "Quake2";

  default:
    return "unknown";
  }
}


