/* XQF - Quake server browser and launcher
 * Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <sys/types.h>	/* chmod, waitpid, kill */
#include <stdio.h>	/* FILE, sprintf, perror */
#include <string.h>	/* strtok, memchr, strlen */
#include <unistd.h>	/* execvp, fork, _exit, chdir, stat, read, close */
                        /* write, fcntl, sleep */
#include <fcntl.h>	/* fcntl */
#include <sys/stat.h>	/* stat, chmod */
#include <sys/wait.h>	/* waitpid */
#include <errno.h>	/* errno */
#include <signal.h>     /* kill, signal... */

#include <gtk/gtk.h>

#include "xqf.h"
#include "pref.h"
#include "utils.h"
#include "dialogs.h"
#include "source.h"
#include "server.h"
#include "qrun.h"


#define CLIENT_ERROR_BUFFER	256
#define	CLIENT_ERROR_MSG_HEAD	"<XQF ERROR> "


struct running_client {
  pid_t	pid;
  int fd;
  int tag;

  char *buffer;
  int pos;

  struct server *server;
};


static GList *clients = NULL;


int quake_config_is_valid (enum server_type type) {
  struct stat stat_buf;
  char *game;
  char *dir;
  char *cmd;
  char *cfgdir;
  char *path;

  if (type == Q2_SERVER) {
    game = "Quake2";
    dir = real_q2_dir;
    cmd = default_q2_cmd;
    cfgdir = "baseq2";
  }
  else {
    game = "QuakeWorld";
    dir = real_quake_dir;
    cmd = default_qw_cmd;
    cfgdir = "id1";
  }

  if (cmd == NULL || cmd[0] == '\0') {
    dialog_ok (NULL, "%s command line is empty.", game);
    return FALSE;
  }

  if (dir != NULL && dir[0] != '\0') {
    if (stat (dir, &stat_buf) != 0 || !S_ISDIR (stat_buf.st_mode)) {
      dialog_ok (NULL, "\"%s\" is not a directory\n"
	              "Please specify correct %s working directory.", 
                       dir, game);
      return FALSE;
    }
  }

  path = file_in_dir (dir, cfgdir);

  if (stat (path, &stat_buf) || !S_ISDIR (stat_buf.st_mode)) {
    if (!dir || dir[0] == '\0') {
      dialog_ok (NULL, "Please specify correct %s working directory.", game);
    }
    else {
      dialog_ok (NULL,  
                 "Directory \"%s\" doesn\'t contain \"%s\" subdirectory.\n"
                 "Please specify correct %s working directory.", 
                 dir, cfgdir, game);
    }
    g_free (path);
    return FALSE;
  }

  g_free (path);
  return TRUE;
}


static FILE *open_cfg (const char *filename, int secure) {
  FILE *f;

  f = fopen (filename, "w");
  if (f) {
    if (secure)
      chmod (filename, S_IRUSR | S_IWUSR);

    fprintf (f, "//\n// generated by XQF, do not modify\n//\n");
  }
  return f;
}


static int real_password (const char *password) {
  if (!password || (password[0] == '1' && password[1] == '\0'))
    return FALSE;
  return TRUE;
}


static int write_passwords (const char *filename, const struct condef *con) {
  FILE *f;

  f = open_cfg (filename, TRUE);
  if (!f) 
    return FALSE;

  if (con->observe)
    fprintf (f, "spectator \"%s\"\n", con->observe);

  if (con->password)
    fprintf (f, "password \"%s\"\n", con->password);

  if (con->rcon_password)
    fprintf (f, "rcon_password \"%s\"\n", con->rcon_password);

  fclose (f);
  return TRUE;
}


static int write_qw_vars (const char *filename, const struct condef *con) {
  FILE *f;
  int plat;

  f = open_cfg (filename, FALSE);
  if (!f)
    return FALSE;

  if (default_name)
    fprintf (f, "name \"%s\"\n", default_name);

  if (default_qw_skin)
    fprintf (f, "skin \"%s\"\n", default_qw_skin);

  fprintf (f, "team \"%s\"\n", (default_team)? default_team : "");
  fprintf (f, "topcolor    \"%d\"\n", default_top_color);
  fprintf (f, "bottomcolor \"%d\"\n", default_bottom_color);

  if (default_pushlatency) {
    if (con->s->ping <= 0)
      plat = -50;		/* "min" value */
    else if (con->s->ping >= 2000)
      plat = -1000;		/* "max" value */
    else {
      plat = con->s->ping / 2;
      plat = ((plat+49)/50)*50;	/* beautify it */
    }
    fprintf (f, "pushlatency %d\n", -plat);
  }

  fprintf (f, "rate        \"%d\"\n", default_rate);
  fprintf (f, "cl_nodelta  \"%d\"\n", default_cl_nodelta);
  fprintf (f, "cl_predict_players \"%d\"\n", default_cl_predict);
  fprintf (f, "noaim       \"%d\"\n", default_noaim);
  fprintf (f, "noskins     \"%d\"\n", default_noskins);
  if (default_w_switch >= 0)
    fprintf (f, "setinfo w_switch \"%d\"\n", default_w_switch);
  if (default_b_switch >= 0)
    fprintf (f, "setinfo b_switch \"%d\"\n", default_b_switch);
  fprintf (f, "_windowed_mouse \"%d\"\n", default_windowed_mouse);

  if (default_qw_cfg)
    fprintf (f, "exec \"%s\"\n", default_qw_cfg);

  if (con->custom_cfg)
    fprintf (f, "exec \"%s\"\n", con->custom_cfg);

  fclose (f);
  return TRUE;
}


static int write_q2_vars (const char *filename, const struct condef *con) {
  FILE *f;

  f = open_cfg (filename, FALSE);
  if (!f)
    return FALSE;

  if (default_name)
    fprintf (f, "set name \"%s\"\n", default_name);

  if (default_q2_skin)
    fprintf (f, "set skin \"%s\"\n", default_q2_skin);

  fprintf (f, "set rate        \"%d\"\n", default_rate);
  fprintf (f, "set cl_nodelta  \"%d\"\n", default_cl_nodelta);
  fprintf (f, "set cl_predict  \"%d\"\n", default_cl_predict);
  fprintf (f, "set cl_noskins  \"%d\"\n", default_noskins);
  fprintf (f, "set _windowed_mouse \"%d\"\n", default_windowed_mouse);

  if (default_q2_cfg)
    fprintf (f, "exec \"%s\"\n", default_q2_cfg);

  if (con->custom_cfg)
    fprintf (f, "exec \"%s\"\n", con->custom_cfg);

  fclose (f);
  return TRUE;
}


static void dialog_failed (char *func, char *arg) {
  dialog_ok ("XQF: ERROR!", "ERROR!\n\n%s(%s) failed: %s", 
                                   func, (arg)? arg : "", g_strerror (errno));
}


static void client_free (struct running_client *cl) {

#ifdef DEBUG
  fprintf (stderr, "client detached (pid:%d)\n", cl->pid);
#endif

  if (cl->fd >= 0) {
    gdk_input_remove (cl->tag);
    close (cl->fd);
  }

  if (cl->buffer)
    g_free (cl->buffer);

  if (cl->server)
    server_unref (cl->server);

  g_free (cl);
}


static void client_detach (struct running_client *cl) {
  client_free (cl);
  clients = g_list_remove (clients, cl);
}


void client_detach_all (void) {
  g_list_foreach (clients, (GFunc) client_free, NULL);
  g_list_free (clients);
  clients = NULL;
}


static void client_sigchild_handler (int signum) {
  int pid;
  int status;
  GList *list;
  struct running_client *cl;

  while ((pid = waitpid (WAIT_ANY, &status, WNOHANG)) > 0) {

    for (list = clients; list; list = list->next) {
      cl = (struct running_client *) list->data;
      if (cl->pid == pid) {
	client_detach (cl);
	break;
      }
    }

  }
}


void client_init (void) {
  on_sig (SIGCHLD, client_sigchild_handler);
}


static void client_input_callback (struct running_client *cl, int fd, 
                                                GdkInputCondition condition) {
  int res;
  int pid;
  char *tmp;
  
  if (!cl->buffer)
    cl->buffer = g_malloc (CLIENT_ERROR_BUFFER);

  res = read (fd, cl->buffer + cl->pos, CLIENT_ERROR_BUFFER - 1 - cl->pos);

  if (res <= 0) {	/* read error or EOF */
    if (errno == EAGAIN || errno == EWOULDBLOCK)
      return;

    client_detach (cl);
    return;
  }

  if (cl->pos + res == CLIENT_ERROR_BUFFER - 1) {
    tmp = &cl->buffer[CLIENT_ERROR_BUFFER - 1];
    *tmp = '\0';
  }
  else {
    tmp = memchr (cl->buffer + cl->pos, '\0', res);
    cl->pos += res;
  }

  if (tmp) {
    gdk_input_remove (cl->tag);
    close (cl->fd);
    cl->fd = -1;

    if (!strncmp (cl->buffer, 
                     CLIENT_ERROR_MSG_HEAD, strlen (CLIENT_ERROR_MSG_HEAD))) {
      dialog_ok ("XQF: ERROR!", "ERROR!\n\n%s", 
	                         cl->buffer + strlen (CLIENT_ERROR_MSG_HEAD));

      pid = cl->pid;  /* save PID value */
      client_detach (cl);
      kill (pid, SIGTERM);
    }
    else {
      client_detach (cl);
    }
  }
}


static void client_attach (pid_t pid, int fd, struct server *s) {
  struct running_client *cl;

  cl = g_malloc0 (sizeof (struct running_client));

  cl->fd = fd;
  cl->pid = pid;
  cl->tag = gdk_input_add (cl->fd, GDK_INPUT_READ, 
                                (GdkInputFunction) client_input_callback, cl);

  cl->server = s;
  s->ref_count++;

#ifdef DEBUG
  fprintf (stderr, "client attached (pid:%d)\n", cl->pid);
#endif

  clients = g_list_append (clients, cl);
}


static int launch_quake_exec (int forkit, char *dir, char *argv[], 
                                                           struct server *s) {
  int pid;
  int pipefds[2];
  int flags;
  char msg[CLIENT_ERROR_BUFFER];

  if (dir && dir[0] != '\0') {
    if (chdir (dir) != 0) {
      dialog_failed ("chdir", dir);
      return -1;
    }
  }

#ifdef DEBUG
  {
    char **argptr = argv;

    fprintf (stderr, "EXEC> ");
    while (*argptr)
      fprintf (stderr, "%s ", *argptr++);
    fprintf (stderr, "\n");
  }
#endif

  if (forkit) {

    if (pipe (pipefds) < 0) {
      dialog_failed ("pipe", NULL);
      return -1;
    }

    pid = fork ();

    if (pid == -1) {
      dialog_failed ("fork", NULL);
      return -1;
    }

    if (pid) {	/* parent */
      close (pipefds[1]);

      flags = fcntl (pipefds[0], F_GETFL, 0);
      if (flags < 0 || fcntl (pipefds[0], F_SETFL, flags | O_NONBLOCK) < 0) {
	dialog_failed ("fcntl", NULL);
	return -1;
      }

      client_attach (pid, pipefds[0], s);
    }
    else {	/* child */
      close (pipefds[0]);

      execvp (argv[0], argv);
  
      g_snprintf (msg, CLIENT_ERROR_BUFFER, "%sexec(%s) failed: %s", 
                          CLIENT_ERROR_MSG_HEAD, argv[0], g_strerror (errno));

#ifdef DEBUG
      fprintf (stderr, "%s\n", msg);
#endif

      write (pipefds[1], msg, strlen (msg) + 1);
      close (pipefds[1]);

      on_sig (SIGHUP,  _exit);
      on_sig (SIGINT,  _exit);
      on_sig (SIGQUIT, _exit);
      on_sig (SIGBUS,  _exit);
      on_sig (SIGSEGV, _exit);
      on_sig (SIGPIPE, _exit);
      on_sig (SIGTERM, _exit);
      on_sig (SIGALRM, _exit);
      on_sig (SIGCHLD, SIG_DFL);

      sleep (15);
      _exit (1);
    }

    return pid;
  }

  execvp (argv[0], argv);

  dialog_failed ("exec", argv[0]);
  return -1;
}


static int already_running (enum server_type type) {
  int another = FALSE;
  struct server *s;
  GList *tmp;
  int res;

  if (!clients)
    return FALSE;

  for (tmp = clients; tmp; tmp = tmp->next) {
    s = (struct server *) ((struct running_client *) tmp->data) -> server;
    if (s->type == type) {
      another = TRUE;
      break;
    }
  }

  s = (struct server *) ((struct running_client *) clients->data) -> server;

  res = dialog_yesno (NULL, 1, "Launch", "Cancel", 
		    "There is %s client running.\n\nLaunch %s client?",
		    (another)? game_type_str (type) : game_type_str (s->type),
		    (another)? "another" : game_type_str (type));

  return TRUE - res;
}


void launch_quake (const struct condef *con, int forkit) {
  char *cmd;
  char *argv[32];
  int argi = 0;
  const char delim[] = " \t\n\r";
  char *q_cmd;
  char *q_dir;
  char *q_conf;
  char *q_passwd;
  char *file;

  if (!con || !con->server || !con->s)
    return;

  if (already_running (con->s->type))
    return;

  switch (con->s->type) {

  case QW_SERVER:
    q_cmd = default_qw_cmd;
    q_dir = real_quake_dir;
    q_conf = "id1/" EXEC_CFG;
    q_passwd = "id1/" PASSWORD_CFG;
    break;

  case Q2_SERVER:
    q_cmd = default_q2_cmd;
    q_dir = real_q2_dir;
    q_conf = "baseq2/" EXEC_CFG;
    q_passwd = "baseq2/" PASSWORD_CFG;
    break;

  default:
    return;
  }

  if (!q_cmd || q_cmd[0] == '\0')
    return;

  file = file_in_dir (q_dir, q_conf);

  if ((con->s->type == QW_SERVER && !write_qw_vars (file, con)) ||
      (con->s->type == Q2_SERVER && !write_q2_vars (file, con))) {
    if (!dialog_yesno (NULL, 1, "Launch", "Cancel", 
             "Cannot write to file \"%s\".\n\nLaunch client anyway?", file)) {
      g_free (file);
      return;
    }
  }

  g_free (file);

  cmd = strdup_strip (q_cmd);

  argv[argi++] = strtok (cmd, delim);
  while ((argv[argi] = strtok (NULL, delim)) != NULL)
    argi++;

  switch (con->s->type) {

  case QW_SERVER:
    if (default_nosound)
      argv[argi++] = "-nosound";

    if (default_nocdaudio)
      argv[argi++] = "-nocdaudio";

    if (con->gamedir) {
      argv[argi++] = "-gamedir";
      argv[argi++] = con->gamedir;
    }

    break;

  case Q2_SERVER:
    if (default_nosound) {
      argv[argi++] = "+set";
      argv[argi++] = "s_initsound";
      argv[argi++] = "0";
    }

    argv[argi++] = "+set";
    argv[argi++] = "cd_nocd";
    argv[argi++] = (default_nocdaudio)? "1" : "0";

    if (con->gamedir) {
      argv[argi++] = "+set";
      argv[argi++] = "game";
      argv[argi++] = con->gamedir;
    }

    break;

  default:
    break;

  }

  if (con->password || con->rcon_password || real_password (con->observe)) {
    file = file_in_dir (q_dir, q_passwd);

    if (!write_passwords (file, con)) {
      if (!dialog_yesno (NULL, 1, "Launch", "Cancel", 
             "Cannot write to file \"%s\".\n\nLaunch client anyway?", file)) {
	g_free (file);
	g_free (cmd);
	return;
      }
    }

    g_free (file);

    argv[argi++] = "+exec";
    argv[argi++] = PASSWORD_CFG;
  }
  else {
    if (con->observe) {
      argv[argi++] = "+spectator";
      argv[argi++] = con->observe;
    }
  }

  if (con->s->type == Q2_SERVER || 
                               (con->s->type == QW_SERVER && !con->gamedir)) {
    argv[argi++] = "+exec";
    argv[argi++] = EXEC_CFG;
  }

  if (con->server) {
    if (!con->demo || con->s->type == Q2_SERVER) {
      argv[argi++] = "+connect";
      argv[argi++] = con->server;
    }
    if (con->demo) {
      argv[argi++] = "+record";
      argv[argi++] = con->demo;
      if (con->s->type == QW_SERVER)
	argv[argi++] = con->server;
    }
  }

  argv[argi] = NULL;

  launch_quake_exec (forkit, q_dir, argv, con->s);

  g_free (cmd);
}


struct condef *condef_new (struct server *s) {
  struct condef *con;

  con = g_malloc (sizeof (struct condef));

  con->s = s;
  con->server = NULL;
  con->gamedir = NULL;
  con->observe = NULL;
  con->password = NULL;
  con->rcon_password = NULL;
  con->custom_cfg = NULL;
  con->demo = NULL;

  return con;
}


void condef_free (struct condef *con) {
  if (con->server) {
    g_free (con->server);
    con->server = NULL;
  }

  if (con->gamedir) {
    g_free (con->gamedir);
    con->gamedir = NULL;
  }

  if (con->observe) {
    g_free (con->observe);
    con->observe = NULL;
  }

  if (con->password) {
    g_free (con->password);
    con->password = NULL;
  }

  if (con->rcon_password) {
    g_free (con->rcon_password);
    con->rcon_password = NULL;
  }

  if (con->custom_cfg) {
    g_free (con->custom_cfg);
    con->custom_cfg = NULL;
  }

  if (con->demo) {
    g_free (con->demo);
    con->demo = NULL;
  }

  g_free (con);
}



