/* XQF - Quake server browser and launcher
 * Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <sys/types.h>
#include <string.h>	/* strstr, strtok, strlen, strcmp */
#include <ctype.h>	/* tolower */
#include <regex.h>

#include <gtk/gtk.h>

#include "xqf.h"
#include "utils.h"
#include "psearch.h"
#include "dialogs.h"
#include "xutils.h"


enum psearch_mode { PSEARCH_MODE_SUBSTRING = 0, PSEARCH_MODE_REGEXP = 1 };
enum psearch_case { PSEARCH_CASE_SENSITIVE = 0, PSEARCH_CASE_INSENSITIVE = 1 };


struct psearch_data {
  enum psearch_mode mode;
  enum psearch_case csens;
  void *pattern;
};


static struct psearch_data psearch = { 
  PSEARCH_MODE_SUBSTRING, 
  PSEARCH_CASE_INSENSITIVE, 
  NULL
};


static GList *psearch_history = NULL;


static char *lowcasestrstr (const char *str, const char *substr) {
  int slen = strlen (str);
  int sublen = strlen (substr);
  const char *end;
  int i;

  if (slen < sublen)
    return NULL;

  end = &str[slen - sublen + 1];

  while (str < end) {
    for (i = 0; i < sublen; i++) {
      if (substr[i] != tolower (str[i]))
	goto loop;
    }
    return (char *) str;

  loop:
    str++;
  }

  return NULL;
}


static void psearch_history_free (void) {
  g_list_foreach (psearch_history, (GFunc) g_free, NULL);
  g_list_free (psearch_history);
  psearch_history = NULL;
}


static char *psearch_history_add (char *str) {
  GList *list;

  if (!str)
    return NULL;

  for (list = psearch_history; list; list = list->next) {
    if (!strcmp (str, (char *) list->data)) {
      g_free (list->data);
      psearch_history = g_list_remove_link (psearch_history, list);
      break;
    }
  }

  while (psearch_history && 
                g_list_length (psearch_history) >= PSEARCH_HISTORY_MAXITEMS) {
    list = g_list_last (psearch_history);
    g_free (list->data);
    psearch_history = g_list_remove_link (psearch_history, list); 
  }
  
  psearch_history = g_list_prepend (psearch_history, g_strdup (str));
  return str;
}


int test_player (struct player *p) {
  GSList *tmp;

  if (!psearch.pattern || !p->name)
    return FALSE;

  switch (psearch.mode) {

  case PSEARCH_MODE_SUBSTRING:

    switch (psearch.csens) {

    case PSEARCH_CASE_SENSITIVE:
      for (tmp = (GSList *) psearch.pattern; tmp; tmp = tmp->next) {
	if (strstr (p->name, (char *) tmp->data))
	  return TRUE;
      }
      return FALSE;

    case PSEARCH_CASE_INSENSITIVE:
      for (tmp = (GSList *) psearch.pattern; tmp; tmp = tmp->next) {
	if (lowcasestrstr (p->name, (char *) tmp->data))
	  return TRUE;
      }
      return FALSE;
    }
    return FALSE;

  case PSEARCH_MODE_REGEXP:
    if (regexec ((regex_t *) psearch.pattern, p->name, 0, NULL, 0) == 0)
      return TRUE;
    return FALSE;
  }
  return FALSE;
}


static void psearch_free_pattern (struct psearch_data *ps) {
  if (!ps)
    return;

  if (ps->pattern) {
    switch (ps->mode) {

    case PSEARCH_MODE_SUBSTRING:
      g_slist_foreach ((GSList *) ps->pattern, (GFunc) g_free, NULL);
      g_slist_free ((GSList *) ps->pattern);
      break;

    case PSEARCH_MODE_REGEXP:
      regfree ((regex_t *) ps->pattern);
      g_free (ps->pattern);
      break;
    }

    ps->pattern = NULL;
  }
}


static enum psearch_res psearch_set_pattern (char *str, 
                             enum psearch_mode mode, enum psearch_case csens) {
  const char delim[] = " \t\n\r";
  char *tmp, *tok;
  int regex_flags;
  regex_t *preg;
  int res;

  if (psearch.pattern)
    psearch_free_pattern (&psearch);

  tmp = strdup_strip (str);

  if (!tmp)
    return PSEARCH_CANCEL;

  switch (mode) {

  case PSEARCH_MODE_SUBSTRING:
    if (csens == PSEARCH_CASE_INSENSITIVE) {
      for (tok = tmp; *tok; tok++)
	*tok = tolower (*tok);
    }

    tok = strtok (tmp, delim);
    (GSList *) psearch.pattern = 
                 g_slist_prepend ((GSList *) psearch.pattern, g_strdup (tok));
    
    while ((tok = strtok (NULL, delim)) != NULL) {
      (GSList *) psearch.pattern = 
	         g_slist_prepend ((GSList *) psearch.pattern, g_strdup (tok));
    }

    g_free (tmp);
    break;

  case PSEARCH_MODE_REGEXP:
    preg = g_malloc (sizeof (regex_t));

    regex_flags = (csens == PSEARCH_CASE_INSENSITIVE)? REG_ICASE : 0;
    regex_flags |= REG_NOSUB;

    res = regcomp (preg, tmp, regex_flags);
    g_free (tmp);

    if (res) {
      g_free (preg);
      dialog_ok (NULL, "Bad regular expression");
      return PSEARCH_CANCEL;
    }

    psearch.pattern = preg;
    break;
  }

  psearch.mode = mode;
  psearch.csens = csens;
  return PSEARCH_OK;
}


static GtkWidget *case_buttons[2];
static GtkWidget *mode_buttons[2];
static GtkWidget *psearch_combo;


static enum psearch_res psearch_ok;


static void psearch_combo_activate_callback (GtkWidget *widget, 
                                                              gpointer data) {
  enum psearch_mode mode;
  enum psearch_case csens;
  char *str;

  mode  = GTK_TOGGLE_BUTTON (mode_buttons[1])->active;
  csens = GTK_TOGGLE_BUTTON (case_buttons[1])->active;

  str = strdup_strip (
           gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (psearch_combo)->entry)));

  if (str) {
    if (psearch.mode == mode && psearch.csens == csens && psearch_history && 
                              !strcmp (str, (char *) psearch_history->data)) {
      if (psearch.mode == PSEARCH_MODE_REGEXP) { 
	psearch_ok = psearch_set_pattern (str, mode, csens);
	if (psearch_ok != PSEARCH_OK)
	  psearch_ok = PSEARCH_CANCEL;
      }
      else {
	psearch_ok = PSEARCH_USELAST;
      }
    }
    else {
      psearch_history_add (str);
      psearch_ok = psearch_set_pattern (str, mode, csens);
    }	
    g_free (str);
  }
}


enum psearch_res psearch_dialog (void) {
  GtkWidget *window;
  GtkWidget *main_vbox;
  GtkWidget *hbox;
  GtkWidget *vbox;
  GtkWidget *button;
  GtkWidget *label;
  GtkWidget *frame;
  GSList *group;

  psearch_ok = PSEARCH_CANCEL;

  window = dialog_create_modal_transient_window ("Find Player", TRUE, FALSE);

  main_vbox = gtk_vbox_new (FALSE, 8);
  gtk_container_border_width (GTK_CONTAINER (main_vbox), 16);
  gtk_container_add (GTK_CONTAINER (window), main_vbox);

  hbox = gtk_hbox_new (FALSE, 8);
  gtk_box_pack_start (GTK_BOX (main_vbox), hbox, TRUE, TRUE, 0);

  /* Message */

  label = gtk_label_new ("Find Player:");
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  /* ComboBox */

  psearch_combo = gtk_combo_new ();
  gtk_entry_set_max_length (GTK_ENTRY (GTK_COMBO (psearch_combo)->entry), 32);
  gtk_widget_set_usize (GTK_COMBO (psearch_combo)->entry, 120, -1);
  gtk_combo_disable_activate (GTK_COMBO (psearch_combo));
  if (psearch_history) {
    gtk_combo_set_popdown_strings (GTK_COMBO (psearch_combo), 
                                                             psearch_history);
    gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (psearch_combo)->entry), 
                                              (char *) psearch_history->data);
    gtk_entry_select_region (GTK_ENTRY (GTK_COMBO (psearch_combo)->entry), 0, 
                                     strlen ((char *) psearch_history->data));
  }
  gtk_signal_connect (GTK_OBJECT (GTK_COMBO (psearch_combo)->entry), 
         "activate", GTK_SIGNAL_FUNC (psearch_combo_activate_callback), NULL);
  gtk_signal_connect_object (GTK_OBJECT (GTK_COMBO (psearch_combo)->entry), 
       "activate", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (window));
  gtk_box_pack_start (GTK_BOX (hbox), psearch_combo, TRUE, TRUE, 0);
  gtk_widget_grab_focus (GTK_COMBO (psearch_combo)->entry);
  gtk_widget_show (psearch_combo);

  /* OK Button */

  button = gtk_button_new_with_label (" OK ");
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
		            GTK_SIGNAL_FUNC (psearch_combo_activate_callback),
			    GTK_OBJECT (psearch_combo));
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
	           GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (window));
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  gtk_widget_show (button);

  /* Cancel Button */

  button = gtk_button_new_with_label (" Cancel ");
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                   GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (window));
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  gtk_widget_show (button);

  gtk_widget_show (hbox);

  /* Mode Buttons */

  hbox = gtk_hbox_new (FALSE, 4);
  gtk_box_pack_start (GTK_BOX (main_vbox), hbox, FALSE, FALSE, 0);

  /* Substring/RegExp Mode Buttons */

  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (hbox), frame, FALSE, FALSE, 0);

  vbox = gtk_vbox_new (FALSE, 2);
  gtk_container_border_width (GTK_CONTAINER (vbox), 4);
  gtk_container_add (GTK_CONTAINER (frame), vbox);

  mode_buttons[0] = gtk_radio_button_new_with_label (NULL, "substring match");
  group = gtk_radio_button_group (GTK_RADIO_BUTTON (mode_buttons[0]));
  mode_buttons[1] = gtk_radio_button_new_with_label (group,
                                                  "regular expression match");
  gtk_toggle_button_set_state (
                        GTK_TOGGLE_BUTTON (mode_buttons[psearch.mode]), TRUE);

  gtk_box_pack_start (GTK_BOX (vbox), mode_buttons[0], TRUE, TRUE, 0);
  gtk_widget_show (mode_buttons[0]);

  gtk_box_pack_start (GTK_BOX (vbox), mode_buttons[1], TRUE, TRUE, 0);
  gtk_widget_show (mode_buttons[1]);

  gtk_widget_show (vbox);
  gtk_widget_show (frame);

  /* Case Sensitivity Buttons */

  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (hbox), frame, FALSE, FALSE, 0);

  vbox = gtk_vbox_new (FALSE, 2);
  gtk_container_border_width (GTK_CONTAINER (vbox), 4);
  gtk_container_add (GTK_CONTAINER (frame), vbox);

  case_buttons[0] = gtk_radio_button_new_with_label (NULL, "case sensitive");
  group = gtk_radio_button_group (GTK_RADIO_BUTTON (case_buttons[0]));
  case_buttons[1] = 
                  gtk_radio_button_new_with_label (group, "case insensitive");

  gtk_toggle_button_set_state (
                       GTK_TOGGLE_BUTTON (case_buttons[psearch.csens]), TRUE);

  gtk_box_pack_start (GTK_BOX (vbox), case_buttons[0], TRUE, TRUE, 0);
  gtk_widget_show (case_buttons[0]);

  gtk_box_pack_start (GTK_BOX (vbox), case_buttons[1], TRUE, TRUE, 0);
  gtk_widget_show (case_buttons[1]);

  gtk_widget_show (vbox);
  gtk_widget_show (frame);

  gtk_widget_show (hbox);

  gtk_widget_show (main_vbox);
  gtk_widget_show (window);

  gtk_main ();

  unregister_window (window);

  return psearch_ok;
}


void psearch_free_data (void) {
  if (psearch.pattern) {
    psearch_free_pattern (&psearch);
  }
  psearch_history_free ();
}

