/*
    SABRE Fighter Plane Simulator 
    Copyright (c) 1997 Dan Hammer
    Portions Donated By Antti Barck

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/*************************************************
 *           SABRE Fighter Plane Simulator       *
 * Version: 0.1                                  *
 * File   : fltmngr.C                            *
 * Date   : March, 1997                          *
 * Author : Dan Hammer                           *
 *************************************************/
#include <stdio.h>
#include <iostream.h>
#include <fstream.h>
#include <math.h>
#include <limits.h>
#include <values.h>
#include "defs.h"
#include "sim.h"
#include "vmath.h"
#include "port_3d.h"
#include "cpoly.h"
#include "obj_3d.h"
#include "copoly.h"
#include "flight.h"
#include "fltlite.h"
#include "weapons.h"
#include "pilot.h"
#include "zview.h"
#include "fltzview.h"
#include "unguided.h"
#include "simfile.h"
#include "simsnd.h"
#include "fltmngr.h"
#include "siminput.h"
#include "simfilex.h"

extern int gpause;
extern int do_random;
extern int no_crash;

#define F2FIX(a) (a)
#define FIX2F(a) (a)

void Flight_Node::read(istream &is)
{
  char buff[64];
  int n;
  char c;
  Flight_Specs *spcs = NULL;
  Z_Node_Manager *zm = NULL;

  READ_TOKI('[',is,c)
    
  if (simfileX::readinput(is,buff,sizeof(buff),n) == simfileX::STR_INPUT)
    {
      spcs = Flight_Specs::getSpecs(buff);
      if (spcs == NULL)
	{
	  printf("Couln't locate specs: %s\n",
		 buff);
	  spcs = Flight_Specs::g_specs[0];
	}
    }
  else
    {
      MYCHECK(n >= 0 && n < Flight_Specs::nspecs);
      spcs = Flight_Specs::g_specs[n];
    }
  MYCHECK(spcs != NULL);
  flight = new Flight(spcs);

  // read in flight params
  is >> *(flight);
  // RANDOMize params
  if (do_random)
    {
      Port_3D *prt = & flight->state.flight_port;
      Vector v(frand(10000.0*world_scale),frand(10000.0*world_scale),
	       frand(3000.0*world_scale));
      prt->delta_look_from(v);
      prt->set_theta(frand(_2PI));
      flight->state.velocity.direction = prt->view_normal;

    }

  if (simfileX::readinput(is,buff,sizeof(buff),n) == simfileX::STR_INPUT)
    {
      zm = Z_Node_Manager::getZNode(buff);
      if (zm == NULL)
	{
	  printf("Couln't locate zm: %s\n",
		 buff);
	  zm = Z_Node_Manager::g_zmanagers[0];
	}
    }
  else
    {
      MYCHECK(n >= 0 && n < Z_Node_Manager::nzmanagers);
      zm = Z_Node_Manager::g_zmanagers[n];
    }
  viewer = new Flight_ZViewer(zm);
  viewer->set_flight(flight);


  // Get weapon list
  if (simfileX::readinput(is,buff,sizeof(buff),n) == simfileX::STR_INPUT)
    wi = wpm.build_instance_list(n,&n_weaps,buff);
  else
    wi = wpm.build_instance_list(n,&n_weaps);

  MYCHECK(wi != NULL);

  viewer->setWeaponInstanceList(new Weapon_Instance_List(wi,n_weaps));

  // Create pilot. If this is not a user node,
  // set the controls of the flight to autopilot
  is >> n;
  MYCHECK(pilot_params[n] != NULL);
  pilot = new Pilot(flight,pilot_params[n],
		       wi,n_weaps,
		       handle,
		       viewer);
  if (Pilot::npilots < Pilot::maxpilots - 1)
    Pilot::pilots[Pilot::npilots++] = pilot;
  // check for waypoint
  is >> n;
  if (n >= 0)
    {
      MYCHECK(WayPoint::gwaypoints[n] != NULL);
      pilot->set_waypoint(WayPoint::gwaypoints[n]);
    }
  is >> handle;
  handle[16] = '\0';
  is >> c;
  READ_TOK(']',is,c)
    /* set the affiliation */
    viewer->affiliation = pilot->params->affiliation;
}

istream &operator >>(istream &is, Flight_Node &fn)
{
  fn.read(is);
  return is;
}

/*
#define DEBUG_WEAPS
*/

/******************************************************
 * Update the flight                                  *
 *****************************************************/
void Flight_Node::update(Unguided_Manager *um)
{
  int r;
  // Skip if crashed
  if (flight->state.crashed && pilot->functioning)
    {
      /* 
	 If the pilot's still in there,
	 well, no more cloud-hopping for
	 him
	 */
      pilot->functioning = 0;
      pilot->set_target(NULL);
      pilot->dbg = "DEAD";
      return;
    }
  
  // Make sure the Target::position member is updated!
  viewer->position = flight->state.flight_port.look_from;

  if (view_node)
    {
      if ((flight->controls.throttle > 0) &&
	  (viewer->hurt < viewer->max_damage))
	sound_on(JET_ENGINE,LOOP);
      else
	sound_off(JET_ENGINE);
    }

  /* See if autopilot is on */
  if (pilot && flight->controls.autopilot != 0)
    {
      if (user_node && !user_auto)
	{
	  pilot->start();
	  user_auto = 1;
	}
      pilot->update(1);
    }
  else 
    {
      if (user_auto)
	{
	  pilot->pause();
	  user_auto = 0;
	}
      // Maintain tracking of target for hud
      pilot->update(0);
    }
  r = 0;

  /* Check for a weapons launch */
  if (um != NULL && pilot->sel_weapon != NULL)
    {
      /* 
	  don't let user fire unless they've armed the weapon
         (controls.gunsight is 1)
      */
      if (view_node)
	{
	  if (!flight->controls.armed_w && !flight->controls.autopilot)
	    flight->controls.bang_bang = 0;
	}
#ifdef DEBUG_WEAPS
      if (!flight->controls.bang_bang && pilot->n_weaps > 1 &&
	  RANDOM(5) == 1)
	{
	  flight->controls.bang_bang = 1;
	  int nweap = RANDOM(pilot->n_weaps - 1) + 1;
	  if (nweap >= pilot->n_weaps)
	    nweap = pilot->n_weaps - 1;
	  pilot->weapons[nweap].update(*flight,um,viewer);
	  flight->controls.bang_bang = 0;
	}
#endif
      r = pilot->sel_weapon->update(*flight,um,viewer);
      if (r)
	{
	  if (view_node)
	    sound_on(GUN,NORM);
	  viewer->shots += r;
	} 
    }

  /*If crashing possible, check for one*/
  
  if (!flight->state.crashed && (!no_crash || !view_node))
    {
      r = viewer->landing_report(lr);
      if (r && lr.score < 0)
	sound_on(CRASH,NORM);
      if (flight->state.crashed)
	{
	  R_3DPoint p = flight->state.flight_port.look_from;
	  //	  p.z = 0.0;
	  //	  p.z += 75.0 * world_scale;
	  if (um)
	    um->boom(p,0);
	  sound_on(CANNON,NORM);
	}
    }

}

void Flight_Node::add_draw_list(DrawList &dlist, Port_3D &port)
{
  if (view_node && flight->controls.view <= back 
      && !gpause)
    return;
  viewer->draw_prep(port);
  viewer->cpk_flg = 0;
  if (viewer->visible_flag)
    dlist.add_object(viewer);
}

Flight_Manager::~Flight_Manager()
{
  if (specs != NULL)
    {
      for (int i=0;i<n_specs;i++)
	{
	  if (specs[i] != NULL)
	    delete specs[i];
	}
      delete [] specs;
      Flight_Specs::g_specs = NULL;
    }
  if (z_managers != NULL)
    {
      for (int i=0;i<n_managers;i++)
	if (z_managers[i] != NULL)
	  delete z_managers[i];
      delete [] z_managers;
      Z_Node_Manager::g_zmanagers = NULL;
    }
  if (p_params != NULL)
    {
      for (int i=0;i<n_pparams;i++)
	if (p_params[i])
	  delete p_params[i];
      delete [] p_params;
    }
  if (flight_nodes != NULL)
    {
      for (int i=0;i<n_flights;i++)
	if (flight_nodes[i])
	  delete flight_nodes[i];
      delete [] flight_nodes;
    }
}

int Flight_Manager::read_file(char *path)
{
  int result = 0;
  ifstream infile;

  if (open_is(infile,path))
    {
      result = 1;
      infile >> *this;
      infile.close();
    }
  return result;
}

void Flight_Manager::read(istream &is)
{
  char buff[BUFSIZ];
  int got_line;
  ifstream is2;
  int i;

  is.seekg(0L);
  // Get flight specification file(s)
  got_line = get_line(is,buff,sizeof(buff));
  MYCHECK(got_line);
  n_specs = atoi(buff);
  MYCHECK(n_specs > 0);
  // Create specs
  specs = new Flight_Specs *[n_specs];
  MYCHECK(specs != NULL);
  Flight_Specs::nspecs = n_specs;
  Flight_Specs::g_specs = specs;
  for (i = 0; i < n_specs; i++)
    {
      got_line = get_line(is,buff,sizeof(buff));
      MYCHECK(got_line);
      specs[i] = new Flight_Specs();
      MYCHECK(specs[i] != NULL);
      specs[i]->read_file(buff);
    }

  // Get shape file(s)
  got_line = get_line(is,buff,sizeof(buff));
  MYCHECK(got_line);
  // Create z_managers
  n_managers = atoi(buff);
  MYCHECK(n_managers > 0);
  z_managers = new Z_Node_Manager *[n_managers];
  MYCHECK(z_managers != NULL);
  for (i=0;i<n_managers;i++)
    {
      z_managers[i] = new Z_Node_Manager();
      MYCHECK(z_managers[i] != NULL);
      got_line = get_line(is,buff,sizeof(buff));
      MYCHECK(got_line);
      z_managers[i]->read_file(buff);
      z_managers[i]->setId(buff);
    }
  Z_Node_Manager::nzmanagers = n_managers;
  Z_Node_Manager::g_zmanagers = z_managers;

  // Get pilot specification file
  got_line = get_line(is,buff,sizeof(buff));
  MYCHECK(got_line);
  if (open_is(is2,buff))
    {
      // Create pilot parameters array
      n_pparams = read_int(is2);
      p_params = new Pilot_Params *[n_pparams];
      MYCHECK(p_params != NULL);
      for (i=0;i<n_pparams;i++)
	{
	  p_params[i] = new Pilot_Params();
	  MYCHECK(p_params[i] != NULL);
	  is2 >> *(p_params[i]);
	}
      is2.close();
    }

  // Get weapons specification file
  got_line = get_line(is,buff,sizeof(buff));
  MYCHECK(got_line);
  wm.read_file(buff);

  // Get waypoints specification file
  got_line = get_line(is,buff,sizeof(buff));
  MYCHECK(got_line);
  if (*buff == '@')
    WayPoint::readWayPoints(is);
  else
    WayPoint::readWayPointFile(buff);
  // Get nodes file
  got_line = get_line(is,buff,sizeof(buff));
  MYCHECK(got_line);
  if (*buff == '@')
    readNodes(is);
  else
    readNodeFile(buff);
}

istream & operator >>(istream &is, Flight_Manager &fm)
{
  fm.read(is);
  return is;
}

void Flight_Manager::readNodes(istream &is)
{
  int i;

  Pilot::npilots = 0;
  // Create flight nodes
  is >> n_flights;
  flight_nodes = new Flight_Node *[n_flights];
  MYCHECK(flight_nodes != NULL);
  for (i=0;i<n_flights;i++)
    {
      flight_nodes[i] = new Flight_Node(specs,z_managers,
					p_params,wm);
      MYCHECK(flight_nodes[i] != NULL);
      if (i == 0)
	{
	  flight_nodes[i]->user_node = 1;
	  flight_nodes[i]->view_node = 1;
	}
      is >> *(flight_nodes[i]);
    }
  
  // See if any targeting to be done
  int n;
  is >> n;
  if (n > -1)
    {
      int x,y;
      for (i=0;i<n;i++)
	{
	  char c = ' ';
	  READ_TOK('(',is,c);
	  is >> x >> y;
	  READ_TOK(')',is,c);
	  MYCHECK(x != y);
	  MYCHECK(x <= n_flights && y <= n_flights);
	  flight_nodes[x]->pilot->set_target(flight_nodes[y]->pilot);
	  flight_nodes[x]->target_node = y;
	}
    }
}

void Flight_Manager::readNodeFile(char *path)
{
  ifstream is;
  if (open_is(is,path))
    readNodes(is);
}

void Flight_Manager::set_view_node(int vn)
{
  if (vn >= n_flights)
    vn = 0;
  else if (vn < 0)
    vn = n_flights - 1;
  flight_nodes[view_node]->view_node = 0;
  view_node = vn;
  flight_nodes[view_node]->view_node = 1;
}

int Flight_Manager::select_next_target(int fr)
{
  int i;
  int foundit = 0;
  int ii = flight_nodes[fr]->target_node;
  for (i=ii+1;i<n_flights;i++)
    {
      if (
	  (i != fr)
	  && (flight_nodes[fr]->pilot->target_flight !=
	      flight_nodes[i]->flight)
	  )
	{
	  flight_nodes[fr]->pilot->set_target(flight_nodes[i]->pilot);
	  flight_nodes[fr]->target_node = i;
	  foundit = 1;
	  break;
	}
    }

  if (!foundit)
    {
      for (i=0;i<ii;i++)
	{
	  if (
	      (i != fr)
	      && (flight_nodes[fr]->pilot->target_flight !=
		  flight_nodes[i]->flight)
	      )
	    {
	      flight_nodes[fr]->pilot->set_target(flight_nodes[i]->pilot);
	      flight_nodes[fr]->target_node = i;
	      foundit = 1;
	      break;
	    }
	}
    }
  return (foundit);
}

void Flight_Manager::update(Unguided_Manager *um)
{
  um->check_hits();
  for (int i=0;i<n_flights;i++)
    {
      if (!flight_nodes[i]->flight->state.crashed)
	flight_nodes[i]->flight->update();
    }
  um->update();
  for (int i=0;i<n_flights;i++)
    flight_nodes[i]->update(um);
}

void Flight_Manager::start()
{
  for (int i=0;i<n_flights;i++)
    flight_nodes[i]->start();
}

void Flight_Manager::pause()
{
  for (int i=0;i<n_flights;i++)
    flight_nodes[i]->pause();
}

void Flight_Manager::add_draw_list(DrawList &dlist, Port_3D &port)
{
  for (int i=0;i<n_flights;i++)
    flight_nodes[i]->add_draw_list(dlist,port);
}


void Flight_Manager::set_flight_view(Port_3D &vport)
{
  R_3DPoint lf,la,r1,r2,*tpoint;
  Vector v,v1;
  float dx,dy;

  if (view_node != v_4_node)
    {
      v_4_flag = 0;
      v_4_node = -1;
    }

  Flight &flt = *(flight_nodes[view_node]->flight);

  vport = flt.state.flight_port;
  Port_3D &fport = flt.state.flight_port;

  if (flt.controls.view > outside_4)
    flt.controls.view = front;
  if (flt.controls.view != outside_4)
    v_4_flag = 0;

  if (flt.controls.view <= back)
    {
      R_3DPoint vp = flt.specs->view_point;
      Z_Node_Manager &zm = *(flight_nodes[view_node]->viewer->z_manager);
      vp *= zm.scaler * world_scale;
      fport.port2world(vp,&lf);
      v = Vector(lf - flt.state.flight_port.look_from);
      vport.delta_look_from(v);
    }

  //For left and right views, we need to switch the roll
  //and the pitch.
  REAL_TYPE roll,phi;

  switch (flt.controls.view)
    {
    case left:
    case right:
      vport.port2world(R_3DPoint(0,0,0),&lf);
      if (flt.controls.view == left)
	vport.port2world(R_3DPoint(-1.0,0,0),&la);
      else
	vport.port2world(R_3DPoint(1.0,0,0),&la);
      vport.set_view(lf,la);
      vport.set_roll(0.0);
      if (flt.controls.view == left)
	{
	  vport.world2port(fport.look_from,&r1);
	  vport.world2port(fport.look_at,&r2);
	}
      else
	{
	  vport.world2port(fport.look_from,&r2);
	  vport.world2port(fport.look_at,&r1);
	}
      dx = r2.x - r1.x;
      dy = r2.y - r1.y;
      roll = atan2(dy,dx);
      if (roll < 0.0)
	roll = _2PI + roll;
      vport.set_roll(roll);
      break;

    case back:
      phi = vport.slook_from.phi;
      phi = _PI2 - phi;
      vport > (2*phi);
      vport < _PI;
      vport / (-2 * vport.roll);
      break;

    case outside_1:
    l1:
    vport.set_roll(0);
    vport.set_phi(1.57);
    vport.set_theta(0);
    vport > flt.controls.vphi;
    vport < flt.controls.vtheta;
    v = to_nvector(vport.look_from,vport.look_at);
    v *= (flt.controls.vdist * world_scale);
    r1 = R_3DPoint(v) + vport.look_from;
    v1 = to_nvector(r1,vport.look_from);
    r2 = R_3DPoint(v1) + r1;
    vport.set_view(r1,r2);
    break;

    case outside_2:
    case outside_3:
      Pilot *pilot;
      Port_3D *vp;
      if (flt.controls.view == outside_2)
	{
	  pilot = flight_nodes[view_node]->pilot;
	  vp = &vport;
	}
      else
	{
	  pilot = flight_nodes[view_node]->pilot->target_pilot;
	  if (pilot == NULL)
	    goto l1;
	  vp = &pilot->flight->state.flight_port;
	  vport = *vp;
	}

      if (pilot->target_flight)
	{
	  REAL_TYPE vx = 12.0 * world_scale;
	  R_3DPoint p1 = vp->look_from + R_3DPoint(vx,vx,vx);
	  tpoint = pilot->target->get_position();
	  v = to_nvector(*tpoint,p1);
	  v *= (flt.controls.vdist * world_scale);
	  r1 = R_3DPoint(v) + p1;
	  v1 = to_nvector(r1,*tpoint);
	  r2 = R_3DPoint(v1) + r1;
	  vport.set_view(r1,r2);
	  vport.set_roll(0);
	}
      else
	goto l1;
      break;

    case satellite:
      {
	float x,y,z;
	float d;
	x = vport.look_from.x;
	y = vport.look_from.y;
	z = vport.look_from.z;
	d = flt.controls.vdist * world_scale;
	z += d;
	lf = R_3DPoint(x,y,z);
	la = seg2point(lf,vport.look_from);
	vport.set_view(lf,la);
	vport.set_roll(0);
	break;
      }

    case outside_4:
      if (!v_4_flag)
	{
	  vport.set_roll(0);
	  vport.set_phi(1.57);
	  vport.set_theta(0);
	  vport > flt.controls.vphi;
	  vport < flt.controls.vtheta;
	  v = to_nvector(vport.look_from,vport.look_at);
	  v *= (flt.controls.vdist * world_scale);
	  r1 = R_3DPoint(v) + vport.look_from;
	  v1 = to_nvector(r1,vport.look_from);
	  r2 = R_3DPoint(v1) + r1;
	  vport.set_view(r1,r2);
 	  v_4_port = vport;
	  v_4_flag = 1;
	  v_4_node = view_node;
	}
      else
	{
	  v_4_port.set_view(v_4_port.look_from,vport.look_from);
	  vport = v_4_port;
	}
      break;
      
      // Wall warning
    case front:
      break;

    }
}
