/* Risk! for Linux..
 * newgame.c
 *
 * Copyright (C) 1998 Joel Nordell
 *
 * This file is protected under the GPL license.
 */

#include "newgame.h"
#include "util.h"

typedef enum {
  OFF,
  HUMAN,
  CPU
} PlayerType;

typedef struct _PlayerDialog {
  gdouble color[4];	// Current color
  gdouble color2[4];	// New color
  GtkWidget *colordialog; // Color dialog window
  GtkWidget *colorbox;	// Color box drawing area
  GtkWidget *colorframe;// Color frame widget
  GtkWidget *name;	// Player name input widget
  GtkWidget *select;	// Select widget
  PlayerType type;	// Player type selection
} PlayerDialog;

static
PlayerDialog PlayerDialogs[6] = { // One for each player
  { { 1, 0, 0, 0 },
    { 0, 0, 0, 0 },
    NULL, NULL, NULL, NULL, NULL, OFF },
  { { 0, 1, 0, 0 },
    { 0, 0, 0, 0 },
    NULL, NULL, NULL, NULL, NULL, OFF },
  { { 0, 0, 1, 0 },
    { 0, 0, 0, 0 },
    NULL, NULL, NULL, NULL, NULL, OFF },
  { { 1, 1, 0, 0 },
    { 0, 0, 0, 0 },
    NULL, NULL, NULL, NULL, NULL, OFF },
  { { 1, 0, 1, 0 },
    { 0, 0, 0, 0 },
    NULL, NULL, NULL, NULL, NULL, OFF },
  { { 0, 1, 1, 0 },
    { 0, 0, 0, 0 },
    NULL, NULL, NULL, NULL, NULL, OFF }
};

GtkWidget *NewGameDialog = NULL;

#define color_box_width 32
#define color_box_height 18

static gint color_box_close(int player);

static
void new_game_dialog_close() {
  int i;

  // Only close it if it's already open!
  if (NewGameDialog != NULL) {
    gtk_widget_destroy(NewGameDialog);
    NewGameDialog = NULL;
  }
  for (i=0; i<6; i++)
    color_box_close(i);
}

static gint
expose_event(GtkWidget *widget, GdkEventExpose *event, gpointer data) {
  GdkColor c;
  GdkGC *gc;
  int player = (int)data;

  c.pixel = 0;
  c.red = (PlayerDialogs[player].color[0] * 65535);
  c.green = (PlayerDialogs[player].color[1] * 65535);
  c.blue = (PlayerDialogs[player].color[2] * 65535);

  gc = gdk_gc_new(widget->window);
  gdk_gc_copy(gc, widget->style->white_gc);

  gdk_color_alloc(
    gdk_window_get_colormap(widget->window),
    &c);

  gdk_gc_set_foreground(gc, &c);

  gdk_draw_rectangle(widget->window, gc, TRUE, 
                  event->area.x, event->area.y,
                  event->area.width, event->area.height);
}

static
void new_game_cancel_callback(GtkWidget *widget, gpointer data) {
  new_game_dialog_close();
}

static
void new_game_ok_callback(GtkWidget *widget, gpointer data) {
  new_game_dialog_close();
}

static
gint color_box_close(int player) {
  if (PlayerDialogs[player].colordialog == NULL) return;

  gtk_widget_destroy(PlayerDialogs[player].colordialog);
  PlayerDialogs[player].colordialog = NULL;

  return FALSE; // To stop this from being re-called as an idle function
}

static
void color_box_ok(GtkWidget *widget, gpointer data) {
  gtk_color_selection_get_color(
    GTK_COLOR_SELECTION(
      GTK_COLOR_SELECTION_DIALOG(PlayerDialogs[(int)data].colordialog)
      ->colorsel),
    PlayerDialogs[(int)data].color);

  gtk_widget_queue_draw(GTK_WIDGET(PlayerDialogs[(int)data].colorbox));
  gtk_idle_add((GtkFunction)color_box_close, data);
}

static
void color_box_cancel(GtkWidget *widget, gpointer data) {
  gtk_idle_add((GtkFunction)color_box_close, data);
}

static
void color_box_color_change(GtkWidget *widget, gpointer data) {
  gtk_color_selection_get_color(
    GTK_COLOR_SELECTION(widget), PlayerDialogs[(int)data].color2);
}

static
void destroy_window(GtkWidget *widget, GtkWidget **window)
{
  *window = NULL;
}

GtkWidget *make_color_dialog(int player) {
  GtkWidget *dialog;
  char title[80];

  if (PlayerDialogs[player].colordialog != NULL &&
      PlayerDialogs[player].colordialog->window != NULL) {
    gdk_window_raise(PlayerDialogs[player].colordialog->window);
    return;
  }

  sprintf(title, "Choose color for player %d", player+1);

  dialog = gtk_color_selection_dialog_new(title);
  gtk_window_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);

  // Set color twice so color_selection's old color is correct...
  gtk_color_selection_set_color(
    GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(dialog)->colorsel),
    PlayerDialogs[player].color);
  gtk_color_selection_set_color(
    GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(dialog)->colorsel),
    PlayerDialogs[player].color);

  gtk_signal_connect (GTK_OBJECT (dialog), "destroy",
                      GTK_SIGNAL_FUNC(destroy_window),
                      &dialog);
  gtk_signal_connect (GTK_OBJECT (dialog), "delete_event",
                      GTK_SIGNAL_FUNC(destroy_window),
                      &dialog);
  gtk_signal_connect (
    GTK_OBJECT (GTK_COLOR_SELECTION_DIALOG (dialog)->colorsel),
    "color_changed",
    GTK_SIGNAL_FUNC(color_box_color_change),
    (gpointer)player);
  gtk_signal_connect (
    GTK_OBJECT (GTK_COLOR_SELECTION_DIALOG (dialog)->ok_button),
    "clicked",
    GTK_SIGNAL_FUNC(color_box_ok),
    (gpointer)player);
  gtk_signal_connect (
    GTK_OBJECT (GTK_COLOR_SELECTION_DIALOG (dialog)->cancel_button),
    "clicked",
    GTK_SIGNAL_FUNC(color_box_cancel),
    (gpointer)player);

  gtk_widget_show(dialog);
  return dialog;
}

static
void open_color_dialog(GtkWidget *widget, GdkEvent *e, gpointer data) {
  PlayerDialogs[(int)data].colordialog = make_color_dialog((int)data);
}

GtkWidget *color_box(int playernum) {
  GtkWidget *frame;
  GtkWidget *drawable;

  drawable = gtk_drawing_area_new();
  gtk_drawing_area_size(GTK_DRAWING_AREA(drawable),
                        color_box_width, color_box_height);
  gtk_widget_set_events(drawable, GDK_BUTTON_PRESS_MASK | GDK_EXPOSURE_MASK);
  gtk_widget_show(drawable);

  gtk_signal_connect(GTK_OBJECT(drawable), "button_press_event",
                     GTK_SIGNAL_FUNC(open_color_dialog),
                     (gpointer)playernum);
  gtk_signal_connect(GTK_OBJECT(drawable), "expose_event",
                     GTK_SIGNAL_FUNC(expose_event),
                     (gpointer)playernum);

  frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
  gtk_widget_show(frame);

  gtk_container_add(GTK_CONTAINER(frame), drawable);

  PlayerDialogs[playernum].colorbox = drawable;
  PlayerDialogs[playernum].colorframe = frame;
  return frame;
}

static
void select_off(GtkWidget *widget, gpointer data) {
  PlayerDialogs[(int)data].type = OFF;
  gtk_widget_hide(PlayerDialogs[(int)data].name);
  gtk_widget_hide(PlayerDialogs[(int)data].colorframe);
}

static
void select_human(GtkWidget *widget, gpointer data) {
  PlayerDialogs[(int)data].type = HUMAN;
  gtk_widget_show(PlayerDialogs[(int)data].name);
  gtk_widget_show(PlayerDialogs[(int)data].colorframe);
}

static
void select_cpu(GtkWidget *widget, gpointer data) {
  PlayerDialogs[(int)data].type = CPU;
  gtk_widget_show(PlayerDialogs[(int)data].name);
  gtk_widget_show(PlayerDialogs[(int)data].colorframe);
}

static
GtkWidget *select_box(int playernum) {
  GtkWidget *option;
  GtkWidget *menu;
  GtkWidget *opt1;
  GtkWidget *opt2;
  GtkWidget *opt3;

  opt1 = gtk_menu_item_new_with_label("Off");
  opt2 = gtk_menu_item_new_with_label("Human");
  opt3 = gtk_menu_item_new_with_label("CPU");
  gtk_widget_show(opt1);
  gtk_widget_show(opt2);
  gtk_widget_show(opt3);
  gtk_signal_connect(GTK_OBJECT(opt1), "activate",
                     GTK_SIGNAL_FUNC(select_off), (gpointer)playernum);
  gtk_signal_connect(GTK_OBJECT(opt2), "activate",
                     GTK_SIGNAL_FUNC(select_human), (gpointer)playernum);
  gtk_signal_connect(GTK_OBJECT(opt3), "activate",
                     GTK_SIGNAL_FUNC(select_cpu), (gpointer)playernum);

  menu = gtk_menu_new();
  gtk_menu_append(GTK_MENU(menu), opt1);
  gtk_menu_append(GTK_MENU(menu), opt2);
  gtk_menu_append(GTK_MENU(menu), opt3);

  option = gtk_option_menu_new();
  gtk_option_menu_set_menu(GTK_OPTION_MENU(option), menu);
  gtk_option_menu_set_history(GTK_OPTION_MENU(option), 3);
  gtk_widget_show(option);

  return option;
}

static
GtkWidget *player_field(int playernum) {
  GtkWidget *box;
  GtkWidget *select;
  GtkWidget *color;
  GtkWidget *name;
  char nametext[10];

  box = gtk_hbox_new(FALSE, 5);
  gtk_widget_show(box);

  color = color_box(playernum);
  select = select_box(playernum);

  name = gtk_entry_new();
  gtk_widget_set_usize(name, 100, 0);
  gtk_widget_show(name);
  sprintf(nametext, "Player%d", playernum+1);
  gtk_entry_set_text(GTK_ENTRY(name), nametext);

  PlayerDialogs[playernum].name = name;
  PlayerDialogs[playernum].select = select;

  gtk_widget_hide(color);
  gtk_widget_hide(name);

  gtk_box_pack_start(GTK_BOX(box), color, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(box), name, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(box), select, TRUE, TRUE, 0);

  return box;
}

static
GtkWidget *players_tab() {
  GtkWidget *box;
  GtkWidget *frame;

  frame = gtk_frame_new("Player Information");
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);
  gtk_container_border_width(GTK_CONTAINER(frame), 10);
  gtk_widget_show(frame);

  box = gtk_vbox_new(FALSE, 5);
  gtk_widget_show(box);
  gtk_container_border_width(GTK_CONTAINER(box), 10);
  gtk_container_add(GTK_CONTAINER(frame), box);

  gtk_box_pack_start(GTK_BOX(box), player_field(0), FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(box), player_field(1), FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(box), player_field(2), FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(box), player_field(3), FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(box), player_field(4), FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(box), player_field(5), FALSE, FALSE, 0);

  return frame;
}

static
GtkWidget *variations_tab() {
  GtkWidget *box;
  GtkWidget *frame;
  GtkWidget *option;

  frame = gtk_frame_new("Game Variations");
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);
  gtk_container_border_width(GTK_CONTAINER(frame), 10);
  gtk_widget_show(frame);

  box = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(box);
  gtk_container_border_width(GTK_CONTAINER(box), 10);
  gtk_container_add(GTK_CONTAINER(frame), box);

  option = gtk_check_button_new_with_label("Retreating");
  gtk_widget_show(option);
  gtk_box_pack_start(GTK_BOX(box), option, FALSE, FALSE, 0);

  option = gtk_check_button_new_with_label("Capitals");
  gtk_widget_show(option);
  gtk_box_pack_start(GTK_BOX(box), option, FALSE, FALSE, 0);
  
  gtk_widget_show(frame);
  return frame;
}

GtkWidget *new_game_dialog() {
  GtkWidget *dialog;
  GtkWidget *ok;
  GtkWidget *cancel;
  GtkWidget *box;
  GtkWidget *variations;
  GtkWidget *notebook;
  GtkWidget *label;
  GtkWidget *padding;
  GList *dialog_contents; // for removing the separator!

  box = gtk_hbox_new(FALSE, 8);
  gtk_widget_show(box);

  ok = gtk_button_new_with_label("OK");
  gtk_widget_set_usize(ok, 64, 0);
  gtk_widget_show(ok);
  gtk_signal_connect(GTK_OBJECT(ok), "clicked",
                     GTK_SIGNAL_FUNC(new_game_ok_callback), NULL);

  cancel = gtk_button_new_with_label("Cancel");
  gtk_widget_set_usize(cancel, 64, 0);
  gtk_widget_show(cancel);
  gtk_signal_connect(GTK_OBJECT(cancel), "clicked",
                     GTK_SIGNAL_FUNC(new_game_cancel_callback), NULL);

  dialog = gtk_dialog_new_no_separator();
  gtk_box_pack_end(GTK_BOX(GTK_DIALOG(dialog)->action_area), box,
                     TRUE, TRUE, 0);
  gtk_box_pack_end(GTK_BOX(box), cancel,
                     FALSE, FALSE, 0);
  gtk_box_pack_end(GTK_BOX(box), ok,
                     FALSE, FALSE, 0);
  notebook = gtk_notebook_new();
  gtk_container_border_width(GTK_CONTAINER(notebook), 10);
  gtk_widget_show(notebook);
  gtk_widget_set_usize(notebook, 320, 0);

  label = gtk_label_new("Players");
  gtk_widget_show(label);
  gtk_widget_set_usize(label, 64, 0);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), players_tab(), label);

  label = gtk_label_new("Variations");
  gtk_widget_set_usize(label, 64, 0);
  gtk_widget_show(label);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), variations_tab(), label);

  gtk_box_pack_end(GTK_BOX(GTK_DIALOG(dialog)->vbox), notebook,
                     FALSE, FALSE, 0);

  gtk_window_set_title(GTK_WINDOW(dialog), "New Game Dialog");

  gtk_widget_grab_focus(ok);
  return dialog;
}

void new_game_cmd_callback(GtkWidget *widget, gpointer data) {
  if (NewGameDialog == NULL || NewGameDialog->window == NULL) {
    NewGameDialog = new_game_dialog();
    gtk_window_position (GTK_WINDOW (NewGameDialog), GTK_WIN_POS_MOUSE);
    gtk_widget_show(NewGameDialog);
  } else {
    gdk_window_raise(NewGameDialog->window);
  }
}

