/*
 * Copyright 1996, 1997 by Daniel B. Suthers, Pleasanton Ca. 94588 USA
 * E-MAIL dbs@tanj.com
 *
 * You may freely copy, use, and distribute this software
 * subject to the following restrictions:
 *
 *  1)	You may not charge money for it.
 *  2)	You may not remove or alter this copyright notice.
 *  3)	You may not claim you wrote it.
 *  4)	If you make improvements (or other changes), you are requested
 *	to send them to me, so there's a focal point for distributing
 *	improved versions.
 */

/* This module is to be called by the first process to run under HEYU.
 * It:
 * 1) Locks the TTY port by putting it's pid in LOCKDIR/LCK..ttyX
 * 2) Validates any existing HEYU locks in LOCKDIR/LCK..heyu.mon
 *    and sets a lock in LOCKDIR/LCK..heyu.mon  with it's PID if none exists.
 * 3) Starts reading from the TTY port associated with the CM11A
 *    and writing the raw bytes to SPOOLDIR/heyu.out
 *    The heyu.out file will be deleted if it exists and created new.
 * 4) Upon SIGHUP signal will truncate the .in file.... someday, but not yet
 * 5) Upon SIGTERM or SIGINT will...
 *    Close the tty port
 *    unlink the TTY lock 
 *    unlink the heyu.mon lock
 *    unlink the heyu.out file
 *    unlink the x10_tty file
 */

#include <stdio.h>
#include <ctype.h>
#include "x10.h"
#include <signal.h>
#include <string.h>
#include <syslog.h>
#ifdef LINUX
#include <sys/resource.h>
#endif
#include <limits.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <time.h>


extern int tty;
extern int verbose;
extern unsigned char cm11map[];
extern int i_am_relay;

extern char x10_tty[50];

char spoolfile[PATH_MAX];

int interrupted = 0;
void flag_intr();


/* tty should be the device that we are going to use.  It should be a fully
 * qualified path name (/dev/tty2), but may be just the device (tty2)
 */
start_relay(tty_name)
char *tty_name;
{
   unsigned char ibuff[80];
   long child;
   int cnt;
   long pid;
   int outfd;
   int powerfail;
   int count_5a;
   int first_byte;
   char argv[2][5];
   extern char *argptr;
   off_t file_ofset;
   time_t pfail_time;
   char RCSID[]= "@(#) $Id: relay.c,v 1.10 1998/05/08 01:50:08 dbs Exp dbs $\n";
   
   first_byte = 1;
   
   /* set up the spool file name */
   spoolfile[0] = '\0';
   strcat(spoolfile, SPOOLDIR);
   if( spoolfile[strlen(spoolfile) - 1] != '/' )
       strcat(spoolfile, "/");
   strcat(spoolfile, "heyu.out");

   /* is a relay in place ? */
    if( lockpid("heyu.mon") > 1)
    {
        if( verbose )
	    printf("The was already a monitor running (pid = %d)\n",
		lockpid("heyu.mon") );
        return(-1);		/* there was a valid monitor running */
    }
    else
    {  
		/* we will spawn a monitor process */
	child = fork();
	if( child > 0 )
	{
	    sleep(3);		/* give child time to set up */
	    return(1);		/* this is parent process */
	}
	if( child < 0 )	     /* This is an error */
	{
	    perror("I could not spawn process");
	    syslog(LOG_DAEMON | LOG_ERR, "I could not spawn process.\n");
	    quit();
	}
    }
    /* from this point out, it should be the child. */
    close(0);
    close(1);
    close(2);
    strcpy(argptr, "heyu_relay");
    pid = setsid();   /* break control terminal affiliation */
    openlog( "heyu_relay", 0, LOG_DAEMON);
    if (pid == -1)
    {
	syslog(LOG_ERR, "relay setsid failed--\n");
	quit(1);
    }
    else
	syslog(LOG_ERR, "relay setting up-\n");


    /* Ok. We're alone now. */
    
    if( ttylock("heyu.mon") < 1)
    {
	syslog(LOG_ERR, "Could not set up the heyu.mon lock-");
         exit(0);	/* a competeing process must have started up
	                 * in the lastfew milliseconds
			 */
    }
    setup_tty(1);	/* open the real tty */
    i_am_relay = 1;	/* set flag so calling function will clean up. */

    unlink(spoolfile);
    outfd=open(spoolfile, O_WRONLY|O_CREAT|O_EXCL|O_APPEND, 0777);
    if( outfd < 0 )
    {
	sprintf(ibuff, "Trouble creating spoolfile (%s)", spoolfile);
        syslog(LOG_ERR, ibuff);
	quit();
    }
    chmod(spoolfile, 0777);

    (void) signal(SIGINT, flag_intr);
    (void) signal(SIGTERM, flag_intr);
    (void) signal(SIGHUP, flag_intr);


    /* certain codes come out 1 second apart.  These are the 5a and a5
     * codes.  They indicate the CM11A wants a response, ie a polling 
     * sequence indicator.
     * In order to handle powerfails, we have to timestamp each a5 character
     * as it comes in.  Three 0xa5 characters in a row, 1 second apart
     * would indicate a power fail condition that needs a reply.
     * If the very first byte received is a5 or 5a, it's a condition
     * that needs a reply.
     * As an alternative, a leading byte that's more than 1 second from the
     * previous one __may__ be a polling sequence.
     */
    powerfail = 0;	/* increment this each time a 0xa5 is seen */
    strcpy(argv[0], " ");/* set a vector that can be used by c_setclock() */
    strcpy(argv[1], " ");/* set a vector that can be used by c_setclock() */
    count_5a = 0;
    while(1)
    {
        read(tty, ibuff, 1);
	if( interrupted == 1 )
	    break;
	if( (ibuff[0] == 0x5a && count_5a == 0) || (ibuff[0] != 0x5a) )
	    write(outfd,ibuff, 1);
	if( ibuff[0] == 0x5a )	/* CM11A has a character to be read */
	{
	    if( ++count_5a > 1) 
	    {
		ibuff[0] = 0xc3;	/* tell the CM11A to send it */
		write(tty,ibuff, 1);
	    }
	}
	else
	{
	    count_5a = 0;
	}
	if(ibuff[0] == 0xa5)	/* CM11A reports a power fail */
	{
	    if( powerfail == 0 )	/* set timestamp for the first poll */
	    {
	        pfail_time = time(NULL);
	    }
	    if( (first_byte == 1) || (powerfail++ == 2) )
	    {
		if  ( (powerfail == 3) && 
		    ((pfail_time != 0) && ((time(NULL) - pfail_time) > 2))
		    )
		{
			 /* 3 bytes of 'a5' in a row over a period of 3 seconds
			  means a power failure*/
		    powerfail = 0;
		    c_setclock(1, &argv[0]);
		    pfail_time = 0;
		}
	    }
	}
	else
	{
	    powerfail = 0;
	    pfail_time = 0;
	}
	first_byte = 0;
	ibuff[0] = '\0';
    }
    munlock("heyu.mon");
    munlock(x10_tty);
    unlink(spoolfile);
    if( interrupted == 1 )
        exit(0);
    
    return(0);

} 

void flag_intr()
{
    interrupted = 1;
    (void) signal(SIGTERM, flag_intr);
    syslog(LOG_ERR, "interrupt received\n");
}
