-- This file is  free  software, which  comes  along  with  SmallEiffel. This
-- software  is  distributed  in the hope that it will be useful, but WITHOUT 
-- ANY  WARRANTY;  without  even  the  implied warranty of MERCHANTABILITY or
-- FITNESS  FOR A PARTICULAR PURPOSE. You can modify it as you want, provided
-- this header is kept unaltered, and a notification of the changes is added.
-- You  are  allowed  to  redistribute  it and sell it, alone or as a part of 
-- another product.
--          Copyright (C) 1994-98 LORIA - UHP - CRIN - INRIA - FRANCE
--            Dominique COLNET and Suzanne COLLIN - colnet@loria.fr 
--                       http://www.loria.fr/SmallEiffel
--
class ARRAY2[E]
   --
   -- Implementation of COLLECTION[E] based on ARRAY[E].
   --

inherit COLLECTION2[E] redefine copy end;
   
creation make, array2

feature {ARRAY2}
   
   storage: ARRAY[ARRAY[E]];
   
feature 
   
   make(l1, u1, l2, u2: INTEGER) is
	 -- Reset all bounds `lower1' / `upper1' / `lower2' and
	 -- `upper2' using arguments as new values.
	 -- All elements are set to the default value of type E.
      require
	 l1 <= u1;
	 l2 <= u2;
      local
	 i: INTEGER;
	 sa: ARRAY[like item];
      do
	 from
	    !!storage.make(l1,u1);
	    i := storage.lower;
	 until
	    i > storage.upper
	 loop
	    !!sa.make(l2,u2);
	    storage.put(sa,i);
	    i := i + 1;
	 end;
      ensure
	 lower1 = l1;
	 upper1 = u1;
	 lower2 = l2;
	 upper2 = u2
      end;
   
   array2(s: ARRAY[ARRAY[E]]) is
	 -- The array `s' is used as internal memory of Current.
	 -- Assume all sub-arrays have the same indexing.
	 -- Also assume `s' is not shared with another object.
      require
	 s /= Void;
	 s.count > 0;
      local
	 i: INTEGER;
      do
	 storage := s;
	 debug
	    from  
	       i := storage.lower + 1;
	    until
	       i > storage.upper
	    loop
	       if lower2 /= storage.item(i).lower then
		  std_error.put_string("ARRAY2.array2: Bad `lower2' at line ")
		  std_error.put_integer(i);
		  std_error.put_new_line;
		  crash;
	       elseif upper2 /= storage.item(i).upper then
		  std_error.put_string("ARRAY2.array2: Bad `upper2' at line ")
		  std_error.put_integer(i);
		  std_error.put_new_line;
		  crash;
	       end;
	       i := i + 1;
	    end;
	 end;
      ensure
	 storage = s;
      end;
   
feature 
   
   lower1: INTEGER is
      do
	 Result := storage.lower;
      end;
   
   upper1: INTEGER is
      do
	 Result := storage.upper;
      end;
   
   count1: INTEGER is
      do
	 Result := storage.count;
      end;
   
   lower2: INTEGER is
      do
	 Result := storage.first.lower;
      end;
   
   upper2: INTEGER is
      do
	 Result := storage.first.upper;
      end;
   
   count2: INTEGER is
      do
	 Result := storage.first.count;
      end;
   
   item(i, j: INTEGER): E is
      do
	 Result := storage.item(i).item(j);
      end;
   
feature 
   
   put(x: E; i, j: INTEGER) is
      do
	 storage.item(i).put(x,j);	 
      end;
   
   copy(other: like Current) is
      local
	 i: INTEGER;
      do
	 make(other.lower1,other.upper1,other.lower2,other.upper2);
	 from  
	    i := other.lower1;
	    i := storage.lower;
	 until
	    i > storage.upper
	 loop
	    storage.item(i).copy(other.storage.item(i));
	    i := i + 1;
	 end;
      end;
   
feature 

   set_all_with(x: E) is
      local 
         i: INTEGER;
      do
	 from 
	    i := storage.lower
	 until
	    i > storage.upper
         loop
            storage.item(i).set_all_with(x)
	    i := i + 1
         end;
      end;

   nb_occurrences(elt: E): INTEGER is
	 -- Number of occurrences using `equal'.
	 -- See also `fast_nb_occurrences' to chose
	 -- the apropriate one.
      local
	 i: INTEGER;
      do
	 from  
	    i := lower1;
	 until
	    i > upper1
	 loop
	    Result := Result + storage.item(i).nb_occurrences(elt);
	    i := i + 1;
	 end;
      ensure
	 Result >= 0;
      end;
      
   fast_nb_occurrences(elt: E): INTEGER is
	 -- Number of occurrences using `='.
      local
	 i: INTEGER;
      do
	 from  
	    i := lower1;
	 until
	    i > upper1
	 loop
	    Result := Result + storage.item(i).fast_nb_occurrences(elt);
	    i := i + 1;
	 end;
      ensure
	 Result >= 0;
      end;

feature -- Resizing :

   resize(l1, u1, l2, u2: INTEGER) is
      require
	 u1 >= l1 - 1;
	 u2 >= l2 - 1;
      local
	 i: INTEGER;
	 oldl1, oldu1: INTEGER;
	 tmp: ARRAY[like item];
      do
	 oldl1 := lower1;
	 oldu1 := upper1;
	 storage.resize (l1,u1); 
	 from
	    i := l1;
	 until
	    i > u1
	 loop
	    if i < oldl1 or i > oldu1 then
	       !!tmp.make (lower2, upper2);
	       storage.put (tmp, i)
	    else
	       storage.item (i).resize (l2,u2);
	    end;
	    i := i + 1;
	 end;
      ensure
	 lower1 = l1;
	 upper1 = u1;
	 lower2 = l2;
	 upper2 = u2;
      end;

   resize1(l, u: INTEGER) is
      require
	 u >= l - 1;
      local
	 i, oldl, oldc: INTEGER;
	 tmp: ARRAY[like item];
      do
	 oldl := lower1;
	 oldc := upper1;
	 storage.resize (l,u);
	 if l < oldl then
	    from
	       i := l;
	    until
	       i = lower1
	    loop
	       !!tmp.make (lower2, upper2);
	       storage.put (tmp, i)
	       i := i + 1;
	    end;
	 end;
	 if u > oldc then
	    from
	       i := oldc + 1;
	    until
	       i > u
	    loop
	       !!tmp.make (lower2, upper2);
	       storage.put (tmp, i)
	       i := i + 1;
	    end;
	 end;
      end;
   
   resize2 (l, u : INTEGER) is
      require
	 u > l - 1;
      local
	 i : INTEGER;
      do
	 from
	    i := lower1;
	 until
	    i > upper1
	 loop
	    storage.item (i).resize (l,u);
	    i := i + 1;
	 end;
      ensure
	 lower2 = l;
	 upper2 = u;
      end;

feature -- Other features :
      
   replace_all(x, r: like item) is
      local
	 i: INTEGER;
      do
	 from 
	    i := lower1;
	 until
	    i > upper1
	 loop
	    storage.item(i).replace_all(x,r);
	    i := i + 1;
	 end;
      end;
   
   fast_replace_all(x, r: like item) is
      local
	 i: INTEGER;
      do
	 from 
	    i := lower1;
	 until
	    i > upper1
	 loop
	    storage.item(i).fast_replace_all(x,r);
	    i := i + 1;
	 end;
      end;
	 
   transpose is
      local
	 i,j : INTEGER;
	 oldc1, oldc2 : INTEGER;
      do
	 oldc1 := count1;
	 oldc2 := count2;
	 if count1 > count2 then
	    resize2 (lower2, lower2 + count1 -1);
	 elseif count2 > count1 then
	    resize1 (lower1, lower1 + count2 - 1);
	 end;
	 from  
	    i := lower1;
	 until 
	    i > upper1 - 1
	 loop
	    from  
	       j := i + 1;
	    until 
	       j > upper2
	    loop
	       swap(i,j,j,i);
	       j := j + 1;
	    end;
	    i := i + 1;
	 end;
	 resize(lower1,lower1 + oldc2 - 1,lower2,lower2 + oldc1 - 1);
      ensure
	 count = oldc1 * oldc2;	 
      end;

invariant
   
   storage.count >= 1;
   
end -- ARRAY2[E]

