/* Quinn Diff, Compares two Packages files looking for differences between Archs */
/* Copyright (C) 1997, 1998, James Troup <james@nocrew.org> */

/* This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "common.h"
#include "error.h"
#include "xmalloc.h"
#include "list.h"
#include "main.h"
#include "utils.h"

/*****************************************************************************
 * function  : parse_package
 * arguments : pointer to the buffer, position in buffer, pointer to
 *             `want list', length of packages file, package name
 *             (used in error messages)
 * results   : 
 * purpose   : non-destructively searches one packages' entry for
 *             interesting (as defined by the want list) information.
 * caveats   : 
 ***************************************************************************** */

void parse_package (const char *buffer, int buffer_position, const int buffer_length, 
		    Qlist *want_list, const char *package_name)
{

  char line[PACKAGE_NAME_LENGTH];
  Qlist *want_list_pointer; 
  int i;

  /* Initialize the want list */
  want_list_pointer = want_list;

  while (want_list_pointer)
    {
      ((want_node *)(want_list_pointer->data))->found = FALSE;
      ((want_node *)(want_list_pointer->data))->destination_string[0] = '\0';
      if (want_list_pointer->next)
	want_list_pointer = want_list_pointer->next;
      else
	break;
    }

  debug (debug_utils, "parse_package: parsing %s", package_name);

  while (buffer[buffer_position] != '\n'  && buffer_position < buffer_length)
    {

      want_list_pointer = want_list;

      /* We're not interested in descriptions, so we skip any lines with "^ " */
      if (buffer[buffer_position] == ' ')  
	{
	  skip_line(buffer, &buffer_position, buffer_length);
	  continue;  
	}

      i = 0;

      while (buffer[buffer_position] != '\n' && i < PACKAGE_NAME_LENGTH)
	line[i++] = buffer[buffer_position++];

      line[i] = '\0'; /* Terminate the line */
      buffer_position++; /* Skip the trailing \n */

      /* Check the want list */
      /* This code is horrible, sorry */
      while (want_list_pointer)
	{
	  if (!((want_node *)(want_list_pointer->data))->found)
	    {
	      if (!strncmp (line, 
			    ((want_node *)(want_list_pointer->data))->search_string,
			    strlen (((want_node *)(want_list_pointer->data))->search_string)))
		{
		  strcpy (((want_node *)(want_list_pointer->data))->destination_string, 
			  line + (strlen (((want_node *)(want_list_pointer->data))->search_string)));
		  ((want_node *)(want_list_pointer->data))->found = TRUE;
		  break;
		}
	    }
	  
	  if (want_list_pointer->next)
	    want_list_pointer = want_list_pointer->next;
	  else 
	    break;
	}
    }
 
}

/*****************************************************************************
 * function  : get_blah
 * arguments : pointer to the buffer, position in buffer, pointer to
 *             string to return result in, string to search for,
 *             length of packages file, package name (used in error
 *             messages)
 * results   : 
 * purpose   : non-destructively searches one packages' entry for a line
 *              beginning with the string as passed to "input"
 * caveats   : 
 ***************************************************************************** */

void get_blah (const char *buffer, int buffer_position, char *output, const char *input, 
	       const int length, const char *package_name)
{
  output[0] = '\0'; 

  debug (debug_utils, "get_blah: searching for %s in %s", input, package_name);

  while (buffer_position < length)
    {
      /* If we encounter a blank line, we've overun the package description; give up */
      if (buffer[buffer_position] == '\n')
	{
	  output[0] = '\0';
	  break;
	}

      if (buffer[buffer_position] == input[0])
	{
	  read_line (buffer, &buffer_position, output);

	  /* If we have a match stop searching, if not blank the output string */
	  if (!(strncmp (input, output, strlen(input)))) 
	    {
	      strcpy (output, output + (strlen (input)));
	      break;
	    }
	  else
	    output[0] = '\0';
	}
      else
	skip_line(buffer, &buffer_position, length);
    }
}

void skip_line (const char *buffer, int *counter, const long buffer_length)
{
  
  while (*counter < buffer_length && buffer[*counter] != '\n')
    (*counter)++;

  if (*counter < buffer_length)
    (*counter)++;          /* Skip the newline */

}

/*****************************************************************************
 * function  : skip_to_next_package
 * arguments : pointer to the buffer, position in buffer
 * results   : 
 * purpose   : Skips to the next package
 * caveats   : 
 ***************************************************************************** */

void skip_to_next_package (const char *buffer, int *buffer_position, const long buffer_length)
{
 
  while (*buffer_position < buffer_length && buffer[*buffer_position] !='\n')
    skip_line (buffer, buffer_position, buffer_length);

  if (*buffer_position < buffer_length)
    (*buffer_position)++;

}


void read_line (const char *buffer, int *counter, char *string)
{

  int i = 0;

  while (buffer[*counter] != '\n')
    string[i++] = buffer[(*counter)++];
  
  (*counter)++;           /* Skip the newline */
  string[i] = '\0';       /* Terminate the string */

}

void determine_needsx (const char *suggests, const char *recommends, const char *depends, 
		       const char *predepends, int *needs_x)
{
  
  if (strstr (suggests, "xlib6") != NULL ||
      strstr (recommends, "xlib6") != NULL ||
      strstr (depends, "xlib6") != NULL || 
      strstr (predepends, "xlib6") != NULL)
    *needs_x = 1;
  else
    *needs_x = 0;

}

/*****************************************************************************
 * function  : priority_to_int
 * arguments : priority (as a string)
 * results   : an integer value for the priority 
 * purpose   : returns an int for a priority, used to make comparisons
 *             of priorties possible
 * caveats   : 
 ***************************************************************************** */

int
priority_to_int (const char *priority)
{

  if (!strcmp (priority, "-"))
    return PRIORITY_UNKNOWN;
  else if (!strcmp (priority, "extra"))
    return PRIORITY_EXTRA;
  else if (!strcmp (priority, "optional"))
    return PRIORITY_OPTIONAL;
  else if (!strcmp (priority, "standard"))
    return PRIORITY_STANDARD;
  else if (!strcmp (priority, "important"))
    return PRIORITY_IMPORTANT;
  else if (!strcmp (priority, "required"))
    return PRIORITY_REQUIRED;
  else
    {
      debug (debug_warn, "priority_to_int: unknown priority: %s, returning PRIORITY_UNKNOWN", priority);
      return PRIORITY_UNKNOWN;
    }

}

void read_file (const char *filename, FILE *fp, char **buffer_p, long *file_length)
{

  int n_read;                   

  fp = fopen (filename, "r");
  if (fp == NULL)
    fubar (SYSERR, "read_file: couldn't open file \"%s\"", filename);
  if ((fseek (fp, 0, SEEK_END)) != 0)
    fubar (SYSERR, "read_file: couldn't seek to the end of file \"%s\"", filename);
  *file_length = ftell (fp);
  rewind (fp);

  if (*file_length == 0)
    fubar (NONSYS, "read_file: \"%s\" is zero-length.", filename);

  /* We don't do any memory management, we just ask for a chunk of
   * memory the size of the file */

  *buffer_p = (char *) xmalloc (*file_length);

  /* RUARI QUINN this is ignoring the possibility that ftell could be long */

  n_read = fread (*buffer_p, sizeof (char), *file_length, fp);
  if (n_read == 0)
    fubar (SYSERR, "read_file: couldn't read file \"%s\" into buffer", filename);
  else if (n_read < *file_length)
    fubar (NONSYS, "read_file: short read (found %d, expected %ld) in file \"%s\"",
	   n_read, *file_length, filename);

  if (fclose (fp) == EOF)
    fubar (SYSERR, "read_file: couldn't close file \"%s\"", filename);

}
