/*
 * ULIO.C - handle curve I/O for ULTRA
 *
 * Source Version: 4.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "ultra.h"

#define write_int(n, fp)   io_write((char *) &n, sizeof(int), 1, fp)
#define read_int(n, fp)    io_read((char *) &n, sizeof(int), 1, fp)

int
 UL_n_curves_read = 0,
 _UL_next_available_number = 1,
 _UL_table_n = 0;

long
 _UL_table_ln = 0;

char *UL_table_name = NULL;

FILE
 *UL_out_ascii,
 *UL_out_bin;

PDBfile
 *UL_out_pdb;
    
static PDBfile
 *UL_cache_file = NULL;

static PM_matrix
 *UL_current_table = NULL;

static char
 Ultra_Hdr[] = "ULTRA II - BINARY FILE";

static pcons
 *file_list = NULL;

static unsigned
 dataptr;                               /* counts the number of data points */

static REAL
 SC_DECLARE(*UL_extract_vector, 
            (PM_matrix *a, int o, int s, int n));

static FILE
 SC_DECLARE(*UL_open_for_reading, (char *str, char *mode));

static void
 SC_DECLARE(UL_wrt_pdb_curve, 
            (PDBfile *fp, curve *crv, int icurve)),
 SC_DECLARE(UL_read_pdb, (PDBfile *fp, char *fname)),
 SC_DECLARE(UL_read_bin, (FILE *fp, char *fname)),
 SC_DECLARE(UL_read_ascii, (FILE *fp, char *fname)),
 SC_DECLARE(UL_wrt_pdb, (PDBfile *fp, object *argl)),
 SC_DECLARE(UL_wrt_bin, (FILE *fp, object *argl)),
 SC_DECLARE(UL_wrt_ascii, (FILE *fp, object *argl)),
 SC_DECLARE(UL_cache_curve, (curve *crv, int type));

static int
 SC_DECLARE(UL_next_number, (int flag)),
 SC_DECLARE(UL_termdata, (int *aryptr, REAL *xbuff, REAL *ybuff)),
 SC_DECLARE(UL_read_pdb_curve,
         (PDBfile *fp, char *fname, char *cname,
          curve *crv, int flag)),
 SC_DECLARE(UL_ultra_binary_filep, (FILE *fp)),
 SC_DECLARE(UL_ultra_ascii_filep, (FILE *fp)),
 SC_DECLARE(UL_file_open, (FILE *fp)),
 SC_DECLARE(UL_remove_file, (FILE *fp)),
 SC_DECLARE(UL_find_ascii_table,
            (FILE *fp, int n, long nl, int *pfn, int *pnr, int *pnc,
             long *paddrt, int nlab, long *paddrl));

/*--------------------------------------------------------------------------*/

/*                            LOW-LEVEL ROUTINES                            */

/*--------------------------------------------------------------------------*/

/* UL_NEXT_NUMBER - return the index for the next available curve menu slot */

static int UL_next_number(flag)
   int flag;
   {int i;

    for (i = _UL_next_available_number; i < UL_N_Curves; i++)
        if (UL_number[i] == -1)
           {if (flag)
               _UL_next_available_number = i + 1;
            return(i);};

    UL_enlarge_dataset();
    return(UL_next_number(flag));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_TERMDATA - terminate one set of data and prepare for the next
 *             - return TRUE iff successful
 */

static int UL_termdata(aryptr, xbuff, ybuff)
   int *aryptr;
   REAL *xbuff, *ybuff;
   {int i, j;
    REAL *xp, *yp;
    REAL xmin, xmax, ymin, ymax;

    if (dataptr <= 0)
       {dataptr = 0;
        return(FALSE);};

    if ((xbuff == NULL) || (ybuff == NULL))
       {dataptr = 0;
        return(FALSE);};

    j = *aryptr;

    REMAKE_N(xbuff, REAL, dataptr);
    xp = UL_dataset[j].xp = xbuff;

    REMAKE_N(ybuff, REAL, dataptr);
    yp = UL_dataset[j].yp = ybuff;

/* bail out if not enough memory */
    if ((xp == NULL) || (yp == NULL))
       SS_error("INSUFFICIENT MEMORY - UL_TERMDATA", SS_null);

    if ((xbuff[dataptr-1] == -999) && (ybuff[dataptr-1] == -999))
       --dataptr;

    PM_maxmin(UL_dataset[j].xp, &xmin, &xmax, dataptr);
    PM_maxmin(UL_dataset[j].yp, &ymin, &ymax, dataptr);

    UL_dataset[j].id    = ' ';
    UL_dataset[j].n     = dataptr;
    UL_dataset[j].xmin  = xmin;
    UL_dataset[j].xmax  = xmax;
    UL_dataset[j].ymin  = ymin;
    UL_dataset[j].ymax  = ymax;

    SC_CHANGE_VALUE_ALIST(UL_dataset[j].info,
			  int, SC_INTEGER_P_S,
			  "LINE-COLOR", 0);

/* put this curve's data in a cache somewhere (let us defer the question
 * about memory management and also to handle all curve sources the same
 * way
 */
    UL_cache_curve(&UL_dataset[j], ASCII);

    dataptr      = 0;
    i            = UL_next_number(TRUE);
    UL_number[i] = j;
    *aryptr      = UL_next_space();

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_READ_PDB_CURVE - read a specifed curve (NAME) from the PDBfile, FP,
 *                   - into the curve pointed to by CRV
 *                   - return TRUE if successful and FALSE otherwise
 */

static int UL_read_pdb_curve(fp, fname, cname, crv, flag)
   PDBfile *fp;
   char *fname, *cname;
   curve *crv;
   int flag;
   {int n;
    REAL xmin, xmax, ymin, ymax, *xp, *yp;
    char label[MAXLINE];

    if (!PD_read_pdb_curve(fp, cname, &xp, &yp, &n, label,
                           &xmin, &xmax, &ymin, &ymax, flag))
       return(FALSE);

    crv->xp = xp;
    crv->yp = yp;

/* set up the rest of the curve data */
    if (fname != NULL)
       {if ((xmin == HUGE) && (xmax == -HUGE) && (xp != NULL))
           PM_maxmin(xp, &xmin, &xmax, n);

        if ((ymin == HUGE) && (ymax == -HUGE) && (yp != NULL))
           PM_maxmin(yp, &ymin, &ymax, n);

        crv->id   = ' ';
        crv->n    = n;
        crv->xmin = xmin;
        crv->xmax = xmax;
        crv->ymin = ymin;
        crv->ymax = ymax;
        crv->text = SC_strsavef(label, "char*:UL_READ_PDB_CURVE:label");
        crv->file = SC_strsavef(fname, "char*:UL_READ_PDB_CURVE:fname");};

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_WRT_PDB_CURVE - write the specified curve out to the given file
 *                  - as the icurve'th curve in the file
 */

static void UL_wrt_pdb_curve(fp, crv, icurve)
   PDBfile *fp;
   curve *crv;
   int icurve;
   {if (!PD_wrt_pdb_curve(fp, crv->text, crv->n, crv->xp, crv->yp, icurve))
       {PRINT(stdout, "%s\n", PD_err);
        SS_error("CAN'T WRITE THE CURVE - WRT_PDB_CURVE", SS_null);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_ULTRA_BINARY_FILEP - test for binary file
 *                       - return TRUE if this is an ULTRA binary file
 *                       - that is it MUST be one of
 *                       -   1) PDB file
 *                       -   2) ASCII ULTRA file
 *                       -   3) ULTRA II binary file (not ULTRA I file)
 *                       - the reason is that a random binary file
 *                       - is indistinguishable from an ULTRA I file
 *                       - by any test save to coredump ULTRA II
 */

static int UL_ultra_binary_filep(fp)
   FILE *fp;
   {char bf[MAXLINE];
    int hdr_sz;

    hdr_sz = strlen(Ultra_Hdr) + 1;
    if (io_read(bf, (size_t) sizeof(char), (size_t) MAXLINE, fp) != MAXLINE)
       return(FALSE);

/* check for ULTRA II binary files */
    if (strncmp(bf, Ultra_Hdr, hdr_sz) == 0)

/* set the file pointer back to the beginning of the data */
       {io_seek(fp, (long) hdr_sz, SEEK_SET);
        return(TRUE);};

    return(FALSE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_ULTRA_ASCII_FILEP - test for ascii file
 *                      - return TRUE if this is an ULTRA ascii file
 */

static int UL_ultra_ascii_filep(fp)
   FILE *fp;
   {char bf[MAXLINE];
    int i, n;
    long addr;

    while (TRUE)
       {addr = io_tell(fp);

        if (io_gets(bf, MAXLINE, fp) == NULL)
           break;

        if (bf[0] != '#')
           continue;

        n = strlen(bf) - 1;
        for (i = 1; i < n; i++)
            if (!isprint(bf[i]))
               return(FALSE);

        io_seek(fp, addr, SEEK_SET);

        return(TRUE);};

    return(FALSE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_OPEN_FOR_READING - open a file so that it can be read AND have
 *                     - its pointer moved around
 */

static FILE *UL_open_for_reading(str, mode)
   char *str, *mode;
   {FILE *fp;

    fp = io_open(str, mode);
    if (fp == NULL)
       SS_error("CAN'T OPEN FILE - UL_OPEN_FOR_READING", SS_mk_string(str));

    return(fp);}

/*--------------------------------------------------------------------------*/

/*                            TOP-LEVEL ROUTINES                            */

/*--------------------------------------------------------------------------*/

/* UL_READ_VER1 - Read data from ULTRA I file
 *              - and put it in the available space
 */

object *UL_read_ver1(obj)
   object *obj;
   {FILE *fp;
    char fname[MAXLINE], name[MAXLINE], *path;
    int j;

    j = UL_next_number(FALSE);

    strcpy(fname, UL_get_string(obj));
    path = SC_search_file(NULL, fname);
    if (path == NULL)
       SS_error("CAN'T FIND FILE - UL_READ_VER1", obj);

    strcpy(name, path);

    fp = UL_open_for_reading(name, "r");
    if (fp == NULL)
       SS_error("NON EXISTENT FILE - UL_READ_VER1", obj);

/* if this dies it is because you lied in claiming this to be a
 * version 1 file
 * there is no safety net here - version 1 files are not to be supported
 * in perpetuity!
 */
    UL_read_bin(fp, fname);
    SX_prefix_list[UL_next_prefix()] = j;

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_READ_DATA - Read data from file and put it in the available space */

object *UL_read_data(obj)
   object *obj;
   {FILE *fp;
    PDBfile *pfp;
    char fname[MAXLINE], name[MAXLINE], *path;
    int j;

    strcpy(fname, UL_get_string(obj));
    path = SC_search_file(NULL, fname);
    if (path == NULL)
       SS_error("CAN'T FIND FILE - UL_READ_DATA", obj);
    strcpy(name, path);

    j = UL_next_number(FALSE);

/* test for a PDB file */    
    pfp = PD_open(name, "r");
    if (pfp != NULL)
       {UL_read_pdb(pfp, fname);
        UL_push_open_file(pfp->stream);
        SX_prefix_list[UL_next_prefix()] = j;
        return(SS_t);};

/* test for other file types
 * remake the name first because a strcpy in PD_open clobbers it
 */
    fp = UL_open_for_reading(name, BINARY_MODE_R);

    if (UL_ultra_binary_filep(fp))
       {UL_read_bin(fp, fname);
        SX_prefix_list[UL_next_prefix()] = j;
        return(SS_t);};

    io_close(fp);

    fp = UL_open_for_reading(name, "r");
    if (UL_ultra_ascii_filep(fp))
       {UL_read_ascii(fp, fname);
        SX_prefix_list[UL_next_prefix()] = j;
        return(SS_t);};

    SS_error("FILE NOT LEGAL ULTRA II FILE - UL_READ_DATA", obj);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_VALID_ULTRA_FILEP - return #t iff the file is a valid ULTRA II
 *                      - input file
 */

object *UL_valid_ultra_filep(obj)
   object *obj;
   {FILE *fp;
    PDBfile *pfp;
    char fname[MAXLINE], name[MAXLINE], *path;

    strcpy(fname, UL_get_string(obj));
    path = SC_search_file(NULL, fname);
    if (path == NULL)
       return(SS_f);

    strcpy(name, path);

/* test for a PDB file */    
    pfp = PD_open(name, "r");
    if (pfp != NULL)
       {PD_close(pfp);
        return(SS_t);};

/* test for other file types
 * remake the name first because a strcpy in PD_open clobbers it
 */
    fp = UL_open_for_reading(name, BINARY_MODE_R);
    if (UL_ultra_binary_filep(fp))
       {io_close(fp);
        return(SS_t);};

    io_close(fp);

    if (SC_isfile_ascii(name))
      {fp = UL_open_for_reading(name, "r");
       if (UL_ultra_ascii_filep(fp))
         {io_close(fp);
          return(SS_t);};};

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_FILE_INFO - report the basic info for a PDB file */

object *UL_file_info(obj)
   object *obj;
   {PDBfile *pfp;
    char fname[MAXLINE], *path;

    strcpy(fname, UL_get_string(obj));
    path = SC_search_file(NULL, fname);
    if (path == NULL)
       SS_error("CAN'T FIND FILE - UL_FILE_INFO", obj);

/* test for a PDB file */    
    pfp = PD_open(path, "r");
    if (pfp != NULL)
       {PRINT(stdout, "\n");
        PRINT(stdout, "PDB System Version: %d\n", pfp->system_version);
        PRINT(stdout, "Creation date: %s\n", pfp->date);
        if (pfp->major_order == ROW_MAJOR_ORDER)
           PRINT(stdout, "Written by C interface (row major order)\n");
        else
           PRINT(stdout, "Written by FORTRAN interface (column major order)\n");
        PRINT(stdout, "\n");

        PD_close(pfp);
        return(SS_t);}

    else
       return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_READ_PDB - read the curves from a PDB file */

static void UL_read_pdb(fp, fname)
   PDBfile *fp;
   char *fname;
   {int i, j, k;
    int icurve = 0;
    char **names;
    pdb_info *ppi;

/* get an alphabetic list of curves */
    names = SC_hash_dump(fp->symtab, "*curve*");
      
    if (names == NULL)
       SS_error("NO CURVES IN FILE OR INSUFFICIENT MEMORY - UL_READ_PDB", SS_null);

    for (i = 0; (names[i] != NULL) && (i < fp->symtab->nelements); i++)

/* get the next open space */
        {j = UL_next_space();

         if (!UL_read_pdb_curve(fp, fname, names[i], &UL_dataset[j], FALSE))
            PRINT(stdout, "WARNING: NO DATA READ FOR CURVE %s - SKIPPED\n",
                  names[i]);
         else
            {icurve++;

/* set up the file info for this curve */
             ppi = FMAKE(pdb_info, "UL_READ_PDB:ppi");
             ppi->file = fp;
             ppi->curve_name = SC_strsavef(names[i],
                               "char*:UL_READ_PDB:cur_name");
             UL_dataset[j].file_info = (byte *) ppi;
             UL_dataset[j].file_type = PDB;

/* put this curve's data in a cache somewhere (let's us defer the question
 * about memory management) and also handle all curve sources the same way
 */
             UL_cache_curve(&UL_dataset[j], PDB);

             k = UL_next_number(TRUE);
             UL_number[k] = j;};};

    UL_n_curves_read += icurve;
    SFREE(names);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_READ_BIN - read a binary file */

static void UL_read_bin(fp, fname)
   FILE *fp;
   char *fname;
   {int n, len, j, i, icurve;
    REAL xmin, xmax, ymin, ymax;
    char c;
    bin_info *pbi;

/* remember this file pointer */
    UL_push_open_file(fp);

    icurve = 0;
    while (!feof(fp))
       {j = UL_next_space();
        read_int(len, fp);
        if (io_read(pbuffer, sizeof(char), len, fp) != len)
           break;
        pbuffer[len] = '\0';
        UL_dataset[j].text = SC_strsavef(pbuffer,
                             "char*:UL_READ_BIN:text");
        UL_dataset[j].file = SC_strsavef(fname,
                             "char*:UL_READ_BIN:fname");
                
        read_int(n, fp);
        UL_dataset[j].n = n;
        UL_dataset[j].xp = FMAKE_N(REAL, n, "UL_READ_BIN:xp");
        UL_dataset[j].yp = FMAKE_N(REAL, n, "UL_READ_BIN:yp");
        if ((UL_dataset[j].xp == NULL) || (UL_dataset[j].yp == NULL))
           SS_error("INSUFFICIENT MEMORY - READ_BIN", SS_null);

/* set up the file info for this curve */
        pbi = FMAKE(bin_info, "UL_READ_BIN:pbi");
        pbi->stream = fp;
        pbi->fileaddr = io_tell(fp);
        UL_dataset[j].file_info = (byte *) pbi;
        UL_dataset[j].file_type = BINARY;

        if (io_read((char *) UL_dataset[j].xp, sizeof(REAL), n, fp) != n)
           {PRINT(stdout, "WARNING: INCOMPLETE CURVE %d IN BINARY FILE",
                          icurve + 1);
            break;};
        if (io_read((char *) UL_dataset[j].yp, sizeof(REAL), n, fp) != n)
           {PRINT(stdout, "WARNING: INCOMPLETE CURVE %d IN BINARY FILE",
                          icurve + 1);
            break;};
                
        PM_maxmin(UL_dataset[j].xp, &xmin, &xmax, n);
        PM_maxmin(UL_dataset[j].yp, &ymin, &ymax, n);

        UL_dataset[j].id = ' ';
        UL_dataset[j].n = n;
        UL_dataset[j].xmin = xmin;
        UL_dataset[j].xmax = xmax;
        UL_dataset[j].ymin = ymin;
        UL_dataset[j].ymax = ymax;

/* put this curve's data in a cache somewhere (let's us defer the question
 * about memory management) and also handle all curve sources the same way
 */
        UL_cache_curve(&UL_dataset[j], BINARY);

        icurve++;

        i = UL_next_number(TRUE);
        UL_number[i] = j;
        c = io_getc(fp);                                    /* look for eof */
        io_ungetc(c, fp);};

    UL_n_curves_read += icurve;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_READ_ASCII - read an ASCII input file */

static void UL_read_ascii(fp, fname)
   FILE *fp;
   char *fname;
   {char instring[MAXLINE];
    int j, c, icurve;
    char *pin, *text, *s;
    unsigned current_size;                        /* current size of buffer */
    REAL *xbuff, *ybuff, *xp, *yp, xval, yval;

    j = UL_next_space();
    dataptr = 0;
    icurve = 0;

    xp = NULL;
    yp = NULL;
    while ((io_gets(instring, MAXLINE, fp) != NULL) && (j >= 0))
       {pin = instring;
        c   = *pin;

        if (c == EOF)
           break;

        if (c == '#')

/* Don't terminate the first one */
           {if (dataptr != 0)
               {if (UL_termdata(&j, xbuff, ybuff))
                   icurve++;};

/* strip off leading whitespace from label */
            pin += strspn(pin, "# \t");

/* GOTCHA - If there are multiple consecutive lines beginning with '#'
 *          (i.e. comments), the following code allocates, but never
 *          releases space for text, fname, xbuff, and ybuff.
 */

/* get rid of \n */
            text = SC_strtok(pin, "\n", s);
            if (text == NULL)
	       UL_dataset[j].text = SC_strsavef("",
                                    "char*:UL_READ_ASCII:NULL");
	    else
	       UL_dataset[j].text = SC_strsavef(text,
                                    "char*:UL_READ_ASCII:text");
            UL_dataset[j].file = SC_strsavef(fname,
                                 "char*:UL_READ_ASCII:fname");

/* allocate space for buffer */
            current_size = MAXPTS;
            xp = xbuff = FMAKE_N(REAL, current_size,
                                 "UL_READ_ASCII:xp");
            yp = ybuff = FMAKE_N(REAL, current_size,
                                 "UL_READ_ASCII:yp");}
        else
           {if ((xp == NULL) || (yp == NULL))
               continue;

            for (text = SC_strtok(instring, " \t\n\r", s);
                 text != NULL;
                 text = SC_strtok(NULL, " \t\n\r", s))
                {if (SC_fltstrp(text))
                    xval = ATOF(text);
                 else
                    break;

                 text = SC_strtok(NULL, " \t\n\t", s);
                 if (text == NULL)
                    break;
                 else if (SC_fltstrp(text))
                    yval = ATOF(text);
                 else
                    break;
                 *xp++ = xval;
                 *yp++ = yval;
                 dataptr++;

/* get more space if needed */
                 if (dataptr >= current_size)
                    {current_size += MAXPTS;
                     REMAKE_N(xbuff, REAL, current_size);
                     REMAKE_N(ybuff, REAL, current_size);
                     xp = xbuff + dataptr;
                     yp = ybuff + dataptr;};};};};

    if (UL_termdata(&j, xbuff, ybuff))
       icurve++;
    io_close(fp);

    if (icurve == 0)
       PRINT(stdout, "%s FILE HAS NO LEGAL CURVES - UL_READ_ASCII: %s\n\n",
                     "WARNING:", fname);

    UL_n_curves_read += icurve;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_READ_ASCII_TABLE - read a table of numbers from an  ASCII input file */

object *UL_read_ascii_table(argl)
   object *argl;
   {int i, j, n, nc, nr, fn, nlabel;
    long addrt, addrl, nl;
    char *name, *token, label[MAXLINE];
    FILE *fp;

    memset(label, 0, MAXLINE);

    if (UL_current_table != NULL)
       {PM_destroy(UL_current_table);
        UL_current_table = NULL;};

    name   = NULL;
    n      = 1;
    nl     = 1;
    nlabel = 0;
    SS_args(argl,
            SC_STRING_I, &name,
            SC_INTEGER_I, &n,
            SC_INTEGER_I, &nlabel,
            SC_LONG_I, &nl,
            0);

    UL_table_name = SC_strsavef(name, "char*:UL_READ_ASCII_TABLE:name");
    name = SC_search_file(NULL, UL_table_name);

    if (name == NULL)
       {if (SS_interactive == ON)
	   PRINT(stdout, "\n No file name given\n");

	return(SS_f);};

    fp = io_open(name, "r");
    if (fp == NULL)
       {if (SS_interactive == ON)
	   PRINT(stdout, "\n Can't open file %s\n", name);

	return(SS_f);};

    if (!UL_find_ascii_table(fp, n, nl, &fn, &nr, &nc, &addrt, nlabel, &addrl))
       {if (SS_interactive == ON)
	   PRINT(stdout, "\n No table #%d in file %s\n", n, name);

	return(SS_f);};

    UL_current_table = PM_create(nr, nc);

    if (addrl != -1)
       {if (io_seek(fp, addrl, SEEK_SET))
	   return(FALSE);

	GETLN(label, MAXLINE, fp);}
    else
        label[0] = '\0';

    if (io_seek(fp, addrt, SEEK_SET))
       return(FALSE);

    for (i = 1; i <= nr; i++)
        {GETLN(SC_line, MAXLINE, fp);

         for (j = 1; j <= nc; j++)
             {token = SC_firsttok(SC_line, " \t\n\r");
              if ((j == 1) && !fn)
                 token = SC_firsttok(SC_line, " \t\n\r");

              PM_element(UL_current_table, i, j) = ATOF(token);};};

    io_close(fp);

    if (SS_interactive == ON)
       {if (label[0] == '\0')
           PRINT(stdout,
                 "\n Table %d : %d rows and %d columns\n\n",
                 n, nr, nc);
        else
           PRINT(stdout,
                 "\n Table %d : %d rows and %d columns\n Label: %s\n\n",
                 n, nr, nc, label);};

    _UL_table_n = n;
    _UL_table_ln = nl;

    return(SS_make_list(SC_INTEGER_I, &nr,
                        SC_INTEGER_I, &nc,
                        0));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_FIND_ASCII_TABLE - find the specified table in the given file
 *                     - return TRUE if successful, otherwise return FALSE
 *
 * table-row :  line of one or more tokens, all numeric
 *                or
 *              line of two or more tokens, all numeric except the first
 * first-row :  table-row
 *                and
 *              (previous line not table-row
 *                 or
 *               number-of-tokens(this line) != number-of-tokens(previous line)
 *                 or
 *               number-numeric-tokens(this line) != number-numeric-tokens(previous line))
 * continuation-row :  table-row
 *                       and
 *                     not first-row
 * table :  first-row followed by zero or more continuation-rows
 * nc    :  number of numeric tokens per table-row
 * nr    :  one plus number of continuation-rows
 * 
 */

static int UL_find_ascii_table(fp, n, nl, pfn, pnr, pnc, paddrt, nlab, paddrl)
   FILE *fp;
   int n, *pfn, *pnr, *pnc;
   long nl, *paddrt;
   int nlab;
   long *paddrl;
   {int i, j, nc, nr, nt, firstnum, nbefore, nafter, nlb;
    long *addr, naddr, addrt, addrl;
    char *token;

    addrl   = -1;
    nbefore = 0;
    nafter  = 0;
    if (nlab > 0)
       nbefore = nlab;
    else
       nafter = -nlab;

    addr = FMAKE_N(long, nbefore+1,
                   "UL_FIND_ASCII_TABLE:addr");

    nr  = 0;
    nc  = 0;
    nlb = 0;

/* skip to specified line */
    for (i = 1; i < nl; i++)
	{addr[nlb++] = io_tell(fp);
	 if (nlb > nbefore)
	    nlb = 0;
	 if (GETLN(SC_line, MAXLINE, fp) == NULL)
            return(FALSE);};

/* loop over tables */
    for (j = 0; j < n; j++)

/* loop over lines until first-row found or EOF */
        {while (TRUE)
            {nc       = 0;
             firstnum = FALSE;
             addrt    = io_tell(fp);

             addr[nlb++] = addrt;
             if (nlb > nbefore)
                nlb = 0;

             if (GETLN(SC_line, MAXLINE, fp) == NULL)
                break;

/* loop over tokens */
             for (nt = 0; TRUE; nt++)
                 {token = SC_firsttok(SC_line, " \t\n\r");
                  if (token == NULL)
                     break;
                  if (nt == 0)
                     {if (SC_numstrp(token))
                         firstnum = TRUE;}
                  else if (!SC_numstrp(token))
                     {nt = 0;
                      break;};};

             nc = firstnum ? nt : nt - 1;
             if (nc > 0)
                break;};
            
/* loop over lines of table */
         for (nr = 1; TRUE; nr++)
             {naddr = io_tell(fp);
              if (GETLN(SC_line, MAXLINE, fp) == NULL)
                 break;

              for (nt = 0; TRUE; nt++)
                  {token = SC_firsttok(SC_line, " \t\n\r");
                   if (token == NULL)
                       break;
                   if (nt == 0)
                      {if ((SC_numstrp(token) && !firstnum) ||
                          (!SC_numstrp(token) && firstnum))
                          break;}
                   else
                      if (!SC_numstrp(token))
                         {nt = 0;
                          break;};};

              nt = firstnum ? nt : nt - 1;
              if (nt != nc)
                 break;

/* if this table is before the one we want, buffer up addresses. */
	      if (j < n-1)
		{addr[nlb++] = naddr;
		 if (nlb > nbefore)
		   nlb = 0;};};

         io_seek(fp, naddr, SEEK_SET);};
         
    if (nbefore > 0)
       {nlb--;
        if (nlb < 0)
           nlb += nbefore + 1;
        addrt = addr[nlb];

        nlb -= nbefore;
        if (nlb < 0)
           nlb += nbefore + 1;
        addrl = addr[nlb];};

    if (nafter > 0)
       {addrt = addr[0];
        for (i = 0; i < nafter; i++)
            {addrl = io_tell(fp);
             if (GETLN(SC_line, MAXLINE, fp) == NULL)
                return(FALSE);};};

    SFREE(addr);

    if ((j == n) && (nc > 0))
       {*pfn    = firstnum;
        *pnc    = nc;
        *pnr    = nr;
        *paddrt = addrt;
        *paddrl = addrl;
        return(TRUE);}
    else
       return(FALSE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_TABLE_CURVE - create and return a curve from the current table using
 *                - the given specifications
 *                -
 *                - table-curve <n> [<y-offs> [<y-stride> [<x-offs> [<x-stride>]]]]
 *                -
 */

object *UL_table_curve(argl)
   object *argl;
   {int k, na, yo, ys, xo, xs;
    REAL *xa, *ya;
    char label[MAXLINE];
    object *ret;

    yo = 0;
    ys = 1;
    xs = 1;
    xo = -1;
    SS_args(argl,
            SC_INTEGER_I, &na,
            SC_INTEGER_I, &yo,
            SC_INTEGER_I, &ys,
            SC_INTEGER_I, &xo,
            SC_INTEGER_I, &xs,
            0);

    ya = UL_extract_vector(UL_current_table, yo, ys, na);

    if (xo != -1)
       xa = UL_extract_vector(UL_current_table, xo, xs, na);
    else
       {xo = 0;
        xa = FMAKE_N(REAL, na, "UL_TABLE_CURVE:xa");
        for (k = 0; k < na; k++)
            xa[k] = (double) (k+1);};

    sprintf(label, "Table %d:%ld (%d:%d) vs (%d:%d)",
            _UL_table_n, _UL_table_ln, yo, ys, xo, xs);

    ret = _UL_mk_curve(na, xa, ya, label, UL_table_name);

    SFREE(ya);
    SFREE(xa);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_EXTRACT_VECTOR - extract a vector from the given array */

static REAL *UL_extract_vector(a, o, s, n)
   PM_matrix *a;
   int o, s, n;
   {int i;
    REAL *val, *src;

    src = a->array + o;
    val = FMAKE_N(REAL, n, "UL_EXTRACT_VECTOR:val");
    for (i = 0; i < n; i++)
        val[i] = src[i*s];

    return(val);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_TABLE_ATTR - return a list of descriptive table parameters
 *               - 
 *               - (n-rows n-cols table-#)
 */

object *UL_table_attr()
   {object *ret;
    int nrows, ncols, table_n;

    if (UL_current_table == NULL)
       ret = SS_null;

    else
       {nrows   = UL_current_table->nrow;
        ncols   = UL_current_table->ncol;
        table_n = _UL_table_n;

        if (SS_interactive == ON)
           PRINT(stdout, "\n Table %d : %d rows and %d columns\n\n",
                 table_n, nrows, ncols);

        ret = SS_make_list(SC_INTEGER_I, &nrows,
                           SC_INTEGER_I, &ncols,
                           SC_INTEGER_I, &table_n,
                           0);};

    return(ret);}

/*--------------------------------------------------------------------------*/

/*                         CURVE OUTPUT ROUTINES                            */

/*--------------------------------------------------------------------------*/

/* UL_WRITE_DATA - write out curves in specified format */

object *UL_write_data(argl) 
   object *argl;
   {int imode;
    char *mode, *fname, *type;
    object *fobj, *frst;
    static HASHTAB *files = NULL;

    if (files == NULL)
       files = SC_make_hash_table(HSZSMALL, NODOC);

    UL_prep_arg(argl);
    mode  = NULL;
    fname = NULL;
    SS_args(argl,
	    SC_STRING_I, &mode,
	    SC_STRING_I, &fname,
	    0);

/* sort out the file type */
    if (strcmp(mode, "ascii") == 0)
       {imode = ASCII;
        fobj  = SS_cadr(argl);
        argl  = SS_cddr(argl);}
    else if (strcmp(mode, "bin") == 0)
       {imode = BINARY;
        fobj  = SS_cadr(argl);
        argl  = SS_cddr(argl);}
    else if (strcmp(mode, "pdb") == 0)
       {imode = PDB;
        fobj  = SS_cadr(argl);
        argl  = SS_cddr(argl);}
    else
       {imode = PDB;
	fname = mode;
	mode  = SC_strsavef("pdb", "char*:UL_WRITE_DATA:mode");
        fobj  = SS_car(argl);
        argl  = SS_cdr(argl);};

    if (strcmp(fname, "- no print name -") == 0)
       SS_error("BAD FILE NAME - UL_WRITE_DATA", fobj);

/* check to see whether the file is a first time encounter */
    type = (char *) SC_def_lookup(fname, files);
    if (type == NULL)
       {if (SC_isfile(fname))
           {switch (SX_file_exist_action)
               {case DESTROY :
		     SC_remove(fname);
		     break;

                case APPEND :
		     break;

                case FAIL:
		default :
		     SS_error("FILE ALREADY EXISTS - UL_WRITE_DATA", fobj);
		     break;};};

	SC_install(fname, mode, SC_STRING_S, files);}

    else
       {if (strcmp(type, mode) != 0)
           SS_error("FILE PREVIOUSLY OPENED WITH ANOTHER TYPE - UL_WRITE_DATA",
		    fobj);};

/* flatten out the curve list */
    frst = SS_car(argl);
    if (strcmp(UL_get_string(frst), "thru") == 0)
       argl = SS_exp_eval(argl);

    switch (imode)
       {case ASCII :
	     {FILE *fp;

/* check for request to write an ASCII file */
	      fp = io_open(fname, "a");
	      if (fp == NULL)
                 SS_error("CAN'T CREATE ASCII FILE - UL_WRITE_DATA", fobj);

	      UL_wrt_ascii(fp, argl);
	      io_close(fp);};

	     break;

        case BINARY :
	     {FILE *fp;

/* check for request to write an ULTRA binary file */
	      fp = io_open(fname, BINARY_MODE_APLUS);
	      if (fp == NULL)
                 SS_error("CAN'T CREATE BINARY FILE - UL_WRITE_DATA", fobj);

/* if this is the first time with this file, put out the header */
	      if (type == NULL)
		 io_write(Ultra_Hdr, (size_t) sizeof(char),
			  (size_t) (strlen(Ultra_Hdr) + 1), fp);

	      UL_wrt_bin(fp, argl);
	      io_close(fp);};

	     break;

        case PDB :
        default  :
	     {PDBfile *fp;

/* check for request to write a pdb file */
	      fp = PD_open(fname, "a");
	      if (fp == NULL)
		 SS_error("CAN'T OPEN PDB FILE - UL_WRITE_DATA", fobj);

	      UL_wrt_pdb(fp, argl);
	      PD_close(fp);};

	     break;};

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_WRT_PDB - write curves to a ULTRA PDB file */

static void UL_wrt_pdb(fp, argl) 
   PDBfile *fp;
   object *argl;
   {int i, j, uncached;
    object *obj;
    static int icurve = 0;

    uncached = FALSE;

    for ( ; SS_consp(argl); argl = SS_cdr(argl))
        {obj = SS_car(argl);
         j = -1;

         if (_SS_numberp(obj))
            {i = SS_INTEGER_VALUE(obj);
             if ((i >= 1) && (i <= UL_n_curves_read))
                j = UL_number[i];
             if ((j != -1) && (UL_dataset[j].xp == NULL))
                {uncached = TRUE;
                 UL_uncache_curve(&UL_dataset[j]);};}
         else
            j = UL_get_curve(obj);

         if (j >= 0)
            UL_wrt_pdb_curve(fp, &UL_dataset[j], icurve++);

         if (uncached)
            {uncached = FALSE;
             SFREE(UL_dataset[j].xp);
             SFREE(UL_dataset[j].yp);
             UL_zero_curve(j);};};

    PD_flush(fp);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_WRT_BIN - write curves to an ULTRA binary file
 *            - format for binary files:
 *            -        int i - length of id string
 *            -        id string
 *            -        int n - number of datapoints
 *            -        REAL n x's
 *            -        REAL n y's
 */

static void UL_wrt_bin(fp, argl)
   FILE *fp;
   object *argl;
   {int i, j, n, uncached, err;
    object *obj;

    uncached = FALSE;

    for ( ; SS_consp(argl); argl = SS_cdr(argl))
        {obj = SS_car(argl);
         j = -1;

         if (_SS_numberp(obj))
            {i = (int) *SS_GET(long, obj);
             if ((i >= 1) && (i <= UL_n_curves_read))
                j = UL_number[i];
             if ((j != -1) && (UL_dataset[j].xp == NULL))
                {uncached = TRUE;
                 UL_uncache_curve(&UL_dataset[j]);};}
         else
            j = UL_get_curve(obj);

         if (j >= 0)
            {n = UL_dataset[j].n;
             i = strlen(UL_dataset[j].text);

             write_int(i, fp);  
             err = io_write(UL_dataset[j].text, sizeof(char), i, fp);
             write_int(n, fp);  
             err = io_write((char *) UL_dataset[j].xp, sizeof(REAL), n, fp);
             err = io_write((char *) UL_dataset[j].yp, sizeof(REAL), n, fp);};

         if (uncached)
            {uncached = FALSE;
             SFREE(UL_dataset[j].xp);
             SFREE(UL_dataset[j].yp);
             UL_zero_curve(j);};};

    err = io_flush(fp);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_WRT_ASCII - write curves to an ASCII file */

static void UL_wrt_ascii(fp, argl)
   FILE *fp;
   object *argl;
   {int i, j, n, uncached;
    REAL *x, *y;
    object *obj;

    uncached = FALSE;

    for ( ; SS_consp(argl); argl = SS_cdr(argl))
        {obj = SS_car(argl);
         j = -1;

         if (_SS_numberp(obj))
            {i = (int) *SS_GET(long, obj);
             if ((i >= 1) && (i <= UL_n_curves_read))
                j = UL_number[i];
             if ((j != -1) && (UL_dataset[j].xp == NULL))
                {uncached = TRUE;
                 UL_uncache_curve(&UL_dataset[j]);};}
         else
            j = UL_get_curve(obj);

         if (j >= 0)
            {n = UL_dataset[j].n;
             x = UL_dataset[j].xp;
             y = UL_dataset[j].yp;

             if (UL_dataset[j].text[0] != '#')
                io_printf(fp, "# ");

             io_printf(fp, "%s\n", UL_dataset[j].text);
             for (i = 0; i < n; i++)
                 {io_printf(fp, UL_ascii_output_format, x[i]);
                  io_printf(fp, " ");
                  io_printf(fp, UL_ascii_output_format, y[i]);
                  io_printf(fp, "\n");};};

         if (uncached)
            {uncached = FALSE;
             SFREE(UL_dataset[j].xp);
             SFREE(UL_dataset[j].yp);
             UL_zero_curve(j);};};

    io_flush(fp);

    return;}

/*--------------------------------------------------------------------------*/

/*                     CURVE MEMORY MANAGEMENT ROUTINES                     */

/*--------------------------------------------------------------------------*/

/* UL_CACHE_CURVE - figure out what/how to do with the curve data
 *                - in a uniform way for any curve source (ASCII, BINARY,
 *                - PDB) this will have to communicate with UL_SELECT when
 *                - the curves are called up
 */

static void UL_cache_curve(crv, type)
   curve *crv;
   int type;
   {static int icurve = 0;
    char bf[20];
    pdb_info *ppi;

    switch (type)
       {case ASCII :
	     if (UL_cache_file == NULL)
	        {icurve = 0;
		 UL_cache_file = PD_create("curves.a");
		 if (UL_cache_file == NULL)
		    SS_error("CAN'T CREATE curves.a - UL_CACHE_CURVE",
			     SS_null);
		 PD_close(UL_cache_file);
#ifdef LINUX
/* GOTCHA: Without this delay the open will fail,
 *         iff file accessed across the network
 */
		 sleep(1);
#endif
		 UL_cache_file = PD_open("curves.a", "a");
		 if (UL_cache_file == NULL)
		    SS_error("CAN'T OPEN curves.a - UL_CACHE_CURVE",
			     SS_null);
		 UL_push_open_file(UL_cache_file->stream);};

	     sprintf(bf, "curve%04d", icurve);
	     ppi = FMAKE(pdb_info, "UL_CACHE_CURVE:ppi");
	     ppi->file = UL_cache_file;
	     ppi->curve_name = SC_strsavef(bf, "char*:UL_CACHE_CURVE:bf");
	     crv->file_info = (byte *) ppi;
	     crv->file_type = PDB;

	     UL_wrt_pdb_curve(UL_cache_file, crv, icurve++);

        case BINARY :
        case PDB    :
	     SFREE(crv->xp);
	     SFREE(crv->yp);
	     crv->xp = NULL;
	     crv->yp = NULL;

        default :
	     break;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_UNCACHE_CURVE - get the curve data out of wherever it has been stashed
 *                  - and connect it up to the curve
 */

void UL_uncache_curve(crv)
   curve *crv;
   {int n;
    bin_info *pbi;
    pdb_info *ppi;
    FILE *stream;
    PDBfile *file;

    switch (crv->file_type)
       {case BINARY :
	     pbi = (bin_info *) (crv->file_info);
	     stream = pbi->stream;
	     io_seek(stream, pbi->fileaddr, SEEK_SET);
	     n = crv->n;
	     io_read((char *) crv->xp, sizeof(REAL), n, stream);
	     io_read((char *) crv->yp, sizeof(REAL), n, stream);
	     break;

        case ASCII :
	     break;

        default  :
        case PDB :
	     ppi = (pdb_info *) (crv->file_info);
	     file = ppi->file;
	     if (!UL_read_pdb_curve(file, (byte *) NULL,
				    ppi->curve_name, crv, TRUE))
	        SS_error("PDB READ FAILED - UL_UNCACHE-CURVE",
			 SS_null);
	     break;};

    return;}

/*--------------------------------------------------------------------------*/

/*                           FILE_LIST ROUTINES                             */

/*--------------------------------------------------------------------------*/

/* UL_FILE_OPEN - check the given file pointer against any others
 *              - in the list of open files and return TRUE if
 *              - it is there and FALSE otherwise
 */

static int UL_file_open(fp)
   FILE *fp;
   {pcons *lst;

    for (lst = file_list; lst != NULL; lst = (pcons *) (lst->cdr))
        if (fp == (FILE *) (lst->car))
           return(TRUE);

    return(FALSE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_REMOVE_FILE - check the given file pointer against any others
 *                - in the list of open files and remove it if it is there
 */

static int UL_remove_file(fp)
   FILE *fp;
   {pcons *lst, *nxt;

/* if the first file on the list is the one, take care of it */
    if (fp == (FILE *) (file_list->car))
       {nxt = (pcons *) (file_list->cdr);
	SC_rl_pcons(file_list, 0);
        file_list = nxt;
        return(TRUE);};

/* otherwise cdr down the list in search of the file */
    for (lst = file_list, nxt = (pcons *) (lst->cdr);
         nxt != NULL;
         lst = nxt, nxt = (pcons *) (nxt->cdr))
        if (fp == (FILE *) (nxt->car))
           {lst->cdr = nxt->cdr;
            SC_rl_pcons(nxt, 0);
            return(TRUE);};

    return(FALSE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_PUSH_OPEN_FILE - check the given file pointer against any others
 *                   - in the list of open file and cons it on if it is
 *                   - not already there
 */

void UL_push_open_file(fp)
   FILE *fp;
   {if (!UL_file_open(fp))
       file_list = SC_mk_pcons(NULL, fp, SC_PCONS_P_S, file_list);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_CLOSE_OPEN_FILES - cdr down the list of open files and close each one
 *                     - NOTE: this should be rewritten to go off file_list
 *                     -       and save the time of going over the curves!!!
 */

void UL_close_open_files()
   {pdb_info *ppi;
    bin_info *pbi;
    FILE *fp;
    int i;

    for (i = 0; i < UL_N_Curves; i++)
        {switch (UL_dataset[i].file_type)
            {case BINARY :
	          pbi = (bin_info *) UL_dataset[i].file_info;
		  fp = pbi->stream;
		  if (UL_file_open(fp))
		     {UL_remove_file(fp);
		      io_close(fp);};
		  SFREE(pbi);
		  break;

             case PDB :
	          {PDBfile *file;

		   ppi = (pdb_info *) UL_dataset[i].file_info;
		   file = ppi->file;
		   if (file != NULL)
		     {fp = file->stream;
		      if (UL_file_open(fp))
			 {UL_remove_file(fp);
			  PD_close(file);
			  if (file == UL_cache_file)
                             UL_cache_file = NULL;};

		      SFREE(ppi->curve_name);
		      SFREE(ppi);};};

		  break;

             default    :
             case ASCII : 
	          break;};

       UL_dataset[i].file_info = NULL;
       UL_dataset[i].file_type = NO_FILE;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
