/*
 * ULFUNC.C - basic functions for Ultra
 *
 * Source Version: 4.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "ultra.h"

#ifdef DOS
#include <process.h>
#define TOO_MANY_POINTS 1000
#endif

#define UL_COPY_CURVE(x)     UL_copy_curve(UL_get_curve(x));
#define advance(j)           (j < (UL_dataset[l].n-1)) ? j++ : j 

#ifndef TOO_MANY_POINTS
#define TOO_MANY_POINTS 1000000
#endif

char
 macro_line[MAXLINE];

static int
 old_autoplot;

static double
 SC_DECLARE(_UL_pow, (double x, double a)),
 SC_DECLARE(UL_compose, (double a, int i));

static void
 SC_DECLARE(UL_plot_off, (byte)),
 SC_DECLARE(UL_restore_plot, (byte));

static int
 SC_DECLARE(UL_expunge, (int j));

static object
 *SC_DECLARE(UL_print_labels,
          (int *indx, int nc, char *regx, char *file,
           int id_flag, char *name)),
 *SC_DECLARE(UL_select, (object *s)),
 *SC_DECLARE(UL_menui, (object *s)),
 *SC_DECLARE(UL_expunge_macro, (object *argl)),
 *SC_DECLARE(UL_compress_numbers, (byte)),
 *SC_DECLARE(UL_erase, (byte)),
 *SC_DECLARE(UL_color, (object *obj, object *color)),
 *SC_DECLARE(UL_scatter, (object *obj, object *flag)),
 *SC_DECLARE(UL_marker, (object *obj, object *flag)),
 *SC_DECLARE(UL_hist, (object *obj, object *flag)),
 *SC_DECLARE(UL_lnwidth, (object *obj, object *width)),
 *SC_DECLARE(UL_lnstyle, (object *obj, object *style)),
 *SC_DECLARE(UL_range, (object *argl)),
 *SC_DECLARE(UL_domain, (object *argl)),
 *SC_DECLARE(UL_open_device, (object *argl)),
 *SC_DECLARE(UL_close_device, (object *arg)),
 *SC_DECLARE(UL_quit, (object *arg)),
 *SC_DECLARE(UL_menu, (object *argl)),
 *SC_DECLARE(UL_prefix, (object *argl)),
 *SC_DECLARE(UL_list_curves, (object *argl)),
 *SC_DECLARE(UL_derivative, (int j)),
 *SC_DECLARE(UL_thin, (int j, object *argl)),
 *SC_DECLARE(UL_filter, (int j, object *argl)),
 *SC_DECLARE(UL_integrate, (int j, double d1, double d2)),
 *SC_DECLARE(UL_label, (object *argl)),
 *SC_DECLARE(UL_average, (object *s)),
 *SC_DECLARE(UL_system, (object *s)),
 *SC_DECLARE(UL_xmm, (int j, double d1, double d2)),
 *SC_DECLARE(UL_filter_coef, (int l, object *argl)),
 *SC_DECLARE(UL_smooth, (int l, object *argl)),
 *SC_DECLARE(UL_reverse, (int j)),
 *SC_DECLARE(UL_smp_append, (object *a, object *b)),
 *SC_DECLARE(UL_pr_append, (object *a, object *b)),
 *SC_DECLARE(UL_append, (object *argl)),
 *SC_DECLARE(UL_hide, (int j)),
 *SC_DECLARE(UL_show, (int j)),
 *SC_DECLARE(UL_make_ln, (object *argl)),
 *SC_DECLARE(UL_mk_curve, (object *argl)),
 *SC_DECLARE(UL_curve_list, (object *arg));

/*--------------------------------------------------------------------------*/

/*                          RUDAMENTARY FUNCTIONS                           */

/*--------------------------------------------------------------------------*/

/* UL_SELECT - get the designated curve from the menu if it exists */

static object *UL_select(s)
   object *s;
   {int i, j;
    object *ret;

    j = -1;
    SS_args(s,
            SC_INTEGER_I, &j,
            0);

    if ((j < 1) || (j > UL_n_curves_read))
       return(SS_t);

    i = UL_number[j];

/* this indicates that there is no curve */
    if (i == -1)
       return(SS_t);

    if (UL_dataset[i].n < 2)
       SS_error("CURVE HAS < 2 POINTS - UL_SELECT", s);

/* make a copy of the curve and read the data into the copy */
    ret = UL_copy_curve(i);
    j = UL_get_curve(ret);

/* fetch the curve data out of the cache wherever and however it is done */
    UL_uncache_curve(&UL_dataset[j]);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _UL_PRINT_LABEL - print one curve label out to the terminal
 *                 - see print_curve_labels
 */

static void _UL_print_label(i, j, md, s, id_flag, fp, f, id)
   int i, j, md;
   char *s;
   int id_flag;
   FILE *fp;
   char f[MAXLINE];
   int id;
   {char label[MAXLINE];
    int k;

/* prep the label text */
    memset(label, '\0', MAXLINE);
    if (SX_squeeze_labels)
      SC_squeeze_blanks(s);
    strncpy(label, s, SX_label_length);
    for (k = 0; k < SX_label_length; k++)
      if (label[k] == '\0')
	label[k] = ' ';

    switch (id_flag)
      {case 1  : PRINT(fp, "%4d", j);
	break;
      case 2  : if (md)
	PRINT(fp, "* %c", id);
      else
	PRINT(fp, "  %c", id);
	break;
      case 3  : if (md)
	PRINT(fp, "%4d * %c", j, id);
      else
	PRINT(fp, "%4d   %c", j, id);
	default : break;};

    PRINT(fp, " %s", label);
    if (SX_label_length < 40)
      PRINT(fp, " %10.2e %10.2e %10.2e %10.2e %s",
	    UL_dataset[i].xmin, UL_dataset[i].xmax,
	    UL_dataset[i].ymin, UL_dataset[i].ymax, f);

    PRINT(fp, "\n");

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_MENUI - display the menu of the selected curves */

static object *UL_menui(s)
   object *s;
   {int i, j, md, id, id_flag;
    char *ss, f[MAXLINE];
    FILE *fp;

    j = -1;
    SS_args(s,
            SC_INTEGER_I, &j,
            0);

    if ((j < 1) || (j > UL_n_curves_read))
      return(SS_t);

    i = UL_number[j];

/* this indicates that there is no curve */
    if (i == -1)
       return(SS_t);

    id_flag = 1;
    fp = stdout;
    f[0] = '\0';
    if (UL_dataset[i].file != NULL)
       strcpy(f, UL_dataset[i].file);
    id = UL_dataset[i].id;
    md = UL_dataset[i].modified;
    ss = UL_dataset[i].text;

    _UL_print_label(i, j, md, ss, id_flag, fp, f, id);

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_EXPUNGE_MACRO - executes SIMPLE_EXPUNGE and COMPRESS_NUMBERS to
 *                  - give the expunge that you really want.
 */

static object *UL_expunge_macro(argl)
   object *argl;
   {object *s, *t;
    int limit, j;

    argl = _SX_prep_arg(argl);

    limit = UL_n_curves_read;
    if (!SS_nullobjp(SS_car(argl)))
       {for (s = argl;
             SS_consp(s) && !SS_nullobjp(SS_car(s));
             s = SS_cdr(s))
	    {t = SS_car(s);
             SS_args(t,
		     SC_INTEGER_I, &j,
		     0);
	     if ((0 <= j) && (j <= limit))
	        {if (!UL_expunge(j))
		    {UL_compress_numbers();
		     SS_error("INVALID CURVE NUMBER - UL_EXPUNGE_MACRO",
			      argl);};};};

        UL_compress_numbers();}

    SS_Assign(argl, SS_null);

    return(argl);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_EXPUNGE - delete the given curve from the menu */

static int UL_expunge(j)
   int j;
   {int i;

/* if the requested curve number is zero, kill them all */
    if (j == 0)
       {UL_close_open_files();
        for (j = 0; j < UL_N_Curves; j++)
            {i = UL_number[j];
             if (i != -1)
                {if (UL_dataset[i].xp != NULL)
                    {SFREE(UL_dataset[i].xp);};
                 if (UL_dataset[i].yp != NULL)
                    {SFREE(UL_dataset[i].yp);};

                 UL_zero_curve(i);
                 UL_number[j] = -1;};};
       UL_n_curves_read = 0;}

    else
       {i = UL_number[j];
        if (i != -1)
           {if (UL_dataset[i].xp != NULL)
               {SFREE(UL_dataset[i].xp);};
            if (UL_dataset[i].yp != NULL)
               {SFREE(UL_dataset[i].yp);};

            UL_zero_curve(i);
            UL_number[j] = -1;
            UL_n_curves_read--;};}

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_COMPRESS_NUMBERS - after killing curves compress the numbers to be
 *                     - in contiguous sequence from 1
 */

static object *UL_compress_numbers()
   {int i, lasti;

    lasti = -1;
    for (i = 1; i < UL_N_Curves; i++)
        {if ((UL_number[i] != -1) && (lasti != -1))
            {UL_number[lasti] = UL_number[i];
             UL_number[i] = -1;
             for (; (UL_number[lasti] != -1) && (lasti < UL_N_Curves);
                  lasti++);}

         else if ((UL_number[i] == -1) && (lasti == -1))
            lasti = i;};

    _UL_next_available_number = max(1, lasti);

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_DELETE - delete the curve from the curve list (but not the menu)
 *           - and rebind the variable whose name is a lower case version
 *           - of the curve label to itself
 */

object *UL_delete(s)
   object *s;
   {int i;

    if (SS_nullobjp(s))
       return(SS_null);

    if (!UL_curvep_a(s))
       SS_error("BAD CURVE - UL_DELETE", s);

    i = UL_curve_id(s);

    return(UL_rl_curve(i));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_ERASE - erase all the currently displayed curves */

static object *UL_erase()
   {int j;

    for (j = 0; j < NDISPLAY; j++)
        if (UL_data_index[j] != -1)
           UL_rl_curve(j);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_COLOR - set the color of the specified curve */

static object *UL_color(obj, color)
   object *obj, *color;
   {int i;

    i = UL_get_curve(obj);

    SC_CHANGE_VALUE_ALIST(UL_dataset[i].info, int, SC_INTEGER_P_S,
			  "LINE-COLOR", SS_INTEGER_VALUE(color));

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_SCATTER - set the scatter attribute of the specified curve */

static object *UL_scatter(obj, flag)
   object *obj, *flag;
   {int i;

    i = UL_get_curve(obj);

    SC_CHANGE_VALUE_ALIST(UL_dataset[i].info, int, SC_INTEGER_P_S,
			  "SCATTER", SS_INTEGER_VALUE(flag));

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_MARKER - set the marker of the specified curve */

static object *UL_marker(obj, flag)
   object *obj, *flag;
   {int i, mrk;

    i = UL_get_curve(obj);

    mrk  = SS_INTEGER_VALUE(flag);
    if ((mrk < 0) || (mrk >= _PG_marker_index))
       SS_error("BAD MARKER VALUE - UL_MARKER", flag);

    SC_CHANGE_VALUE_ALIST(UL_dataset[i].info, int, SC_INTEGER_P_S,
			  "MARKER-INDEX", mrk);

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_HIST - set UL_plot the curve as a histogram */

static object *UL_hist(obj, flag)
   object *obj, *flag;
   {int i, fl, mode, side;
    pcons *info;

    fl = SS_INTEGER_VALUE(flag);
    if (fl == HIST_LEFT)
       {mode = HISTOGRAM;
        side = 0;}

    else if (fl == HIST_RIGHT)
       {mode = HISTOGRAM;
        side = 1;}

    else if (fl == HIST_CENTER)
       {mode = HISTOGRAM;
        side = 2;}

    else
       {mode = CARTESIAN;
        side = 0;}

    i = UL_get_curve(obj);

    info = UL_dataset[i].info;
    info = PG_set_plot_type(info, mode, mode);

    SC_CHANGE_VALUE_ALIST(UL_dataset[i].info, int, SC_INTEGER_P_S,
			  "HIST-START", side);

    UL_dataset[i].info = info;

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_LNWIDTH - set line width of specified curve */

static object *UL_lnwidth(obj, width)
   object *obj, *width;
   {int i;
    double wd;

    i = UL_get_curve(obj);

    wd = 0.0;
    SS_args(width,
	    SC_DOUBLE_I, &wd,
	    0);

    SC_CHANGE_VALUE_ALIST(UL_dataset[i].info, double, SC_DOUBLE_P_S,
			  "LINE-WIDTH", wd);

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_LNSTYLE - set line style of specified curve */

static object *UL_lnstyle(obj, style)
   object *obj, *style;
   {int i;

    i = UL_get_curve(obj);

    SC_CHANGE_VALUE_ALIST(UL_dataset[i].info, int, SC_INTEGER_P_S,
			  "LINE-STYLE", SS_INTEGER_VALUE(style));

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_RANGE - set the range of the plot */

static object *UL_range(argl)
   object *argl;
   {object *s;
    PG_device *dev;
    REAL t, xmin, xmax, ymin, ymax;

    UL_prep_arg(argl);

    if (UL_graphics_device != NULL)
       dev = UL_graphics_device;
    else if (UL_PS_device != NULL)
       dev = UL_PS_device;
    else if (UL_CGM_device != NULL)
       dev = UL_CGM_device;
#ifdef HAVE_JPEGLIB
    else if (UL_JPEG_device != NULL)
       dev = UL_JPEG_device;
#endif
    else
       dev = NULL;

    if (SS_nullobjp(argl))
       UL_plot_limits(dev, FALSE, &xmin, &xmax, &ymin, &ymax);
    else
       {s = SS_car(argl);

	if (!_SS_numberp(s))
	   {if (SC_str_icmp(UL_get_string(s), "de") == 0)
	       {SX_autorange = TRUE;
		UL_plot_limits(dev, FALSE, &xmin, &xmax, &ymin, &ymax);}
	    else
	       SS_error("BAD ARGUMENTS - UL_RANGE", s);}
	else
	   {ymin = HUGE;
	    ymax = -HUGE;
	    SS_args(argl,
		    SC_REAL_I, &ymin,
		    SC_REAL_I, &ymax,
		    0);
	    if (ymin == HUGE)
	       SS_error("BAD NUMBER LOWER LIMIT - UL_RANGE", argl);

	    if (ymax == -HUGE)
	       SS_error("BAD NUMBER UPPER LIMIT - UL_RANGE", argl);

	    if (ymin == ymax)
	       SS_error("LOWER LIMIT EQUALS UPPER LIMIT - UL_RANGE", argl);

	    if (ymax < ymin)
	       {t = ymin;
		ymin = ymax;
		ymax = t;};

	    UL_gymin = ymin;
	    UL_gymax = ymax;
	    SX_autorange = FALSE;};};

    return(SS_mk_cons(SS_mk_float(ymin), SS_mk_float(ymax)));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_DOMAIN - set the domain of the plot */

static object *UL_domain(argl)
   object *argl;
   {object *s;
    PG_device *dev;
    REAL t, xmin, xmax, ymin, ymax;

    UL_prep_arg(argl);

    if (UL_graphics_device != NULL)
       dev = UL_graphics_device;
    else if (UL_PS_device != NULL)
       dev = UL_PS_device;
    else if (UL_CGM_device != NULL)
       dev = UL_CGM_device;
#ifdef HAVE_JPEGLIB
    else if (UL_JPEG_device != NULL)
       dev = UL_JPEG_device;
#endif
    else
       dev = NULL;

    if (SS_nullobjp(argl))
       UL_plot_limits(dev, FALSE, &xmin, &xmax, &ymin, &ymax);
    else
       {s = SS_car(argl);

	if (!_SS_numberp(s))
	   {if (SC_str_icmp(UL_get_string(s), "de") == 0)
	       {SX_autodomain = TRUE;
		UL_plot_limits(dev, FALSE, &xmin, &xmax, &ymin, &ymax);}
	    else
	       SS_error("BAD ARGUMENTS - UL_DOMAIN", s);}
	else
	   {xmin = HUGE;
	    xmax = -HUGE;
	    SS_args(argl,
		    SC_REAL_I, &xmin,
		    SC_REAL_I, &xmax,
		    0);
	    if (xmin == HUGE)
	       SS_error("BAD NUMBER LOWER LIMIT - UL_DOMAIN", argl);

	    if (xmax == -HUGE)
	       SS_error("BAD NUMBER UPPER LIMIT - UL_DOMAIN", argl);

	    if (xmin == xmax)
	       SS_error("LOWER LIMIT EQUALS UPPER LIMIT - UL_DOMAIN", argl);

	    if (xmax < xmin)
	       {t = xmin;
		xmin = xmax;
		xmax = t;};

	    UL_gxmin = xmin;
	    UL_gxmax = xmax;
	    SX_autodomain = FALSE;};};

    return(SS_mk_cons(SS_mk_float(xmin), SS_mk_float(xmax)));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_OPEN_DEVICE - open the graphics window for ULTRA
 *                - arguments:
 *                -    DEVICE_NAME - X display name or one of the PGS names
 *                -    DEVICE_TYPE - primarily "COLOR" or "MONOCHROME"
 *                -    DISPLAY_TITLE - a title for identification purposes
 */

static object *UL_open_device(argl)
   object *argl;
   {char *name, *type, *title;

    UL_prep_arg(argl);

    name  = NULL;
    type  = NULL;
    title = NULL;
    SS_args(argl,
            SC_STRING_I, &name,
            SC_STRING_I, &type,
            SC_STRING_I, &title,
            0);

    SC_str_upper(name);
    SC_str_upper(type);
    if (strcmp(name, "PS") == 0)
       {SFREE(SX_ps_type);
        SX_ps_type = type;

        SFREE(SX_ps_name);   
        SX_ps_name = title;

        SFREE(name);}

    else if (strcmp(name, "CGM") == 0)
       {SFREE(SX_cgm_type);
        SX_cgm_type = type;

        SFREE(SX_cgm_name);   
        SX_cgm_name = title;

        SFREE(name);}

#ifdef HAVE_JPEGLIB
    else if (strcmp(name, "JPEG") == 0)
       {SFREE(SX_jpeg_type);
        SX_jpeg_type = type;

        SFREE(SX_jpeg_name);   
        SX_jpeg_name = title;

        SFREE(name);}
#endif

    else
       {SFREE(UL_display_type);
        UL_display_type = type;

        SFREE(UL_display_title);   
        UL_display_title = title;

        SFREE(UL_display_name);
        UL_display_name = name;

        if (UL_gr_mode)
           UL_mode_graphics();};

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_CLOSE_DEVICE - close the named graphics device for ULTRA */

static object *UL_close_device(arg)
   object *arg;
   {char *name;

    name = SC_strsavef(UL_get_string(arg),
                      "char*:UL_CLOSE_DEVICE:name");

    SC_str_upper(name);
    if (strcmp(name, "PS") == 0)
       {SFREE(SX_ps_type);
        SFREE(SX_ps_name);   
         if (UL_PS_device != NULL)
           {PG_close_device(UL_PS_device);
            UL_PS_device = NULL;};}

    else if (strcmp(name, "CGM") == 0)
       {SFREE(SX_cgm_type);
        SFREE(SX_cgm_name);   
        if (UL_CGM_device != NULL)
           {PG_close_device(UL_CGM_device);
            UL_CGM_device = NULL;};}

#ifdef HAVE_JPEGLIB
    else if (strcmp(name, "JPEG") == 0)
       {SFREE(SX_jpeg_type);
        SFREE(SX_jpeg_name);   
        if (UL_JPEG_device != NULL)
           {PG_close_device(UL_JPEG_device);
            UL_JPEG_device = NULL;};}
#endif

    else
       {if (UL_graphics_device != NULL)
           {PG_close_device(UL_graphics_device);
            UL_graphics_device = NULL;};};

    SFREE(name);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _UL_QUIT - gracefully exit from Ultra */

void _UL_quit(i)
   int i;
   {UL_mode_text();

/* check the need to close the command log */
    if (SX_command_log != NULL)
       io_close(SX_command_log);

/* check the need to close the ascii output file */
    if (UL_out_ascii != NULL)
       io_close(UL_out_ascii);

/* check the need to close the binary output file */
    if (UL_out_bin != NULL)
       io_close(UL_out_bin);

/* check the need to close the PDB output file */
    if (UL_out_pdb != NULL)
       PD_close(UL_out_pdb);

/* check the need to close the PostScript device */
    if (UL_PS_device != NULL)
       PG_close_device(UL_PS_device);

/* check the need to close the CGM device */
    if (UL_CGM_device != NULL)
       PG_close_device(UL_CGM_device);

#ifdef HAVE_JPEGLIB
/* check the need to close the JPEG device */
    if (UL_JPEG_device != NULL)
       PG_close_device(UL_JPEG_device);
#endif

/* close the cache file and any open data files */
    UL_close_open_files();

    PC_exit_all();

    exit(i);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_QUIT - gracefully exit from Ultra */

static object *UL_quit(arg)
   object *arg;
   {int exit_val;

    exit_val = 0;
    SS_args(arg,
            SC_INTEGER_I, &exit_val,
            0);

    _UL_quit(exit_val);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_PRINT_LABELS - print the curve labels out to the terminal
 *                 - see print_curve_labels
 */

static object *UL_print_labels(indx, nc, regx, file, id_flag, name)
   int *indx, nc;
   char *regx, *file;
   int id_flag;
   char *name;
   {int i, j, np, id, md, nmore, nlp;
    char bf[10];
    char f[MAXLINE], *s;
    FILE *fp;
    object *ret;

    ret = SS_null;

    if (name != NULL)

       {fp = io_open(name, "w");
        if (fp == NULL)
           SS_error("CANNOT OPEN FILE - UL_PRINT_LABELS", SS_null);
        PRINT(fp, "\n\n");}
    else
       fp = stdout;

    if (SS_lines_page == 0)
       nlp = INT_MAX;
    else
       nlp = max(26, SS_lines_page);

    np    = 0;
    nmore = 0;
    for (j = 0; j < nc; j++)
        {i = indx[j];
         if (i >= 0)
            {id = UL_dataset[i].id;
             md = UL_dataset[i].modified;
             s  = UL_dataset[i].text;
             if (UL_dataset[i].file != NULL)
                strcpy(f, UL_dataset[i].file);
             else
                f[0] = '\0';
             if (regx != NULL)
                {if (!SC_regx_match(s, regx))
                    continue;};
             if (file != NULL)
                {if (!SC_regx_match(f, file))
                    continue;};

             if (id_flag == 1)
                {SS_Assign(ret, SS_mk_cons(SS_mk_integer(j), ret));}
             else if (id_flag == 2)
                {SS_Assign(ret, SS_mk_cons(UL_dataset[i].obj, ret));}

             if ((SS_interactive == ON) || (fp != stdout))

/* prep the label text */
	       {_UL_print_label(i, j, md, s, id_flag, fp, f, id);
		if (fp == stdout)
		  {nmore++;
		   if (nmore >= nlp)
		     {PRINT(fp, "More ... (n to stop)");
		      GETLN(bf, 10, stdin);
		      if (bf[0] == 'n')
			break;
		      nmore = 0;};};

		if (name != NULL)
		  {np++;
		   if (np >= nlp)
		     {PRINT(fp, "\014\n\n");
		      np = 0;};};};};};

    if (name != NULL)
      io_close(fp);

    SS_Assign(ret, SS_reverse(ret));

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_MENU - display the menu of the available curves */

static object *UL_menu(argl)
   object *argl;
   {char *pr, *pf, *pn;
    object *ret;

    if (!SS_nullobjp(argl))
       {UL_prep_arg(argl);};

    pr = NULL;
    pf = NULL;
    pn = NULL;
    SS_args(argl,
            SC_STRING_I, &pr,
            SC_STRING_I, &pf,
            SC_STRING_I, &pn,
            0);

    ret = UL_print_labels(UL_number, UL_N_Curves, pr, pf, 1, pn);
    SX_plot_flag = FALSE;

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_PREFIX - set or display menu prefixes */

static object *UL_prefix(argl)
   object *argl;
   {object *arg1;
    object *arg2;
    object *ret;
    char pre, prefix[MAXLINE];
    int  mindex, i;
    char *fname, *s;

    ret    = SS_null;
    fname  = "";
    mindex = 0;

    if (SS_consp(argl))
       {arg1  = SS_car(argl);
        argl = SS_cdr(argl);
        strcpy(prefix, UL_get_string(arg1));
        if (strlen(prefix) != 1)
           SS_error("BAD PREFIX - UL_PREFIX", arg1);
        pre = tolower(prefix[0]);
        if ((pre < 'a') || (pre > 'z'))
           SS_error("BAD PREFIX - UL_PREFIX", arg1);
        if (SS_consp(argl))
           {arg2 = SS_car(argl);
            s = UL_get_string(arg2);
            if (SC_intstrp(s, 10))
               mindex = atoi(s);
            else
               {if (strcmp(s, "off") != 0)
                   SS_error("BAD INDEX - UL_PREFIX", arg2);}
            if ((mindex > 0) && (mindex <= UL_n_curves_read))
               fname = UL_dataset[UL_number[mindex]].file;
            SX_prefix_list[pre - 'a'] = mindex;}
        else
           {if ((mindex = SX_prefix_list[pre - 'a']) > 0)
               {if (mindex <= UL_n_curves_read)
                    fname = UL_dataset[UL_number[mindex]].file;
                if (SS_interactive == ON)
                   PRINT(stdout, " %c%6d    %s\n", pre, mindex, fname);}
            else
               {if (SS_interactive == ON)
                   PRINT(stdout, " Prefix %c is not assigned\n", pre);};}
        {SS_Assign(ret,
                   SS_mk_cons(arg1,
                              SS_mk_cons(SS_mk_integer(mindex),
                                         SS_mk_cons(SS_mk_string(fname),
                                                    ret))));};}
    else
       {for (i = 0; i < NPREFIX; i++)
            if ((mindex = SX_prefix_list[i]) > 0)
               {pre = 'a' + i;
                fname = "";
                if (mindex <= UL_n_curves_read)
                   fname = UL_dataset[UL_number[mindex]].file;
                if (SS_interactive == ON)
                   PRINT(stdout, " %c%6d    %s\n", pre, mindex, fname);
                arg1 = SS_mk_char((int) pre);
                {SS_Assign(ret,
                           SS_mk_cons(
                                      SS_mk_cons(arg1,
                                                 SS_mk_cons(SS_mk_integer(mindex),
                                                            SS_mk_cons(SS_mk_string(fname),
                                                                       SS_null))),
                                      ret));};}
        {SS_Assign(ret, SS_reverse(ret));};}

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_LIST_CURVES - display the curve list */

static object *UL_list_curves(argl)
   object *argl;
   {char *pr, *pf;
    object *ret;

    if (!SS_nullobjp(argl))
       {UL_prep_arg(argl);};

    pr = NULL;
    pf = NULL;
    SS_args(argl,
            SC_STRING_I, &pr,
            SC_STRING_I, &pf,
            0);

    ret = UL_print_labels(UL_data_index, NDISPLAY, pr, pf, 2, NULL);
    SX_plot_flag = FALSE;

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _UL_POW - a callable version of pow (why?) */

static double _UL_pow(x, a)
   double x, a;
   {return(POW(a, x));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_COMPOSE - a functional composition of curves */

static double UL_compose(a, i)
   double a;
   int i;
   {int j, n;
    REAL xta, xtb, xtc, yta, ytb, val;
    REAL *x, *y;

    n = UL_dataset[i].n;
    x = UL_dataset[i].xp;
    y = UL_dataset[i].yp;
    j = PM_find_index(x, a, n);
    if (j == 0)
       {PM_interp(val, a, x[0], y[0], x[1], y[1]);}

    else if (j == n)
       {PM_interp(val, a, x[n-2], y[n-2], x[n-1], y[n-1]);}

    else
       {PM_interp(val, a, x[j-1], y[j-1], x[j], y[j]);};

    return(val);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_DERIVATIVE - take the derivative of a list of curves */

static object *UL_derivative(j)
   int j;
   {REAL *xp, *yp;
    int n;
    object *ch;

    n  = UL_dataset[j].n;
    xp = UL_dataset[j].xp;
    yp = UL_dataset[j].yp;
    UL_buf1x = FMAKE_N(REAL, n+5, "UL_DERIVATIVE:buf1x");
    UL_buf1y = FMAKE_N(REAL, n+5, "UL_DERIVATIVE:buf1y");

    PM_derivative(n, xp, yp, UL_buf1x, UL_buf1y);

    sprintf(pbuffer, "Derivative %c", UL_dataset[j].id);

    if (n == 2)
       {n = 3;
	UL_buf1x[0] = xp[0];
	UL_buf1x[1] = xp[1];
	UL_buf1y[1] = UL_buf1y[0];};

    ch = _UL_mk_curve(n-1, UL_buf1x, UL_buf1y, pbuffer, NULL);

    SFREE(UL_buf1x);
    SFREE(UL_buf1y);      

    return(ch);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_THIN - thin a curve */

static object *UL_thin(j, argl)
   int j;
   object *argl;
   {REAL *xp, *yp;
    int n, m;
    double toler;
    char *type;
    object *ch;

    type  = NULL;
    toler = 0.02;
    SS_args(argl,
            SC_STRING_I, &type,
            SC_DOUBLE_I, &toler,
            0);

    n  = UL_dataset[j].n;
    xp = UL_dataset[j].xp;
    yp = UL_dataset[j].yp;
    UL_buf1x = FMAKE_N(REAL, n, "UL_THIN:buf1x");
    UL_buf1y = FMAKE_N(REAL, n, "UL_THIN:buf1y");

    if (strncmp(type, "int", 3) == 0)
       m = PM_thin_1d_int(n, xp, yp, UL_buf1x, UL_buf1y, toler);
    else
       m = PM_thin_1d_der(n, xp, yp, UL_buf1x, UL_buf1y, toler);

    if (m < 1)
       SS_error("THIN FAILED - UL_THIN", argl);

    sprintf(pbuffer, "Thinned %c", UL_dataset[j].id);
    ch = _UL_mk_curve(m, UL_buf1x, UL_buf1y, pbuffer, NULL);
    SFREE(UL_buf1x);
    SFREE(UL_buf1y);      

    return(ch);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_FILTER - filter out points from curve that fail domain or range test */

static object *UL_filter(j, argl)
   int j;
   object *argl;
   {int i, k, n;
    REAL *xp, *yp;
    REAL xmin, xmax, ymin, ymax;
    object *dom_pred, *ran_pred, *xexpr, *yexpr;

    dom_pred = SS_null;
    ran_pred = SS_null;

    SS_args(argl,
            SS_OBJECT_I, &dom_pred,
            SS_OBJECT_I, &ran_pred,
            0);

    if (SS_nullobjp(dom_pred))
       SS_error("BAD DOMAIN PREDICATE ARGUMENT - UL_FILTER", argl);
    if (SS_nullobjp(ran_pred))
       SS_error("BAD RANGE PREDICATE ARGUMENT - UL_FILTER", argl);

    n  = UL_dataset[j].n;
    xp = UL_dataset[j].xp;
    yp = UL_dataset[j].yp;
    UL_buf1x = FMAKE_N(REAL, n, "UL_FILTER:buf1x");
    UL_buf1y = FMAKE_N(REAL, n, "UL_FILTER:buf1y");

    xexpr = SS_null;
    yexpr = SS_null;
    for (i = 0, k = 0; i < n; i++)
        {SS_Assign(xexpr, SS_make_list(SS_OBJECT_I, dom_pred,
				       SC_REAL_I, xp + i,
				       0));
         SS_Assign(yexpr, SS_make_list(SS_OBJECT_I, ran_pred,
				       SC_REAL_I, yp + i,
				       0));
         if ((SS_true(SS_exp_eval(xexpr))) && (SS_true(SS_exp_eval(yexpr))))
            {UL_buf1x[k] = xp[i];
             UL_buf1y[k] = yp[i];
             k++;};};
    SS_Assign(xexpr, SS_null);
    SS_Assign(yexpr, SS_null);

    if (k < 2)
       {SFREE(UL_buf1x);
        SFREE(UL_buf1y);
        SS_error("FEWER THAN TWO POINTS REMAIN - UL_FILTER", SS_null);};

    UL_dataset[j].n  = k;
    UL_dataset[j].xp = UL_buf1x;
    UL_dataset[j].yp = UL_buf1y;
    PM_maxmin(UL_dataset[j].xp, &xmin, &xmax, k);
    PM_maxmin(UL_dataset[j].yp, &ymin, &ymax, k);
    UL_dataset[j].xmin = xmin;
    UL_dataset[j].xmax = xmax;
    UL_dataset[j].ymin = ymin;
    UL_dataset[j].ymax = ymax;
        
    SFREE(xp);
    SFREE(yp);

    return((object *) UL_dataset[j].obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_INTEGRATE - integrate a curve between the specified limits */

static object *UL_integrate(j, d1, d2)
   int j;
   double d1, d2;
   {int n;
    REAL *xp, *yp;
    object *ch;

    n = UL_dataset[j].n;

    UL_buf1x = FMAKE_N(REAL, n+5, "UL_INTEGRATE:buf1x");
    UL_buf1y = FMAKE_N(REAL, n+5, "UL_INTEGRATE:buf1y");

    xp = UL_dataset[j].xp;
    yp = UL_dataset[j].yp;

/* take care of some bad cases */
    if ((UL_dataset[j].xmin >= d2) || (UL_dataset[j].xmax <= d1))
       SS_error("XMIN GREATER THAN XMAX - UL_INTEGRATE", SS_null);

    PM_integrate_tzr(d1, d2, &n, xp, yp, UL_buf1x, UL_buf1y);

    sprintf(pbuffer, "Integrate %c", UL_dataset[j].id);
    ch = _UL_mk_curve(n, UL_buf1x, UL_buf1y, pbuffer, NULL);

    SFREE(UL_buf1x);
    SFREE(UL_buf1y);
        
    return(ch);}
        
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_LABEL - change the label of the given curve */

static object *UL_label(argl)
   object *argl;
   {char *labl;
    int j;

    UL_prep_arg(argl);

    j    = -1;
    labl = NULL;
    SS_args(argl,
            UL_CURVE_INDEX_I, &j,
            SC_STRING_I, &labl,
            0);

    if (j < 0)
       SS_error("BAD CURVE ARGUMENT - UL_LABEL", argl);

    if (labl == NULL)
       SS_error("BAD LABEL ARGUMENT - UL_LABEL", argl);

    SFREE(UL_dataset[j].text);

    UL_dataset[j].text     = labl;
    UL_dataset[j].modified = FALSE;

    return(SS_car(argl));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_PLOT_OFF - turn off autoplotting */

static void UL_plot_off()
   {old_autoplot = SX_autoplot;
    SX_autoplot  = OFF;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_RESTORE_PLOT - turn on autoplotting */

static void UL_restore_plot()
   {SX_autoplot = old_autoplot;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_AVERAGE - return a new curve containing the average of
 *            - the curves specified
 */

static object *UL_average(s)
   object *s;
   {int j;
    object *c, *numtoks;
    char *t;

    UL_prep_arg(s);
    UL_plot_off();

    numtoks = SS_mk_integer((long) _SS_length(s));
    c = UL_bc((PFInt) PM_fplus, s);
    if (SS_true(c))
       {c = UL_opyc((PFInt) PM_fdivide,
                    SS_make_list(SS_OBJECT_I, c,
                                 SS_OBJECT_I, numtoks,
                                 0));
        j = UL_get_curve(c);

        SC_strtok(UL_dataset[j].text, " ", t);
        sprintf(pbuffer, "Append %s", SC_strtok(NULL, "\n", t));
        SFREE(UL_dataset[j].text);

        UL_dataset[j].text = SC_strsavef(pbuffer,
                              "char*:UL_AVERAGE:pbuffer");
        UL_restore_plot();

        return(c);}

    else
       {return(SS_null);};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_SYSTEM - exec a process */

static object *UL_system(s)
   object *s;
   {char local[MAXLINE];
   
    strcpy(local, UL_get_string(s));
    SYSTEM(local);

    UL_pause(FALSE);

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_XMM - extract a portion of the given curve */

static object *UL_xmm(j, d1, d2)
   int j;
   double d1, d2;
   {REAL xta, xtb, xtc, yta, ytb;
    REAL *xp, *yp, *xrev, *yrev, tempx, tempy;
    int i = 0, l, k = 0, n, decreasing = FALSE;
    object *ch;

    UL_buf1x = FMAKE_N(REAL, UL_dataset[j].n+1, "UL_XMM:buf1x");
    UL_buf1y = FMAKE_N(REAL, UL_dataset[j].n+1, "UL_XMM:buf1y");
    xp       = UL_dataset[j].xp;
    yp       = UL_dataset[j].yp;

/* take care of dumb case */
    if ((UL_dataset[j].xmin >= d2) || (UL_dataset[j].xmax <= d1))
       SS_error("XMIN GREATER THAN XMAX - UL_XMM", SS_null);

/* check to see if x is decreasing */
    if (xp[0] > xp[1])
       {xrev = FMAKE_N(REAL, UL_dataset[j].n, "UL_XMM:xrev");
        yrev = FMAKE_N(REAL, UL_dataset[j].n, "UL_XMM:yrev");
        n    = UL_dataset[j].n;
        for (l = 0; l < n; l++)
            {xrev[l] = xp[n-(l+1)];
             yrev[l] = yp[n-(l+1)];};
        xp   = xrev;
        yp   = yrev;
        decreasing = TRUE;}

/* first point */
    if (d1 < *xp)
       {UL_buf1x[k] = *xp;
        UL_buf1y[k++] = *yp;
        i++;}
    else
       {while ((xp[i] <= d1) && (i < UL_dataset[j].n))
           ++i;
        UL_buf1x[k] = d1;
        PM_interp(UL_buf1y[k++], d1, xp[i-1], yp[i-1], xp[i], yp[i]);};

/* all the rest */
    while ((i < UL_dataset[j].n) && (xp[i] < d2))
       {UL_buf1x[k] = xp[i];
        UL_buf1y[k++] = yp[i];
        i++;};

    if ((d2 <= xp[i]) && (i < UL_dataset[j].n))
       {UL_buf1x[k] = d2;
        PM_interp(UL_buf1y[k++], d2, xp[i-1], yp[i-1], xp[i], yp[i]);};

    sprintf(pbuffer, "Extract %c", UL_dataset[j].id);

/* reverse points if decreasing */
    if (decreasing)
       {for (l = 0; l < k/2; l++)
            {tempx           = UL_buf1x[l];
             tempy           = UL_buf1y[l];
             UL_buf1x[l]     = UL_buf1x[k-l-1];
             UL_buf1y[l]     = UL_buf1y[k-l-1];
             UL_buf1x[k-l-1] = tempx;
             UL_buf1y[k-l-1] = tempy;};}
             

    ch = _UL_mk_curve(k, UL_buf1x, UL_buf1y, pbuffer, NULL);

    SFREE(UL_buf1x);
    SFREE(UL_buf1y);      

    if (decreasing)
       {SFREE(xrev);
        SFREE(yrev);}

    return(ch);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _UL_FILTER_COEFF - worker for coefficient based filters */

static void _UL_filter_coeff(l, arr, ntimes)
   int l;
   C_array *arr;
   int ntimes;
   {int i, n, ne, nc, nh, ne0;
    REAL *coeff;

    if (arr != NULL)
       {ne    = arr->length;
	coeff = NULL;
	CONVERT(SC_REAL_S, &coeff, arr->type, arr->data, ne, FALSE);
        ne--;

        nc  = coeff[0];
        nh  = nc >> 1;
        ne0 = nc + nh*(3*nh + 1);
        if (ne != ne0)
           SS_error("INCORRECT FILTER SIZE - _UL_FILTER_COEF", SS_null);}

    else
       SS_error("BAD COEFFICIENT ARRAY - _UL_FILTER_COEF", SS_null);

    n = UL_dataset[l].n;

    for (i = 0; i < ntimes; i++)
        PM_filter_coeff(UL_dataset[l].yp, n, coeff + 1, nc);

    SFREE(coeff);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_FILTER_COEF - the n point filter routine */

static object *UL_filter_coef(l, argl)
   int l;
   object *argl;
   {int ntimes;
    C_array *arr;

    arr    = NULL;
    ntimes = 1;
    SS_args(argl,
            G_NUM_ARRAY, &arr,
	    SC_INTEGER_I, &ntimes,
	    0);

    _UL_filter_coeff(l, arr, ntimes);

    UL_lmt(UL_dataset[l].yp, UL_dataset[l].n,
	   &UL_dataset[l].ymin, &UL_dataset[l].ymax);

    return((object *) UL_dataset[l].obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_SMOOTH - the n point smoothing routine */

static object *UL_smooth(l, argl)
   int l;
   object *argl;
   {object *obj;
    char bf[MAXLINE];
    int i, pts;
    int n, ntimes;

    pts = 3;
    ntimes = 1;

    SS_args(argl,
	    SC_INTEGER_I, &pts,
	    SC_INTEGER_I, &ntimes,
	    0);

    n = UL_dataset[l].n;

    if (SC_str_icmp(SX_smooth_method, "fft") == 0)

/* in the future let the user specify the filter function */
       {for (i = 0; i < ntimes; i++)
            PM_smooth_fft(UL_dataset[l].xp, UL_dataset[l].yp,
                          n, pts, PM_smooth_filter);}

    else if (SC_str_icmp(SX_smooth_method, "averaging") == 0)
       {for (i = 0; i < ntimes; i++)
            PM_smooth_int_ave(UL_dataset[l].xp, UL_dataset[l].yp,
                              n, pts);}

    else
       {C_array *arr;

        obj = SS_INQUIRE_OBJECT(SX_smooth_method);
        if (obj == NULL)
           {sprintf(bf, "NO FILTER NAMED %s EXISTS - UL_SMOOTH",
		    SX_smooth_method);
	    SS_error(bf, SS_null);};

        SS_args(SS_lk_var_val(obj, SS_Env),
                G_NUM_ARRAY, &arr,
		0);

        if (arr == NULL)
           {sprintf(bf, "%s IS NOT A FILTER - UL_SMOOTH",
		    SX_smooth_method);
	    SS_error(bf, SS_null);};

	_UL_filter_coeff(l, arr, ntimes);};

    UL_lmt(UL_dataset[l].xp, n, &UL_dataset[l].xmin, &UL_dataset[l].xmax);
    UL_lmt(UL_dataset[l].yp, n, &UL_dataset[l].ymin, &UL_dataset[l].ymax);

    return((object *) UL_dataset[l].obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_REVERSE - interchange the x and y data for the given curve */

static object *UL_reverse(j)
   int j;
   {REAL *temp;
    REAL treal;

    temp = UL_dataset[j].xp;
    UL_dataset[j].xp = UL_dataset[j].yp;
    UL_dataset[j].yp = temp;

    treal = UL_dataset[j].xmin;                  /* switch limits */
    UL_dataset[j].xmin = UL_dataset[j].ymin;
    UL_dataset[j].ymin = treal;

    treal = UL_dataset[j].xmax;
    UL_dataset[j].xmax = UL_dataset[j].ymax;
    UL_dataset[j].ymax = treal;

    return((object *) UL_dataset[j].obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_SORT - order the points in a curve */

object *UL_sort(k)
   int k;
   {PM_val_sort(UL_dataset[k].n, UL_dataset[k].xp, UL_dataset[k].yp);

    return((object *) UL_dataset[k].obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_SMP_APPEND - simple append only concatenates curves it does
 *               - not concern itself with overlaps
 */

static object *UL_smp_append(a, b)
   object *a, *b;
   {object *c;
    int i, j, n, na, nb, l;
    REAL *xp, *yp;

    i = UL_get_curve(a);
    j = UL_get_curve(b);

    na = UL_dataset[i].n;
    nb = UL_dataset[j].n;
    n  = na + nb;

    UL_buf1x = FMAKE_N(REAL, n, "UL_SMP_APPEND:buf1x");
    UL_buf1y = FMAKE_N(REAL, n, "UL_SMP_APPEND:buf1y");

/* insert the first curve */
    xp = UL_dataset[i].xp;
    yp = UL_dataset[i].yp;
    for (l = 0, n = 0; l < na; l++, n++)
        {UL_buf1x[n] = *xp++;
	 UL_buf1y[n] = *yp++;};

/* insert the second curve */
    xp = UL_dataset[j].xp;
    yp = UL_dataset[j].yp;
    for (l = 0; l < nb; l++, n++)
        {UL_buf1x[n] = *xp++;
	 UL_buf1y[n] = *yp++;};

    sprintf(pbuffer, "Append %c %c", UL_dataset[i].id, UL_dataset[j].id);
    c = _UL_mk_curve(n, UL_buf1x, UL_buf1y, pbuffer, NULL);

    SFREE(UL_buf1x);
    SFREE(UL_buf1y);      

    return(c);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_PR_APPEND - primitive append is the binary append operation
 *              - it appends two curves together and returns a
 *              - newly created curve
 */

static object *UL_pr_append(a, b)
   object *a, *b;
   {object *c, *tmp;
    int i, j, k, n, na, nb, nc, l;
    REAL xmn, xmx, xv, yv;
    REAL *xp, *yp;

    tmp = SS_make_list(SS_OBJECT_I, a,
                       SS_OBJECT_I, b,
                       0);
    SS_MARK(tmp);
    c = UL_average(tmp);
    SS_GC(tmp);

/* no overlap of curves */
    if (!UL_curvep_a(c))
       c = UL_COPY_CURVE(a);

    i = UL_get_curve(a);
    j = UL_get_curve(b);
    k = UL_get_curve(c);

    na  = UL_dataset[i].n;
    nb  = UL_dataset[j].n;
    nc  = UL_dataset[k].n;

    xmn = UL_dataset[k].xmin;
    xmx = UL_dataset[k].xmax;

    n = na + nb + nc + 10;
    UL_buf1x = FMAKE_N(REAL, n, "UL_PR_APPEND:buf1x");
    UL_buf1y = FMAKE_N(REAL, n, "UL_PR_APPEND:buf1y");

/* insert region of A less than averaged region */
    xp = UL_dataset[i].xp;
    yp = UL_dataset[i].yp;
    for (n = 0; *xp < xmn; n++)
        {UL_buf1x[n] = *xp++;
	 UL_buf1y[n] = *yp++;};

/* insert region of B less than averaged region */
    xp = UL_dataset[j].xp;
    yp = UL_dataset[j].yp;
    for (; *xp < xmn; n++)
        {UL_buf1x[n] = *xp++;
	 UL_buf1y[n] = *yp++;};

/* insert averaged region */
    xp = UL_dataset[k].xp;
    yp = UL_dataset[k].yp;
    for (l = 0; l < nc; l++, n++)
        {UL_buf1x[n] = *xp++;
         UL_buf1y[n] = *yp++;};

/* insert region of A greater than averaged region */
    xp = UL_dataset[i].xp;
    yp = UL_dataset[i].yp;
    for (l = 0; l < na; l++)
        {xv = *xp++;
         yv = *yp++;
	 if (xv > xmx)
            {UL_buf1x[n] = xv;
             UL_buf1y[n] = yv;
             n++;};};

/* insert region of B greater than averaged region */
    xp = UL_dataset[j].xp;
    yp = UL_dataset[j].yp;
    for (l = 0; l < nb; l++)
        {xv = *xp++;
         yv = *yp++;
	 if (xv > xmx)
            {UL_buf1x[n] = xv;
             UL_buf1y[n] = yv;
             n++;};};

    UL_delete(c);
    sprintf(pbuffer, "Append %c %c", UL_dataset[i].id, UL_dataset[j].id);
    c = _UL_mk_curve(n, UL_buf1x, UL_buf1y, pbuffer, NULL);

    SFREE(UL_buf1x);
    SFREE(UL_buf1y);      

    return(c);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_APPEND - sticks curves together recursively
 *           - where they overlap, it averages otherwise takes from
 *           - either curve
 */

static object *UL_append(argl)
   object *argl;
   {object *b, *acc, *target, *tmp;
    int j;
    char local[MAXLINE];

    UL_prep_arg(argl);
    SX_autoplot = OFF;

    if (SS_nullobjp(argl))
       {SX_autoplot = ON;
        return(SS_null);};

/* get out the first curve and make two copies:
 * the first for the ultimate return curve; and
 * the second so that later deletions don't kill this first curve.
 */
    acc = SS_car(argl);
    if (!UL_curvep_a(acc))
       SS_error("BAD FIRST CURVE -  UL_APPEND", acc);
    strcpy(local, "Append");
    target = UL_COPY_CURVE(acc);
    acc = UL_COPY_CURVE(acc);

    for ( ; SS_consp(argl); argl = SS_cdr(argl))
        {b = SS_car(argl);
         sprintf(local, "%s %c", local, UL_dataset[UL_get_curve(b)].id);

         if (UL_curvep_a(b))
            {if (UL_simple_append)
	        tmp = UL_smp_append(acc, b);
             else
	        tmp = UL_pr_append(acc, b);
             UL_delete(acc);
             acc = tmp;};};

    UL_delete(target);
    target = UL_COPY_CURVE(acc);
    UL_delete(acc);

    j = UL_get_curve(target);
    SFREE(UL_dataset[j].text);
    UL_dataset[j].text = SC_strsavef(local, "char*:UL_APPEND:local");

    SX_autoplot = ON;

    return(target);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_HIDE - suppress (hide) the plotting of the given curve */

static object *UL_hide(j)
   int j;
   {

    SC_CHANGE_VALUE_ALIST(UL_dataset[j].info, int, SC_INTEGER_P_S,
			  "LINE-COLOR", -1);

    return((object *) UL_dataset[j].obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_SHOW - undo a HIDE operation on the given curve */

static object *UL_show(j)
   int j;
   {

    SC_CHANGE_VALUE_ALIST(UL_dataset[j].info, int, SC_INTEGER_P_S,
			  "LINE-COLOR",
			  _SX_next_color(UL_graphics_device));

    return((object *) UL_dataset[j].obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _UL_MAKE_LN - given a slope, intercept, xmin, xmax, and number of points
 *             - return a new curve
 */

object *_UL_make_ln(slope, interc, first, last, n)
   double slope, interc, first, last;
   int n;
   {REAL *xp, *yp;
    REAL step, x;
    object *ch;
    int i;
   
    step = (last - first)/(n-1);
    UL_buf1x = FMAKE_N(REAL, n, "_UL_MAKE_LN:buf1x");
    UL_buf1y = FMAKE_N(REAL, n, "_UL_MAKE_LN:buf1y");
    xp = UL_buf1x;
    yp = UL_buf1y;
    for (x = first, i = 0; i < n; i++, x += step, xp++, yp++)
        {*xp = x;
         *yp = slope*x + interc;};

/* without the next two lines, the final point isn't exactly at last */
    *(xp - 1) = last;
    *(yp - 1) = slope*last + interc;

    ch = _UL_mk_curve(i, UL_buf1x, UL_buf1y, "Straight line", NULL);
    SFREE(UL_buf1x);
    SFREE(UL_buf1y);      

    return(ch);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_MAKE_LN - draw a line with given slope and intercept */

static object *UL_make_ln(argl)
   object *argl;
   {REAL first, last, slope, interc;
    int n;

    slope  = 1.0;
    interc = 0.0;
    first  = 0.0;
    last   = 1.0;
    n      = UL_default_npts;
    SS_args(argl,
            SC_REAL_I, &slope,
            SC_REAL_I, &interc,
            SC_REAL_I, &first,
            SC_REAL_I, &last,
            SC_INTEGER_I, &n,
            0);

    return(_UL_make_ln(slope, interc, first, last, n));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_MK_CURVE - make an ULTRA curve out of two lists of numbers */

static object *UL_mk_curve(argl)
   object *argl;
   {REAL *xp, *yp;
    object *x, *y, *xvals, *yvals;
    int n;
    object *ch;
    char *labls;

    labls = "Curve";
    SS_args(argl,
            SS_OBJECT_I, &xvals,
            SS_OBJECT_I, &yvals,
            SC_STRING_I, &labls,
            0);

    if (!SS_consp(xvals))
       SS_error("BAD LIST OF X-VALUES - UL_MK_CURVE", xvals);

    if (!SS_consp(yvals))
       SS_error("BAD LIST OF Y-VALUES - UL_MK_CURVE", yvals);

    n = min(_SS_length(xvals), _SS_length(yvals));

    UL_buf1x = FMAKE_N(REAL, n, "UL_MK_CURVE:buf1x");
    UL_buf1y = FMAKE_N(REAL, n, "UL_MK_CURVE:buf1y");
    for (xp = UL_buf1x, yp = UL_buf1y;
         SS_consp(xvals) && SS_consp(yvals);
         xvals = SS_cdr(xvals), yvals = SS_cdr(yvals))
        {x = SS_car(xvals);
         if (!_SS_numberp(x))
            SS_error("BAD X-VALUE - UL_MK_CURVE", x);
         if (SS_integerp(x))
            *xp++ = SS_INTEGER_VALUE(x);
         else
            *xp++ = SS_FLOAT_VALUE(x);

         y = SS_car(yvals);
         if (!_SS_numberp(y))
            SS_error("BAD Y-VALUE - UL_MK_CURVE", y);
         if (SS_integerp(y))
            *yp++ = SS_INTEGER_VALUE(y);
         else
            *yp++ = SS_FLOAT_VALUE(y);};

    ch = _UL_mk_curve(n, UL_buf1x, UL_buf1y, labls, NULL);

    SFREE(UL_buf1x);
    SFREE(UL_buf1y);

    return(ch);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_CURVE_LIST - make a list of the list of x values and the list of 
 *               - y values from a given ULTRA curve
 */

static object *UL_curve_list(arg)
   object *arg;
   {int i, j, n;
    REAL *xp, *yp;
    object *xvals, *yvals;

    j = UL_get_curve(arg);
    if (j < 0)
       SS_error("BAD CURVE - UL_CURVE_LIST", arg);

    n  = UL_dataset[j].n;
    xp = UL_dataset[j].xp;
    yp = UL_dataset[j].yp;
    xvals = SS_null;
    yvals = SS_null;
    for (i = 0; i < n; i++)
        {xvals = SS_mk_cons(SS_mk_float(xp[i]), xvals);
         yvals = SS_mk_cons(SS_mk_float(yp[i]), yvals);};

    return(SS_make_list(SS_OBJECT_I, SS_reverse(xvals),
                        SS_OBJECT_I, SS_reverse(yvals),
                        0));}

/*--------------------------------------------------------------------------*/

/*                            INSTALL FUNCTIONS                             */

/*--------------------------------------------------------------------------*/

/* UL_INSTALL_SCHEME_FUNCS - install some Ultra functions directly in the 
 *                         - Scheme hash table
 */

void UL_install_scheme_funcs()
   {SX_install_global_funcs();

    SS_install("curve?",
               "Prodedure: Return #t iff the argument is an Ultra curve object\n     Usage: curve? <object>",
               SS_sargs,
               UL_curveobjp, SS_PR_PROC);

    SS_install("curve->list",
               "Prodedure: Given a curve return a list of the lists of x and y values\n     Usage: curve->list <curve>",
               SS_sargs,
               UL_curve_list, SS_PR_PROC);

    SS_install("menu",
               "Macro: List the available curves\n     Usage: menu [<label-pattern> [<file-pattern>]]",
               SS_znargs,
               UL_menu, SS_UR_MACRO);

    SS_install("menu*",
               "Procedure: List the available curves\n     Usage: menu* [<label-pattern> [<file-pattern>]]",
               SS_znargs,
               UL_menu, SS_PR_PROC);

    SS_install("lst",
               "Macro: Display curves in list\n     Usage: lst [<label-pattern> [<file-pattern>]]",
               SS_znargs,
               UL_list_curves, SS_UR_MACRO);

    SS_install("lst*",
               "Procedure: Display curves in list\n     Usage: lst* [<label-pattern> [<file-pattern>]]",
               SS_znargs,
               UL_list_curves, SS_PR_PROC);

    SS_install("prefix",
               "Macro: List or set menu prefixes\n     Usage: prefix [a | b | ... | z [<menu-index> | off]]",
               SS_znargs,
               UL_prefix, SS_UR_MACRO);

    SS_install("prefix*",
               "Procedure: List or set menu prefixes\n     Usage: prefix* [a | b | ... | z [<menu-index> | off]]",
               SS_znargs,
               UL_prefix, SS_PR_PROC);

    SS_install("thru",
               "Procedure: Return an expanded list of curves\n     Usage: thru <first-curve> <last-curve>",
               SS_nargs,
               UL_thru, SS_PR_PROC);

    SS_install("ifft",
               "Procedure: Compute Inverse FFT. Return real and imaginary parts.\n     Usage: ifft <real-curve> <imaginary-curve>",
               SS_nargs,
               UL_ifft, SS_PR_PROC);

    SS_install("cfft",
               "Procedure: Compute Complex FFT. Return real and imaginary parts.\n     Usage: cfft <real-curve> <imaginary-curve>",
               SS_nargs,
               UL_cfft, SS_PR_PROC);

    SS_install("open-device*",
               "Procedure: Open graphics display device\n     Usage: open-device* <name> <type> <title>",
               SS_nargs,
               UL_open_device, SS_PR_PROC);

    SS_install("open-device",
               "Macro: Open graphics device\n     Usage: open-device <name> <type> <title>",
               SS_nargs,
               UL_open_device, SS_UR_MACRO);

    SS_install("close-device*",
#ifdef HAVE_JPEGLIB
               "Procedure: Close graphics device\n     Usage: close-device* ps | cgm | jpeg | <name>",
#else
               "Procedure: Close graphics device\n     Usage: close-device* ps | cgm | <name>",
#endif
               SS_sargs,
               UL_close_device, SS_PR_PROC);

    SS_install("close-device",
#ifdef HAVE_JPEGLIB
               "Macro: Close graphics device\n     Usage: close-device ps | cgm | jpeg | <name>",
#else
               "Macro: Close graphics device\n     Usage: close-device ps | cgm | <name>",
#endif
               SS_sargs,
               UL_close_device, SS_UR_MACRO);

    SS_install("domain",
               "Procedure: Set the domain for plotting\n     Usage: domain <low-lim> <high-lim> or\n     Usage: domain de",
               SS_znargs,
               UL_domain, SS_PR_PROC);

    SS_install("range",
               "Procedure: Set the range for plotting\n     Usage: range <low-lim> <high-lim> or\n     Usage: range de",
               SS_znargs,
               UL_range, SS_PR_PROC);

    SS_install("label",
               "Procedure: Change the label displayed for a curve by lst command\n     Usage: label <curve> <new-label>",
               SS_nargs,
               UL_label, SS_PR_PROC);

    SS_install("set-id",
               "Procedure: Change the data-id of a curve\n     Usage: set-id <curve> a | b | ... | z",
               SS_nargs,
               UL_set_id, SS_PR_PROC);

/* these are the no argument functions */
    SS_install("replot",
               "Macro: Replot curves in list\n     Usage: replot",
               SS_zargs,
               UL_plot, SS_UR_MACRO);

    SS_install("end",
               "Macro: Exit ULTRA with optional exit status value\n     Usage: end [<integer>]",
               SS_znargs,
               UL_quit, SS_PR_PROC);

    SS_install("erase",
               "Macro: Erases all curves on the screen but leaves the limits untouched\n     Usage: erase",
               SS_zargs,
               UL_erase, SS_UR_MACRO);

    SS_install("extract-curve",
               "Procedure: Extracts a curve with the specified points\n     Usage: extract-curve <curve> <x-start> <x-stop> <x-step>",
               SS_nargs,
               UL_extract_curve, SS_PR_PROC);

    SS_install("text",
               "Macro: Enter text mode - no graphics are available\n     See graphics\n     Usage: text",
               SS_zargs,
               UL_mode_text, SS_UR_MACRO);

    SS_install("graphics",
               "Macro: Enter graphics mode\n     To enter text mode type: text\n     Usage: graphics",
               SS_zargs,
               UL_mode_graphics, SS_UR_MACRO);

    SS_install("print",
               "Macro: Print the current plot.\n     Usage: print",
               SS_zargs,
               UL_printscr, SS_UR_MACRO);

/* these are the former Ultra macros */
    SS_install("fitcurve",
               "Procedure: Fit first curve to other curves\n     Usage: fitcurve <curve to fit> <curve-list for fit>",
               SS_nargs,
               UL_fit_curve, SS_PR_PROC);

    SS_install("stats",
               "Procedure: Calculate mean and standard deviation for curves\n     Usage: stats <curve-list>",
               SS_nargs,
               UL_stats, SS_PR_PROC);

    SS_install("kill",
               "Procedure: Delete entries from the menu.\n     Usage: kill <number-list> | all",
               SS_nargs,
               UL_expunge_macro, SS_PR_PROC);

    SS_install("rd",
               "Macro: Read curve data file\n     Usage: rd <file-name>",
               SS_sargs,
               UL_read_data, SS_UR_MACRO);

    SS_install("rd1",
               "Macro: Read ULTRA I curve data file\n     Usage: rd1 <file-name>",
               SS_sargs,
               UL_read_ver1, SS_UR_MACRO);

    SS_install("ultra-file?",
               "Procedure: Return #t iff the file is a valid ULTRA II file\n     Usage: ultra-file? <file-name>",
               SS_sargs,
               UL_valid_ultra_filep, SS_PR_PROC);

    SS_install("open",
               "Procedure: Read curve data file\n     Usage: open <file-name>",
               SS_sargs,
               UL_read_data, SS_PR_PROC);

    SS_install("open1",
               "Procedure: Read ULTRA I curve data file\n     Usage: open1 <file-name>",
               SS_sargs,
               UL_read_ver1, SS_PR_PROC);

    SS_install("wrt",
               "Procedure: Save curves to file\n     Usage: wrt [<type>] <file-name> <curve-list>",
               SS_nargs,
               UL_write_data, SS_PR_PROC);

    SS_install("save",
               "Macro: Save curves to file\n     Usage: save [<type>] <file-name> <curve-lst>",
               SS_nargs,
               UL_write_data, SS_UR_MACRO);

    SS_install("average",
               "Procedure: Average curves\n     Usage: average <curve-list>",
               SS_nargs,
               UL_average, SS_PR_PROC);

    SS_install("system",
               "Procedure: Pass command to the operating system\n     Usage: system <unix-command>",
               SS_sargs,
               UL_system, SS_PR_PROC);

    SS_install("append-curves",
               "Procedure: Merge curves over union of their domains \n     Where they overlap, take the average\n     Usage: append <curve-list>",
               SS_nargs,
               UL_append, SS_PR_PROC);

    SS_install("describe*",
               "Procedure: Describe an Ultra function or variable\n     Usage: describe* <function-list> <variable-list>",
               SS_nargs,
               UL_describe, SS_PR_PROC);

    SS_install("describe",
               "Macro: Describe an Ultra function or variable\n     Usage: describe <function-list> <variable-list>",
               SS_nargs,
               UL_describe, SS_UR_MACRO);

    SS_install("apropos",
               "Macro: List all functions or variables containing the designated substring\n     Usage: apropos <string>",
               SS_sargs,
               UL_apropos, SS_UR_MACRO);

    SS_install("make-curve*",
               "Procedure: Make an Ultra curve from two lists of numbers\n     Usage: make-curve* '(<list of x-values>) '(<list of y-values>)",
               SS_nargs,
               UL_mk_curve, SS_PR_PROC);

    SS_install("make-curve",
               "Macro: Make an Ultra curve from two lists of numbers\n     Usage: make-curve (<list of x-values>) (<list of y-values>)",
               SS_nargs,
               UL_mk_curve, SS_UR_MACRO);

    SS_install("make-filter",
               "Procedure: make an array of filter coefficients\n     Usage: make-filter <val1> ...",
               SS_nargs,
               SX_list_array, SS_PR_PROC);

    SS_install("line",
               "Procedure: Generate a curve with y = mx + b\n     Usage: line <m> <b> <low-lim> <high-lim> [<no.-pts>]",
               SS_nargs,
               UL_make_ln, SS_PR_PROC);

    SS_install("read-table*",
               "Procedure: Read nth table that starts at or after specified line in ASCII file\n     Usage: read-table* <file> [<n> [<label-line> [<line>]]]",
               SS_nargs,
               UL_read_ascii_table, SS_PR_PROC);

    SS_install("read-table",
               "Macro: Read nth table that starts at or after specified line in ASCII file\n     Usage: read-table <file> [<n> [<label-line> [<line>]]]",
               SS_nargs,
               UL_read_ascii_table, SS_UR_MACRO);

    SS_install("table-curve",
               "Procedure: Extract curve from current table\n     Usage: table-curve <num-points> [<y-off> [<y-stride> [<x-off> [<x-stride>]]]]",
               SS_nargs,
               UL_table_curve, SS_PR_PROC);

    SS_install("table-attributes",
               "Procedure: Return the attributes of the current table\n     Usage: table-attributes",
               SS_zargs,
               UL_table_attr, SS_PR_PROC);

    SS_install("file-info",
               "Macro: Print the descriptive information for an ULTRA file\n     Usage: file-info <name>",
               SS_sargs,
               UL_file_info, SS_UR_MACRO);

    SS_install("file-info*",
               "Procedure: Print the descriptive information for an ULTRA file\n     Usage: file-info* <name>",
               SS_sargs,
               UL_file_info, SS_PR_PROC);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* INSTALL_FUNCS - install Ultra functions in the Ultra hash table */

void UL_install_funcs()
   {UL_install_aux_funcs();

/* US handled functions */

    SS_install_cf("select",
                  "Procedure: Select curves from the menu for plotting\n     Usage: select <list-of-menu-numbers>",
                  UL_us, 
                  UL_select);
    SS_install_cf("menui",
                  "Procedure: List selected curves\n     Usage: menui <list-of-menu-numbers>",
                  UL_us, 
                  UL_menui);
    SS_install_cf("del",
                  "Procedure: Delete curves from list\n     Usage: del <curve-list>",
                  UL_us, 
                  UL_delete);

/* OPXC, OPYC handled functions */

    SS_install_cf("dx",
                  "Procedure: Shift x values of curves by a constant\n     Usage: dx <curve-list> <value>",
                  UL_opxc,
                  PM_fplus);
    SS_install_cf("dy",
                  "Procedure: Shift y values of curves by a constant\n     Usage: dy <curve-list> <value>",
                  UL_opyc, 
                  PM_fplus);
    SS_install_cf("mx",
                  "Procedure: Scale x values of curves by a constant\n     Usage: mx <curve-list> <value>",
                  UL_opxc, 
                  PM_ftimes);
    SS_install_cf("my",
                  "Procedure: Scale y values of curves by a constant\n     Usage: my <curve-list> <value>",
                  UL_opyc, 
                  PM_ftimes);
    SS_install_cf("divx",
                  "Procedure: Divide x values of curves by a constant\n     Usage: divx <curve-list> <value>",
                  UL_opxc, 
                  PM_fdivide);
    SS_install_cf("divy",
                  "Procedure: Divide y values of curves by a constant\n     Usage: divy <curve-list> <value>",
                  UL_opyc, 
                  PM_fdivide);
    SS_install_cf("powr",
                  "Procedure: Raise y values of curves to a power, y=y^a\n     Usage: powr <curve-list> <a>",
                  UL_opyc, 
                  POW);
    SS_install_cf("powrx",
                  "Procedure: Raise x values of curves to a power, x=x^a\n     Usage: powrx <curve-list> <a>",
                  UL_opxc, 
                  POW);
    SS_install_cf("powa",
                  "Procedure: Raise a to the power of the y values of curves, y=a^y\n     Usage: powa <curve-list> <a>",
                  UL_opyc, 
                  _UL_pow);
    SS_install_cf("powax",
                  "Procedure: Raise a to the power of the x values of curves, x=a^x\n     Usage: powax <curve-list> <a>",
                  UL_opxc, 
                  _UL_pow);

/* UOPXC, UOPYC handled functions */

    SS_install_cf("ln",
                  "Procedure: Take natural log of y values of curves\n     Usage: ln <curve-list>",
                  UL_uopyc, 
                  PM_ln);
    SS_install_cf("lnx",
                  "Procedure: Take natural log of x values of curves\n     Usage: lnx <curve-list>",
                  UL_uopxc, 
                  PM_ln);
    SS_install_cf("log10",
                  "Procedure: Take base 10 log of y values of curves\n     Usage: log10 <curve-list>",
                  UL_uopyc, 
                  PM_log);
    SS_install_cf("log10x",
                  "Procedure: Take base 10 log of x values of curves\n     Usage: log10x <curve-list>",
                  UL_uopxc, 
                  PM_log);
    SS_install_cf("exp",
                  "Procedure: Take exponential of y values of curves, y=e^y\n     Usage: exp <curve-list>",
                  UL_uopyc, 
                  exp);
    SS_install_cf("expx",
                  "Procedure: Take exponential of x values of curves, x=e^x\n     Usage: expx <curve-list>",
                  UL_uopxc, 
                  exp);
    SS_install_cf("sqrt",
                  "Procedure: Take square root of y values of curves\n     Usage: sqrt <curve-list>",
                  UL_uopyc, 
                  PM_sqrt);
    SS_install_cf("sqrtx",
                  "Procedure: Take square root of x values of curves\n     Usage: sqrtx <curve-list>",
                  UL_uopxc, 
                  PM_sqrt);
    SS_install_cf("sqr",
                  "Procedure: Square y values of curves\n     Usage: sqr <curve-list>",
                  UL_uopyc, 
                  PM_sqr);
    SS_install_cf("sqrx",
                  "Procedure: Square x values of curves\n     Usage: sqrx <curve-list>",
                  UL_uopxc, 
                  PM_sqr);
    SS_install_cf("abs",
                  "Procedure: Take absolute value of y values of curves\n     Usage: abs <curve-list>",
                  UL_uopyc, 
                  ABS);
    SS_install_cf("absx",
                  "Procedure: Take absolute value of x values of curves\n     Usage: absx <curve-list>",
                  UL_uopxc, 
                  ABS);
    SS_install_cf("recip",
                  "Procedure: Take reciprocal of y values of curves\n     Usage: recip <curve-list>",
                  UL_uopyc, 
                  PM_recip);
    SS_install_cf("recipx",
                  "Procedure: Take reciprocal of x values of curves\n     Usage: recipx <curve-list>",
                  UL_uopxc, 
                  PM_recip);
    SS_install_cf("random",
                  "Procedure: Generate random y values between -1 and 1 for curves\n     Usage: random <curve-list>",
                  UL_uopyc, 
                  PM_random);
    SS_install_cf("cos",
                  "Procedure: Take cosine of y values of curves\n     Usage: cos <curve-list>",
                  UL_uopyc, 
                  cos);
    SS_install_cf("cosx",
                  "Procedure: Take cosine of x values of curves\n     Usage: cosx <curve-list>",
                  UL_uopxc, 
                  cos);
    SS_install_cf("acos",
                  "Procedure: Take ArcCos of y values of curves\n     Usage: acos <curve-list>",
                  UL_uopyc, 
                  acos);
    SS_install_cf("acosx",
                  "Procedure: Take ArcCos of x values of curves\n     Usage: acosx <curve-list>",
                  UL_uopxc, 
                  acos);
    SS_install_cf("cosh",
                  "Procedure: Take hyperbolic cosine of y values of curves\n     Usage: cosh <curve-list>",
                  UL_uopyc, 
                  cosh);
    SS_install_cf("coshx",
                  "Procedure: Take hyperbolic cosine of x values of curves\n     Usage: coshx <curve-list>",
                  UL_uopxc, 
                  cosh);
    SS_install_cf("sin",
                  "Procedure: Take sine of y values of curves\n     Usage: sin <curve-list>",
                  UL_uopyc, 
                  sin);
    SS_install_cf("sinx",
                  "Procedure: Take sine of x values of curves\n     Usage: sinx <curve-list>",
                  UL_uopxc, 
                  sin);
    SS_install_cf("asin",
                  "Procedure: Take ArcSin of y values of curves\n     Usage: asin <curve-list>",
                  UL_uopyc, 
                  asin);
    SS_install_cf("asinx",
                  "Procedure: Take ArcSin of x values of curves\n     Usage: asinx <curve-list>",
                  UL_uopxc, 
                  asin);
    SS_install_cf("sinh",
                  "Procedure: Take hyperbolic sine of y values of curves\n     Usage: sinh <curve-list>",
                  UL_uopyc, 
                  sinh);
    SS_install_cf("sinhx",
                  "Take hyperbolic sine of x values of curves\n     Usage: sinhx <curve-list>",
                  UL_uopxc, 
                  sinh);
    SS_install_cf("tan",
                  "Procedure: Take tangent of y values of curves\n     Usage: tan <curve-list>",
                  UL_uopyc, 
                  tan);
    SS_install_cf("tanx",
                  "Procedure: Take tangent of x values of curves\n     Usage: tanx <curve-list>",
                  UL_uopxc, 
                  tan);
    SS_install_cf("atan",
                  "Procedure: Take ArcTan of y values of curves\n     Usage: atan <curve-list>",
                  UL_uopyc, 
                  atan);
    SS_install_cf("atanx",
                  "Procedure: Take ArcTan of x values of curves\n     Usage: atanx <curve-list>",
                  UL_uopxc, 
                  atan);
    SS_install_cf("tanh",
                  "Procedure: Take hyperbolic tangent of y values of curves\n     Usage: tanh <curve-list>",
                  UL_uopyc, 
                  tanh);
    SS_install_cf("tanhx",
                  "Procedure: Take hyperbolic tangent of x values of curves\n     Usage: tanhx <curve-list>",
                  UL_uopxc, 
                  tanh);
    SS_install_cf("j0",
                  "Procedure: Take zeroth order Bessel function of the first kind of y values of curves\n     Usage: j0 <curve-list>",
                  UL_uopyc, 
                  PM_j0);
    SS_install_cf("j0x",
                  "Procedure: Take zeroth order Bessel function of the first kind of x values of curves\n     Usage: j0x <curve-list>",
                  UL_uopxc, 
                  PM_j0);
    SS_install_cf("j1",
                  "Procedure: Take first order Bessel function of the first kind of y values of curves\n     Usage: j1 <curve-list>",
                  UL_uopyc, 
                  PM_j1);
    SS_install_cf("j1x",
                  "Procedure: Take first order Bessel function of the first kind of x values of curves\n     Usage: j1x <curve-list>",
                  UL_uopxc, 
                  PM_j1);
    SS_install_cf("jn",
                  "Procedure: Take nth order Bessel function of the first kind of y values of curves\n     Usage: jn <curve-list> <n>",
                  UL_opyc, 
                  PM_jn);
    SS_install_cf("jnx",
                  "Procedure: Take nth order Bessel function of the first kind of x values of curves\n     Usage: jnx <curve-list> <n>",
                  UL_opxc, 
                  PM_jn);
    SS_install_cf("tchn",
                  "Procedure: Take nth order Tchebyshev function of y values of curves\n     Usage: jn <curve-list> <n>",
                  UL_opyc, 
                  PM_tchn);
    SS_install_cf("tchnx",
                  "Procedure: Take nth order Tchebyshev function of x values of curves\n     Usage: jn <curve-list> <n>",
                  UL_opxc,
                  PM_tchn);
    SS_install_cf("yn",
                  "Procedure: Take nth order Bessel function of the second kind of y values of curves\n     Usage: yn <curve-list> <n>",
                  UL_opyc, 
                  PM_yn);
    SS_install_cf("ynx",
                  "Procedure: Take nth order Bessel function of the second kind of x values of curves\n     Usage: yn <curve-list> <n>",
                  UL_opxc, 
                  PM_yn);
    SS_install_cf("y0",
                  "Procedure: Take zeroth order Bessel function of the second kind of y values of curves\n     Usage: y0 <curve-list>",
                  UL_uopyc, 
                  PM_y0);
    SS_install_cf("y0x",
                  "Procedure: Take zeroth order Bessel function of the second kind of x values of curves\n     Usage: y0x <curve-list>",
                  UL_uopxc, 
                  PM_y0);
    SS_install_cf("y1",
                  "Procedure: Take first order Bessel function of the second kind of y values of curves\n     Usage: y1 <curve-list>",
                  UL_uopyc, 
                  PM_y1);
    SS_install_cf("y1x",
                  "Procedure: Take first order Bessel function of the second kind of x values of curves\n     Usage: y1x <curve-list>",
                  UL_uopxc, 
                  PM_y1);

/* BC handled functions */

    SS_install_cf("+",
                  "Procedure: Take sum of curves\n     Usage: + <curve-list>",
                  UL_bc, 
                  PM_fplus);
    SS_install_cf("-",
                  "Procedure: Take difference of curves\n     Usage: - <curve-list>",
                  UL_bc, 
                  PM_fminus);
    SS_install_cf("*",
                  "Procedure: Take product of curves\n     Usage: * <curve-list>",
                  UL_bc, 
                  PM_ftimes);
    SS_install_cf("/",
                  "Procedure: Take quotient of curves\n     Usage: / <curve-list>",
                  UL_bc, 
                  PM_fdivide);
    SS_install_cf("min",
                  "Procedure: Construct curve from minima of y values of curves\n     Usage: min <curve-list>",
                  UL_bc, 
                  PM_fmin);
    SS_install_cf("max",
                  "Procedure: Construct curve from maxima of y values of curves\n     Usage: max <curve-list>",
                  UL_bc, 
                  PM_fmax);
    SS_install_cf("hypot",
                  "Procedure: Calculate harmonic average of two curves, sqrt(a^2+b^2)\n     Usage: hypot <a> <b>",
                  UL_bc, 
                  HYPOT);

/* BLTOC handled functions */

    SS_install_cf("color",
                  "Procedure: Set the color of curves\n     Usage: color <curve-list> <color-number>",
                  UL_bltoc, 
                  UL_color);
    SS_install_cf("scatter",
                  "Procedure: Plot curves as scatter plots\n     Usage: scatter <curve-list> on | off",
                  UL_bltoc, 
                  UL_scatter);
    SS_install_cf("marker",
                  "Procedure: Set the marker of curves\n     Usage: marker <curve-list> plus | star | triangle",
                  UL_bltoc, 
                  UL_marker);
    SS_install_cf("histogram",
                  "Procedure: Plot curves as histograms\n     Usage: histogram <curve-list> off | left | right | center",
                  UL_bltoc, 
                  UL_hist);
    SS_install_cf("lnwidth",
                  "Procedure: Set the line widths of curves\n     Usage: lnwidth <curve-list> <width-number>",
                  UL_bltoc, 
                  UL_lnwidth);
    SS_install_cf("lnstyle",
                  "Procedure: Set the line styles of curves\n     Usage: lnstyle <curve-list> solid | dotted | dashed | dotdashed",
                  UL_bltoc, 
                  UL_lnstyle);

/* UL2TOC handled functions */

    SS_install_cf("integrate",
                  "Procedure: Integrate curves\n     Usage: integrate <curve-list> <low-lim> <high-lim>",
                  UL_ul2toc, 
                  UL_integrate);
    SS_install_cf("xmm",
                  "Procedure: Excerpt part of curves\n     Usage: xmm <curve-list> <low-lim> <high-lim>",
                  UL_ul2toc, 
                  UL_xmm);

/* ULNTOC handled functions */

    SS_install_cf("filter",
                  "Procedure: Return a filtered curve\n     Usage: filter <curve-list> <dom-pred> <ran-pred>",
                  UL_ulntoc,
                  UL_filter);
    SS_install_cf("filter-coef",
                  "Procedure: Return a curve filtered through coefficents\n     Usage: filter-coef <curve-list> <coeff-array> <ntimes>",
                  UL_ulntoc,
                  UL_filter_coef);
    SS_install_cf("smoothn",
                  "Procedure: Smooth curves using user specified smooth-method\n     Usage: smoothn <curve-list> <n> <ntimes>",
                  UL_ulntoc, 
                  UL_smooth);

/* UC handled functions */

    SS_install_cf("derivative",
                  "Procedure: Take derivative of curves\n     Usage: derivative <curve-list>",
                  UL_uc, 
                  UL_derivative);
    SS_install_cf("rev",
                  "Procedure: Swap x and y values for curves\n     You may want to sort after this\n     Usage: rev <curve-list>",
                  UL_uc, 
                  UL_reverse);
    SS_install_cf("copy",
                  "Procedure: Copy curves\n     Usage: copy <curve-list>",
                  UL_uc, 
                  UL_copy_curve);
    SS_install_cf("xindex",
                  "Procedure: Create curves with y values vs integer index values\n     Usage: xindex <curve-list>",
                  UL_uc, 
                  UL_xindex_curve);
    SS_install_cf("sort",
                  "Procedure: Sort curves' points into ascending order based on x values\n     Usage: sort <curve-list>",
                  UL_uc, 
                  UL_sort);
    SS_install_cf("compose",
                  "Functional composition f(g(x))\n     Usage: compose <f> <g>",
                  UL_bcxl, 
                  UL_compose);
    SS_install_cf("hide",
                  "Procedure: Hide curves from view\n     Usage: hide <curve-list>",
                  UL_uc, 
                  UL_hide);
    SS_install_cf("show",
                  "Procedure: Reveal curves hidden by hide command\n     Usage: show <curve-list>",
                  UL_uc, 
                  UL_show);
    SS_install_cf("thin",
                  "Procedure: Represent a curve with fewer points\n     Usage: thin <curve-list> <mode> <val>",
                  UL_ulntoc,
                  UL_thin);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
