/*
 * SXCONT.C - control interface routines for SX
 *
 * Source Version: 3.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "sx.h"

#define EOE(str) ((*SS_PTR(str) == '\n') || (*SS_PTR(str) == '\0'))

int
 SX_autorange,
 SX_autodomain,
 SX_autoplot,
 SX_background_color_white,
 SX_border_width,
 SX_cgm_background_color,
 SX_cgm_flag,
 SX_data_id,
 SX_default_color,
 SX_file_exist_action,
 SX_gr_mode,
 SX_grid,
 SX_label_color_flag,
 SX_label_length,
 SX_label_type_size,
 SX_plot_flag,
 SX_plot_type_size,
 SX_ps_flag,
 SX_qflag = FALSE,
 SX_render_def,
 SX_render_1d_1d,
 SX_render_2d_1d,
 SX_render_2d_2d,
 SX_render_3d_1d,
 SX_show_mouse_location,
 SX_squeeze_labels,
 SX_x_log_scale,
 SX_y_log_scale;

REAL
 SX_botspace,
 SX_chi,
 SX_label_space,
 SX_leftspace,
 SX_marker_scale,
 SX_marker_orientation,
 SX_phi,
 SX_rightspace,
 SX_show_mouse_location_x,
 SX_show_mouse_location_y,
 SX_theta,
 SX_topspace;

char
 *SX_cgm_name,
 *SX_cgm_type,
 *SX_plot_type_style,
 *SX_promotion_type,
 *SX_ps_name,
 *SX_ps_type,
 *SX_smooth_method;

static char
 *SX_MAPPING_S = "PM_mapping *",
 *SX_IMAGE_S = "PG_image *",
 *SX_CURVE_S = "PG_curve";

static char
 SC_DECLARE(*_SX_isthrough, (char *s)),
 SC_DECLARE(*_SX_expand_first, (char *s, char *sp));

static int
 SC_DECLARE(_SX_isodd, (int n)),
 SC_DECLARE(_SX_no_argsp, (object *obj));

static void
 SC_DECLARE(_SX_list_vobjects, 
            (char *patt, g_file *po, int type)),
 SC_DECLARE(_SX_sort_lists, (SX_menu_item *a, int n)),
 SC_DECLARE(_SX_free_menu, (g_file *po));

#ifdef HAVE_JPEGLIB
int
 SX_jpeg_flag;

char
 *SX_jpeg_name,
 *SX_jpeg_type;
#else
static object
 *SC_DECLARE(_SX_no_jpeg_support, (byte));
#endif

char
 *SX_jpeglib_flag;

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_ARGS - get a C level data item from a single Scheme object */

void _SX_args(obj, v, type)
   object *obj;
   byte *v;
   int type;
   {byte **pv;
    char *s;

    pv = (byte **) v;
    if (SS_nullobjp(obj) && (type != G_FILE))
       {*pv = NULL;
        return;};

    switch (type)
       {case G_FILE       :
             if (!SS_nullobjp(obj) && !SX_FILEP(obj))
                SS_error("OBJECT NOT FILE - _SX_ARGS", obj);

             if (SS_nullobjp(obj))
                *pv = (byte *) SX_gvif;
             else
                *pv = (byte *) SS_GET(g_file, obj);
             break;

        case G_DEVICE     :
             if (!SX_DEVICEP(obj))
                SS_error("OBJECT NOT PG_DEVICE - _SX_ARGS", obj);
             *pv = (byte *) SS_GET(PG_device, obj);
             break;

        case G_SET        :
             if (!SX_SETP(obj))
                SS_error("OBJECT NOT PM_SET - _SX_ARGS", obj);
             *pv = (byte *) SS_GET(PM_set, obj);
             break;

        case G_MAPPING    :
             if (SX_GRAPHP(obj))
                *pv = (byte *) (SS_GET(PG_graph, obj)->f);
             else if (SX_MAPPINGP(obj))
                *pv = (byte *) SS_GET(PM_mapping, obj);
             else
                SS_error("OBJECT NOT PM_MAPPING - _SX_ARGS", obj);
             break;

        case G_GRAPH      :
             if (SX_GRAPHP(obj))
                *pv = (byte *) SS_GET(PG_graph, obj);
             else
                SS_error("OBJECT NOT PG_GRAPH - _SX_ARGS", obj);
             break;

        case G_NUM_ARRAY  :
             if (!SX_NUMERIC_ARRAYP(obj))
                SS_error("OBJECT NOT C_ARRAY - _SX_ARGS", obj);
             *pv = (byte *) SS_GET(C_array, obj);
             break;

        case G_IMAGE      :
             if (!SX_IMAGEP(obj))
                SS_error("OBJECT NOT PG_IMAGE - _SX_ARGS", obj);
             *pv = (byte *) SS_GET(PG_image, obj);
             break;

        case G_PACKAGE    :
             if (!SX_PACKAGEP(obj))
                {s   = SS_get_string(obj);
                 *pv = (byte *) PA_INQUIRE_VARIABLE(s);
                 if (pv == NULL)
                    SS_error("BAD PACKAGE - _SX_ARGS", obj);}
             else
                *pv = (byte *) SS_GET(PA_package, obj);
             break;

        case G_PANVAR     :
             if (!SX_PANVARP(obj))
                {s   = SC_strsavef(SS_get_string(obj),
                       "char*:_SX_ARGS:s");
                 *pv = (byte *) PA_INQUIRE_VARIABLE(s);
                 if (pv == NULL)
                    SS_error("NOT PANACEA VARIABLE - _SX_ARGS",
                             obj);}
             else
                *pv = (byte *) SS_GET(PA_variable, obj);
             break;

        case G_PDBDATA    :
             if (!SX_PDBDATAP(obj))
                SS_error("NOT PDBDATA OBJECT - _SX_ARGS", obj);

             *pv = (byte *) SS_GET(g_pdbdata, obj);
             break;

        case G_DEFSTR     :
             if (!SX_DEFSTRP(obj))
                SS_error("NOT DEFSTR OBJECT - _SX_ARGS", obj);

             *pv = (byte *) SS_GET(defstr, obj);
             break;

        case G_SYMENT     :
        case G_PLOT_MAP   :
        case G_PLT_CRV    :
        case G_FUNCTION   :

        case G_DEV_ATTRIBUTES   :
        case G_SOURCE_VARIABLE  :
        case G_IV_SPECIFICATION :
        case G_PLOT_REQUEST     :

        default                 :
             SS_error("BAD TYPE - _SX_ARGS", obj);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_CALL_ARGS - make a SCHEME level object from a C level one */

object *_SX_call_args(type, v)
   int type;
   byte *v;
   {object *obj;

    if (v == NULL)
       obj = SS_null;

    switch (type)
       {case G_FILE       :
             if (v == NULL)
                obj = SX_ovif;
             else
                obj = SX_mk_gfile((g_file *) v);
             break;

        case G_DEVICE     :
             if (v != NULL)
                obj = SX_mk_graphics_device((PG_device *) v);
             break;

        case G_SET        :
             if (v != NULL)
                obj = SX_mk_set((PM_set *) v);
             break;

        case G_MAPPING    :
             if (v != NULL)
                obj = SX_mk_mapping((PM_mapping *) v);
             break;

        case G_GRAPH      :
             if (v != NULL)
                obj = SX_mk_graph((PG_graph *) v);
             break;

        case G_NUM_ARRAY  :
             if (v != NULL)
                obj = SX_mk_C_array((C_array *) v);
             break;

        case G_IMAGE      :
             if (v != NULL)
                obj = SX_mk_image((PG_image *) v);
             break;

        case G_PACKAGE    :
             if (v != NULL)
                obj = SX_mk_package((PA_package *) v);
             break;

        case G_PANVAR     :
             if (v != NULL)
                obj = SX_mk_variable((PA_variable *) v);
             break;

        case G_DEFSTR     :
             if (v != NULL)
                obj = _SX_mk_gdefstr((defstr *) v);
             break;

        case G_SYMENT     :
             if (v != NULL)
                obj = _SX_mk_gsyment((syment *) v);
             break;

        case G_PDBDATA    :
        case G_PLOT_MAP   :
        case G_PLT_CRV    :
        case G_FUNCTION   :

        case G_DEV_ATTRIBUTES   :
        case G_SOURCE_VARIABLE  :
        case G_IV_SPECIFICATION :
        case G_PLOT_REQUEST     :

        default                 :
             SS_error("BAD TYPE - _SX_CALL_ARGS", obj);};

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MENU - print a menu of images, curves, and graphs */

object *SX_menu(argl)
   object *argl;
   {char *patt, *type;
    g_file *po;

    po   = NULL;
    patt = NULL;
    type = NULL;
    SS_args(argl,
            G_FILE, &po,
            SC_STRING_I, &patt,
            SC_STRING_I, &type,
            0);

    if (strcmp(po->type, SX_PDBFILE_S) != 0)
       return(SS_f);

    _SX_get_menu(po);

    if (type != NULL)
       _SX_list_vobjects(patt, po, type[0]);
    else
       _SX_list_vobjects(patt, po, 0);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_GET_MENU - make lists of syments for the mappings, images, and curves 
 *              - in the current directory of the given file
 */

void _SX_get_menu(po)
   g_file *po;
   {int i, n;
    PDBfile *file;
    char *dir, **names;

    file = FILE_FILE(PDBfile, po);
/*
    if ((po->n_menu_items != 0L) && (!file->virtual_internal))
       return;

    if (file->virtual_internal)
*/
       _SX_free_menu(po);

    dir = (PD_has_directories(file)) ? PD_pwd(file) : NULL;

    names = PD_ls(file, dir, "PM_mapping *", &n);
    for (i = 0; i < n; i++)
        _SX_push_menu_item(po, names[i], SX_MAPPING_S);
    SFREE(names);

    names = PD_ls(file, dir, "PG_image *", &n);
    for (i = 0; i < n; i++)
        _SX_push_menu_item(po, names[i], SX_IMAGE_S);
    SFREE(names);

    names = PD_ls(file, dir, "char", &n);
    for (i = 0; i < n; i++)
        if (SC_regx_match(names[i], "curve????"))
	   _SX_push_menu_item(po, names[i], SX_CURVE_S);
    SFREE(names);

    _SX_sort_lists(po->menu, po->n_menu_items);

    PD_reset_ptr_list(file);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_SORT_LISTS - sort the menu according to the alphabetization
 *                - of the labels
 */

static void _SX_sort_lists(a, n)
   SX_menu_item *a;
   int n;
   {int gap, i, j;
    SX_menu_item temp;

    for (gap = n/2; gap > 0; gap /= 2)
        for (i = gap; i < n; i++)
            for (j = i-gap; j >= 0; j -= gap)
                {if (strcmp(a[j].label, a[j+gap].label) <= 0)
                    break;
                 temp     = a[j];
                 a[j]     = a[j+gap];
                 a[j+gap] = temp;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_LIST_VOBJECTS - list the items from the specified list matching the
 *                   - given pattern
 */

static void _SX_list_vobjects(patt, po, type)
   char *patt;
   g_file *po;
   int type;
   {long i, n;
    int t;
    char *s;
    SX_menu_item *mitems;

    mitems = po->menu;
    n      = po->n_menu_items;

    if (n <= 0L)
       return;

    PRINT(stdout, "\n");

    for (i = 0L; i < n; i++)
        {s = mitems[i].label;
         t = mitems[i].type[3];
         if (((type == 0) || (t == type)) && SC_regx_match(s, patt))
            PRINT(stdout, " %ld  %c  %s\n", i+1, t, s);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_PUSH_MENU_ITEM - add the given symbol table entry to the file menu */

void _SX_push_menu_item(po, name, type)
   g_file *po;
   char *name, *type;
   {SX_menu_item mitem;
    char s[MAXLINE], var[MAXLINE], *lb;
    PDBfile *file;

    file = FILE_FILE(PDBfile, po);

    mitem.type  = type;

    strcpy(s, _PD_fixname(file, name));
    mitem.vname = SC_strsavef(s, "char*:_SX_PUSH_MENU_ITEM:vname");

/* get mapping label */
    if (strcmp(type, SX_MAPPING_S) == 0)
       {sprintf(var, "%s[%d].name", s, file->default_offset);
        if (!PD_read(file, var, &lb))
 	   SS_error("FAILED TO READ LABEL - _SX_PUSH_MENU_ITEM", SS_null);
	mitem.label = SC_strsavef(lb, "char*:_SX_PUSH_MENU_ITEM:label");
        SC_REMEMBER(SX_menu_item, mitem, po->menu,
                    po->n_menu_items, po->max_menu_items, 100);}

/* get curve label */
    else if (strcmp(type, SX_CURVE_S) == 0)
       {char *u, t[MAXLINE], *pt;

        if (!PD_read(file, s, t))
           SS_error("FAILED TO READ CURVE HEADER - _SX_PUSH_MENU_ITEM",
		    SS_null);

/* extract the name of the label variable */
        lb = SC_strtok(t, "|", pt);
        if (lb == NULL)
           SS_error("BAD CURVE HEADER - _SX_PUSH_MENU_ITEM", SS_null);

/* get the label */
        if (PD_has_directories(file))
           {u = strrchr(s, '/');
            u = (u == NULL) ? s : u + 1;
            strcpy(u, lb);}
        else
           strcpy(s, lb);

        if (!PD_read(file, s, t))
           SS_error("FAILED TO READ LABEL - _SX_PUSH_MENU_ITEM", SS_null);
        mitem.label = SC_strsavef(t, "char*:_SX_PUSH_MENU_ITEM:label");
        SC_REMEMBER(SX_menu_item, mitem, po->menu,
                    po->n_menu_items, po->max_menu_items, 100);}

/* get image label */
    else if (strcmp(type, SX_IMAGE_S) == 0)
       {sprintf(var, "%s[%d].label", s, file->default_offset);
        if (!PD_read(file, var, &lb))
           SS_error("FAILED TO READ LABEL - _SX_PUSH_MENU_ITEM", SS_null);
	mitem.label = SC_strsavef(lb, "char*:_SX_PUSH_MENU_ITEM:label");
        SC_REMEMBER(SX_menu_item, mitem, po->menu,
                    po->n_menu_items, po->max_menu_items, 100);}

    else
       SS_error("BAD OBJECT TYPE - _SX_PUSH_MENU_ITEM", SS_null);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_FREE_MENU - release the menus associated with a file */

static void _SX_free_menu(po)
   g_file *po;
   {SX_menu_item *mitems;
    int i, n;

    mitems = po->menu;
    n      = po->n_menu_items;
    for (i = 0; i < n; i++)
        {SFREE(mitems[i].vname);
         SFREE(mitems[i].label);};

    SFREE(po->menu);

    po->n_menu_items   = 0;
    po->max_menu_items = 0;

    return;}

/*--------------------------------------------------------------------------*/

/*                     INPUT REPROCESSING ROUTINES                          */

/*--------------------------------------------------------------------------*/

/* SX_PARSE - determine whether or not to reprocess the input for Ultra
 *          - this is the real worker for the SS_post_eval_hook
 *          - since this SS_evobj is not the same as in SS_REPL
 *          - it should be SS_MARK'd as being an additional pointer to its
 *          - respective object
 */

void SX_parse(replot, reproc, strm)
   DECLFPTR(byte, replot, (void));
   DECLFPTR(char *, reproc, (char *s));
   object *strm;
   {char *t, s[MAXLINE], line[MAXLINE], *ptr;

    if (SS_procedurep(SS_evobj))
       {strcpy(s, SS_PP(SS_evobj, name));
        if (_SX_no_argsp(SS_evobj) || !EOE(strm))
           {ptr = SS_BUFFER(strm);
            if (_SX_isodd(SC_char_count(ptr, '\"')))
               {PRINT(stdout, "\nUNMATCHED QUOTE: %s\n\n", ptr);
                *SS_BUFFER(strm) = '\0';
                SS_PTR(strm) = SS_BUFFER(strm);}
            else
               {strcpy(line, ptr);
                while ((t = (*reproc)(line)) != NULL)
                  {strcpy(ptr, t);
                   SS_PTR(strm) = SS_BUFFER(strm);
                   SS_Assign(SS_rdobj, _SS_read(strm));
                   SS_interactive = ON;
                   SX_plot_flag   = TRUE;
                   SS_Assign(SS_evobj, SS_eval(SS_rdobj));
                   SS_interactive = OFF;};

                if (SX_plot_flag && (strcmp(s, "replot") != 0) &&
                    (SX_autoplot == ON) &&
                    (replot != NULL))
                   (*replot)();};};};

    if (PG_console_device != NULL)
       PG_console_device->gprint_flag = TRUE;

    for (t = SS_BUFFER(strm); *t; t++);
    SS_PTR(strm) = t;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_WRAP_PAREN - enclose second arg by first and third args */

char *_SX_wrap_paren(open, form, close)
   char *open, *form, *close;
   {char tmp[MAXLINE];

   strcpy(tmp, form);
   strcpy(form, open);
   strcat(form, tmp);
   strcat(form, close);

   return(form);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_EXPAND_EXPR - expand object lists of the form x:y 
 *                 - which means all object from x through y inclusive
 */

int _SX_expand_expr(s)
   char *s;
   {char *sp;
    int ret;

/* something wrong in input if s is NULL */
    if (s == NULL)
       return(FALSE);

    sp = _SX_isthrough(s);

/* if nothing to expand - termination condition for loop */
    if (sp == NULL)
       ret = TRUE;

    else

/* expand the first missing object */
       {s = _SX_expand_first(s, sp);

/* loop through to find any more instances */
        ret = _SX_expand_expr(s);};

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_ISTHROUGH - return pointer if :range found, otherwise NULL */

static char *_SX_isthrough(s)
   char *s;
   {Register char *sp;

    for (sp = s; *sp != '\0'; sp++)

/* skip quoted strings */
        {if (*sp == '\"')
            for (sp++; *sp != '\"'; sp++);

/* skip index expressions */
	 if (*sp == '[')
            for (sp++; *sp != ']'; sp++);

         if (*sp == ':')
            return(sp);};

    return(NULL);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_EXPAND_FIRST - expand the first missing menu/lst object */

static char *_SX_expand_first(s, sp)
   char *s, *sp;
   {char s1[MAXLINE], s2[MAXLINE], temp[MAXLINE];
    char *tp, *lexpr, *rexpr;
    int unbalanced;

    *sp = '\0';

/* break the portion of the command before the colon into two strings:
 * the expression immediately preceding the colon (lexpr) and the rest.
 */
    strcpy(s1, s);
    for (tp = s1 + (sp - s) - 1; *tp == ' '; tp--);

    if (*tp == ')')
       {*(tp + 2) = '\0';
        *(tp + 1) = ')';
        for (unbalanced = 1; unbalanced; tp--)
            {*tp = *(tp - 1);
             if (*tp == ')')
                unbalanced++;
             else if (*tp == '(')
                unbalanced--;}
        *tp = '\0';
        lexpr = tp + 1;}
    else
       lexpr = SC_lasttok(s1, " ,");

/* break the portion of the command after the colon into two strings:
 * the expression immediately following the colon (rexpr) and the rest.
 */
    for (tp = sp + 1; *tp == ' '; tp++);
    strcpy(s2, tp);

    if (*tp == '(')
       {rexpr = tp++;
        strcpy(temp, rexpr);
        rexpr = temp;
        for (unbalanced = 1; unbalanced; tp++)
            {if (*tp == '(')
                unbalanced++;
             else if (*tp == ')')
                unbalanced--;}
        strcpy(s2, tp);
        *tp = '\0';}
    else
       rexpr = SC_firsttok(s2, " ,");

/* build a new expanded command from all the parts.
 * removing the blank after the ')' causes multiple ranges to give an error.
 * the blank before the '(' improves the aesthetics of command log output.
 */
    sprintf(s, "%s (thru %s %s) %s", s1, lexpr, rexpr, s2);

    return(s);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_SPLIT_COMMAND - return the next command from the string in
 *                   - the given buffer
 *                   - return FALSE if there are no more commands
 */

int _SX_split_command(cmd, lst)
   char *cmd, *lst;
   {char *pc, *pl;

    for (pl = lst, pc = cmd; TRUE; )
        {switch (*pc++ = *pl++)
            {case '\"' : while ((*pc++ = *pl++) != '\"');
                         break;

             case '\r' :
             case '\n' :
             case ';'  : pc--;
                         *pc = '\0';

/* copy the remainder of lst down to the beginning of lst */
                         for (pc = lst; (*pc++ = *pl++) != '\0'; );

                         return(TRUE);

             case '\0' : return(FALSE);

             default   : break;};};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_ISODD - return TRUE iff the given number is odd */

static int _SX_isodd(n)
   int n;
   {return((2*(n/2) == n) ? FALSE : TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_NO_ARGSP - return TRUE iff the object is a procedure object which
 *              - may have zero arguments
 */

static int _SX_no_argsp(obj)
   object *obj;
   {int ret;
    object *params;
    PFPObject hand;

    ret = FALSE;

    switch (SS_PROCEDURE_TYPE(obj))
       {case SS_MACRO : 
        case SS_PROC  : params = SS_params(obj);
                        ret = !SS_consp(params) || (params == SS_null);
                        break;

        default       : hand = SS_C_PROCEDURE_HANDLER(obj);
                        ret = (hand == SS_zargs)      ||
                              (hand == SS_znargs)     ||
                              (hand == SS_acc_char)   ||
                              (hand == SS_acc_int)    ||
                              (hand == SS_acc_long)   ||
                              (hand == SS_acc_REAL)   ||
                              (hand == SS_acc_string) ||
                              (hand == SS_acc_ptr);
                        break;};

    return(ret);}

/*--------------------------------------------------------------------------*/

/*                         INSTALLATION ROUTINES                            */

/*--------------------------------------------------------------------------*/

/* SX_INSTALL_GLOBAL_VARS - install the global variables */

void SX_install_global_vars()
   {

    SX_autodomain             = TRUE;
    SX_autoplot               = ON;
    SX_autorange              = TRUE;
    SX_botspace               = 0.01;
    SX_cgm_background_color   = 1;
    SX_cgm_flag               = FALSE;
    SX_cgm_name               = SC_strsavef("plots",
					    "char*:SX_INSTALL_GLOBAL_VARS:cgm_name");
    SX_cgm_type               = SC_strsavef("COLOR",
					    "char*:SX_INSTALL_GLOBAL_VARS:cgm_type");
    SX_border_width           = 2;
    SX_chi                    = 0.0;
    SX_data_id                = TRUE,
    SX_default_color          = -1;
    SX_GRI_x                  = 0.5;
    SX_GRI_y                  = 0.01;
    SX_GRI_dx                 = 0.0;
    SX_GRI_dy                 = 0.0;
    SX_grid                   = OFF;
#ifdef HAVE_JPEGLIB
    SX_jpeglib_flag           = SC_strsavef("T",
					    "char*:SX_INSTALL_GLOBAL_VARS:jpeglib_flag");
    SX_jpeg_flag              = FALSE;
    SX_jpeg_name              = SC_strsavef("plots",
					    "char*:SX_INSTALL_GLOBAL_VARS:jpeg_name");
    SX_jpeg_type              = SC_strsavef("RGB",
					    "char*:SX_INSTALL_GLOBAL_VARS:jpeg_type");
#else
    SX_jpeglib_flag           = NULL;
#endif
    SX_label_color_flag       = FALSE;
    SX_label_length           = 25;
    SX_label_space            = 0.0;
    SX_label_type_size        = 8;
    SX_leftspace              = 0.01;
    SX_marker_orientation     = 0.0;
    SX_marker_scale           = 0.01;
/*    SX_palette                = SC_strsavef("spectrum",
                                            "char*:SX_INSTALL_GLOBAL_VARS:palette"); */
    SX_phi                    = 0.0;
    SX_plot_type_size         = 12;
    SX_plot_type_style        = SC_strsavef("medium",
					    "char*:SX_INSTALL_GLOBAL_VARS:type_style");
    SX_promotion_type         = SC_strsavef("none",
					    "char*:SX_INSTALL_GLOBAL_VARS:promotion_type");
    SX_ps_flag                = FALSE;
    SX_ps_name                = SC_strsavef("plots",
					    "char*:SX_INSTALL_GLOBAL_VARS:ps_name");
    SX_ps_type                = SC_strsavef("MONOCHROME",
					    "char*:SX_INSTALL_GLOBAL_VARS:ps_type");
    SX_render_def             = PLOT_MESH;
    SX_render_1d_1d           = _PG_plot_type;
    SX_render_2d_1d           = PLOT_CONTOUR;
    SX_render_2d_2d           = PLOT_VECTOR;
    SX_render_3d_1d           = PLOT_MESH;
    SX_rightspace             = 0.01;
    SX_show_mouse_location    = FALSE;
    SX_show_mouse_location_x  = 0.025;
    SX_show_mouse_location_y  = 0.955;
    SX_smooth_method          = SC_strsavef("averaging",
					    "char*:SX_INSTALL_GLOBAL_VARS:method");
    SX_squeeze_labels         = FALSE;
    SX_theta                  = 0.0;
    SX_topspace               = 0.01;
    SX_x_log_scale            = FALSE;
    SX_y_log_scale            = FALSE;

    SX_GRI_title            = SC_strsavef("PDBView Controls",
					  "char*:SX_INSTALL_GLOBAL_VARS:GRI_title");
    SX_GRI_type_face        = SC_strsavef("helvetica",
					  "char*:SX_INSTALL_GLOBAL_VARS:GRI_face");
    SX_GRI_type_style       = SC_strsavef("medium",
					  "char*:SX_INSTALL_GLOBAL_VARS:GRI_style");
    SX_GRI_type_size        = 12;

    _PG_axis_label_x_format = SC_strsavef("%10.2g",
					  "char*:SX_INSTALL_GLOBAL_VARS:xformat");
    _PG_axis_label_y_format = SC_strsavef("%10.2g",
					  "char*:SX_INSTALL_GLOBAL_VARS:yformat");
    _PG_axis_type_face      = SC_strsavef("helvetica",
					  "char*:SX_INSTALL_GLOBAL_VARS:type_face");

    SS_interactive = FALSE;
    SS_print_flag  = FALSE;
    SS_stat_flag   = FALSE;

/* A */

    SS_install_cf("answer-prompt",
                  "Variable: A string printed before the return value\n     Usage: answer-prompt <string>",
                  SS_acc_string,
                  SS_ans_prompt);

    SS_install_cf("autorange",
                  "Variable: Turns on or off autoranging\n     Usage: autorange on | off",
                  SS_acc_int,
                  &SX_autorange);

    SS_install_cf("autodomain",
                  "Variable: Turns on or off autodomain\n     Usage: autodomain on | off",
                  SS_acc_int,
                  &SX_autodomain);

    SS_install_cf("autoplot",
                  "Variable: Controls autoploting of graphs\n     Usage: autoplot on | off",
                  SS_acc_int,
                  &SX_autoplot);

    SS_install_cf("axis",
                  "Variable: Controls drawing the axes\n     Usage: axis on | off ",
                  SS_acc_int,
                  &_PG_axis_on);

    SS_install_cf("axis-max-major-ticks",
                  "Variable: Controls maximum number of major ticks\nUsage: axis-max-major-ticks <integer> ",
                  SS_acc_int,
                  &_PG_axis_max_major_ticks);

    SS_install_cf("axis-number-minor-ticks",
                  "Variable: Controls number of minor axis ticks\nUsage: axis-number-minor-ticks <integer> ",
                  SS_acc_int,
                  &_PG_axis_n_minor_ticks);

    SS_install_cf("axis-number-minor-x-ticks",
                  "Variable: Controls number of minor X axis ticks\nUsage: axis-number-minor-x-ticks <integer> ",
                  SS_acc_int,
                  &_PG_axis_n_minor_x_ticks);

    SS_install_cf("axis-number-minor-y-ticks",
                  "Variable: Controls number of minor Y axis ticks\nUsage: axis-number-minor-y-ticks <integer> ",
                  SS_acc_int,
                  &_PG_axis_n_minor_y_ticks);

    SS_install_cf("axis-n-decades",
                  "Variable: Controls maximum number of log axis decades\nUsage: axis-n-decades <real> ",
                  SS_acc_REAL,
                  &_PG_axis_n_decades);

    SS_install_cf("axis-tick-type",
                  "Variable: Controls location of axis ticks\nUsage: axis-tick-type left-of-axis | right-of-axis | straddle-axis ",
                  SS_acc_int,
                  &_PG_axis_tick_type);

    SS_install_cf("axis-char-size",
                  "Variable: Controls character size for the axes\n     Usage: axis-char-size <integer> ",
                  SS_acc_int,
                  &_PG_axis_char_size);

    SS_install_cf("axis-char-angle",
                  "Variable: Controls angle at which characters are drawn\n     Usage: axis-char-angle <real> ",
                  SS_acc_REAL,
                  &_PG_axis_char_angle);

    SS_install_cf("axis-grid-style",
                  "Variable: Controls line style for the axis grid\n     Usage: axis-grid-style solid | dotted | dashed | dotdashed ",
                  SS_acc_int,
                  &_PG_axis_grid_style);

    SS_install_cf("axis-line-style",
                  "Variable: Controls line style for the axes\n     Usage: axis-line-style solid | dotted | dashed | dotdashed ",
                  SS_acc_int,
                  &_PG_axis_line_style);

    SS_install_cf("axis-line-width",
                  "Variable: Controls line width for the axes\n     Usage: axis-line-width [<real>] ",
                   SS_acc_REAL,
                  &_PG_axis_line_width);

    SS_install_cf("axis-tick-size",
                  "Variable: Controls major tick size for the axes\n     Usage: axis-tick-size <real>",
                  SS_acc_REAL,
                  &_PG_axis_major_tick_size);

    SS_install_cf("axis-type",
                  "Variable: Controls axis type\n     Usage: axis-type cartesian | polar | insel",
                  SS_acc_int,
                  &_PG_axis_type);

    SS_install_cf("axis-x-format",
                  "Variable: Controls format for x axis labels\n     Usage: axis-x-format <format>",
                  SS_acc_ptr,
                  &_PG_axis_label_x_format);

    SS_install_cf("axis-y-format",
                  "Variable: Controls format for y axis labels\n     Usage: axis-y-format <format>",
                  SS_acc_ptr,
                  &_PG_axis_label_y_format);

/* BCD */

    SS_install_cf("background-color-flag",
                  "Variable: Background color - white or black\n     Usage: background-color-flag black | white",
                  SS_acc_int,
                  &SX_background_color_white);

    SS_install_cf("border-width",
                  "Variable: Window border width in pixels\n     Usage: border-width <integer>",
                  SS_acc_int,
                  &SX_border_width);

    SS_install_cf("botspace",
                  "Variable: Fractional space at bottom of screen\n     Usage: botspace <real>",
                  SS_acc_REAL,
                  &SX_botspace);

    SS_install_cf("cgm-background-color-flag",
                  "Variable: CGM background color flag - white, black, auto\n     Usage: cgm-background-color-flag black | white | auto",
                  SS_acc_int,
                  &SX_cgm_background_color);

    SS_install_cf("cgm-flag",
                  "Variable: When TRUE print will generate a plot in a CGM file",
                  SS_acc_int,
                  &SX_cgm_flag);

    SS_install_cf("cgm-name",
                  "Variable: CGM file name\n     Usage: cgm-name <string>",
                  SS_acc_ptr,
                  &SX_cgm_name);

    SS_install_cf("cgm-type",
                  "Variable: CGM file type (monochrome or color)\n     Usage: cgm-type <string>",
                  SS_acc_ptr,
                  &SX_cgm_type);

    SS_install_cf("chi",
                  "Variable: Default chi view angle\n     Usage: chi <real>",
                  SS_acc_REAL,
                  &SX_chi);

    SS_install_cf("console-type",
                  "Variable: Controls console mode\n     Usage: console-type <string>",
                  SS_acc_ptr,
                  &SX_console_type);

    SS_install_cf("console-origin-x",
                  "Variable: X comp of console origin (frac of screen width)\n     Usage: console-origin-x <real>",
                  SS_acc_REAL,
                  &SX_console_x);

    SS_install_cf("console-origin-y",
                  "Variable: Y comp of console origin (frac of screen width)\n     Usage: console-origin-y <real>",
                  SS_acc_REAL,
                  &SX_console_y);

    SS_install_cf("console-width",
                  "Variable: Console width in fraction of screen width\n     Usage: console-width <real>",
                  SS_acc_REAL,
                  &SX_console_width);

    SS_install_cf("console-height",
                  "Variable: Console height in fraction of screen width\n     Usage: console-height <real>",
                  SS_acc_REAL,
                  &SX_console_height);

    SS_install_cf("contour-n-levels",
                  "Variable: Default number of contour levels to plot\n     Usage: contour-n-levels <int>",
                  SS_acc_int,
                  &_PG_contour_n_levels);

    SS_install_cf("contour-ratio",
                  "Variable: Default ratio for conntour levels spacing\n     Usage: contour-ratio <real>",
                  SS_acc_REAL,
                  &_PG_contour_ratio);

    SS_install_cf("data-id",
                  "Variable: Controls display of curve identifiers on graph\n     Usage: data-id on | off",
                  SS_acc_int,
                  &SX_data_id);

    SS_install_cf("default-color",
                  "Variable: The default line color\n     Usage: default-color <color>",
                  SS_acc_int,
                  &SX_default_color);

/* EFGHIJ */

    SS_install_cf("error-bar-cap-size",
                  "Variable: Fractional size of error bar caps\n     Usage: error-bar-cap-size <real>",
                  SS_acc_REAL,
                  &_PG_error_cap_size);

    SS_install_cf("file-exist-action",
                  "Variable: Control action when opening existing files for append\n     Usage: file-exist-action <integer>",
                  SS_acc_int,
                  &SX_file_exist_action);

    SS_install_cf("graphics-mode",
                  "Variable: Graphics mode\n     Usage: graphics-mode [on|off]",
                  SS_acc_int,
                  &SX_gr_mode);

    SS_install_cf("gri-title",
                  "Variable: Graphical interface window title\n     Usage: gri-title <string>",
                  SS_acc_ptr,
                  &SX_GRI_type_face);

    SS_install_cf("gri-type-face",
                  "Variable: Graphical interface type face\n     Usage: gri-type-face medium | italic | bold | bold-italic",
                  SS_acc_ptr,
                  &SX_GRI_type_face);

    SS_install_cf("gri-type-size",
                  "Variable: Graphical interface type size\n     Usage: gri-type-size [<integer>]",
                  SS_acc_int,
                  &SX_GRI_type_size);

    SS_install_cf("gri-type-style",
                  "Variable: Graphical interface type style\n     Usage: gri-type-style medium | italic | bold | bold-italic",
                  SS_acc_ptr,
                  &SX_GRI_type_style);

    SS_install_cf("gri-x",
                  "Variable: Graphical interface window x origin\n     Usage: gri-x <real>",
                  SS_acc_REAL,
                  &SX_GRI_x);

    SS_install_cf("gri-y",
                  "Variable: Graphical interface window y origin\n     Usage: gri-y <real>",
                  SS_acc_REAL,
                  &SX_GRI_y);

    SS_install_cf("gri-width",
                  "Variable: Graphical interface window width\n     Usage: gri-width <real>",
                  SS_acc_REAL,
                  &SX_GRI_dx);

    SS_install_cf("gri-height",
                  "Variable: Graphical interface window height\n     Usage: gri-height <real>",
                  SS_acc_REAL,
                  &SX_GRI_dy);

    SS_install_cf("grid",
                  "Variable: Controls display of grid lines in graph\n     Usage: grid on | off",
                  SS_acc_int,
                  &SX_grid);

    SS_install_cf("hide-rescale",
                  "Variable: Allows plot to rescale when hiding curves\n     Usage: hide-rescale on | off",
                  SS_acc_int,
                  &PG_hide_rescale);

    SS_install_cf("histogram-start",
                  "Variable: Starting point for histogram plots\n     Usage: histogram-start left | center | right | off",
                  SS_acc_int,
                  &_PG_hist_start);

#ifdef HAVE_JPEGLIB
    SS_install_cf("jpeg-flag",
                  "Variable: When TRUE print will generate a plot in a JPEG file",
                  SS_acc_int,
                  &SX_jpeg_flag);

    SS_install_cf("jpeg-name",
                  "Variable: JPEG file name\n     Usage: jpeg-name <string>",
                  SS_acc_ptr,
                  &SX_jpeg_name);

    SS_install_cf("jpeg-type",
                  "Variable: JPEG file type (monochrome or rgb)\n     Usage: jpeg-type <string>",
                  SS_acc_ptr,
                  &SX_jpeg_type);
#else
    SS_install("jpeg-flag",
               "jpeg-flag is not supported at present",
               SS_zargs,
               _SX_no_jpeg_support, SS_PR_PROC);

    SS_install("jpeg-name",
               "jpeg-name is not supported at present",
               SS_zargs,
               _SX_no_jpeg_support, SS_PR_PROC);

    SS_install("jpeg-type",
               "jpeg-type is not supported at present",
               SS_zargs,
               _SX_no_jpeg_support, SS_PR_PROC);
#endif

    SS_install_cf("jpeglib_flag",
                  "Variable: internal use only for JPEG device",
                  SS_acc_ptr,
                  &SX_jpeglib_flag);

/* KLMN */
    SS_install_cf("labels",
                  "Variable: Controls drawing plot labels\n    Usage: labels on | off ",
                  SS_acc_int,
                  &PG_plot_labels);

    SS_install_cf("label-color-flag",
                  "Variable: When TRUE print curve label same color as curve",
                  SS_acc_int,
                  &SX_label_color_flag);

    SS_install_cf("label-length",
                  "Variable: The length of the label shown by lst et al\n     Usage: label-length <integer> ",
                  SS_acc_int,
                  &SX_label_length);

    SS_install_cf("label-space",
                  "Variable: Fractional space for curve labels\n     Usage: label-space <real>",
                  SS_acc_REAL,
                  &SX_label_space);

    SS_install_cf("label-type-size",
                  "Variable: Label type size for plot\n     Usage: label-type-size [<integer>]",
                  SS_acc_int,
                  &SX_label_type_size);

    SS_install_cf("leftspace",
                  "Variable: Fractional space at left of screen\n     Usage: leftspace <real>",
                  SS_acc_REAL,
                  &SX_leftspace);

    SS_install_cf("logical-plot",
                  "Variable: Controls plotting ranges vs indexes\n     Usage: logical-plot <on | off>",
                  SS_acc_int,
                  &_PG_logical_plot);

    SS_install_cf("map-color-to-extrema",
                  "Variable: Map out of bounds values to max or min\n  Usage: map-color-to-extrema <on | off>",
                  SS_acc_int,
                  &_PG_color_map_to_extrema);

    SS_install_cf("marker-index",
                  "Variable: Reference the nth marker\n     Usage: marker-index <integer>",
                  SS_acc_int,
                  &_PG_marker_index);

    SS_install_cf("marker-scale",
                  "Variable: The marker scale factor\n     Usage: marker-scale <real>",
                  SS_acc_REAL,
                  &SX_marker_scale);

    SS_install_cf("marker-orientation",
                  "Variable: The marker orientation angle\n     Usage: marker-orientation <real>",
                  SS_acc_REAL,
                  &SX_marker_orientation);

/* OPQR */

    SS_install_cf("phi",
                  "Variable: Default phi view angle\n     Usage: phi <real>",
                  SS_acc_REAL,
                  &SX_phi);

    SS_install_cf("plot-flag",
                  "Variable: Flag that controls whether or not a handler plots\n     Usage: plot-flag on | off",
                  SS_acc_int,
                  &SX_plot_flag);

    SS_install_cf("plot-labels",
                  "Variable: Print curve labels if TRUE\n     Usage: plot-labels on | off",
                  SS_acc_int,
                  &PG_plot_labels);

    SS_install_cf("plot-type",
                  "Variable: Controls plot type\n     Usage: plot-type cartesian | polar | insel",
                  SS_acc_int,
                  &_PG_plot_type);

    SS_install_cf("print-flag",
                  "Variable: Controls the interpreter value output\n     Usage: print-flag on | off",
                  SS_acc_int,
                  &SS_print_flag);

    SS_install_cf("print-stats",
                  "Variable: Controls the interpreter statistics output\n     Usage: print-flag on | off",
                  SS_acc_int,
                  &SS_stat_flag);

    SS_install_cf("prompt",
                  "Variable: The prompt\n     Usage: prompt <string>",
                  SS_acc_string,
                  SS_prompt);

    SS_install_cf("promotion-type",
                  "Variable: Data type for promotion of sets and arrays (default none)\n     Usage: promotion-type <string>",
                  SS_acc_ptr,
                  &SX_promotion_type);

    SS_install_cf("ps-dots-inch",
                  "Variable: PostScript dots/inch for 8.5 x 11 page\n     Usage: ps-dots-inch <real>",
                  SS_acc_REAL,
                  &_PG_ps_dots_inch);

    SS_install_cf("ps-flag",
                  "Variable: When TRUE print will generate a plot in a PostScript file",
                  SS_acc_int,
                  &SX_ps_flag);

    SS_install_cf("ps-name",
                  "Variable: PostScript file name\n     Usage: ps-name <string>",
                  SS_acc_ptr,
                  &SX_ps_name);

    SS_install_cf("ps-type",
                  "Variable: PostScript file type (monochrome or color)\n     Usage: ps-type <string>",
                  SS_acc_ptr,
                  &SX_ps_type);

    SS_install_cf("ref-mesh",
                  "Variable: Controls drawing a reference mesh\n     Usage: ref-mesh on | off ",
                  SS_acc_int,
                  &_PG_ref_mesh);

    SS_install_cf("ref-mesh-color",
                  "Variable: Controls line color of a reference mesh\n     Usage: ref-mesh <int>",
                  SS_acc_int,
                  &_PG_ref_mesh_color);

    SS_install_cf("render-1d-1d",
                  "Variable: Render 1D vs 1D plots\n     Usage: render-1d-1d <int>",
                  SS_acc_int,
                  &SX_render_1d_1d);

    SS_install_cf("render-2d-1d",
                  "Variable: Render 2D vs 1D plots\n     Usage: render-2d-1d <int>",
                  SS_acc_int,
                  &SX_render_2d_1d);

    SS_install_cf("render-2d-2d",
                  "Variable: Render 2D vs 2D plots\n     Usage: render-2d-2d <int>",
                  SS_acc_int,
                  &SX_render_2d_2d);

    SS_install_cf("render-3d-1d",
                  "Variable: Render 3D vs 1D plots\n     Usage: render-3d-1d <int>",
                  SS_acc_int,
                  &SX_render_3d_1d);

    SS_install_cf("rightspace",
                  "Variable: Fractional space at right of screen\n     Usage: rightspace <real>",
                  SS_acc_REAL,
                  &SX_rightspace);


/* STUV */

    SS_install_cf("scatter-plot",
                  "Variable: Controls scatter plotting\n     Usage: scatter-plot on | off",
                  SS_acc_int,
                  &_PG_scatter_plot);

    SS_install_cf("show-mouse-location",
                  "Variable: Controls display of mouse posision\n     Usage: show-mouse-location on | off",
                  SS_acc_int,
                  &SX_show_mouse_location);

    SS_install_cf("show-mouse-location-x",
                  "Variable: Controls location of mouse posision display\n     Usage: show-mouse-location-x <real>",
                  SS_acc_REAL,
                  &SX_show_mouse_location_x);

    SS_install_cf("show-mouse-location-y",
                  "Variable: Controls location of mouse posision display\n     Usage: show-mouse-location-y <real>",
                  SS_acc_REAL,
                  &SX_show_mouse_location_y);

    SS_install_cf("smooth-method",
                  "Variable: Method for smooth functions\n     Usage: smooth-method \"fft\" | \"averaging\" | \"tchebyshev\" | \"least-sqr\"",
                  SS_acc_ptr,
                  &SX_smooth_method);

    SS_install_cf("squeeze-labels",
                  "Variable: Controls squeezing of extra blanks from labels\n     Usage: squeeze-labels on | off",
                  SS_acc_int,
                  &SX_squeeze_labels);

    SS_install_cf("text-format",
                  "Variable: Controls format for various text\n     Usage: text-format <format>",
                  SS_acc_ptr,
                  &_PG_text_format);

    SS_install_cf("theta",
                  "Variable: Default theta view angle\n     Usage: theta <real>",
                  SS_acc_REAL,
                  &SX_theta);

    SS_install_cf("topspace",
                  "Variable: Fractional space at top of screen\n     Usage: topspace <real>",
                  SS_acc_REAL,
                  &SX_topspace);

    SS_install_cf("type-face",
                  "Variable: Font type face for plot\n     Usage: type-face helvetica | times | courier",
                  SS_acc_ptr,
                  &_PG_axis_type_face);

    SS_install_cf("type-size",
                  "Variable: Font type size for plot\n     Usage: type-size <integer>",
                  SS_acc_int,
                  &SX_plot_type_size);

    SS_install_cf("type-style",
                  "Variable: Font type style for plot\n     Usage: type-style medium | italic | bold | bold-italic",
                  SS_acc_ptr,
                  &SX_plot_type_style);

/* WXYZ */

    SS_install_cf("x-log-scale",
                  "Variable: Controls log scale on x axis\n     Usage: x-log-scale on | off",
                  SS_acc_int,
                  &SX_x_log_scale);

    SS_install_cf("y-log-scale",
                  "Variable: Controls log scale on y axis\n     Usage: y-log-scale on | off",
                  SS_acc_int,
                  &SX_y_log_scale);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_NO_JPEG_SUPPORT - inform that the variable is not supported */

#ifndef HAVE_JPEGLIB
static object *_SX_no_jpeg_support()
   {PRINT(stdout, "JPEG is not supported at present\n\n");
    return(SS_f);}
#endif

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
