/*
 * TRNACA.C - command processors for NACF spoke in TRANSL
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "trnacf.h"

#ifdef HAVE_NACF

char
 *TR_NACFILE_S = "NACfile";

static int
 n_structs = 0;

static char
 *NAC_types[] = {"double", "float", "double",
                 NULL, NULL, NULL, NULL, NULL,
                 "long", "int", "short", "long", 
                 NULL, NULL, NULL, NULL,
                 "char", "char",
                 "char", NULL, NULL, NULL, NULL, NULL,
                 "char", "char", NULL, NULL,
                 NULL, NULL, NULL, NULL, NULL};

extern long
 SC_DECLARE(_PD_extract_field, (char *in, int off, int nbi,
			     int nby, int *ord));

long
 SC_DECLARE(*_NAC_unpack_table,
         (PDBfile *file, char *rt, char *dt,
          int n, int ndt, int *pnum));

void
 SC_DECLARE(_NAC_build_name_table,
         (PDBfile *file, char *names,
          long *ofd, long *ofa, long *ofm, 
          int n, int numdir, int numatt, int numdim));

defstr
 SC_DECLARE(*TR_proc_nac_str,
         (PDBfile *file, char *names,
          long *ofd, long *ofa, long *ofm,
          int n, int numdir, int numatt, int numdim)),
 SC_DECLARE(*TR_find_str, (PDBfile *file, memdes *dp));

int
 SC_DECLARE(_TR_dimens_eq, (dimdes *da, dimdes *db));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* TR_INSTALL_NACF_FUNCS - install the NACF extensions to Scheme */
 
void TR_install_nacf_funcs()
   {TR_file_funcs[N_file_types].open      = (PFPByte) NAC_open;
    TR_file_funcs[N_file_types].close     = (PFPByte) NAC_close;
    TR_file_funcs[N_file_types].type      = TR_NACFILE_S;
    TR_file_funcs[N_file_types].type_hook = TR_nacfilep;
    N_file_types++;

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TR_NACFILEP - return TRUE iff an object is a G_FILE of type NAC file */

int TR_nacfilep(obj)
   object *obj;
   {if (SX_FILEP(obj))
       {if (strcmp(FILE_EXT_TYPE(obj), TR_NACFILE_S) == 0)
           return(TRUE);};

    return(FALSE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* NAC_CLOSE - close an NAC file */

int NAC_close(file)
   PDBfile *file;
   {FILE *fp;

    switch (setjmp(_PD_close_err))
       {case ABORT    : return(FALSE);
        case ERR_FREE : return(TRUE);
        default       : memset(PD_err, 0, MAXLINE);
                        break;};

    fp = file->stream;
    if (io_close(fp) != 0)
       PD_error("CAN'T CLOSE FILE - NAC_CLOSE", PD_CLOSE);

/* free the space */
    _PD_rl_pdb(file);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* NAC_OPEN - open an existing NAC File */

PDBfile *NAC_open(name, mode)
   char *name, *mode;
   {unsigned char str[MAXLINE];
    char fname[MAXLINE], id[MAXLINE], *idmf, *ifdm;
    char *idf, *ifd, *iaf, *ifa;
    char *names, *bf;
    long SS_params[10], *tp, *ofd, *ofa, *ofm;
    long tptlen, tptaddr, dftaddr, dftlen, ntaddr, ntlen;
    long dtaddr, dtlen, aftaddr, aftlen, ataddr, atlen;
    long dmftaddr, dmftlen, dmaddr, dmlen;
    int numdir, numatt, numdim;
    PDBfile *file;
    FILE *fp;

    file = NULL;
    fp   = NULL;
    switch (setjmp(_PD_open_err))
       {case ABORT    : io_close(fp);
                        return(NULL);
        case ERR_FREE : return(file);
        default       : memset(PD_err, 0, MAXLINE);
                        break;};

/* open the file */
    strcpy(fname, name);

#ifdef PCC

    fp = io_open(fname, BINARY_MODE_RPLUS);
    if (fp == NULL)
       PD_error("CAN'T OPEN FILE - NAC_OPEN", PD_OPEN);

#endif

#ifdef ANSI

    fp = io_open(fname, BINARY_MODE_RPLUS);
    if (fp == NULL)
       PD_error("CAN'T OPEN FILE - NAC_OPEN", PD_OPEN);

#endif

    if (PD_buffer_size != -1)
       if (setvbuf(fp, NULL, _IOFBF, (size_t) PD_buffer_size*BUFSIZ))
          PD_error("CAN'T SET FILE BUFFER - NAC_OPEN", PD_OPEN);

/* attempt to read an ASCII header */
    if (io_seek(fp, 0L, SEEK_SET))
       PD_error("FSEEK FAILED TO FIND ORIGIN - NAC_OPEN", PD_OPEN);
    if (io_read(str, CRAY_BYTES_WORD, 10, fp) != 10)
       PD_error("CAN'T READ THE FILE HEADER - NAC_OPEN", PD_OPEN);

/* grab the ASCII id */
    strncpy(id, (char *) &str[8], CRAY_BYTES_WORD);
    id[CRAY_BYTES_WORD] = '\0';

    if ((str[0] != 0x6E) || (str[1] != 0x91) ||
        (strcmp(id, "nac-file") != 0))
       PD_error("NOT NAC FILE - NAC_OPEN", PD_OPEN);

    file = _PD_mk_pdb(name);
    if (file == NULL)
       PD_error("CAN'T ALLOCATE NACFILE - NAC_OPEN", PD_OPEN);

    file->stream         = fp;
    file->std            = _PD_copy_standard(&CRAY_STD);
    file->align          = _PD_copy_alignment(&UNICOS_ALIGNMENT);
    file->default_offset = 1;
    file->major_order    = COLUMN_MAJOR_ORDER;
    file->type           = SC_strsavef(TR_NACFILE_S,
                           "char*:NAC_OPEN:type");
    if (*mode == 'a')
       file->mode = PD_APPEND;
    else
       file->mode = PD_OPEN;

    if (_PD_compare_std(file->host_std, file->std,
                        file->host_align, file->align))
       {file->std   = file->host_std;
        file->align = file->host_align;};

    _PD_init_chrt(file);

    TR_conv_in(file, SS_params, &str[16], "long", 8L);

/* get the table pointer table */
    tptlen  = SS_params[4];
    tptaddr = SS_params[5]*CRAY_BYTES_WORD;
    bf = FMAKE_N(char, tptlen*CRAY_BYTES_WORD, "NAC_OPEN:bf");
    tp = FMAKE_N(long, tptlen, "NAC_OPEN:tp");
    if (io_seek(fp, tptaddr, SEEK_SET))
       PD_error("FAILED TO FIND TABLE POINTER TABLE - NAC_OPEN", PD_OPEN);
    if (io_read(bf, CRAY_BYTES_WORD, tptlen, fp) != tptlen)
       PD_error("FAILED TO READ TABLE POINTER TABLE - NAC_OPEN", PD_OPEN);
    TR_conv_in(file, tp, bf, "long", (long) tptlen);
    SFREE(bf);

/* get the directory field table */
    dftaddr = tp[19]*CRAY_BYTES_WORD;
    dftlen  = tp[20];
    idf = FMAKE_N(char, dftlen*CRAY_BYTES_WORD, "NAC_OPEN:idf");
    if (io_seek(fp, dftaddr, SEEK_SET))
       PD_error("FAILED TO FIND DIRECTORY FIELD TABLE - NAC_OPEN", PD_OPEN);
    if (io_read(idf, CRAY_BYTES_WORD, dftlen, fp) != dftlen)
       PD_error("FAILED TO READ DIRECTORY FIELD TABLE - NAC_OPEN", PD_OPEN);

/* get the directory table */
    dtaddr = tp[3]*CRAY_BYTES_WORD;
    dtlen  = tp[4];
    ifd = FMAKE_N(char, dtlen*CRAY_BYTES_WORD, "NAC_OPEN:ifd");
    if (io_seek(fp, dtaddr, SEEK_SET))
       PD_error("FAILED TO FIND DIRECTORY TABLE - NAC_OPEN", PD_OPEN);
    if (io_read(ifd, CRAY_BYTES_WORD, dtlen, fp) != dtlen)
       PD_error("FAILED TO READ DIRECTORY TABLE - NAC_OPEN", PD_OPEN);

/* get the attribute field table */
    aftaddr = tp[22]*CRAY_BYTES_WORD;
    aftlen  = tp[23];
    iaf = FMAKE_N(char, aftlen*CRAY_BYTES_WORD, "NAC_OPEN:iaf");
    if (io_seek(fp, aftaddr, SEEK_SET))
       PD_error("FAILED TO FIND ATTRIBUTE FIELD TABLE - NAC_OPEN", PD_OPEN);
    if (io_read(iaf, CRAY_BYTES_WORD, aftlen, fp) != aftlen)
       PD_error("FAILED TO READ ATTRIBUTE FIELD TABLE - NAC_OPEN", PD_OPEN);

/* get the attribute table */
    ataddr = tp[7]*CRAY_BYTES_WORD;
    atlen  = tp[8];
    ifa = FMAKE_N(char, atlen*CRAY_BYTES_WORD, "NAC_OPEN:ifa");
    if (io_seek(fp, ataddr, SEEK_SET))
       PD_error("FAILED TO FIND ATTRIBUTE TABLE - NAC_OPEN", PD_OPEN);
    if (io_read(ifa, CRAY_BYTES_WORD, atlen, fp) != atlen)
       PD_error("FAILED TO READ ATTRIBUTE TABLE - NAC_OPEN", PD_OPEN);

    ofd = _NAC_unpack_table(file, idf, ifd, dftlen, dtlen, &numdir);
    ofa = _NAC_unpack_table(file, iaf, ifa, aftlen, atlen, &numatt);

/* get the dimensionality field table */
    dmftaddr = tp[25]*CRAY_BYTES_WORD;
    dmftlen  = tp[26];
    idmf = FMAKE_N(char, dmftlen*CRAY_BYTES_WORD, "NAC_OPEN:idmf");
    if (io_seek(fp, dmftaddr, SEEK_SET))
       PD_error("FAILED TO FIND DIMENSIONALITY FIELD TABLE - NAC_OPEN", PD_OPEN);
    if (io_read(idmf, CRAY_BYTES_WORD, dmftlen, fp) != dmftlen)
       PD_error("FAILED TO READ DIMENSIONALITY FIELD TABLE - NAC_OPEN", PD_OPEN);

/* get the dimensionality table if there is one */
    dmaddr = tp[10]*CRAY_BYTES_WORD;
    dmlen  = tp[11];
    ofm    = NULL;
    if (dmlen > 0)
       {ifdm   = FMAKE_N(char, dmlen*CRAY_BYTES_WORD, "NAC_OPEN:ifdm");
        if (io_seek(fp, dmaddr, SEEK_SET))
           PD_error("FAILED TO FIND DIMENSIONALITY TABLE - NAC_OPEN", PD_OPEN);
        if (io_read(ifdm, CRAY_BYTES_WORD, dmlen, fp) != dmlen)
           PD_error("FAILED TO READ DIMENSIONALITY TABLE - NAC_OPEN", PD_OPEN);
        ofm = _NAC_unpack_table(file, idmf, ifdm, dmftlen, dmlen, &numdim);};

/* get the name table */
    ntaddr = tp[13]*CRAY_BYTES_WORD;
    ntlen  = tp[14];
    names  = FMAKE_N(char, ntlen, "NAC_OPEN:names");
    if (io_seek(fp, ntaddr, SEEK_SET))
       PD_error("FAILED TO FIND NAME TABLE - NAC_OPEN", PD_OPEN);
    if (io_read(names, 1, ntlen, fp) != ntlen)
       PD_error("FAILED TO READ NAME TABLE - NAC_OPEN", PD_OPEN);

    _NAC_build_name_table(file, names, ofd, ofa, ofm, tp[6],
                          numdir, numatt, numdim);

    SFREE(idf);
    SFREE(ifd);
    SFREE(iaf);
    SFREE(ifa);
    SFREE(ofd);
    SFREE(ofa);
    SFREE(names);

    return(file);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _NAC_BUILD_NAME_TABLE - build the NAC name table (ala symbol table) */

void _NAC_build_name_table(file, names, ofd, ofa, ofm, n,
                           numdir, numatt, numdim)
   PDBfile *file;
   char *names;
   long *ofd, *ofa, *ofm;
   int n, numdir, numatt, numdim;
   {long next, itype, mult;
    long *pd, *pa, *pm;
    long addr, numb, mini, leng, width;
    char name[MAXLINE], *type;
    HASHTAB *nt;
    syment *ep;
    dimdes *dims, *nxt, *prev;
    defstr *dp;

    n_structs = 0;

    nt = file->symtab;

    next = n;
    while (next)
       {pd = ofd + (next - 1)*numdir;
        pa = ofa + (pd[5] - 1)*numatt;

        strncpy(name, names + pd[3], pd[4]);
        name[pd[4]] = '\0';

        dims  = NULL;
        addr  = pd[0]*CRAY_BYTES_WORD;
        width = pa[2];
        numb  = pa[1];
        itype = pa[0];
        type  = NAC_types[itype];
        switch (itype)
           {case NAC_ASCII : mult  = CRAY_BYTES_WORD*((width + 7)/CRAY_BYTES_WORD);
                             numb *= mult;
                             dims  = _PD_mk_dimensions(1L, mult);
                             break;

            case NAC_CHARS : mult  = width;
                             numb *= mult;
                             dims  = _PD_mk_dimensions(1L, mult);
                             break;

            case NAC_LJBF  : mult  = CRAY_BYTES_WORD*((width + 63)/64);
                             numb *= mult;
                             dims  = _PD_mk_dimensions(1L, mult);
                             break;

            case NAC_BIT   : numb = (numb + 7)/CRAY_BYTES_WORD;
                             dims = _PD_mk_dimensions(1L, numb);
                             break;

            case NAC_REC   : dp = TR_proc_nac_str(file, names,
                                                  ofd, ofa, ofm, next,
                                                  numdir, numatt, numdim);
                             if (dp == NULL)
                                {PRINT(stdout,
                                       "WARNING: VARIABLE %s HAS BAD STRUCTURE - _NAC_BUILD_NAME_TABLE\n", name);
                                 next = pd[7];
                                 continue;};

                             type = dp->type;

                             break;};

        next = pd[7];
        prev = dims;
        if (pd[6] == 0)
           {leng = numb;
	    dims = _PD_mk_dimensions(1L, leng);}

        else
           {pm = ofm + (pd[6] - 1)*numdim;
            while (TRUE)

/* check to see whether minimum dimension index is stored */
               {if (pm[0] == 0)
                   mini = 1L;
                else
                   mini = pm[2] ? pm[1] : -pm[1];

/* check to see whether maximum dimension index is stored */
                if (pm[5] == 0)
                   leng = numb;
                else
                   {leng  = pm[7] ? pm[6] : -pm[6];
                    leng -= (mini - 1L);};

/* skip variables with dimensions of length less than one */
		if (leng <= 0)
		   break;

                nxt = _PD_mk_dimensions(mini, leng);
                if (dims == NULL)
                   dims = nxt;
                else
                   prev->next = nxt;
                prev = nxt;

                if (pm[13] != 0)
                   pm = ofm + (pm[13] - 1)*numdim;
                else
                   break;};};

/* skip variables with dimensions of length less than one */
	if (leng <= 0)
	   continue;

        if (type == NULL)
           {PRINT(stdout, "UNKNOWN TYPE %d - _NAC_BUILD_NAME_TABLE\n", itype);
            continue;};

        ep = _PD_mk_syment(type, numb, addr, 0L, dims);

        SC_install(name, ep, SYMENT, nt);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _NAC_UNPACK_TABLE - unpack a table
 *                   - FT  - the field table
 *                   - RT  - the raw table data
 *                   - DT  - the packed data table (byte stream)
 *                   - N   - the number of CRAY words in RT
 *                   - NDT - the number of CRAY words in DT
 */

long *_NAC_unpack_table(file, rt, dt, nrt, ndt, pnum)
   PDBfile *file;
   char *rt, *dt;
   int nrt, ndt, *pnum;
   {int i, j, n, bit_off, n_bits, count, first;
    char *nb, *pdt;
    long *out, *prt, *ft, *pout, *pft;
    long v, nwpe, nbpe, nitems;
    int *ord;
    static int size_max = SC_BITS_BYTE*sizeof(long);

    n  = nrt/5;
    ft = FMAKE_N(long, nrt, "_NAC_UNPACK_TABLE:ft");
    TR_conv_in(file, ft, rt, "long", (long) nrt);

/* compute the number of words per entry (nwpe) */
    nwpe = 0L;
    for (i = 0; i < n; i++)
        {v    = ft[5*i];
         nwpe = max(nwpe, v);};

/* compute the number of entries in the table, dt */
    nitems = ndt/nwpe;
    nbpe   = CRAY_BYTES_WORD*nwpe;

/* compute the byte order */
    ord = FMAKE_N(int, nbpe, "_NAC_UNPACK_TABLE:ord");
    for (i = 0; i < nbpe; i++)
        ord[i] = i+1;

/* compute the number of bits in each mask */
    nb = FMAKE_N(char, n, "_NAC_UNPACK_TABLE:nb");
    for (i = 0; i < n; i++)
        {prt   = (long *) (rt + i*40 + 16);
         switch (sizeof(long))
            {case 4  : count  = SC_bit_count(*prt++, sizeof(long));
                       count += SC_bit_count(*prt++, sizeof(long));
                       break;
             case 8  : count = SC_bit_count(*prt++, CRAY_BYTES_WORD);
                       break;
             default : PD_error("DIDN'T EXPECT THIS - _NAC_UNPACK_TABLE",
                                PD_OPEN);};
         nb[i] = count;};

    out = FMAKE_N(long, nitems*n, "_NAC_UNPACK_TABLE:out");

    first = TRUE;
    for (i = 0; i < nitems; i++)
        {pout = out + i*n;
         pdt  = dt  + i*nbpe;
         for (j = 0; j < n; j++)
             {pft     = ft + j*5;
              n_bits  = nb[j];
              bit_off = pft[0]*64 - n_bits - pft[1];
              if (n_bits >= size_max)
                 {if (first)
                     {PRINT(stdout, "WARNING: TRUNCATED A FIELD - _NAC_UNPACK_TABLE\n");
                      first = FALSE;};
                  bit_off += (n_bits - size_max);
                  n_bits   = size_max;};

              pout[j] = _PD_extract_field(pdt, bit_off, n_bits,
                                          nbpe, ord);};};
                                          
    SFREE(ord);
    SFREE(ft);
    SFREE(nb);

    *pnum = n;

    return(out);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TR_PROC_NAC_STR - process NAC record structure defining defstr's as
 *                 - it goes
 */

defstr *TR_proc_nac_str(file, names, ofd, ofa, ofm, n,
                        numdir, numatt, numdim)
   PDBfile *file;
   char *names;
   long *ofd, *ofa, *ofm;
   int n, numdir, numatt, numdim;
   {char memb[MAXLINE], mname[MAXLINE], mdims[MAXLINE], type_name[MAXLINE];
    char *mtype;
    int doffs, ndims, skip;
    long next, itype, mini, maxi, width, numb, mult;
    long *pd, *pa, *pm;
    HASHTAB *fchrt;
    defstr *dp;
    memdes *desc, *lst, *prev;

    prev  = NULL;
    lst   = NULL;
    fchrt = file->chart;
    doffs = file->default_offset;

    pd = ofd + (n - 1)*numdir;

/* start with the first child */
    next = pd[8];

    while (next)
       {skip = 0;

	pd = ofd + (next - 1)*numdir;
        pa = ofa + (pd[5] - 1)*numatt;

/* get the member name */
        strncpy(mname, names + pd[3], pd[4]);
        mname[pd[4]] = '\0';

/* if the member is a struct recurse down */
        sprintf(mdims, "(");
        ndims = 0;
        width = pa[2];
        numb  = pa[1];
        itype = pa[0];
        mtype = NAC_types[itype];
        switch (itype)
           {case NAC_ASCII : mult = CRAY_BYTES_WORD*((width + 7)/CRAY_BYTES_WORD);
                             if (mult > 1)
                                {sprintf(memb, "%ld", mult);
                                 strcat(mdims, memb);
                                 ndims++;};
                             break;

            case NAC_CHARS : if (width > 1)
                                {sprintf(memb, "%ld", width);
                                 strcat(mdims, memb);
                                 ndims++;};
                             break;

            case NAC_LJBF  : mult = CRAY_BYTES_WORD*((width + 63)/64);
                             if (mult > 1)
                                {sprintf(memb, "%ld", mult);
                                 strcat(mdims, memb);
                                 ndims++;};
                             break;

            case NAC_BIT   : numb  = (numb + 7)/CRAY_BYTES_WORD;
                             pd[6] = 0;
                             break;

            case NAC_REC   : dp = TR_proc_nac_str(file, names, ofd, ofa, ofm,
                                                  next,
                                                  numdir, numatt, numdim);
                             if (dp == NULL)
                                return(NULL);

                             mtype = dp->type;
                             break;};

        if (mtype == NULL)
           return(NULL);

        next = pd[7];

/* get the member dimensions dimensions */
        if (pd[6] == 0)
           {if (ndims == 1)
               strcat(mdims, ",");
            sprintf(memb, "%ld", numb);
            strcat(mdims, memb);
            ndims++;
            strcat(mdims, ")");}

        else
           {if (ndims == 1)
               strcat(mdims, ",");

            pm = ofm + (pd[6] - 1)*numdim;
            while (TRUE)

/* check to see whether minimum dimension index is stored */
               {if (pm[0] == 0)
                   mini = 1L;
                else
                   mini = pm[2] ? pm[1] : -pm[1];

/* check to see whether maximum dimension index is stored */
                if (pm[5] == 0)
                   maxi = pa[1];
                else
                   maxi = pm[7] ? pm[6] : -pm[6];

		if (mini > maxi)
		   {skip = 1;
		    break;};

                if (mini == 1)
                   sprintf(memb, "%ld", maxi);
                else
                   sprintf(memb, "%ld:%ld", mini, maxi);
                strcat(mdims, memb);
                ndims++;

                if (pm[13] != 0)
                   {pm = ofm + (pm[13] - 1)*numdim;
                    strcat(mdims, ",");}
                else
                   {strcat(mdims, ")");
                    break;};};};
	
/* skip members that have dimensions of length less than one */
	if (skip)
	   continue;

        if ((ndims == 1) && (numb == 1))
           mdims[0] = '\0';

        sprintf(memb, "%s %s%s", mtype, mname, mdims);
        desc = _PD_mk_descriptor(memb, doffs);

        if (lst == NULL)
           lst = desc;
        else
           prev->next = desc;
        prev = desc;};

/* make the type name */
    sprintf(type_name, "str-%d", n_structs);

    dp = TR_find_str(file, lst);
    if (dp == NULL)
       {n_structs++;

/* install the type in both charts */
        dp = _PD_defstr_inst(type_name, lst, -1, NULL, NULL,
                             fchrt, file->host_chart,
                             file->align, file->host_align, FALSE);}
    else
       for (desc = lst; desc != NULL; desc = lst)
	   {lst = desc->next;
	    _PD_rl_descriptor(desc);};

    return(dp);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TR_FIND_STR - if the given memdes matches an existing one
 *               return a pointer to it otherwise return NULL
 */

defstr *TR_find_str(file, dp)
   PDBfile *file;
   memdes *dp;
   {int differ;
    defstr *ndp;
    memdes *ndsc, *odsc;
    hashel *hp;
    HASHTAB *chrt;

    chrt = file->chart;

    for (hp = *(chrt->table); hp != NULL; hp = hp->next)
        {ndp = (defstr *) hp->def;

         differ = FALSE;
         ndsc   = ndp->members;
         odsc   = dp;
         while (TRUE)
            {if ((ndsc == NULL) || (odsc == NULL))
                break;

#if 0
             if ((strcmp(ndsc->type, odsc->type) != 0) ||
                 (ndsc->number != odsc->number) ||
                 (!_TR_dimens_eq(ndsc->dimensions, odsc->dimensions)))
#endif

             if (strcmp(ndsc->member, odsc->member) != 0)
                {differ = TRUE;
                 break;};

             ndsc = ndsc->next;
             odsc = odsc->next;};

         if ((ndsc != NULL) || (odsc != NULL))
            continue;

         if (!differ)
            break;};

    return((hp == NULL) ? NULL : ndp);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _TR_DIMENS_EQ - return TRUE if the two dimension specifications are
 *               - the same and FALSE otherwise
 */

int _TR_dimens_eq(da, db)
   dimdes *da, *db;
   {dimdes *nda, *ndb;
    int same;

    same = TRUE;
    nda  = da;
    ndb  = db;
    while (TRUE)
       {if ((nda == NULL) || (ndb == NULL))
           break;

        if ((nda->index_min != ndb->index_min) ||
            (nda->index_max != ndb->index_max) ||
            (nda->number != ndb->number))
           {same = FALSE;
            break;};

        nda = nda->next;
        ndb = ndb->next;};

    if ((nda != NULL) || (ndb != NULL))
       same = FALSE;

    return(same);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

#endif

