;
; SXDM.SCM - define routines to compare PM_mappings in PDB files
;          - and display mappings whose difference measure exceeds
;          - the specified tolerance (default 1.0e-8)
;
; Source Version: 2.0
; Software Release #92-0043
;
; #include <pact-copyright.h>
; 

(define HUGE 1.0e100)
(define SMALL 1.0e-100)
(define TOLERANCE 1.0e-8)
(define PRECISION 1.0e15)

(define displays-off #t)
(define dev-a  (pg-make-device "unix:0.0" "COLOR" "New Result"))
(define dev-b  (pg-make-device "unix:0.0" "COLOR" "Standard Result"))
(define dev-c  (pg-make-device "unix:0.0" "COLOR" "Difference"))
;(define dev-d  (pg-make-device "unix:0.0" "COLOR" "Integral of Difference"))

(define contour    11)
(define image      12)
(define wire-frame 13)
(define shaded     14)
(define vector     15)

(define theta 45.0)
(define phi -45.0)

; --------------------------------------------------------------------------
; --------------------------------------------------------------------------

; DIFF-MEASURE - something like the chi squared between two mappings

(define (diff-measure a b)
   (let* ((tc (norm a))
	  (td (norm b))
	  (diff (norm (- a b)))
	  (den (+ tc td))

; diff variables
	  (domain-volume (pm-set-volume (pm-mapping-domain diff)))
	  (diff-range-volume (pm-set-volume (pm-mapping-range diff)))

; den variables
	  (ave-range-volume (pm-set-volume (pm-mapping-range den)))

	  meas integ)

     (cond ((eqv? diff-range-volume 0.0)
	    (list 0.0 nil nil))

	   ((or (> (- diff-range-volume ave-range-volume) PRECISION)
		(> diff-range-volume HUGE)
		(> ave-range-volume HUGE))
	    (list PRECISION nil nil))

	   (#t (set! meas (/ diff (pm-shift-range den SMALL)))
	       (set! integ (integrate meas))
	       (list (/ (pm-set-volume (pm-mapping-range integ))
			domain-volume)
		     meas integ)))))

; --------------------------------------------------------------------------
; --------------------------------------------------------------------------

; DIFF-MAPPINGS - the read-eval-print loop for the validation process
;               -
;               - the special commands are:
;               -
;               - NEXT/NO - these curves don't compare and go on to the next pair
;               - OK/YES  - these curves do compare and go on to the next pair
;

(define (diff-mappings file1 file2)
  (let* ((correct 0)
	 (differ nil)
	 (get-response
	  (lambda (n a dc)
	    (printf nil  "\n[%d | %s | %g]-> "
		    n a dc)
	    (let ((form (read)))
	      (cond ((or (eq? form 'next)
			 (eq? form 'n))
		     (set! differ (cons n differ)))
		    ((or (eq? form 'ok)
			 (eq? form 'o)
			 (eq? form 'y))
		     (set! correct (+ correct 1)))
		    (#t
		     (get-response n a dc))))))
	 (diff-mappings-aux
	  (lambda (n file1 file2)
	    (let ((a (pdbdata->pm-mapping file1 n))
		  (b (pdbdata->pm-mapping file2 n))
		  c d dc form)
	      (if (and (not (null? a)) (not (null? b))
		       (not (pm-grotrian-mapping? a))
		       (not (pm-grotrian-mapping? b)))
		  (begin
		    (printf nil "%d " n)
		    (set! form (diff-measure a b))
		    (set! dc (car form))
		    (set! c  (cadr form))
		    (set! d  (caddr form))
		    (if (< dc TOLERANCE)
			(begin
			  (set! correct (+ correct 1))
			  (diff-mappings-aux (+ n 1) file1 file2))
			(begin
			  (if displays-off
			      (open-devices))
			  (show-mappings a b c d)
			  (get-response n (pm-mapping-name a) dc)
			  (diff-mappings-aux (+ n 1)
					     file1 file2)))))))))
    (diff-mappings-aux 1 file1 file2)
    (if (not displays-off)
	(close-devices))
    (newline)
    (list correct differ)))

; --------------------------------------------------------------------------
; --------------------------------------------------------------------------

; OPEN-DEVICES - open four devices to show the mapping from the
;              - new and old files, the difference, and the integral
;              - of the difference

(define-macro (open-devices)
   (pg-open-device dev-a  0.02 0.04 0.3  0.3)
   (pg-open-device dev-b  0.34 0.04 0.3  0.3)
   (pg-open-device dev-c  0.66 0.04 0.3  0.3)
;   (pg-open-device dev-d  0.5  0.54 0.3  0.3)
   (set! displays-off #f))

; --------------------------------------------------------------------------
; --------------------------------------------------------------------------

; CLOSE-DEVICES - close the four devices

(define-macro (close-devices)
   (pg-close-device dev-a)
   (pg-close-device dev-b)
   (pg-close-device dev-c)
;   (pg-close-device dev-d)
   (set! displays-off #t))

; --------------------------------------------------------------------------
; --------------------------------------------------------------------------

; SHOW-MAPPINGS - display the four mappings


(define (show-mappings a b c d)
   (if (not (null? a))
       (pg-draw-graph dev-a a))
   (if (not (null? b))
       (pg-draw-graph dev-b b))
   (if (not (null? c))
       (pg-draw-graph dev-c c))
;   (if (not (null? c))
;       (pg-draw-graph dev-d d))
)

; --------------------------------------------------------------------------
; --------------------------------------------------------------------------

; COMPARE - read in two files and display each pair of matching curves
;          - as prompted
;          - when all have been checked kill all of them and erase the
;          - screen
;          - there is some development to be done here

(define-macro (compare new old)
  (if (ascii-file? new)
      (printf nil "File %s is ascii\n\n" new)
      (if (ascii-file? old)
	  (printf nil "File %s is ascii\n\n" old)
	  (let* (n-total n-correct n-differ differing result
			 (file1 (open-pdbfile new))
			 (file2 (open-pdbfile old))
			 (snew (if (symbol? new)
				   (symbol->string new)
				   new))
			 (file (string-append snew ".chk"))
			 (log (open-output-file file)))
	    (printf nil "\nComparing %s with %s\n" new old)
	    (set! result (diff-mappings file1 file2))
	    (set! n-correct (car result))
	    (set! differing (cadr result))
	    (set! n-differ (if (pair? differing) (length differing) 0))
	    (set! n-total (+ n-correct n-differ))
	    (if (pair? differing)
		(begin
		  (printf log
			  "%d mappings out of %d differ between %s and %s\n"
			  n-differ n-total new old)
		  (for-each '(lambda (n) (printf log "   %d differs\n" n))
			    differing)
		  (printf nil "%s differs from %s\n\n" new old))
		(begin
		  (printf nil "%s matches %s\n\n" new old)
		  (printf log "%s matches %s\n\n" new old)))

	    (close-output-file log)
	    (= n-differ 0)))))

; --------------------------------------------------------------------------
; --------------------------------------------------------------------------

