/*
 * SHPROC.C - C and Scheme Process control routines
 *
 * Source Version: 4.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "scheme.h"

static int
  n_tries;

static void
 SC_DECLARE(_SS_wr_process, (object *obj, object *strm)),
 SC_DECLARE(_SS_rl_process, (object *obj));

static object
 SC_DECLARE(*SS_opn_pr, (object *argl)),
 SC_DECLARE(*SS_pr_runp, (object *obj)),
 SC_DECLARE(*SS_cls_pr, (object *obj)),
 SC_DECLARE(*SS_pr_rd_tries, (object *argl)),
 SC_DECLARE(*SS_prp, (object *obj)),
 SC_DECLARE(*SS_pr_stat, (object *obj)),
 SC_DECLARE(*SS_pr_rd_line, (object *obj)),
 SC_DECLARE(*SS_pr_sn_line, (object *argl));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_INST_PROC - install the primitives for process control */

void SS_inst_proc()
   {
    SS_install("process-open",
               "Procedure: Exec a process and communicate with it",
               SS_nargs,
               SS_opn_pr, SS_PR_PROC);

    SS_install("process-running?",
               "Procedure: Returns #t if the process is still running",
               SS_sargs,
               SS_pr_runp, SS_PR_PROC);

    SS_install("process-status",
               "Procedure: Returns a list of the process id, in, out, status, reason",
               SS_sargs,
               SS_pr_stat, SS_PR_PROC);

    SS_install("process-close",
               "Procedure: Terminate a process",
               SS_sargs,
               SS_cls_pr, SS_PR_PROC);

    SS_install("process?",
               "Procedure: Returns #t if the object is a PROCESS_OBJ",
               SS_sargs,
               SS_prp, SS_PR_PROC);

    SS_install("process-read-line",
               "Procedure: Returns a string recieved from a process",
               SS_sargs,
               SS_pr_rd_line, SS_PR_PROC);

    SS_install("process-send-line",
               "Procedure: Send a string to a process",
               SS_nargs,
               SS_pr_sn_line, SS_PR_PROC);

    SS_install("process-read-tries",
               "Procedure: Get/Set number of attempts to read from unblocked process",
               SS_nargs,
               SS_pr_rd_tries, SS_PR_PROC);

    n_tries = 100;

    return;}

/*--------------------------------------------------------------------------*/

/*                           PROCESS PREDICATES                             */

/*--------------------------------------------------------------------------*/

/* SS_PRP - process? at Scheme level */

static object *SS_prp(obj)
   object *obj;
   {return(SS_processp(obj) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_PR_RUNP - process-running? in Scheme */

static object *SS_pr_runp(obj)
    object *obj;
    {if (!SS_processp(obj))
        SS_error("OBJECT NOT PROCESS - PROCESS-RUNNING?", obj);

     return((SS_PROCESS_STATUS(obj) == RUNNING) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/

/*                           PROCESS PRIMITIVES                             */

/*--------------------------------------------------------------------------*/

/* SS_OPN_PR - process-open in Scheme */

static object *SS_opn_pr(argl)
   object *argl;
   {PROCESS *pp;
    int n, i;
    char **argv, *mode;
    object *obj;

/* pull the mode off */
    mode = NULL;
    SS_args(argl,
            SC_STRING_I, &mode,
            0);

    if (strchr("rwa", mode[0]) == NULL)
       SS_error("BAD MODE - SS_OPN_PR", SS_car(argl));

    argl = SS_cdr(argl);

/* the rest of the args constitute the command line */
    n = (int) _SS_length(argl);

/* we need one extra for a NULL argument to terminate argv */
    argv = FMAKE_N(char *, n+1, "SS_OPN_PR:argv");

    for (i = 0 ; i < n; argl = SS_cdr(argl))
        {obj = SS_car(argl);

         if (SS_stringp(obj))
            argv[i++] = SS_STRING_TEXT(obj);
         else if (SS_variablep(obj))
            argv[i++] = SS_VARIABLE_NAME(obj);
         else
            SS_error("BAD OBJECT - PROCESS-OPEN", obj);};

    argv[i] = NULL;

    pp = PC_open(argv, NULL, mode);
    PC_block_file(stdin);   
    if (pp == NULL)
       SS_error("CAN'T OPEN PROCESS - PROCESS-OPEN", obj);

    SFREE_N(argv, n);

    obj = SS_mk_object(pp, PROCESS_OBJ, SELF_EV, NULL);
    obj->print   = _SS_wr_process;
    obj->release = _SS_rl_process;

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_CLS_PR - process-close in Scheme */

static object *SS_cls_pr(obj)
   object *obj;
   {

    if (!SS_processp(obj))
       SS_error("BAD PROCESS - PROCESS-CLOSE", obj);

    PC_close(SS_PROCESS_VALUE(obj));

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_PR_RD_TRIES - process-read-tries in Scheme */

static object *SS_pr_rd_tries(argl)
   object *argl;
   {

    if (!SS_nullobjp(argl))
       SS_args(argl,
               SC_INTEGER_I, &n_tries,
               0);

    return(SS_mk_integer((long) n_tries));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_WR_PROCESS - print a process object */

static void _SS_wr_process(obj, strm)
   object *obj, *strm;
   {PROCESS *pp;
    int flag;
    FILE *str;

    str = SS_OUTSTREAM(strm);

    pp = SS_PROCESS_VALUE(obj);
    PRINT(str, "<PROCESS|%d-%d-%d-", pp->id, pp->in, pp->out);

    flag = (pp->status & ~CHANGED);
    switch (flag)
       {case RUNNING                 : PRINT(str, "Running");
                                       break;
        case STOPPED                 : PRINT(str, "Stopped-");
                                       break;
        case EXITED                  : PRINT(str, "Exited-");
                                       break;
        case (EXITED | COREDUMPED)   : PRINT(str, "Exited-Core-Dumped-");
                                       break;
        case SIGNALED                : PRINT(str, "Signaled-");
                                       break;
        case (SIGNALED | COREDUMPED) : PRINT(str, "Signaled-Core-Dumped-");
                                       break;
        default                      : PRINT(str, "Unknown");
                                       break;};
    if (flag != RUNNING)
       switch (flag)
          {case STOPPED                 :
           case EXITED                  :
           case (EXITED | COREDUMPED)   :
           case SIGNALED                :
           case (SIGNALED | COREDUMPED) : PRINT(str, "%d", pp->reason);
           default                      : break;};

    PRINT(str, ">");

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_RL_PROCESS - this is need for the garbage collector */

static void _SS_rl_process(obj)
   object *obj;
   {PROCESS *pp;

    pp = SS_PROCESS_VALUE(obj);
    PC_close(pp);

    SFREE(pp);
    SFREE(SS_OBJECT(obj));

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_PR_STAT - process-status at the Scheme level */

static object *SS_pr_stat(obj)
   object *obj;
   {PROCESS *pp;
    int st, rs;
    object *ret;

    if (!SS_processp(obj))
       SS_error("ARGUMENT NOT PROCESS - PROCESS-STATUS", obj);

    pp = SS_PROCESS_VALUE(obj);
    st = pp->status;
    rs = (st == RUNNING) ? 0 : pp->reason;

    ret = SS_make_list(SC_INTEGER_I, &(pp->id),
                       SC_INTEGER_I, &(pp->in),
                       SC_INTEGER_I, &(pp->out),
                       SC_INTEGER_I, &st,
                       SC_INTEGER_I, &rs,
                       0);

    return(ret);}    

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_PR_RD_LINE - process-read-line at Scheme level */

static object *SS_pr_rd_line(obj)
   object *obj;
   {int i;
    char bf[LRG_TXT_BUFFER];

    if (!SS_processp(obj))
       SS_error("ARGUMENT NOT PROCESS - PROCESS-READ-LINE", obj);

    for (i = 0; i < n_tries; i++)
        {if (PC_gets(bf, LRG_TXT_BUFFER, SS_PROCESS_VALUE(obj)) != NULL)
            return(SS_mk_string(bf));};

    return(SS_null);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_PR_SN_LINE - process-send-line at Scheme level */

static object *SS_pr_sn_line(argl)
   object *argl;
   {int ret;
    PROCESS *pp;
    char *s;

    pp = NULL;
    s  = NULL;
    SS_args(argl,
            PROCESS_OBJ, &pp,
            SC_STRING_I, &s,
            0);

    ret = PC_printf(pp, "%s\n", s);

    return(ret ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
