/*
 * PPCAUX.C - Portable Process Control system auxilliary routines
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "ppc.h"

#ifndef INADDR_NONE
# define INADDR_NONE -1
#endif

static PC_poll_desc
 *_PC_fds = NULL;

static PFVoid
 *_PC_fd_fnc = NULL;

static int
 _PC_nfds = 0,
 _PC_maxfds = 0,
 _PC_nfd_fnc = 0,
 _PC_maxfd_fnc = 0;

FILE
 *_PC_diag;

int
 _PC_debug = FALSE,
 _PC_unblock = FALSE,
 PC_io_interrupts_on = FALSE;

jmp_buf
 _PC_err_state;

static void
 SC_DECLARE(_PC_SIGIO_handler, (int sig));

void
 SC_DECLARE(PC_rl_io_callback_lists, (byte));

long
 SC_DECLARE(_PC_transfer_bytes, (int cfd, long nbe, FILE *fp)),
 SC_DECLARE(_PC_get_cmd_resp, (PROCESS *pp, char *msg));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_ERROR - handle PPC error conditions */

#ifdef ANSI

void PC_error(char *fmt, ...)

#endif

#ifdef PCC

void PC_error(fmt, va_alist)
   char *fmt;
   va_dcl

#endif

   {char err[MAXLINE];

    sprintf(PC_err, "\nERROR: ");
    SC_VA_START(fmt);
    SC_VSPRINTF(err, fmt);
    SC_VA_END;

    strcat(PC_err, err);

    longjmp(_PC_err_state, ABORT);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_STRING_TO_ARGS - crack a string and return an array of tokens from
 *                   - the string
 */

char **PC_string_to_args(s, delim)
   char *s, *delim;
   {char **argv, *token;
    int i, n, count;

/* count the number of delimiters in the string */
    n     = strlen(s);
    count = 1;
    for (i = 0; i < n; i++)
        if (strchr(delim, (int) s[i]) != NULL)
           count++;
        
/* allocate the token array to be the number of delimiters long */
    argv = FMAKE_N(char *, count,
           "PC_STRING_TO_ARGS:argv");

/* tokenize the string and copy them into the token array */
    for (i = 0; TRUE; i++)
        {token = SC_firsttok(s, delim);
         if (token != NULL)
            argv[i] = SC_strsavef(token, "char*:PC_STRING_TO_ARGS:token");
         else
            break;};

    return(argv);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_SET_ATTR - set the control flags on a PROCESS
 *             - the following are more or less portable
 *             -
 *             - PC_LINE    line at a time input
 *             - PC_NDELAY  non-blocking I/O
 *             - PC_APPEND  append (writes guaranteed at the end)
 *             - PC_SYNC    synchronous write option
 *             - PC_ASYNC   interrupt-driven I/O for sockets
 */

int PC_set_attr(pp, attr, val)
   PROCESS *pp;
   int attr, val;
   {switch (attr)
       {case PC_LINE  : pp->line_mode = val;
                        break;

        default       : return(PC_set_fd_attr(pp->in, val, TRUE));};

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_SET_IO_ATTR - set the specified attribute to be ON or OFF */

int PC_set_io_attr(fd, attr, state)
   int fd, attr, state;
   {int status;

    PC_ERR_TRAP(FALSE);

#ifndef OPENNT
#ifdef UNIX
    {TERMINAL s;

     status = ioctl(fd, TIOCGETP, &s);
     if (status == -1)
        PC_error("COULDN'T GET STATE - PC_SET_IO_ATTR");

     if (state)

# ifdef BSD_TERMINAL
        s.sg_flags |= attr;
# endif

# ifdef SYSV_TERMINAL
        s.c_lflag |= attr;
# endif

     else

# ifdef BSD_TERMINAL
        s.sg_flags &= ~attr;
# endif

# ifdef SYSV_TERMINAL
        s.c_lflag &= ~attr;
# endif

     status = ioctl(fd, TIOCSETP, &s);
     if (status == -1)
        PC_error("COULDN'T SET STATE - PC_SET_IO_ATTR");
     else
        status = TRUE;};
#endif

    return(status);}
#else

    return(FALSE);}

#endif

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_SET_FD_ATTR - set the control flags on a file descriptor
 *                - the following are more or less portable
 *                -
 *                - PC_LINE    line at a time input
 *                - PC_NDELAY  non-blocking I/O
 *                - PC_APPEND  append (writes guaranteed at the end)
 *                - PC_SYNC    synchronous write option
 *                - PC_ASYNC   interrupt-driven I/O for sockets
 *                -
 *                - the STATE argument specifies whether the flag is to
 *                - be set ON or OFF
 *                - return TRUE iff successful
 */

int PC_set_fd_attr(fd, i, state)
   int fd, i, state;
   {PC_ERR_TRAP(FALSE);

#ifndef DOS
#ifndef MAC

    if (i != PC_ASYNC)
       {int arg, status;

        arg = fcntl(fd, F_GETFL);
        if (arg < 0)
           PC_error("COULDN'T GET DESCRIPTOR FLAG - PC_SET_FD_ATTR");

        if (state)
            arg |= i;
        else
            arg &= ~i;

        status = fcntl(fd, F_SETFL, arg);
        if (status < 0)
           PC_error("COULDN'T SET DESCRIPTOR FLAG - PC_SET_FD_ATTR");}

    else

#ifdef HAVE_STREAMS_P

       {int pid, arg, status;
        PFVoid errf;
        static int first = TRUE;

        if (PC_io_interrupts_on)
           {arg    = 0;
            status = ioctl(fd, I_GETSIG, &arg);
            if (status < 0)
               arg = 0;

            if (state)
               arg |= (S_INPUT | S_HIPRI);
            else
               arg &= ~(S_INPUT | S_HIPRI);

            status = ioctl(fd, I_SETSIG, arg);
            if (status < 0)
               PC_error("SETSIG IOCTL FAILED - PC_SET_FD_ATTR");

            if (first & state)
               {errf = SIGNAL(PC_SIGIO, _PC_SIGIO_handler);
                if (errf == SIG_ERR)
                   PC_error("CAN'T SET SIGIO HANDLER - PC_SET_FD_ATTR");
                first = FALSE;};

            return(TRUE);}

        else
            return(FALSE);};

#else

#ifdef FASYNC

       {int arg, status;
        PFVoid errf;
        static int first = TRUE;

        if (PC_io_interrupts_on)
           {arg = fcntl(fd, F_GETFL);
            if (arg == -1)
               PC_error("GETFL FCNTL FAILED - PC_SET_FD_ATTR");

            if (state)
               arg |= FASYNC;
            else
               arg &= ~FASYNC;

            status = fcntl(fd, F_SETFL, arg);
            if (status == -1)
               PC_error("SETFL FCNTL FAILED - PC_SET_FD_ATTR");

            if (first & state)
               {errf = SIGNAL(PC_SIGIO, _PC_SIGIO_handler);
                if (errf == SIG_ERR)
                   PC_error("CAN'T SET SIGIO HANDLER - PC_SET_FD_ATTR");
                first = FALSE;};

            return(TRUE);}

        else
            return(FALSE);};

#else

       {sprintf(PC_err,
                "INTERRUPT DRIVEN I/O NOT SUPPORTED HERE - PC_SET_FD_ATTR");

        return(FALSE);};

#endif

#endif
#endif

    return(TRUE);}

#else

    return(FALSE);}

#endif

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_ECHO_ON_FILE - set the file input to be echoed */

int PC_echo_on_file(fp)
   FILE *fp;
   {int fd;

    fd = fileno(fp);
    return(PC_echo_on_fd(fd));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_ECHO_ON_FD - set the file descriptor input to be echoed */

int PC_echo_on_fd(fd)
   int fd;

#if defined(DOS) || defined(OPENNT)
   {return(FALSE);}
#else                                                      
   {return(PC_set_io_attr(fd, ECHO, TRUE));}
#endif
        

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_ECHO_OFF_FILE - set the file input to be echoed */

int PC_echo_off_file(fp)
   FILE *fp;
   {int fd;

    fd = fileno(fp);
    return(PC_echo_off_fd(fd));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_ECHO_OFF_FD - set the file descriptor input to be echoed */

int PC_echo_off_fd(fd)
   int fd;

#if defined(DOS) || defined(OPENNT)
   {return(FALSE);}
#else
   {return(PC_set_io_attr(fd, ECHO, FALSE));}
#endif

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_UNBLOCK - set the PROCESS to be unblocked */

int PC_unblock(pp)
   PROCESS *pp;
   {pp->blocking = FALSE;

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_UNBLOCK_FILE - set the file to be unblocked */

int PC_unblock_file(fp)
   FILE *fp;
   {int fd;

/* useful for debug
   if (fp == stdin) return(TRUE);
 */

    fd = fileno(fp);
    return(PC_unblock_fd(fd));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_UNBLOCK_FD - set the file descriptor to be unblocked */

int PC_unblock_fd(fd)
   int fd;
   {int status;

    status = 0;

#ifdef UNIX
#ifndef OPENNT
    status = fcntl(fd, F_GETFL, status);
    status = fcntl(fd, F_SETFL, status | O_NDELAY);
    if (status == -1)
       status = FALSE;
    else
       status = TRUE;
#endif
#endif

    return(status);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_BLOCK - set the PROCESS to be blocked */

int PC_block(pp)
   PROCESS *pp;
   {pp->blocking = TRUE;

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_BLOCK_FILE - set the file to be blocked */

int PC_block_file(fp)
   FILE *fp;
   {int fd;

    fd = fileno(fp);
    return(PC_block_fd(fd));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_BLOCK_FD - set the file descriptor to be blocked */

int PC_block_fd(fd)
   int fd;
   {int status;

    status = 0;

#ifdef UNIX
#ifndef OPENNT
    status = fcntl(fd, F_GETFL, status);
    status = fcntl(fd, F_SETFL, status & ~O_NDELAY);
    if (status == -1)
       status = FALSE;
    else
       status = TRUE;
#endif
#endif

    return(status);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_IO_CALLBACK_FILE - set the file I/O to be interrupt driven */

int PC_io_callback_file(fp, fnc)
   FILE *fp;
   PFVoid fnc;
   {int fd;

    fd = fileno(fp);
    return(PC_io_callback_fd(fd, fnc));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_IO_CALLBACK_FD - set the file descriptor I/O to be interrupt driven */

int PC_io_callback_fd(fd, fnc)
   int fd;
   PFVoid fnc;
   {PC_poll_desc pd;

    pd.fd     = fd;
    pd.events = POLLIN | POLLPRI;

/* remember everyone in this session who is interruptable */
    SC_REMEMBER(PC_poll_desc, pd, _PC_fds, _PC_nfds, _PC_maxfds, 10);
    SC_REMEMBER(PFVoid, fnc, _PC_fd_fnc, _PC_nfd_fnc, _PC_maxfd_fnc, 10);

    return(PC_set_fd_attr(fd, PC_ASYNC, TRUE));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_RL_IO_CALLBACK - remove any callback associated with
 *                   - a file descriptor
 */

void PC_rl_io_callback(fd)
   int fd;
   {int i;

    for (i = 0; i < _PC_nfds; i++)
        {if (_PC_fds[i].fd == fd)
            {_PC_fds[i]    = _PC_fds[--_PC_nfds];
             _PC_fd_fnc[i] = _PC_fd_fnc[--_PC_nfd_fnc];
	     break;};};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_RL_IO_CALLBACK_LISTS - free memory associated with I/O callback lists
 *
 * WARNING: This function is provided for regression test bookkeeping purposes
 * only. Real applications should not use it.
 *
 */

void _PC_rl_io_callback_lists()
   {

    SFREE(_PC_fds);
    SFREE(_PC_fd_fnc);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_SIGIO_HANDLER - respond to an input available interrupt */

static void _PC_SIGIO_handler(sig)
   int sig;
   {

    PC_poll_descriptors(-1);
    PC_catch_io_interrupts(TRUE);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_CATCH_IO_INTERRUPTS - activate the SIGIO handler */

void PC_catch_io_interrupts(flag)
   int flag;
   {if (flag && PC_io_interrupts_on)
       SIGNAL(PC_SIGIO, _PC_SIGIO_handler);
    else
       SIGNAL(PC_SIGIO, SIG_IGN);

/* printf("Catch IO interrupts: %d\n", PC_io_interrupts_on&&flag); */

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_POLL_DESCRIPTORS - poll the registered file descriptors and dispatch
 *                     - to their assigned functions
 *                     - these are registered via PC_io_callback_file and
 *                     - PC_io_callback_fd calls
 *                     - TO is in milliseconds
 *                     - TO of -1 means block
 */

int PC_poll_descriptors(to)
   int to;
   {int i, n;
    short rev;

    PC_err[0] = '\0';
    n = PC_poll(_PC_fds, _PC_nfds, to);
    if (n > 0)
       {for (i = 0; i < _PC_nfds; i++)
            {rev = _PC_fds[i].revents;
             if ((rev & POLLIN) || (rev & POLLPRI))
                {(*_PC_fd_fnc[i])(_PC_fds[i].fd);
                 PC_catch_io_interrupts(PC_io_interrupts_on);}

             else if ((rev & POLLERR) || (rev & POLLNVAL))
                continue;};};

    return(n);}

/*--------------------------------------------------------------------------*/

#ifndef HAVE_STREAMS_P

#ifdef HAVE_SELECT_P

/*--------------------------------------------------------------------------*/

/* PC_POLL - implement poll in terms of select
 *         - timeout is the number of milliseconds to wait for input
 *         - timeout = -1 means block
 */

int PC_poll(fds, nfds, timeout)
   PC_poll_desc *fds;
   long nfds;
   int timeout;
   {int i, n, nfd;
    short rev;
    fd_set rfd, efd;
    struct timeval to, *pto;

    nfd = min(FD_SETSIZE, nfds);

/* block if timeout is -1 */
    if (timeout == -1)
       pto = NULL;
    else
       {to.tv_sec  = timeout / 1000L;
        to.tv_usec = timeout % 1000L;

        pto = &to;};

/* map the descriptors from the PC_poll_desc */
    FD_ZERO(&rfd);
    FD_ZERO(&efd);
    for (i = 0; i < nfd; i++)
        {FD_SET(fds[i].fd, &rfd);
         FD_SET(fds[i].fd, &efd);};

/* the HP folks do it differently (sigh)
 * and arguably wrong since an fd_set is essentially a long
 * they are exploiting a "coincidence" that int and long are
 * the same
 */
#ifdef HPUX
    n = select(FD_SETSIZE, (int *) &rfd, NULL, (int *) &efd, pto);
#else
    n = select(FD_SETSIZE, &rfd, NULL, &efd, pto);
#endif
    if (n < 0)
       sprintf(PC_err, "SELECT ERROR %d - PC_POLL", errno);

/* map the return values to the PC_poll_desc */
    for (i = 0; i < nfd; i++)
        {rev = 0;
         if (FD_ISSET(fds[i].fd, &rfd))
            rev |= POLLIN;

         if (FD_ISSET(fds[i].fd, &efd))
            rev |= POLLERR;

         fds[i].revents = rev;};

    return(n);}

/*--------------------------------------------------------------------------*/

#else

/*--------------------------------------------------------------------------*/

/* PC_POLL - poll stub for systems without poll or select */

int PC_poll(fds, nfds, timeout)
   PC_poll_desc *fds;
   long nfds;
   int timeout;
   {int i, n, nfd, fd;

    nfd = 0;
    for (i = 0; i < _PC_nfds; i++)
        {fd = _PC_fds[i].fd;
         PC_unblock_fd(fd);

/* ask how many characters are available */
         n = 0;

         PC_block_fd(fd);

         if (n > 0)
            {_PC_fds[i].revents = POLLIN;
             nfd++;}
         else
            _PC_fds[i].revents = 0;};

#ifndef DOS
    sleep(timeout*1000);
#endif

    return(nfd);}

/*--------------------------------------------------------------------------*/

#endif

#endif

/*--------------------------------------------------------------------------*/

/*                         CLIENT/SERVER ROUTINES                           */

/*--------------------------------------------------------------------------*/

/* PC_INIT_SERVER - perform steps to initialize a server
 *                - for GET_PORT return the port for the client connection
 *                - for GET_CONNECTION return the socket for the client
 *                - connection
 */

int PC_init_server(step, closep)
   int step, closep;
   {int sasz, nfd, err, port;
#ifdef GETSOCKNAME_LONG
    unsigned long len;
#else
    int len;
#endif
    static struct sockaddr_in *srvr;
    static int fd = -1;

#ifdef HAVE_PROCESS_CONTROL

    if (_PC_debug)
       {fprintf(_PC_diag, "   PC_init_server: %d\n", step);
        fflush(_PC_diag);};

    sasz = sizeof(struct sockaddr_in);

    PC_err[0] = '\0';
    if (setjmp(_PC_err_state))
       {if (closep)
	   {if (fd >= 0)
	       close(fd);

	    fd = -1;};

        return(fd);};

    switch (step)
       {case GET_PORT :
             fd = socket(PF_INET, SOCK_STREAM, 0);
             if (fd < 0)
                PC_error("COULDN'T OPEN SOCKET - PC_INIT_SERVER");

             if (_PC_debug)
                {fprintf(_PC_diag, "      Socket opened: %d\n", fd);
                 fflush(_PC_diag);};

             srvr = FMAKE(struct sockaddr_in,
                    "PC_INIT_SERVER:srvr");
             srvr->sin_family      = PF_INET;
             srvr->sin_addr.s_addr = htonl(INADDR_ANY);

             for (port = 5000; port < 65000; port++)
                 {srvr->sin_port = htons(port);

                  if (bind(fd, (struct sockaddr *) srvr, sasz) >= 0)
                     {if (_PC_debug)
                         {fprintf(_PC_diag, "      Bind succeeded: %d\n", port);
                          fflush(_PC_diag);};

                      return(port);};};

             PC_error("BIND FAILED - PC_INIT_SERVER");
             break;

        case GET_CONNECTION :
             len = sasz;
             getsockname(fd, (struct sockaddr *) srvr, &len);

             err = listen(fd, 5);
             if (_PC_debug)
                {if (err < 0)
		    fprintf(_PC_diag, "      Listen failed %d (%d)\n",
			    fd, errno);
		 else
		    fprintf(_PC_diag, "      Listen succeeded\n");
                 fflush(_PC_diag);};

             if (err < 0)
                PC_error("LISTEN FAILED - PC_INIT_SERVER");

             nfd = accept(fd, (struct sockaddr *) srvr, &len);
             if (_PC_debug)
                {if (nfd < 0)
		    fprintf(_PC_diag, "      Accept failed %d (%d)\n",
			    fd, errno);
		 else
		    fprintf(_PC_diag, "      Accept succeeded: %d\n", nfd);
                 fflush(_PC_diag);};
             if (nfd < 0)
                PC_error("ACCEPT FAILED - PC_INIT_SERVER");

	     if (closep)
	        {close(fd);

		 if (_PC_debug)
		    {fprintf(_PC_diag, "      Socket closed: %d\n", fd);
		     fflush(_PC_diag);};

		 fd = -1;
		 SFREE(srvr);};

             break;};

#else
	nfd = 0;
#endif

    return(nfd);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_INIT_CLIENT - initialize a client
 *                - return the socket which connects to the server
 */

int PC_init_client(host, port)
   char *host;
   int port;
   {int sasz, fd, err;
    unsigned long haddr;
    struct sockaddr_in *srvr;
    struct hostent *hp;

#ifdef HAVE_PROCESS_CONTROL

    if ((host == NULL) || (port < 0))
       return(-1);

    sasz = sizeof(struct sockaddr_in);

    PC_ERR_TRAP(-1);

    srvr = FMAKE(struct sockaddr_in,
           "PC_INIT_CLIENT:srvr");
    srvr->sin_family = PF_INET;
    srvr->sin_port   = htons(port);

    haddr = inet_addr(host);
    if (haddr == INADDR_NONE)
       {hp = gethostbyname(host);
        if (hp != NULL)
           memcpy(&haddr, hp->h_addr, sasz);};

    if (haddr == INADDR_NONE)
       PC_error("CAN'T GET SERVER ADDRESS - PC_INIT_CLIENT");

    memcpy(&(srvr->sin_addr), &haddr, sizeof(long));

    fd = socket(PF_INET, SOCK_STREAM, 0);
    if (fd < 0)
       PC_error("CAN'T OPEN SOCKET - PC_INIT_CLIENT");

    err = connect(fd, (struct sockaddr *) srvr, sasz);
    if (err < 0)
       {close(fd);
        PC_error("CAN'T CONNECT TO SERVER - PC_INIT_CLIENT");};

#else
	fd = 0;
#endif

    return(fd);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_IO_CONNECT - connect the PDB io hooks for remote access */

int PC_io_connect(flag)
   int flag;
   {int ret;

    ret = TRUE;
    io_printf_hook = (PFfprintf) _PD_pio_printf;

#ifndef HAVE_PROCESS_CONTROL
    if (flag == PC_REMOTE)
       flag = PC_LOCAL;
#endif

    switch (flag)
       {case PC_REMOTE :
	     io_open_hook    = (PFfopen) PC_ropen;
	     io_tell_hook    = (PFftell) PC_rtell;
	     io_read_hook    = (PFfread) PC_rread;
	     io_write_hook   = (PFfwrite) PC_rwrite;
	     io_setvbuf_hook = (PFsetvbuf) PC_rsetvbuf;
	     io_close_hook   = (PFfclose) PC_rclose;
	     io_seek_hook    = (PFfseek) PC_rseek;
	     io_puts_hook    = (PFfputs) PC_rputs;
	     io_getc_hook    = (PFfgetc) PC_rgetc;
	     io_ungetc_hook  = (PFungetc) PC_rungetc;
	     io_flush_hook   = (PFfflush) PC_rflush;
	     io_gets_hook    = (PFfgets) PC_rgets;
	     break;
/*
        case PC_PARALLEL :
	     io_open_hook    = (PFfopen) PC_sopen;
	     io_tell_hook    = (PFftell) PC_stell;
	     io_read_hook    = (PFfread) PC_sread;
	     io_write_hook   = (PFfwrite) PC_swrite;
	     io_setvbuf_hook = (PFsetvbuf) PC_ssetvbuf;
	     io_close_hook   = (PFfclose) PC_sclose;
	     io_seek_hook    = (PFfseek) PC_sseek;
	     io_puts_hook    = (PFfputs) PC_sputs;
	     io_getc_hook    = (PFfgetc) PC_sgetc;
	     io_ungetc_hook  = (PFungetc) PC_sungetc;
	     io_flush_hook   = (PFfflush) PC_sflush;
	     io_gets_hook    = (PFfgets) PC_sgets;
	     break;
*/
        case PC_LOCAL :
	     io_open_hook    = (PFfopen) fopen;
	     io_tell_hook    = (PFftell) ftell;
	     io_read_hook    = (PFfread) fread;
	     io_write_hook   = (PFfwrite) fwrite;
	     io_setvbuf_hook = (PFsetvbuf) setvbuf;
	     io_close_hook   = (PFfclose) fclose;
	     io_seek_hook    = (PFfseek) fseek;
	     io_puts_hook    = (PFfputs) fputs;
	     io_getc_hook    = (PFfgetc) fgetc;
	     io_ungetc_hook  = (PFungetc) ungetc;
	     io_flush_hook   = (PFfflush) fflush;
	     io_gets_hook    = (PFfgets) fgets;
	     break;

	default :
	     ret = FALSE;
	     break;};

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_TRANSFER_BYTES - transfer bytes from the client to the file
 *                    - CFD is the client data socket
 *                    - NBE is the expected number of bytes from the client
 *                    - FP is the output file
 *                    - return the number of bytes successfully written
 */

long _PC_transfer_bytes(cfd, nbe, fp)
   int cfd;
   long nbe;
   FILE *fp;
   {char *bf, *ps;
    long nbw, nbt, nbr;

    bf = FMAKE_N(char, nbe, "_PC_TRANSFER_BYTES:bf");

/* read the data looping until all expected bytes come in */
    nbt = nbe;
    for (ps = bf; TRUE; ps += nbr)
        {nbr  = read(cfd, ps, nbt);
	 nbt -= nbr;
	 if (nbt <= 0)
	    break;};

    nbw = fwrite(bf, 1, nbe, fp);

    SFREE(bf);

    return(nbw);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_GET_CMD_RESP - wait for the response from the server */

long _PC_get_cmd_resp(pp, msg)
   PROCESS *pp;
   char *msg;
   {char s[MAXLINE];
    int lm;

    lm = strlen(msg);

    while (PC_gets(s, MAXLINE, pp) != NULL)
       if (strncmp(s, msg, lm) == 0)
          return(SC_stol(s+lm+1));

#if 0
/* GOTCHA: done this way because PC_gets will not block (FIX PC_GETS) */
    while (TRUE)
       {if (PC_gets(s, MAXLINE, pp) != NULL)
           continue;

        if (strncmp(s, msg, lm) == 0)
           return(SC_stol(s+lm+1));};
#endif

    return(FALSE);}

/*--------------------------------------------------------------------------*/

/*                              ALARM ROUTINES                              */

/*--------------------------------------------------------------------------*/

/* _PC_ALARM_HANDLER - tell somebody that we timed out and exit */

static void _PC_alarm_handler(sig)
   int sig;
   {

#ifdef UNIX
    PRINT(stdout, "\nProcess %d timed out\n\n", getpid());
#endif

    exit(123);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_ALARM - process will timeout and call FNC in TO seconds if
 *          - a second call to PC_set_timeout isn't made with
 *          - a value of 0 for TO before then
 *          - if FNC is NULL the process will exit
 */

void PC_alarm(to, fnc)
   int to;
   PFByte fnc;
   {

#ifdef UNIX

    ALARM(to);
    if (to > 0)
       {PFByte fn;

	if (fnc == NULL)
           fn = (PFByte) _PC_alarm_handler;
	else
           fn = fnc;

	SIGNAL(SIGALRM, fn);};

#endif

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
