/*
 * GSDV_CGM.C - PGS CGM routines
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
  
#include "pgs.h"

/* make CGMs which conform to DOD CALS spec (MIL-D-28003A) */
#define DOD_CALS

/* maximum length of a CELL ARRAY partition */
#define MAX_PARTITION 0x7FFC

REAL
 PG_CGM_text_mag = 1.6;

SC_dynamic_array
 _PG_CGM_x_point_list,
 _PG_CGM_y_point_list;

static PG_device
 SC_DECLARE(*_PG_CGM_open,
	    (PG_device *dev, double xf, double yf, double dxf, double dyf));

static void
 SC_DECLARE(PG_CGM_defaults, (PG_device *dev)),
 SC_DECLARE(_PG_CGM_query_pointer,
	    (PG_device *dev, int *px, int *py, int *pb, int *pq)),
 SC_DECLARE(_PG_CGM_clear_page, (PG_device *dev, int i)),
 SC_DECLARE(_PG_CGM_clear_window, (PG_device *dev)),
 SC_DECLARE(_PG_CGM_clear_viewport, (PG_device *dev)),
 SC_DECLARE(_PG_CGM_clear_region_NDC,
	    (PG_device *dev, double xmn, double xmx,
	     double ymn, double ymx, int pad)),
 SC_DECLARE(_PG_CGM_close_device, (PG_device *dev)),
 SC_DECLARE(_PG_CGM_expose_device, (PG_device *dev)),
 SC_DECLARE(_PG_CGM_finish_plot, (PG_device *dev)),
 SC_DECLARE(_PG_CGM_make_device_current, (PG_device *dev)),
 SC_DECLARE(_PG_CGM_map_to_color_table, 
            (PG_device *dev, PG_palette *pal)),
 SC_DECLARE(_PG_CGM_next_line, (PG_device *dev)),
 SC_DECLARE(_PG_CGM_query, 
            (PG_device *dev, int *pdx, int *pdy, int *pnc)),
 SC_DECLARE(_PG_CGM_release_current_device, (PG_device *dev)),
 SC_DECLARE(_PG_CGM_update_vs, (PG_device *dev)),
 SC_DECLARE(_PG_CGM_write_text, (PG_device *dev, FILE *fp, char *s));

/* from the corresponding PR file */

int
 SC_DECLARE(_PG_CGM_set_font,
	    (PG_device *dev, char *face, char *style, int size));

void
 SC_DECLARE(_PG_CGM_draw_disjoint_polyline_2,
	    (PG_device *dev, REAL *x, REAL *y,
	     long n, int flag, int coord)),
 SC_DECLARE(_PG_CGM_draw_curve, 
            (PG_device *dev, PG_curve *crv, int clip)),
 SC_DECLARE(_PG_CGM_draw_to_abs, (PG_device *dev, double x, double y)),
 SC_DECLARE(_PG_CGM_draw_to_rel, (PG_device *dev, double x, double y)),
 SC_DECLARE(_PG_CGM_get_text_ext_NDC,
	    (PG_device *dev, char *s, REAL *px, REAL *py)),
 SC_DECLARE(_PG_CGM_set_clipping, (PG_device *dev, int flag)),
 SC_DECLARE(_PG_CGM_set_char_line, (PG_device *dev, int n)),
 SC_DECLARE(_PG_CGM_set_char_path, 
            (PG_device *dev, double x, double y)),
 SC_DECLARE(_PG_CGM_set_char_precision, (PG_device *dev, int p)),
 SC_DECLARE(_PG_CGM_set_char_space, (PG_device *dev, double s)),
 SC_DECLARE(_PG_CGM_set_char_size_NDC, 
            (PG_device *dev, double x, double y)),
 SC_DECLARE(_PG_CGM_set_char_up, (PG_device *dev, double x, double y)),
 SC_DECLARE(_PG_CGM_set_fill_color, 
            (PG_device *dev, int color, int mapped)),
 SC_DECLARE(_PG_CGM_set_line_color, 
            (PG_device *dev, int color, int mapped)),
 SC_DECLARE(_PG_CGM_set_line_style, (PG_device *dev, int style)),
 SC_DECLARE(_PG_CGM_set_line_width, (PG_device *dev, double width)),
 SC_DECLARE(_PG_CGM_set_logical_op, (PG_device *dev, int lop)),
 SC_DECLARE(_PG_CGM_set_text_color, 
            (PG_device *dev, int color, int mapped)),
 SC_DECLARE(_PG_CGM_shade_poly, (PG_device *dev, REAL *x, REAL *y, int n)),
 SC_DECLARE(_PG_CGM_fill_curve, (PG_device *dev, PG_curve *crv)),
 SC_DECLARE(_PG_CGM_move_gr_abs, (PG_device *dev, double x, double y)),
 SC_DECLARE(_PG_CGM_move_tx_abs, (PG_device *dev, double x, double y)),
 SC_DECLARE(_PG_CGM_move_tx_rel, (PG_device *dev, double x, double y)),
 SC_DECLARE(_PG_CGM_put_image,
	    (PG_device *dev, unsigned char *bf, int ix, int iy,
	     int nx, int ny));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_SETUP_CGM_DEVICE - do the device dependent device initialization
 *                     - for PG_make_device
 */

void PG_setup_cgm_device(d)
   PG_device *d;
   {d->cgm_page_set            = FALSE;
    d->type_index              = CGMF_DEVICE;

    d->query_pointer           = _PG_CGM_query_pointer;
    d->clear_page              = _PG_CGM_clear_page;
    d->clear_window            = _PG_CGM_clear_window;
    d->clear_viewport          = _PG_CGM_clear_viewport;
    d->clear_region_NDC        = _PG_CGM_clear_region_NDC;
    d->close_device            = _PG_CGM_close_device;
    d->draw_dj_polyln_2        = _PG_CGM_draw_disjoint_polyline_2;
    d->draw_curve              = _PG_CGM_draw_curve;
    d->draw_to_abs             = _PG_CGM_draw_to_abs;
    d->draw_to_rel             = _PG_CGM_draw_to_rel;
    d->expose_device           = _PG_CGM_expose_device;
    d->finish_plot             = _PG_CGM_finish_plot;
    d->get_char                = NULL;
    d->get_image               = NULL;
    d->get_text_ext_NDC        = _PG_CGM_get_text_ext_NDC;
    d->ggets                   = (PFfgets) io_gets_hook;
    d->gputs                   = NULL;
    d->make_device_current     = _PG_CGM_make_device_current;
    d->map_to_color_table      = _PG_CGM_map_to_color_table;
    d->match_rgb_colors        = _PG_match_rgb_colors;
    d->move_gr_abs             = _PG_CGM_move_gr_abs;
    d->move_tx_abs             = _PG_CGM_move_tx_abs;
    d->move_tx_rel             = _PG_CGM_move_tx_rel;
    d->next_line               = _PG_CGM_next_line;
    d->open_screen             = _PG_CGM_open;
    d->put_image               = _PG_CGM_put_image;
    d->query_screen            = _PG_CGM_query;
    d->release_current_device  = _PG_CGM_release_current_device;
    d->resolution_scale_factor = 64;
    d->set_bound               = _PG_set_bound;
    d->set_clipping            = _PG_CGM_set_clipping;
    d->set_char_line           = _PG_CGM_set_char_line;
    d->set_char_path           = _PG_CGM_set_char_path;
    d->set_char_precision      = _PG_CGM_set_char_precision;
    d->set_char_size_NDC       = _PG_CGM_set_char_size_NDC;
    d->set_char_space          = _PG_CGM_set_char_space;
    d->set_char_up             = _PG_CGM_set_char_up;
    d->set_fill_color          = _PG_CGM_set_fill_color;
    d->set_font                = _PG_CGM_set_font;
    d->set_line_color          = _PG_CGM_set_line_color;
    d->set_line_style          = _PG_CGM_set_line_style;
    d->set_line_width          = _PG_CGM_set_line_width;
    d->set_logical_op          = _PG_CGM_set_logical_op;
    d->set_text_color          = _PG_CGM_set_text_color;
    d->shade_poly              = _PG_CGM_shade_poly;
    d->fill_curve              = _PG_CGM_fill_curve;
    d->update_vs               = _PG_CGM_update_vs;
    d->write_text              = _PG_CGM_write_text;
    
    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_CGM_DEFAULTS - set the page defaults for the CGM */

static void PG_CGM_defaults(dev)
   PG_device *dev;
   {int nbytes;

    nbytes = 4;
    PG_CGM_command(dev, METAFILE_DEFAULTS_REPLACEMENT(nbytes));

/* use stroked text */
    PG_CGM_command(dev, TEXT_PRECISION, 2);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_CGM_OPEN - initialize a Computer Graphics Metafile device */
 
static PG_device *_PG_CGM_open(dev, xf, yf, dxf, dyf)
   PG_device *dev;
   double xf, yf, dxf, dyf;
   {FILE *cgm_fp;
    REAL view_x_max, view_y_max, intensity;
    int Lightest, Light, Light_Gray, Dark_Gray, Dark, Darkest;
    int display_width, display_height, n_colors, lst[20];
    int i, nt, nfonts;
    char **font_name, **fs, *token, *name, *date, *s;
    char lname[MAXLINE], fname[MAXLINE], description[MAXLINE];
    PG_font_family *ff;

    PG_setup_markers();

    dev->type_index       = CGMF_DEVICE;
    dev->hard_copy_device = TRUE;
    dev->print_labels     = TRUE;

/* the default CGM uses integer coordinates in the 1rst quadrant */
    dev->quadrant = QUAD_ONE;

/* get the window shape in NDC */
    if ((xf == 0.0) && (yf == 0.0))
       {xf = 0.1;
        yf = 0.0;};

    if ((dxf == 0.0) && (dyf == 0.0))
       {dxf = 0.8;
        dyf = 0.8;};

    _PG_CGM_query(dev, &display_width, &display_height, &n_colors);

    SC_INIT_DYNAMIC_ARRAY(_PG_CGM_x_point_list, REAL, "_PG_CGM_OPEN", 20);
    SC_INIT_DYNAMIC_ARRAY(_PG_CGM_y_point_list, REAL, "_PG_CGM_OPEN", 20);

/* GOTCHA: The following pixel coordinate limits may be incorrect */
/* set device pixel coordinate limits */
    dev->min_pc_x = INT_MIN + display_width;
    dev->max_pc_x = INT_MAX - display_width;
    dev->min_pc_y = INT_MIN + display_height;
    dev->max_pc_y = INT_MAX - display_height;
 
    dev->window_x      = display_width*xf;
    dev->window_y      = display_height*(1.0 - yf);
    dev->window_width  = display_width*dxf;
    dev->window_height = display_width*dyf;

    dev->window_x_off  = dev->window_x;
    dev->window_y_off  = dev->window_y - dev->window_height;

    strcpy(lname, dev->title);
    name = SC_strtok(lname, " \t\n\r", s);
    strcpy(fname, name);

    token = SC_strtok(NULL, " \t\n\r", s);
    if (token != NULL)
       PG_CGM_text_mag = SC_stof(token);

    strcpy(fname, dev->title);
    SC_strtok(fname, " \t\n\r\\?~.,;:<>/'`\"[]{}()=+|!@#$%^&*", s);
    strcat(fname, ".cgm");
    cgm_fp = io_open(fname, "wb");
    if (cgm_fp == NULL)
       return(NULL);

    dev->file = cgm_fp;

/* write out standard CGM prolog */
    PG_CGM_command(dev, BEGIN_METAFILE, fname);
    PG_CGM_command(dev, METAFILE_VERSION, 1);

    date = SC_date();

#ifdef DOD_CALS

    sprintf(description, "Creator: PACT PGS ; Title: %s plots ; CreationDate: %s ; MIL-D-28003A/BASIC-1",
                         dev->title,
                         date);
#else

    sprintf(description, "Creator: PACT PGS ; Title: %s plots ; CreationDate: %s",
                         dev->title,
                         date);
#endif

    SFREE(date);

    PG_CGM_command(dev, METAFILE_DESCRIPTION, description);

    PG_CGM_command(dev, MAXIMUM_COLOUR_INDEX, 255);
    lst[0] =  1;
    lst[1] = -1;
    lst[2] =  1;
    PG_CGM_command(dev, METAFILE_ELEMENT_LIST, lst);
    PG_CGM_defaults(dev);

/* decide on the overall color layout */
    intensity = dev->max_intensity*MAXPIX;
    if (dev->background_color_white)
       {if (strcmp(dev->type, "MONOCHROME") == 0)
           {Color_Map(dev, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1);}
        else
           {Color_Map(dev, 1, 0, 2, 3, 4, 5, 6, 7, 8, 9,
                           10, 11, 12, 13, 14, 15);};
        Lightest   = 0;
        Light      = intensity;
        Light_Gray = 0.8*intensity;
        Dark_Gray  = 0.5*intensity;
        Dark       = 0;
        Darkest    = intensity;}
    else
       {if (strcmp(dev->type, "MONOCHROME") == 0)
/*           {Color_Map(dev, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1);} */
           {Color_Map(dev, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);}
        else
           {Color_Map(dev, 1, 0, 2, 3, 4, 5, 6, 7, 8, 9,
                           10, 11, 12, 13, 14, 15);};
        Lightest   = intensity;
        Light      = intensity;
        Light_Gray = 0.8*intensity;
        Dark_Gray  = 0.5*intensity;
        Dark       = 0;
        Darkest    = 0;};

    dev->ncolor = n_colors;
    dev->absolute_n_color = 256;

/* compute the view port */
    if (dev->view_x == 0.0)
       {dev->view_x = 0.175;
        dev->view_y = 0.175;};

    if (dev->view_width == 0.0)
       dev->view_width = 0.65;

    if (dev->view_height == 0.0)
       dev->view_height = 0.65;

    view_x_max = dev->view_x + dev->view_width;
    view_y_max = dev->view_y + dev->view_height;

/* set the view port */
    PG_set_viewport(dev, dev->view_x, view_x_max, dev->view_y, view_y_max);
    PG_set_window(dev, 0.0, 1.0, 0.0, 1.0);
 
/* set up fonts */

#ifdef DOD_CALS

    ff = PG_make_font_family(dev, "helvetica", NULL, 4,
                             "HELVETICA",
                             "HELVETICA_OBLIQUE",
                             "HELVETICA_BOLD",
                             "HELVETICA_BOLD_OBLIQUE");

    ff = PG_make_font_family(dev, "times", ff, 4,
                             "TIMES_ROMAN",
                             "TIMES_ITALIC",
                             "TIMES_BOLD",
                             "TIMES_BOLD_ITALIC");

    ff = PG_make_font_family(dev, "courier", ff, 4,
                             "COURIER",
                             "COURIER_OBLIQUE",
                             "COURIER_BOLD",
                             "COURIER_BOLD_OBLIQUE");

#else

    ff = PG_make_font_family(dev, "helvetica", NULL, 4,
                             "Helvetica",
                             "Helvetica-Oblique",
                             "Helvetica-Bold",
                             "Helvetica-BoldOblique");

    ff = PG_make_font_family(dev, "times", ff, 4,
                             "Times-Roman",
                             "Times-Italic",
                             "Times-Bold",
                             "Times-BoldItalic");

    ff = PG_make_font_family(dev, "courier", ff, 4,
                             "Courier",
                             "Courier-Oblique",
                             "Courier-Bold",
                             "Courier-BoldOblique");
#endif

    dev->font_family = ff;

/* initialize font list */
    nfonts = 0;
    for (ff = dev->font_family; ff != NULL; ff = ff->next)
        nfonts += ff->n_styles;

    font_name = FMAKE_N(char *, nfonts, "_PG_CGM_OPEN:fontname");
    for (nfonts = 0, ff = dev->font_family; ff != NULL; ff = ff->next)
        {nt = ff->n_styles;
         fs = ff->type_styles;
         for (i = 0; i < nt; i++)
             font_name[nfonts++] = fs[i];};

    PG_CGM_command(dev, FONT_LIST(nfonts), font_name);

    SFREE(font_name);

    PG_set_font(dev, "helvetica", "medium", 12);

/* put in the default palettes */
    PG_setup_standard_palettes(dev, 64,
			       Light, Dark,
			       Light_Gray, Dark_Gray,
			       Lightest, Darkest);

    return(dev);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_CGM_QUERY - query some CGM device characteristics */

static void _PG_CGM_query(dev, pdx, pdy, pnc)
   PG_device *dev;
   int *pdx, *pdy, *pnc;
   {*pdx = 32767;
    *pdy = 32767;

    if (strcmp(dev->type, "MONOCHROME") == 0)
       *pnc = 16;
    else
       *pnc = 256;
        
    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_CGM_MAP_TO_COLOR_TABLE - map the PGS palette to host
 *                            - color table values
 */

static void _PG_CGM_map_to_color_table(dev, pal)
   PG_device *dev;
   PG_palette *pal;
   {return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_CGM_CLOSE_DEVICE - close a device */
 
static void _PG_CGM_close_device(dev)
   PG_device *dev;
   {if (dev->cgm_page_set)
       PG_CGM_command(dev, END_PICTURE);

    PG_CGM_command(dev, END_METAFILE);
    io_close(dev->file);

/* clean up the device */
   _PG_rl_device(dev);

    SC_RELEASE_DYNAMIC(_PG_CGM_x_point_list);
    SC_RELEASE_DYNAMIC(_PG_CGM_y_point_list);

   return;}

/*--------------------------------------------------------------------------*/

/*                            PAGE STATE ROUTINES                           */

/*--------------------------------------------------------------------------*/
 
/* _PG_CGM_CLEAR_WINDOW - clear the screen */
 
static void _PG_CGM_clear_window(dev)
   PG_device *dev;
   {char s[MAXLINE];
    PG_palette *cp;
    double ca[2];
    int ih, fc, al[2], pts[10];
    static int npage = 1;

    PG_make_device_current(dev);

    sprintf(s, "Page %d", npage++);
    PG_CGM_command(dev, BEGIN_PICTURE, s);

/* set line width specification mode */
    PG_CGM_command(dev, LINE_WIDTH_SPECIFICATION_MODE, 0);

    PG_CGM_command(dev, BEGIN_PICTURE_BODY);
    
    cp = dev->current_palette;

/* set the palette
 * due to a GPLOT bug this is the one exceptional case in the CGM
 * command handling scheme
 * the offset will be specially trapped when COLOUR TABLE is parsed
 */
    PG_CGM_command(dev, COLOUR_TABLE(cp->n_pal_colors), cp, 0);

/* start with clipping off */
    PG_CGM_command(dev, CLIP_INDICATOR, FALSE);

    ih = PG_CGM_text_mag*dev->char_height_s*dev->window_height;
    PG_CGM_command(dev, CHARACTER_HEIGHT, ih);

    al[0] = 1;             /* left */
    al[1] = 4;             /* base */
    ca[0] = 0.0;           /* continuous horizontal */
    ca[1] = 0.0;           /* continuous vertical */
    PG_CGM_command(dev, TEXT_ALIGNMENT(al, ca));

    dev->cgm_page_set = TRUE;

/* solid fill */
    PG_CGM_command(dev, INTERIOR_STYLE, 1);

    fc = dev->fill_color;
    if (dev->current_palette != dev->palettes)
       {PG_set_color_fill(dev, 9, FALSE);}
    else
       {PG_set_color_fill(dev, dev->BLACK, FALSE);};

    pts[0] = 0;
    pts[1] = 0;
    pts[2] = 32767;
    pts[3] = 0;
    pts[4] = 32767;
    pts[5] = 32767;
    pts[6] = 0;
    pts[7] = 32767;
    pts[8] = 0;
    pts[9] = 0;
    PG_CGM_command(dev, POLYGON(10), pts);
    PG_set_color_fill(dev, fc, FALSE);

    PG_set_color_text(dev, dev->WHITE, TRUE);
    PG_set_color_line(dev, dev->WHITE, TRUE);

    PG_release_current_device(dev);

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_CGM_CLEAR_VIEWPORT - clear the viewport */
 
static void _PG_CGM_clear_viewport(dev)
   PG_device *dev;
   {return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_CGM_CLEAR_REGION_NDC - clear the rectangular region specified in NDC */
 
static void _PG_CGM_clear_region_NDC(dev, xmn, xmx, ymn, ymx, pad)
   PG_device *dev;
   double xmn, xmx, ymn, ymx;
   int pad;
   {return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_CGM_UPDATE_VS - update the view surface for the given device */
 
static void _PG_CGM_update_vs(dev)
   PG_device *dev;
   {return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_CGM_FINISH_PLOT - do what's necessary to finish up a graphical image
 *                     - and get the device updated with the image
 *                     - after this function nothing more can be added to
 *                     - the image
 */
 
static void _PG_CGM_finish_plot(dev)
   PG_device *dev;
   {long addr;

    PG_make_device_current(dev);

    PG_update_vs(dev);

    PG_CGM_command(dev, END_PICTURE);
    addr = io_tell(dev->file);

    dev->cgm_page_set = FALSE;
    
    PG_CGM_command(dev, END_METAFILE);
    io_seek(dev->file, addr, SEEK_SET);

    PG_release_current_device(dev);

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_CGM_CLEAR_PAGE - clear the current page
 *                    - and go to the line i on the screen
 */
 
static void _PG_CGM_clear_page(dev, i)
   PG_device *dev;
   int i;
   {return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_CGM_EXPOSE_DEVICE - make this device the topmost one */
 
static void _PG_CGM_expose_device(dev)
   PG_device *dev;
   {return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_CGM_MAKE_DEVICE_CURRENT - make this device the current one for drawing
 *                             - purposes
 */
 
static void _PG_CGM_make_device_current(dev)
   PG_device *dev;
   {return;}
        
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_CGM_RELEASE_CURRENT_DEVICE - make no device current
 *                                - (SUN in its brilliance needs this)
 */
 
static void _PG_CGM_release_current_device(dev)
   PG_device *dev;
   {return;}
        
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_CGM_WRITE_TEXT - write out text to the appropriate device */
 
static void _PG_CGM_write_text(dev, fp, s)
   PG_device *dev;
   FILE *fp;
   char *s;
   {int x1, y1, fl, info[3], rot[4];
    REAL x, y;

    x = dev->tcurx;
    y = dev->tcury;

    WtoS(dev, x, y);
    StoP(dev, x, y, x1, y1);
 
    x = dev->char_path_x;
    y = dev->char_path_y;
    fl = !(PM_CLOSETO_ABS(x, 1.0) && PM_CLOSETO_ABS(y, 0.0));

/* if the char path is not oriented along the x axis do the rotation */
    if (fl)
       {int ix, iy;

        x *= dev->bxw_s;
        y *= dev->byw_s;

	ix = floor(x*dev->window_width + 0.5);
	iy = floor(y*dev->window_height + 0.5);

        rot[0] = -iy;
        rot[1] =  ix;
        rot[2] =  ix;
        rot[3] =  iy;

        PG_CGM_command(dev, CHARACTER_ORIENTATION, rot);};

    info[0] = x1;
    info[1] = y1;
    info[2] = 1;
    PG_CGM_command(dev, TEXT, info, 1, SC_STRING_S, s);
 
/* restore the character orientation to horizontal */
    if (fl)
       {rot[0] = 0;
        rot[1] = 1;
        rot[2] = 1;
        rot[3] = 0;

        PG_CGM_command(dev, CHARACTER_ORIENTATION, rot);};

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_CGM_NEXT_LINE - do a controlled newline */
 
static void _PG_CGM_next_line(dev)
   PG_device *dev;
   {return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_CGM_QUERY_POINTER - query the pointer for location and button status */

static void _PG_CGM_query_pointer(dev, px, py, pb, pq)
   PG_device *dev;
   int *px, *py, *pb, *pq;
   {return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
