/*
 * GSCNTR.C - contour plotting routines for PGS
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pgs.h"

#define LABEL_TEXT_SIZE 30

#define PG_ZONE_SIDE(xa, ya, aa, xb, yb, ab)                                 \
    if (((aa[i] < lv) && (lv <= ab[i])) ||                                   \
        ((aa[i] >= lv) && (lv > ab[i])))                                     \
       {fracb = (lv - aa[i])/(ab[i] - aa[i] + SMALL);                        \
        fraca = 1.0 - fracb;                                                 \
        xpt[npt] = fraca*xa[i] + fracb*xb[i];                                \
        ypt[npt] = fraca*ya[i] + fracb*yb[i];                                \
        npt++;}

int
 _PG_contour_n_levels = 10;

REAL
 _PG_contour_ratio = 1.0;

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_CONTOUR_LEVELS - compute an array of contour levels */

static int PG_contour_levels(lev, nlev, fmin, fmax, ratio)
   REAL *lev;
   int nlev;
   double fmin, fmax, ratio;
   {int i;
    double delta, levn;

    lev[0] = fmin;
    levn   = nlev - 1.0;
    if (ratio == 1.0)
       delta = (fmax - fmin)/levn;
    else
       delta = (fmax - fmin)*(1.0 - ratio)/
               (ratio*(1.0 - POW(ratio, levn)));

    for (i = 1; i < nlev; i++)
        {delta *= ratio;
         lev[i] = lev[i-1] + delta;};

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_ISO_LIMIT - find the extreme values of the array A which is NPTS long
 *              - return the min and max thru the pointers PMIN, PMAX
 */

void PG_iso_limit(a, npts, pmin, pmax)
   REAL *a;
   int npts;
   REAL *pmin, *pmax;
   {int i;
    REAL lmin, lmax;
 
    lmax = lmin = a[0];
    for (i = 0; i < npts; i++)
        {lmax = max(lmax, a[i]);
         lmin = min(lmin, a[i]);};

    *pmin = lmin;
    *pmax = lmax;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SETUP_ISO_LEVELS - setup a set of level values for a contour plot */

void PG_setup_iso_levels(dev, data, pd)
   PG_device *dev;
   PG_graph *data;
   PG_picture_desc *pd;
   {int nlev;
    int *pn;
    REAL rlmin, rlmax, ratio;
    REAL *pl, *pr, *lv, *lev, *rextr;
    PM_set *range;
    pcons *alst;

    rlmin =  HUGE;
    rlmax = -HUGE;

    alst = pd->alist;
    pr   = NULL;
    pn   = NULL;
    pl   = NULL;
    SC_assoc_info(alst,
		  "RATIO", &pr,
		  "N-LEVELS", &pn,
		  "LEVELS", &pl,
		  NULL);

    ratio = (pr == NULL) ? _PG_contour_ratio : *pr;
    nlev  = (pn == NULL) ? _PG_contour_n_levels : *pn;
    lv    = (pl == NULL) ? NULL : pl;

    if (nlev <= 0)
       nlev = 10;

/* find the range limits if any */
    range = data->f->range;
    rextr = PM_get_limits(range);

    if (rextr != NULL)
       {rlmin = rextr[0];
	rlmax = rextr[1];};

    SFREE(pd->levels);

    lev = FMAKE_N(REAL, nlev, "PG_SETUP_ISO_LEVELS:lev");
    if (lv == NULL)
       {if (dev->autorange || (rlmin == HUGE) || (rlmax == -HUGE))
	   {rextr = dev->range_extrema;
	    rlmin = rextr[0];
	    rlmax = rextr[1];};

/* compute the contour levels */
	PG_contour_levels(lev, nlev, rlmin, rlmax, ratio);}

    else
       PM_copy_array(lv, lev, nlev);

    pd->n_levels  = nlev;
    pd->levels    = lev;
    pd->legend_fl = TRUE;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SETUP_PICTURE_CONTOUR - setup a window for a contour rendering
 *                          - NOTE: no drawing of any kind is to be done here
 */

PG_picture_desc *PG_setup_picture_contour(dev, data, save, clear)
   PG_device *dev;
   PG_graph *data;
   int save, clear;
   {int nde, nre, change;
    REAL vxmn, vxmx, vymn, vymx;
    REAL *dpex, *ddex, *pdx, *rpex, *rdex, *prx, *vwprt;
    PG_picture_desc *pd;
    PG_par_rend_info *pri;
    PG_device *dd;
    pcons *alst;

    change = !dev->supress_setup;

    pd = PG_get_rendering_properties(dev, data);

    pd->palette_fl = FALSE;
    pd->legend_fl  = TRUE;

    alst = pd->alist;
    pri  = dev->pri;
    if (pri != NULL)
       {dd = pri->dd;
	if (dd != NULL)
	   {dd->pri->alist  = alst;
	    dd->pri->render = PLOT_CONTOUR;};};

/* setup the viewport */
    vwprt = pd->viewport;
    if (vwprt != NULL)
       {vxmn = vwprt[0];
        vxmx = vwprt[1];
        vymn = vwprt[2];
        vymx = vwprt[3];}
    else
       {vxmn = 0.175;
        vxmx = 0.735;
        vymn = 0.175;
        vymx = 0.825;};

    if (change)
       {PG_set_viewport(dev, vxmn, vxmx, vymn, vymx);

/* find the extrema for this frame */
	PG_find_extrema(data, 0.0, &dpex, &rpex, &nde, &ddex, &nre, &rdex);

/* setup the domain limits */
	pdx = ((dev->autodomain == TRUE) || (dpex == NULL)) ? ddex : dpex;
	PG_set_window(dev, pdx[0], pdx[1], pdx[2], pdx[3]);

/* setup the range limits */
	prx = ((dev->autorange == TRUE) || (rpex == NULL)) ? rdex : rpex;
	PG_register_range_extrema(dev, nre, prx);

	SFREE(ddex);
	SFREE(rdex);

/* set up the drawing properties */
	PG_set_palette(dev, "standard");

	PG_set_color_line(dev, dev->WHITE, TRUE);
	PG_set_color_text(dev, dev->WHITE, TRUE);

	PG_setup_iso_levels(dev, data, pd);};

    return(pd);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_ISO_ZC_LR - draw iso contour lines assuming a zone centered
 *                   - Logical-Rectangular mesh
 */

void PG_draw_iso_zc_lr(dev, a, rx, ry, lev, nlev, id, cnnct, alist)
   PG_device *dev;
   REAL *a, *rx, *ry, *lev;
   int nlev, id;
   byte *cnnct;
   pcons *alist;
   {REAL *ap;

    ap = PM_zone_node_lr_2d(a, cnnct, alist);

    PG_draw_iso_nc_lr(dev, ap, rx, ry, lev, nlev, id, cnnct, alist);

    SFREE(ap);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_ISO_NC_LR - plot iso-contours for the function A on the
 *                   - Logical-Rectangular mesh specified by coordinates
 *                   - X and Y which is KMAX by LMAX and node centered
 *                   - plot NLEV contour levels specified in LEV
 *                   - if ID is non-zero label the contour levels starting
 *                   - with ID as the first character
 *                   - the mesh topology is always ignored and is there
 *                   - to be call compatible with the AC version
 */

void PG_draw_iso_nc_lr(dev, a, x, y, lev, nlev, id, cnnct, alist)
   PG_device *dev;
   REAL *a, *x, *y, *lev;
   int nlev, id;
   byte *cnnct;
   pcons *alist;
   {int k, km, l, lm, i, i1, npt, ilev, mcnt, mark, ok_ids;
    int *maxes, kmax, lmax, n, nmap, eflag;
    REAL xpt[4], ypt[4], fraca, fracb, lv;
    REAL *x1, *x2, *x3, *x4;
    REAL *y1, *y2, *y3, *y4;
    REAL *a1, *a2, *a3, *a4;
    REAL xave, yave;
    char *emap;

    maxes = (int *) cnnct;
    kmax  = maxes[0];
    lmax  = maxes[1];
    n     = kmax*lmax;
    nmap  = (kmax - 1) * (lmax - 1);

    LR_MAPPING_INFO(alist, nmap);

    PM_CHECK_EMAP(alist, nmap, eflag, emap);

    PM_LOGICAL_ZONE(x, x1, x2, x3, x4, kmax);
    PM_LOGICAL_ZONE(y, y1, y2, y3, y4, kmax);
    PM_LOGICAL_ZONE(a, a1, a2, a3, a4, kmax);

    ok_ids = dev->data_id;

    for (ilev = 0; ilev < nlev; ilev++)
        {lv = lev[ilev];

         if (id)
            {mark = id + ilev;
             mcnt = 0;};
  
         km = kmax - 1;
         lm = lmax - 1;
         for (l = 0; l < lm; l++)
             for (k = 0; k < km; k++)
                 {i  = l*kmax + k;
                  i1 = l*km +k;
                  if (emap[i1] == 0)
                     continue;

                  npt = 0;

                  PG_ZONE_SIDE(x1, y1, a1, x2, y2, a2);
                  PG_ZONE_SIDE(x2, y2, a2, x3, y3, a3);
                  PG_ZONE_SIDE(x3, y3, a3, x4, y4, a4);
                  PG_ZONE_SIDE(x4, y4, a4, x1, y1, a1);

                  if (npt == 2)
                     {PG_draw_line(dev, xpt[0], ypt[0], xpt[1], ypt[1]);
                      if (id && ok_ids)
                         {if (mcnt > 9)
                             {mcnt = 0;
                              xave = 0.5*(xpt[0] + xpt[1]);
                              yave = 0.5*(ypt[0] + ypt[1]);
                              PG_write_WC(dev, xave, yave, "%c", mark);}
                          else
                             mcnt++;};}
                  else if (npt == 4)
                     {mcnt++;
                      if (a1[i] < lv)
                         {PG_draw_line(dev, xpt[0], ypt[0],
                                            xpt[1], ypt[1]);
                          PG_draw_line(dev, xpt[2], ypt[2],
                                            xpt[3], ypt[3]);}
                      else
                         {PG_draw_line(dev, xpt[1], ypt[1],
                                            xpt[2], ypt[2]);
                          PG_draw_line(dev, xpt[3], ypt[3],
                                            xpt[0], ypt[0]);};};};};

    if (eflag)
        SFREE(emap);

    return;}
  
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_ISO_ZC_AC - draw iso contour lines assuming a zone centered
 *                   - Arbitrarily-Connected mesh
 */

void PG_draw_iso_zc_ac(dev, a, x, y, lev, nlev, id, cnnct, alist)
   PG_device *dev;
   REAL *a, *x, *y, *lev;
   int nlev, id;
   byte *cnnct;
   pcons *alist;
   {REAL *ap;

    ap = PM_zone_node_ac_2d(a, cnnct, alist);

    PG_draw_iso_nc_ac(dev, ap, x, y, lev, nlev, id, cnnct, alist);

    SFREE(ap);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_ISO_NC_AC - plot iso-contours for the function A on the
 *                   - Arbitrarily-Connected mesh specified by coordinates
 *                   - X and Y which is KMAX by LMAX and node centered
 *                   - plot NLEV contour levels specified in LEV
 *                   - if ID is non-zero label the contour levels starting
 *                   - with the character ID
 */

void PG_draw_iso_nc_ac(dev, a, x, y, lev, nlev, id, cnnct, alist)
   PG_device *dev;
   REAL *a, *x, *y, *lev;
   int nlev, id;
   byte *cnnct;
   pcons *alist;
   {int i, npt, ilev, mcnt, mark, ok_ids;
    int is, os, iz, oz, is1, is2, in1, in2;
    int *nc, nz, *np, nzp, nsp;
    long **cells, *zones, *sides;
    REAL *xpt, *ypt, fraca, fracb, lv;
    REAL xave, yave, a1, a2;
    PM_mesh_topology *mt;

    mt = (PM_mesh_topology *) cnnct;

    cells = mt->boundaries;
    zones = cells[2];
    sides = cells[1];

    nc = mt->n_cells;
    nz = nc[2];

    np  = mt->n_bound_params;
    nzp = np[2];
    nsp = np[1];

/* can compute this by looping over the sides */
    xpt = FMAKE_N(REAL, 10, "PG_DRAW_ISO_NC_AC:xpt");
    ypt = FMAKE_N(REAL, 10, "PG_DRAW_ISO_NC_AC:ypt");

    ok_ids = dev->data_id;

    for (ilev = 0; ilev < nlev; ilev++)
        {lv = lev[ilev];

         if (id)
            {mark = id + ilev;
             mcnt = 0;};
  
         for (iz = 0; iz < nz; iz++)
             {oz  = iz*nzp;
              is1 = zones[oz];
              is2 = zones[oz+1];

              npt = 0;
              for (is = is1; is <= is2; is++)
                  {os  = is*nsp;
                   in1 = sides[os];
                   in2 = sides[os+1];

                   a1 = a[in1];
                   a2 = a[in2];
                   if (((a1 < lv) && (lv <= a2)) ||
                       ((a1 >= lv) && (lv > a2)))
                      {fracb = (lv - a1)/(a2 - a1 + SMALL);
                       fraca = 1.0 - fracb;
                       xpt[npt] = fraca*x[in1] + fracb*x[in2];
                       ypt[npt] = fraca*y[in1] + fracb*y[in2];
                       npt++;};};

              if (npt == 2)
                 {PG_draw_line(dev, xpt[0], ypt[0], xpt[1], ypt[1]);
                  if (id && ok_ids)
                     {if (mcnt > 9)
                         {mcnt = 0;
                          xave = 0.5*(xpt[0] + xpt[1]);
                          yave = 0.5*(ypt[0] + ypt[1]);
                          PG_write_WC(dev, xave, yave, "%c", mark);}
                      else
                        mcnt++;};}
              else if (npt > 2)
                 {mcnt++;
                  xave = 0.0;
                  yave = 0.0;
                  for (i = 0; i < npt; i++)
                      {xave += xpt[0];
                       yave += ypt[0];};
                  xave /= (double) npt;
                  yave /= (double) npt;
                  for (i = 0; i < npt; i++)
                      PG_draw_line(dev, xave, yave, xpt[i], ypt[i]);};};};

    SFREE(xpt);
    SFREE(ypt);

    return;}
  
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_ISO_HAND - iso-contour plot handler
 *             - for the graph G on the device DEV
 *             - plot NLEV contour levels
 */

static void PG_iso_hand(dev, g, pd, fnc_zc, fnc_nc)
   PG_device *dev;
   PG_graph *g;
   PG_picture_desc *pd;
   PFByte fnc_zc, fnc_nc;
   {int id, npts;
    int color, style, centering, nlev;
    REAL width;
    REAL *rx, *ry, *f, *lev, **r;
    byte *cnnct;
    pcons *alst;
    PM_mapping *h;
    PM_set *domain, *range;

    PG_set_clipping(dev, TRUE);

    nlev = pd->n_levels;
    lev  = pd->levels;
    id   = g->identifier;

    for (h = g->f; h != NULL; h = h->next)
        {if ((g->info != NULL) &&
	     (strcmp(g->info_type, SC_PCONS_P_S) == 0))
	    {int *pc, *ps;
	     REAL *pw;

	     SC_assoc_info((pcons *) g->info,
			   "LINE-COLOR", &pc,
			   "LINE-WIDTH", &pw,
			   "LINE-STYLE", &ps,
			   NULL);

	     color = (pc == NULL) ? dev->BLUE : *pc;
	     width = (pw == NULL) ? 0.0 : *pw;
	     style = (ps == NULL) ? SOLID : *ps;}

	else
	   {color = dev->BLUE;
	    width = 0.0;
	    style = SOLID;};

	 PG_set_line_style(dev, style);
	 PG_set_line_width(dev, width);

	 domain = h->domain;
	 npts  = domain->n_elements;
	 r     = (REAL **) domain->elements;

	 rx = PM_array_real(domain->element_type, r[0], npts, NULL);
	 ry = PM_array_real(domain->element_type, r[1], npts, NULL);

	 range = h->range;
	 npts  = range->n_elements;

	 f = PM_array_real(range->element_type, DEREF(range->elements), npts, NULL);

/* find the additional mapping information */
	 centering = N_CENT;
	 alst      = PM_mapping_info(h,
				     "CENTERING", &centering,
				     NULL);
	 cnnct     = PM_connectivity(h);

	 PG_set_color_line(dev, color, TRUE);
	 switch (centering)
	    {case Z_CENT :
                  (*fnc_zc)(dev, f, rx, ry, lev, nlev, id, cnnct, alst);
                  break;

             case N_CENT :
                  (*fnc_nc)(dev, f, rx, ry, lev, nlev, id, cnnct, alst);
                  break;

             case F_CENT :
             case U_CENT :
             default     :
                  break;};

	 PG_draw_domain_boundary(dev, h);

	 PG_update_vs(dev);

	 SFREE(rx);
	 SFREE(ry);
	 SFREE(f);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_CONTOUR_CORE - contour plot skeleton routine */

static void _PG_contour_core(dev, data, fnc_zc, fnc_nc)
   PG_device *dev;
   PG_graph *data;
   PFByte fnc_zc, fnc_nc;
   {PG_graph *g;
    PG_picture_desc *pd;

    data->rendering = PLOT_CONTOUR;

    pd = PG_setup_picture(dev, data, TRUE, TRUE, TRUE);

/* this has to be done again here to be correct for parallel renderings */
    PG_setup_iso_levels(dev, data, pd);

/* plot all of the current functions */
    for (g = data; g != NULL; g = g->next)
        PG_iso_hand(dev, g, pd, fnc_zc, fnc_nc);

    PG_finish_picture(dev, data, pd);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_CONTOUR_PLOT - main contour plot control routine */

#ifdef PCC

void PG_contour_plot(dev, data, va_alist)
   PG_device *dev;
   PG_graph *data;
   va_dcl

#endif

#ifdef ANSI

void PG_contour_plot(PG_device *dev, PG_graph *data, ...)

#endif

   {REAL ratio, *pr;
    pcons *info;
        
    info = (pcons *) data->info;
    SC_assoc_info(info,
                  "RATIO", &pr,
                  NULL);

    ratio = (pr == NULL) ? 1.0 : *pr;
    if (ratio == 0.0)
       {SC_VA_START(data);
        pr  = FMAKE(REAL, "PG_CONTOUR_PLOT:pr");
        *pr = SC_VA_ARG(double);
        data->info = (byte *) SC_add_alist(info, "RATIO",
                                           "REAL *", (byte *) pr);
        SC_VA_END;};

    if (strcmp(data->f->category, PM_LR_S) == 0)
       _PG_contour_core(dev, data, PG_draw_iso_zc_lr, PG_draw_iso_nc_lr);

    else if (strcmp(data->f->category, PM_AC_S) == 0)
       _PG_contour_core(dev, data, PG_draw_iso_zc_ac, PG_draw_iso_nc_ac);

    return;}

/*--------------------------------------------------------------------------*/

/*                            FORTRAN API ROUTINES                          */

/*--------------------------------------------------------------------------*/

/* PGCLEV - compute contour levels */

FIXNUM F77_ID(pgclev_, pgclev, PGCLEV)(lev, pn, pfn, pfx, pr)
   REAL *lev;
   FIXNUM *pn;
   REAL *pfn, *pfx, *pr;
   {return((FIXNUM) PG_contour_levels(lev, (int) *pn,
				      (double) *pfn, (double) *pfx,
				      (double) *pr));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGPLCN - low level contour plot routine */

FIXNUM F77_ID(pgplcn_, pgplcn, PGPLCN)(devid, px, py, pa, pl,
                                       pkx, plx, pnl, pli, pal)
   FIXNUM *devid;
   REAL *px, *py, *pa, *pl;
   FIXNUM *pkx, *plx, *pnl, *pli, *pal;
   {PG_device *dev;
    int maxes[2];
    pcons *alst;

    maxes[0] = *pkx;
    maxes[1] = *plx;

    dev  = SC_GET_POINTER(PG_device, *devid);
    alst = SC_GET_POINTER(pcons, *pal);

    PG_draw_iso_nc_lr(dev, pa, px, py, pl,
                      (int) *pnl, (int) *pli, (byte *) maxes, alst);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
