/*
 * GSCLR.C - color handling routines for PGS
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "pgs.h"

#define PG_map_to_color_table(d, pal)                                        \
    if ((d) != NULL)                                                         \
       {if ((d)->map_to_color_table != NULL)                                 \
           (*(d)->map_to_color_table)(d, pal);}

#define PG_match_rgb_colors(d, pal)                                          \
    if ((d) != NULL)                                                         \
       {if ((d)->match_rgb_colors != NULL)                                   \
           (*(d)->match_rgb_colors)(d, pal);}

/*--------------------------------------------------------------------------*/

#define _PG_PSEUDO_SHADE_HUE(g, b, n, ns, fl, fu)                            \
    {a = (fu - fl)/n;                                                        \
     for (i = 1; (i <= n) && (j < ns); i++, j++)                             \
         {pseudo_cm[j].red   = pal->max_red_intensity*(a*i + fl);            \
          pseudo_cm[j].green = pal->max_green_intensity*g;                   \
          pseudo_cm[j].blue  = pal->max_blue_intensity*b;};}

/*--------------------------------------------------------------------------*/

#define _PG_SHADE_HUE(r, g, b, n, ns, fl, fu)                                \
    {a = (fu - fl)/n;                                                        \
     for (i = 1; (i <= n) && (j < ns); i++, j++)                             \
         {true_cm[j].red   = pal->max_red_intensity*r*(a*i + fl);            \
          true_cm[j].green = pal->max_green_intensity*g*(a*i + fl);          \
          true_cm[j].blue  = pal->max_blue_intensity*b*(a*i + fl);           \
          if (pv != NULL)                                                    \
	     pv[j] = j + off + 2;};}

/*--------------------------------------------------------------------------*/

#define INSTALL_COLOR(true_cm, mxr, mxg, mxb, fr, fg, fb)                    \
    {true_cm[j].red   = mxr*fr;                                              \
     true_cm[j].green = mxg*fg;                                              \
     true_cm[j].blue  = mxb*fb;                                              \
     if (pv != NULL)                                                         \
        pv[j] = j + off + 2;                                                 \
     j++;}

/*--------------------------------------------------------------------------*/

extern jmp_buf
 io_avail;

int 
 _PG_color_map_to_extrema  = FALSE;

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_MK_PALETTE - allocate and initialize a new palette */

static PG_palette *PG_mk_palette(dev, name, nclr)
   PG_device *dev;
   char *name;
   int nclr;
   {int npc, ndc, i;
    unsigned short *pv;
    double mxr, mxg, mxb;
    PG_palette *npal;
    RGB_color_map *pcm;

    mxr = dev->max_red_intensity;
    mxg = dev->max_green_intensity;
    mxb = dev->max_blue_intensity;

/* the 256 max is to make it doable at all on a system with
 * 24 bits of color - nobody wants to wait a year for X to
 * allocate 16 million color
 */
    npc = min(nclr, 256);
    ndc = min(dev->absolute_n_color, 256);

    npal = FMAKE(PG_palette, "PG_MK_PALETTE:npal");

    if (strcmp(dev->name, "IMAGE") == 0)
       pv = FMAKE_N(unsigned short, npc+2, "PG_MK_PALETTE:pv");
    else
       pv = NULL;

    npal->max_pal_dims        = 0;
    npal->pal_dims            = NULL;
    npal->pixel_value         = pv;
    npal->true_colormap       = FMAKE_N(RGB_color_map, npc + 2,
					"PG_MK_PALETTE:true_colormap");
    npal->pseudo_colormap     = FMAKE_N(RGB_color_map, npc + 2,
					"PG_MK_PALETTE:pseudo_colormap");
    npal->max_red_intensity   = mxr;
    npal->max_green_intensity = mxg;
    npal->max_blue_intensity  = mxb;
    npal->n_pal_colors        = npc;
    npal->n_dev_colors        = ndc;
    npal->name                = SC_strsavef(name,
					    "char*:PG_MK_PALETTE:name");
    npal->next                = NULL;

    pcm  = npal->true_colormap;

    for (i = 0; i < (npc + 2); i++)
        {pcm[i].red   = 0;
	 pcm[i].green = 0;
	 pcm[i].blue  = 0;};

    pcm  = npal->pseudo_colormap;

    for (i = 0; i < (npc + 2); i++)
        {pcm[i].red   = 0;
	 pcm[i].green = 0;
	 pcm[i].blue  = 0;};

/* load black and white into the palette */

    pcm  = npal->true_colormap;

    pcm[0].red   = 0;
    pcm[0].green = 0;
    pcm[0].blue  = 0;
    pcm[1].red   = MAXPIX*mxr;
    pcm[1].green = MAXPIX*mxg;
    pcm[1].blue  = MAXPIX*mxb;

    return(npal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_RL_PALETTE - release a palette */

void PG_rl_palette(pal)
   PG_palette *pal;
   {int i, **pd;

    SFREE(pal->name);
    SFREE(pal->pixel_value);
    SFREE(pal->true_colormap);
    SFREE(pal->pseudo_colormap);
    pal->next = NULL;

    if (pal->pal_dims != NULL)
       {pd = pal->pal_dims;
        for (i = 0; i < pal->max_pal_dims; i++)
            SFREE(pd[i]);
        SFREE(pd);}

    SFREE(pal);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_ATTACH_PALETTE_DIMS - attach a set of shapes for multiple
 *                         - multi-dimensional palettes
 */

static PG_palette *_PG_attach_palette_dims(pal, nc, nd, sh)
   PG_palette *pal;
   int nc, nd, *sh;
   {int i, j, ncd, **pd, *pp;

    pd = FMAKE_N(int *, nd, "_PG_ATTACH_PALETTE_DIMS:pd");
    for (i = 0; i < nd; i++)
        {if (sh == NULL)
            ncd = POW((double) nc, 1.0/((double) (i + 1)));

         pp = FMAKE_N(int, i+1, "_PG_ATTACH_PALETTE_DIMS:pp");
         for (j = 0; j <= i; j++)
             pp[j] = (sh == NULL) ? ncd : sh[j];

         pd[i] = pp;};

    pal->max_pal_dims = nd;
    pal->pal_dims     = pd;

    return(pal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_TRANS_PALETTE - fill a part of a palette with colors transforming
 *                   - from the first RGB value to the second
 */

static void _PG_trans_palette(cm, nc, r1, g1, b1, r2, g2, b2, nx)
   RGB_color_map *cm;
   int nc;
   double r1, g1, b1, r2, g2, b2, nx;
   {int i;
    double f, x, x1, x2, c;
   
/* control the color distribution */
    if (nx == 0.0)
       {f = 1.0/(nc - 1.0);
	for (i = 0; i < nc; i++)
            {x2 = i*f;
	     x1 = 1.0 - x2;

	     cm[i].red   = r1*x1 + r2*x2;
	     cm[i].green = g1*x1 + g2*x2;
	     cm[i].blue  = b1*x1 + b2*x2;};}

    else if (nx < 0.0)
       {nx = -nx;
	f  = 1.0/(nc - 1.0);
	for (i = 0; i < nc; i++)
            {x  = i*f;
	     x2 = POW(x, nx);
	     x1 = 1.0 - x2;

	     cm[i].red   = r1*x1 + r2*x2;
	     cm[i].green = g1*x1 + g2*x2;
	     cm[i].blue  = b1*x1 + b2*x2;};}

    else
       {f = 1.0/(nc - 1.0);
	c = 1.0/(1.0 - POW(0.5, nx));
	for (i = 0; i < nc; i++)
            {x  = 1.0/(1.0 + i*f);
	     x2 = c*(1.0 - POW(x, nx));
	     x1 = 1.0 - x2;

	     cm[i].red   = r1*x1 + r2*x2;
	     cm[i].green = g1*x1 + g2*x2;
	     cm[i].blue  = b1*x1 + b2*x2;};};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_WB_PALETTE - compute a white to black palette */

static PG_palette *_PG_wb_palette(dev, pal, nc)
   PG_device *dev;
   PG_palette *pal;
   int nc;
   {int ns, reg;
    double mxr, mxg, mxb;
    RGB_color_map *true_cm;

    reg = (pal == NULL);
    if (reg)
       pal = PG_mk_palette(dev, "wb", nc);

    true_cm = pal->true_colormap + 2;

    ns  = MAXPIX;
    mxr = ns*pal->max_red_intensity;
    mxg = ns*pal->max_green_intensity;
    mxb = ns*pal->max_blue_intensity;

    _PG_trans_palette(true_cm, nc, mxr, mxg, mxb, 0.0, 0.0, 0.0, 1.0);

    if (reg)
       _PG_register_palette(dev, pal, FALSE);

    return(pal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RGB_PALETTE - compute a blue -> green -> red palette */

static PG_palette *_PG_rgb_palette(dev, pal, nc)
   PG_device *dev;
   PG_palette *pal;
   int nc;
   {int ns, reg;
    int s1, s2, s3, s4;
    double mxr, mxg, mxb;
    RGB_color_map *true_cm;

    reg = (pal == NULL);
    if (reg)
       pal = PG_mk_palette(dev, "rgb", nc);

    true_cm = pal->true_colormap + 2;

    ns  = MAXPIX;
    mxr = ns*pal->max_red_intensity;
    mxg = ns*pal->max_green_intensity;
    mxb = ns*pal->max_blue_intensity;

    s1 = nc/4.0;
    s2 = (nc - s1)/3.0;
    s3 = (nc - s1 - s2)/2.0;
    s4 = (nc - s1 - s2 - s3);

/* blue -> cyan */
    _PG_trans_palette(true_cm,
		      s1, 0.0, 0.0, mxb, 0.0, mxg, mxb, 2.0);

/* cyan -> green */
    _PG_trans_palette(true_cm + s1 - 1,
		      s2, 0.0, mxg, mxb, 0.0, mxg, 0.0, -2.0);

/* green -> yellow */
    _PG_trans_palette(true_cm + s1 + s2 - 1,
		      s3, 0.0, mxg, 0.0, mxr, mxg, 0.0, 2.0);

/* yellow -> red */
    _PG_trans_palette(true_cm + s1 + s2 + s3 - 1,
		      s4+1, mxr, mxg, 0.0, mxr, 0.0, 0.0, -2.0);

    if (reg)
       _PG_register_palette(dev, pal, FALSE);

    return(pal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_CYM_PALETTE - compute a cyan -> yellow -> magenta palette */

static PG_palette *_PG_cym_palette(dev, pal, nc)
   PG_device *dev;
   PG_palette *pal;
   int nc;
   {int ns, reg;
    int s1, s2, s3, s4;
    double mxr, mxg, mxb;
    RGB_color_map *true_cm;

    reg = (pal == NULL);
    if (reg)
       pal = PG_mk_palette(dev, "cym", nc);

    true_cm = pal->true_colormap + 2;

    ns  = MAXPIX;
    mxr = ns*pal->max_red_intensity;
    mxg = ns*pal->max_green_intensity;
    mxb = ns*pal->max_blue_intensity;

    s1 = nc/4.0;
    s2 = (nc - s1)/3.0;
    s3 = (nc - s1 - s2)/2.0;
    s4 = (nc - s1 - s2 - s3);

/* cyan -> green */
    _PG_trans_palette(true_cm,
		      s1, 0.0, mxg, mxb, 0.0, mxg, 0.0, -2.0);

/* green -> yellow */
    _PG_trans_palette(true_cm + s1 - 1,
		      s2, 0.0, mxg, 0.0, mxr, mxg, 0.0, 2.0);

/* yellow -> red */
    _PG_trans_palette(true_cm + s1 + s2 - 1,
		      s3, mxr, mxg, 0.0, mxr, 0.0, 0.0, -2.0);

/* red -> magenta */
    _PG_trans_palette(true_cm + s1 + s2 + s3 - 1,
		      s4+1, mxr, 0.0, 0.0, mxr, 0.0, mxb, 2.0);

    if (reg)
       _PG_register_palette(dev, pal, FALSE);

    return(pal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_HC_PALETTE - compute a blue -> white -> red palette */

static PG_palette *_PG_hc_palette(dev, pal, nc)
   PG_device *dev;
   PG_palette *pal;
   int nc;
   {int ns, reg;
    int s1, s2, s3;
    double mxr, mxg, mxb;
    RGB_color_map *true_cm;

    reg = (pal == NULL);
    if (reg)
       pal = PG_mk_palette(dev, "hc", nc);

    true_cm = pal->true_colormap + 2;

    ns  = MAXPIX;
    mxr = ns*pal->max_red_intensity;
    mxg = ns*pal->max_green_intensity;
    mxb = ns*pal->max_blue_intensity;

    s1 = nc/4.0;
    s2 = (nc - s1)/3.0;
    s3 = (nc - s1 - s2)/2.0;

/* dark blue -> blue */
    _PG_trans_palette(true_cm,
		      s1, 0.0, 0.0, 0.5*mxb, 0.0, 0.0, mxb, 1.0);

/* blue -> white */
    _PG_trans_palette(true_cm + s1 - 1,
		      s2, 0.0, 0.0, mxb, mxr, mxg, mxb, 1.0);

/* white -> red */
    _PG_trans_palette(true_cm + s1 + s2 - 1,
		      s3, mxr, mxg, mxb, mxr, 0.0, 0.0, 1.0);

/* red -> dark red */
    _PG_trans_palette(true_cm + s1 + s2 + s3 - 1,
		      s2+1, mxr, 0.0, 0.0, 0.5*mxr, 0.0, 0.0, 1.0);

    if (reg)
       _PG_register_palette(dev, pal, FALSE);

    return(pal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_BGY_PALETTE - compute a blue -> cyan -> green -> yellow palette */

static PG_palette *_PG_bgy_palette(dev, pal, nc)
   PG_device *dev;
   PG_palette *pal;
   int nc;
   {int ns, reg;
    int s1, s2, s3;
    double mxr, mxg, mxb;
    RGB_color_map *true_cm;

    reg = (pal == NULL);
    if (reg)
       pal = PG_mk_palette(dev, "bgy", nc);

    true_cm = pal->true_colormap + 2;

    ns  = MAXPIX;
    mxr = ns*pal->max_red_intensity;
    mxg = ns*pal->max_green_intensity;
    mxb = ns*pal->max_blue_intensity;

    s1 = nc/3.0;
    s2 = (nc - s1)/2.0;
    s3 = (nc - s1 - s2);

/* blue -> cyan */
    _PG_trans_palette(true_cm,
		      s1, 0.0, 0.0, mxb, 0.0, mxg, mxb, 2.0);

/* cyan -> green */
    _PG_trans_palette(true_cm + s1 - 1,
		      s2, 0.0, mxg, mxb, 0.0, mxg, 0.0, -2.0);

/* green -> yellow */
    _PG_trans_palette(true_cm + s1 + s2 - 1,
		      s3+1, 0.0, mxg, 0.0, mxr, mxg, 0.0, 2.0);

    if (reg)
       _PG_register_palette(dev, pal, FALSE);

    return(pal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_TRI_PALETTE - compute a blue -> black -> white -> red palette */

static PG_palette *_PG_tri_palette(dev, pal, nc)
   PG_device *dev;
   PG_palette *pal;
   int nc;
   {int ns, reg;
    int s1, s2, s3;
    double mxr, mxg, mxb;
    RGB_color_map *true_cm;

    reg = (pal == NULL);
    if (reg)
       pal = PG_mk_palette(dev, "tri", nc);

    true_cm = pal->true_colormap + 2;

    ns  = MAXPIX;
    mxr = ns*pal->max_red_intensity;
    mxg = ns*pal->max_green_intensity;
    mxb = ns*pal->max_blue_intensity;

    s1 = nc/3.0;
    s2 = 0.66666*(nc - s1);
    s3 = (nc - s1 - s2);

/* blue -> black */
    _PG_trans_palette(true_cm,
		      s1, 0.0, 0.0, mxb, 0.0, 0.0, 0.0, -2.0);

/* black -> white */
    _PG_trans_palette(true_cm + s1 - 1,
		      s2, 0.0, 0.0, 0.0, mxr, mxg, mxb, 1.0);

/* white -> red */
    _PG_trans_palette(true_cm + s1 + s2 - 1,
		      s3+1, mxr, mxg, mxb, mxr, 0.0, 0.0, 0.0);

    if (reg)
       _PG_register_palette(dev, pal, FALSE);

    return(pal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_IRON_PALETTE - compute a black -> red -> yellow -> white palette */

static PG_palette *_PG_iron_palette(dev, pal, nc)
   PG_device *dev;
   PG_palette *pal;
   int nc;
   {int ns, reg;
    int s1, s2, s3;
    double mxr, mxg, mxb;
    RGB_color_map *true_cm;

    reg = (pal == NULL);
    if (reg)
       pal = PG_mk_palette(dev, "iron", nc);

    true_cm = pal->true_colormap + 2;

    ns  = MAXPIX;
    mxr = ns*pal->max_red_intensity;
    mxg = ns*pal->max_green_intensity;
    mxb = ns*pal->max_blue_intensity;

    s1 = 0.5*nc;
    s2 = 0.75*(nc - s1);
    s3 = (nc - s1 - s2);

/* black -> red */
    _PG_trans_palette(true_cm,
		      s1, 0.0, 0.0, 0.0, mxr, 0.0, 0.0, 2.0);

/* red -> yellow */
    _PG_trans_palette(true_cm + s1 - 1,
		      s2, mxr, 0.0, 0.0, mxr, mxg, 0.0, 2.0);

/* yellow -> white */
    _PG_trans_palette(true_cm + s1 + s2 - 1,
		      s3+1, mxr, mxg, 0.0, mxr, mxg, mxb, 2.0);

    if (reg)
       _PG_register_palette(dev, pal, FALSE);

    return(pal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_THRESH_PALETTE - compute a blue -> cyan  yellow -> red palette */

static PG_palette *_PG_thresh_palette(dev, pal, nc)
   PG_device *dev;
   PG_palette *pal;
   int nc;
   {int ns, reg;
    int s1, s2;
    double mxr, mxg, mxb;
    RGB_color_map *true_cm;

    reg = (pal == NULL);
    if (reg)
       pal = PG_mk_palette(dev, "thresh", nc);

    true_cm = pal->true_colormap + 2;

    ns  = MAXPIX;
    mxr = ns*pal->max_red_intensity;
    mxg = ns*pal->max_green_intensity;
    mxb = ns*pal->max_blue_intensity;

    s1 = 0.5*nc;
    s2 = nc - s1;

/* blue -> cyan */
    _PG_trans_palette(true_cm,
		      s1, 0.0, 0.0, mxb, 0.0, mxg, mxb, 2.0);

/* yellow -> red */
    _PG_trans_palette(true_cm + s1 - 1,
		      s2+1, mxr, mxg, 0.0, mxr, 0.0, 0.0, -2.0);

    if (reg)
       _PG_register_palette(dev, pal, FALSE);

    return(pal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RAND_PALETTE - compute a palette with random colors */

static PG_palette *_PG_rand_palette(dev, pal)
   PG_device *dev;
   PG_palette *pal;
   {int i, j, k, ic, nc, reg;
    int *ip; 
    PG_palette *root;
    RGB_color_map *true_cm, *root_cm;

    root    = dev->color_table;
    root_cm = root->true_colormap;
    nc      = root->n_pal_colors;

    ip = FMAKE_N(int, nc, "_PG_RAND_PALETTE:ip");
    for (i = 0; i < nc; i++)
        ip[i] = i;

    for (i = 0, ic = nc-1; i < nc; i++, ic--)
        {j = 0.5*(1.0 + PM_random(-1))*ic;

         k      = ip[j];
	 ip[j]  = ip[ic];
         ip[ic] = k;};

    reg = (pal == NULL);
    if (reg)
       pal = PG_mk_palette(dev, "rand", nc);

    true_cm = pal->true_colormap + 2;

    for (i = 0; i < nc; i++)
        true_cm[i] = root_cm[ip[i]];

    SFREE(ip);

    if (reg)
       _PG_register_palette(dev, pal, FALSE);

    return(pal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_MONO_PALETTES - create the mono chromatic palettes */

static void _PG_mono_palettes(dev, nct)
   PG_device *dev;
   int nct;
   {int l, ns, nc;
    int s1, s2;
    double mxr, mxg, mxb, rf, gf, bf, rw, gw, bw, *pv;
    PG_palette *pal;
    RGB_color_map *true_cm;
    static char *colors[] = {"magentas", "blues", "cyans",
			     "greens", "yellows", "reds"};
    static double val[] = {1.0, 0.0, 1.0,   0.0, 0.0, 1.0,   0.0, 1.0, 1.0,
			   0.0, 1.0, 0.0,   1.0, 1.0, 0.0,   1.0, 0.0, 0.0};

    nc = nct/6.0;
    ns  = MAXPIX;
    mxr = ns*dev->max_red_intensity;
    mxg = ns*dev->max_green_intensity;
    mxb = ns*dev->max_blue_intensity;

    pv = val;
    for (l = 0; l < 6; l++)
        {pal = PG_mk_palette(dev, colors[l], nc);

	 true_cm = pal->true_colormap + 2;

	 s1 = 0.715*nc;
	 s2 = nc - s1;

         rf = *pv++;
         gf = *pv++;
         bf = *pv++;

         rw = max(rf, 0.85);
         gw = max(gf, 0.85);
         bw = max(bf, 0.85);

/* low value -> high value */
	 _PG_trans_palette(true_cm, s1,
			   0.5*rf*mxr, 0.5*gf*mxg, 0.5*bf*mxb,
			   rf*mxr, gf*mxg, bf*mxb,
			   1.0);

/* high saturation -> low saturation */
	 _PG_trans_palette(true_cm + s1 - 1, s2+1,
			   rf*mxr, gf*mxg, bf*mxb,
			   rw*mxr, gw*mxg, bw*mxb,
			   1.0);

	 _PG_register_palette(dev, pal, FALSE);};

    return;}

/*--------------------------------------------------------------------------*/

/*                           PRINCIPAL COLOR SELECTORS                      */

/*--------------------------------------------------------------------------*/

/* _PG_BW_PALETTE - compute a black and white palette
 *                - always add ABSOLUTE black and white in the palette
 *                - as the first two colors
 */

static PG_palette *_PG_bw_palette(dev, pal, off, nc)
   PG_device *dev;
   PG_palette *pal;
   int off, nc;
   {int i, j, ns, reg;
    unsigned short *pv;
    double mxr, mxg, mxb,  scale;
    RGB_color_map *true_cm;

    reg = (pal == NULL);
    if (reg)
       pal = PG_mk_palette(dev, "bw", nc);

    true_cm = pal->true_colormap;
    pv      = pal->pixel_value;

    ns  = MAXPIX;
    mxr = ns*pal->max_red_intensity;
    mxg = ns*pal->max_green_intensity;
    mxb = ns*pal->max_blue_intensity;

    true_cm += (2 + off);
    if (pv != NULL)
       pv += (2 + off);

/* NOTE: there is no sense in repeating full black and full white */
    scale = 1.0/(nc + 1.0);
    for (i = 0; i < nc; i++)
        {j = i + 1;
	 true_cm[i].red   = mxr*j*scale;
         true_cm[i].green = mxg*j*scale;
         true_cm[i].blue  = mxb*j*scale;

         if (pv != NULL)
	    pv[i] = i + off + 2;};

#if 0
    int n_pal_colors, n_dev_colors;
    int white, black;
    RGB_color_map *pseudo_cm;

    pseudo_cm    = pal->pseudo_colormap;
    n_dev_colors = pal->n_dev_colors;
    if (n_dev_colors < nc)
       {switch (n_dev_colors)
           {case  2: white      = 1;
                     black      = 0;
                     break;
            case  4: white      = 2;
                     black      = 0;
                     break;
            default: white      = 7;
                     black      = 0;};

        for (i = 0; i < nc; i++)
            {pseudo_cm[i].red   = pal->max_red_intensity*
                                  ((double) i)/((double) nc);
             pseudo_cm[i].green = pal->max_green_intensity*white;
             pseudo_cm[i].blue  = pal->max_blue_intensity*black;};

        pal->pseudo_colormap = pseudo_cm;};
#endif

    if (reg)
       _PG_register_palette(dev, pal, FALSE);

    return(pal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RAINBOW_PALETTE - set up a chromatic color palette
 *                     - always add ABSOLUTE black and white in the palette
 *                     - as the first two colors
 */

static PG_palette *_PG_rainbow_palette(dev, pal, off, nc)
   PG_device *dev;
   PG_palette *pal;
   int off, nc;
   {int i, j, l, ns, nb, nv, nl, nr;
    int reg;
    unsigned short *pv;
    RGB_color_map *true_cm;
    double a, r, g, b, rx, gx, bx, mxr, mxg, mxb, fl, fu, fs;
    double *pcv;
    static double cv[] = {1.0, 0.0, 1.0,  0.0, 0.0, 1.0,   0.0, 1.0, 1.0,
		          0.0, 1.0, 0.0,  1.0, 1.0, 0.0,   1.0, 0.0, 0.0};

    reg = (pal == NULL);
    if (reg)
       pal = PG_mk_palette(dev, "rainbow", nc);

    true_cm = pal->true_colormap;
    pv      = pal->pixel_value;

    ns  = MAXPIX;
    mxr = ns*pal->max_red_intensity;
    mxg = ns*pal->max_green_intensity;
    mxb = ns*pal->max_blue_intensity;

    true_cm += (2 + off);
    if (pv != NULL)
       pv += (2 + off);

/* number of colors in hue
 * nv and nl coefficients based on 14 colors divided to 10 and 4
 */
    nb = nc/6.0;
    fl = 1.0/1.4;
    nv = fl*((double) nb);
    nl = 0.4*((double) nv);
    nr = nc - 6.0*(nv + nl);

/* set a rainbow spectrum: magenta, blue, cyan, green, yellow, red */
    fl = 0.5;
    fu = 1.0;
    fs = 0.15;
    pcv = cv;
    for (l = 0, j = 0; l < 6; l++)
        {a = (fu - fl)/nv;
         r = (*pcv++)*mxr;
         g = (*pcv++)*mxg;
         b = (*pcv++)*mxb;
	 for (i = 1; (i <= nv) && (j < ns); i++, j++)
	     {rx = r*(a*i + fl);
	      gx = g*(a*i + fl);
	      bx = b*(a*i + fl);
	      true_cm[j].red   = rx;
	      true_cm[j].green = gx;
	      true_cm[j].blue  = bx;

              if (pv != NULL)
		 pv[j] = j + off + 2;};

	 a = ns*(1.0 - fs)/nl;
	 for (i = 1; (i <= nl) && (j < ns); i++, j++)
	     {rx += a;
	      gx += a;
	      bx += a;
	      rx  = min(rx, ns);
	      gx  = min(gx, ns);
	      bx  = min(bx, ns);
	      true_cm[j].red   = rx;
	      true_cm[j].green = gx;
	      true_cm[j].blue  = bx;

              if (pv != NULL)
		 pv[j] = j + off + 2;};};

/* fill the rest of the color map with gray scale */
    if (nr > 0)
       {_PG_SHADE_HUE(ns, ns, ns, nr, ns, 0.5, 1.0);};

    if (reg)
       _PG_register_palette(dev, pal, FALSE);

    return(pal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_2DRAINBOW_PALETTE - set up a chromatic color palette
 *                       - always add ABSOLUTE black and white in the palette
 *                       - as the first two colors
 */

static PG_palette *_PG_2drainbow_palette(dev, pal, off, nc)
   PG_device *dev;
   PG_palette *pal;
   int off, nc;
   {int i, j, l, ns, nb, nv, nl, nr, shape[2];
    int reg;
    unsigned short *pv;
    RGB_color_map *true_cm;
    double a, r, g, b, rx, gx, bx, mxr, mxg, mxb, fl, fu, fs;
    double *pcv;
    static double cv[] = {1.0, 0.0, 1.0,  0.0, 0.0, 1.0,   0.0, 1.0, 1.0,
		          0.0, 1.0, 0.0,  1.0, 1.0, 0.0,   1.0, 0.0, 0.0};

    reg = (pal == NULL);
    if (reg)
       pal = PG_mk_palette(dev, "2drainbow", nc);

    true_cm = pal->true_colormap;
    pv      = pal->pixel_value;

    ns  = MAXPIX;
    mxr = ns*pal->max_red_intensity;
    mxg = ns*pal->max_green_intensity;
    mxb = ns*pal->max_blue_intensity;

    true_cm += (2 + off);
    if (pv != NULL)
       pv += (2 + off);

/* number of colors in hue
 * nv and nl coefficients based on 14 colors divided to 10 and 4
 */
    nb = nc/6.0;
    fl = 1.0/1.4;
    nv = fl*((double) nb);
    nl = 0.4*((double) nv);
    nr = nc - 6.0*(nv + nl);

/* set a rainbow spectrum: magenta, blue, cyan, green, yellow, red */
    fl = 0.5;
    fu = 1.0;
    fs = 0.15;
    pcv = cv;
    for (l = 0, j = 0; l < 6; l++)
        {a = (fu - fl)/nv;
         r = (*pcv++)*mxr;
         g = (*pcv++)*mxg;
         b = (*pcv++)*mxb;
	 for (i = 1; (i <= nv) && (j < ns); i++, j++)
	     {rx = r*(a*i + fl);
	      gx = g*(a*i + fl);
	      bx = b*(a*i + fl);
	      true_cm[j].red   = rx;
	      true_cm[j].green = gx;
	      true_cm[j].blue  = bx;

	      if (pv != NULL)
		 pv[i] = i + off + 2;};

	 a = ns*(1.0 - fs)/nl;
	 for (i = 1; (i <= nl) && (j < ns); i++, j++)
	     {rx += a;
	      gx += a;
	      bx += a;
	      rx  = min(rx, ns);
	      gx  = min(gx, ns);
	      bx  = min(bx, ns);
	      true_cm[j].red   = rx;
	      true_cm[j].green = gx;
	      true_cm[j].blue  = bx;

	      if (pv != NULL)
		 pv[i] = i + off + 2;};};

/* fill the rest of the color map with gray scale */
    if (nr > 0)
       {_PG_SHADE_HUE(ns, ns, ns, nr, ns, 0.5, 1.0);};

    shape[0] = 14;
    shape[1] = 6;

    _PG_attach_palette_dims(pal, nc, 2, shape);

    if (reg)
       _PG_register_palette(dev, pal, FALSE);

    return(pal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_2DCTAB1_PALETTE - set up a 2d color table (as specified by 
 *                     - Pete Keller)
 *                     - always add ABSOLUTE black and white in the palette
 *                     - as the first two colors
 */

static PG_palette *_PG_2dctab1_palette(dev, pal, off)
   PG_device *dev;
   PG_palette *pal;
   int off;
   {int i, j, k, nc, shape[2];
    double r0, g0, b0, dxr, dxg, dxb, saturated;
    int reg;
    RGB_color_map *true_cm;

    nc = 192;

    reg = (pal == NULL);
    if (reg)
       pal = PG_mk_palette(dev, "2dctab1", nc);

    true_cm = pal->true_colormap;

    true_cm += (2 + off);
    
    saturated = MAXPIX;

    r0 = 0.0;
    g0 = saturated;
    b0 = saturated;

/*
    dxr = saturated/23.0;
    dxg = -dxr;
    dxb = -saturated/7.0;

    for (i = 0, k = 0; i < 8; i++)
        for (j = 0; j < 24; j++, k++)
            {true_cm[k].red   = r0 + j * dxr;
             true_cm[k].green = g0 + j * dxg;
             true_cm[k].blue  = b0 + i * dxb;}
*/

    dxr = saturated/11.0;
    dxg = -dxr;
    dxb = -saturated/7.0;

    for (i = 0, k = 0; i < 8; i++)
        {for (j = 0; j < 12; j++, k++)
            {true_cm[k].red   = r0 + j * dxr;
             true_cm[k].green = g0;
             true_cm[k].blue  = b0 + i * dxb;}
         for (j = 0; j < 12; j++, k++)
            {true_cm[k].red   = saturated;
             true_cm[k].green = g0 + j * dxg;
             true_cm[k].blue  = b0 + i * dxb;};} 

    shape[0] = 24;
    shape[1] =  8;

    _PG_attach_palette_dims(pal, nc, 2, shape);

    if (reg)
       _PG_register_palette(dev, pal, FALSE);

    return(pal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_SPECTRUM_PALETTE - a different color palette
 *                      - always add ABSOLUTE black and white in the palette
 *                      - as the first two colors
 */

static PG_palette *_PG_spectrum_palette(dev, pal, off, nc)
   PG_device *dev;
   PG_palette *pal;
   int off, nc;
   {int i, j, reg;
    int ns, s1, s2, s3, s4, s5, s6, ss;
    unsigned short *pv;
    double mxr, mxg, mxb, f, x;
    RGB_color_map *true_cm;

    reg = (pal == NULL);
    if (reg)
       pal = PG_mk_palette(dev, "spectrum", nc);

    true_cm = pal->true_colormap;
    pv      = pal->pixel_value;

    ns  = MAXPIX;
    mxr = ns*pal->max_red_intensity;
    mxg = ns*pal->max_green_intensity;
    mxb = ns*pal->max_blue_intensity;

    true_cm += (off + 2);
    if (pv != NULL)
       pv += (2 + off);

    s1 = 0.20*nc;
    s4 = 0.20*nc;
    s5 = 0.20*nc;
    s6 = 0.20*nc;
    ss = nc - (s1 + s4 + s5 + s6);
    s2 = 0.5*ss;
    s3 = ss - s2;

/* blue -> cyan */
    f = 1.0/((double) s1);
    for (i = 0, j = 0; i < s1; i++, j++)
        {x = 1.0/(1.0 + i*f);
         true_cm[j].blue  = mxb;
         true_cm[j].green = mxg*1.33333*(1.0 - x*x);
         true_cm[j].red   = 0;

         if (pv != NULL)
	    pv[j] = j + off + 2;};

/* cyan -> green */
    f = 1.0/((double) s2);
    for (i = 0; i < s2; i++, j++)
        {x = 1.0/(1.0 + (s2 - i)*f);
         true_cm[j].blue  = mxb*1.33333*(1.0 - x*x);
         true_cm[j].green = mxg;
         true_cm[j].red   = 0;

         if (pv != NULL)
	    pv[j] = j + off + 2;};

/* green -> yellow */
    f = 1.0/((double) s3);
    for (i = 0; i < s3; i++, j++)
        {x = 1.0/(1.0 + i*f);
         true_cm[j].blue  = 0;
         true_cm[j].green = mxg;
         true_cm[j].red   = mxr*1.33333*(1.0 - x*x);

         if (pv != NULL)
	    pv[j] = j + off + 2;};

/* yellow -> red */
    f = 1.0/((double) s4);
    for (i = 0; i < s4; i++, j++)
        {x = 1.0/(1.0 + (s4 - i)*f);
         true_cm[j].blue  = 0;
         true_cm[j].green = mxg*1.33333*(1.0 - x*x);
         true_cm[j].red   = mxr;

         if (pv != NULL)
	    pv[j] = j + off + 2;};

/* red -> magenta */
    f = 1.0/((double) s5);
    for (i = 0; i < s5; i++, j++)
        {x = 1.0/(1.0 + i*f);
         true_cm[j].blue  = mxb*1.33333*(1.0 - x*x);
         true_cm[j].green = 0;
         true_cm[j].red   = mxr;

         if (pv != NULL)
	    pv[j] = j + off + 2;};

/* magenta -> blue */
    f = 1.0/((double) s6);
    for (i = 0; i < s6; i++, j++)
        {x = 1.0/(1.0 + (s6 - i)*f);
         true_cm[j].blue  = mxb;
         true_cm[j].green = 0;
         true_cm[j].red   = mxr*1.33333*(1.0 - x*x);

         if (pv != NULL)
	    pv[j] = j + off + 2;};

    if (reg)
       _PG_register_palette(dev, pal, FALSE);

    return(pal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_CLR_CONTRAST_PALETTE - a color contrast palette
 *                          - always add ABSOLUTE black and white
 *                          - as the first two colors
 */

static PG_palette *_PG_clr_contrast_palette(dev, pal, off, nc)
   PG_device *dev;
   PG_palette *pal;
   int off, nc;
   {int j, reg, ns;
    unsigned short *pv;
    double mxr, mxg, mxb;
    RGB_color_map *true_cm;

    reg = (pal == NULL);
    if (reg)
       pal = PG_mk_palette(dev, "contrast", nc);

    true_cm = pal->true_colormap;
    pv      = pal->pixel_value;

    ns  = MAXPIX;
    mxr = ns*pal->max_red_intensity;
    mxg = ns*pal->max_green_intensity;
    mxb = ns*pal->max_blue_intensity;

    true_cm += (off + 2);
    if (pv != NULL)
       pv += (2 + off);

    j = 0;
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.0000, 0.0000, 1.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.2125, 0.2125, 1.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.4250, 0.4250, 1.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.6375, 0.6375, 1.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.8500, 0.8500, 1.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.8500, 1.0000, 1.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 1.0000, 1.0000, 0.8500);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 1.0000, 1.0000, 0.6375);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 1.0000, 1.0000, 0.4250);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 1.0000, 1.0000, 0.2125);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.9541, 1.0000, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.9000, 0.9000, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.8000, 0.8000, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.7000, 0.7000, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.6000, 0.6000, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.5500, 0.5500, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.5500, 0.5500, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.5500, 0.5500, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.5500, 0.0000, 0.5500);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.5500, 0.0000, 0.5500);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.7000, 0.0000, 0.7000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.9000, 0.0000, 0.9000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 1.0000, 0.0000, 0.8533);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 1.0000, 0.2125, 1.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 1.0000, 0.4250, 1.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 1.0000, 0.6375, 1.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 1.0000, 0.8500, 1.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.8500, 1.0000, 1.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.6375, 1.0000, 1.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.4250, 1.0000, 1.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.2125, 1.0000, 1.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.0000, 0.9500, 0.9500);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.0000, 0.9000, 0.9000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.0000, 0.8000, 0.8000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.0000, 0.6000, 0.6000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.0000, 0.5500, 0.5500);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.0000, 0.5500, 0.5500);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.0000, 0.5500, 0.5500);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.0000, 0.5500, 0.5500);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.5500, 0.0000, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.5500, 0.0000, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.5500, 0.0000, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.6000, 0.0000, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.7000, 0.0000, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.8000, 0.0000, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.9000, 0.0000, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 1.0000, 0.1972, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 1.0000, 0.2125, 0.2125);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 1.0000, 0.4250, 0.4250);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 1.0000, 0.6375, 0.6375);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 1.0000, 0.8500, 0.8500);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.8500, 1.0000, 0.8500);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.6375, 1.0000, 0.6375);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.4250, 1.0000, 0.4250);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.2125, 1.0000, 0.2125);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.0000, 1.0000, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.0000, 0.9000, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.0000, 0.8000, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.0000, 0.7000, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.0000, 0.6000, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.0000, 0.5500, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.0000, 0.5500, 0.0000);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.0000, 0.0000, 0.5500);
    INSTALL_COLOR(true_cm, mxr, mxg, mxb, 0.0000, 0.0000, 0.5500);

    if (reg)
       _PG_register_palette(dev, pal, FALSE);

    return(pal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_STANDARD_PALETTE - set up a standard color palette for dev */

static PG_palette *_PG_standard_palette(dev, pal, off, nc,
                                        Light, Dark,
					Light_Gray, Dark_Gray,
					Lightest, Darkest)
   PG_device *dev;
   PG_palette *pal;
   int off, nc;
   int Light, Dark, Light_Gray, Dark_Gray, Lightest, Darkest;
   {int i, reg;
    unsigned short *pv;
    RGB_color_map *true_cm;

    reg = (pal == NULL);
    if (reg)
       pal = PG_mk_palette(dev, "standard", nc);

    true_cm = pal->true_colormap + off;

/* compute the color maps */
    true_cm[dev->RED].red            = pal->max_red_intensity*Light;
    true_cm[dev->RED].green          = pal->max_green_intensity*Dark;
    true_cm[dev->RED].blue           = pal->max_blue_intensity*Dark;
    
    true_cm[dev->YELLOW].red         = pal->max_red_intensity*Light_Gray;
    true_cm[dev->YELLOW].green       = pal->max_green_intensity*Light_Gray;
    true_cm[dev->YELLOW].blue        = pal->max_blue_intensity*Dark;
    
    true_cm[dev->GREEN].red          = pal->max_red_intensity*Dark;
    true_cm[dev->GREEN].green        = pal->max_green_intensity*Light;
    true_cm[dev->GREEN].blue         = pal->max_blue_intensity*Dark;
    
    true_cm[dev->CYAN].red           = pal->max_red_intensity*Dark;
    true_cm[dev->CYAN].green         = pal->max_green_intensity*Light;
    true_cm[dev->CYAN].blue          = pal->max_blue_intensity*Light;
    
    true_cm[dev->BLUE].red           = pal->max_red_intensity*Dark;
    true_cm[dev->BLUE].green         = pal->max_green_intensity*Dark;
    true_cm[dev->BLUE].blue          = pal->max_blue_intensity*Light;
    
    true_cm[dev->MAGENTA].red        = pal->max_red_intensity*Light;
    true_cm[dev->MAGENTA].green      = pal->max_green_intensity*Dark;
    true_cm[dev->MAGENTA].blue       = pal->max_blue_intensity*Light;
    
    true_cm[dev->DARK_RED].red       = pal->max_red_intensity*Light_Gray;
    true_cm[dev->DARK_RED].green     = pal->max_green_intensity*Dark;
    true_cm[dev->DARK_RED].blue      = pal->max_blue_intensity*Dark;
    
    true_cm[dev->BROWN].red          = pal->max_red_intensity*Light_Gray;
    true_cm[dev->BROWN].green        = pal->max_green_intensity*Dark_Gray;
    true_cm[dev->BROWN].blue         = pal->max_blue_intensity*Dark;
    
    true_cm[dev->DARK_GREEN].red     = pal->max_red_intensity*Dark;
    true_cm[dev->DARK_GREEN].green   = pal->max_green_intensity*Light_Gray;
    true_cm[dev->DARK_GREEN].blue    = pal->max_blue_intensity*Dark;
    
    true_cm[dev->DARK_CYAN].red      = pal->max_red_intensity*Dark;
    true_cm[dev->DARK_CYAN].green    = pal->max_green_intensity*Light_Gray;
    true_cm[dev->DARK_CYAN].blue     = pal->max_blue_intensity*Light_Gray;
    
    true_cm[dev->DARK_BLUE].red      = pal->max_red_intensity*Dark;
    true_cm[dev->DARK_BLUE].green    = pal->max_green_intensity*Dark;
    true_cm[dev->DARK_BLUE].blue     = pal->max_blue_intensity*Light_Gray;
    
    true_cm[dev->DARK_MAGENTA].red   = pal->max_red_intensity*Light_Gray;
    true_cm[dev->DARK_MAGENTA].green = pal->max_green_intensity*Dark;
    true_cm[dev->DARK_MAGENTA].blue  = pal->max_blue_intensity*Dark_Gray;

    true_cm[dev->GRAY].red           = pal->max_red_intensity*Light_Gray;
    true_cm[dev->GRAY].green         = pal->max_green_intensity*Light_Gray;
    true_cm[dev->GRAY].blue          = pal->max_blue_intensity*Light_Gray;
    
    true_cm[dev->DARK_GRAY].red      = pal->max_red_intensity*Dark_Gray;
    true_cm[dev->DARK_GRAY].green    = pal->max_green_intensity*Dark_Gray;
    true_cm[dev->DARK_GRAY].blue     = pal->max_blue_intensity*Dark_Gray;
    
/* these MUST be assigned last in case all dev->RED, etc have the same
 * value!!!
 */
    if (off == 0)
       {true_cm[dev->BLACK].red   = pal->max_red_intensity*Darkest;
	true_cm[dev->BLACK].green = pal->max_green_intensity*Darkest;
	true_cm[dev->BLACK].blue  = pal->max_blue_intensity*Darkest;

	true_cm[dev->WHITE].red   = pal->max_red_intensity*Lightest;
	true_cm[dev->WHITE].green = pal->max_green_intensity*Lightest;
	true_cm[dev->WHITE].blue  = pal->max_blue_intensity*Lightest;}

    pv = pal->pixel_value;
    if (pv != NULL)
       {for (i = 0; i < 16; i++)
            pv[i] = i;};

    dev->line_color = dev->WHITE;
    dev->text_color = dev->WHITE;

    if (reg)
       _PG_register_palette(dev, pal, FALSE);

    return(pal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_REGISTER_PALETTE - enter the given palette into the list of palettes
 *                      - for the given device
 *                      - match the colors of the palette to those from the
 *                      - device color table
 */

void _PG_register_palette(dev, pal, map)
   PG_device *dev;
   PG_palette *pal;
   int map;
   {

/* splice the palette into the ring */
    if (dev->palettes == NULL)
       {pal->next     = pal;
        dev->palettes = pal;}

    else
       {pal->next           = dev->palettes->next;
        dev->palettes->next = pal;};

/* match the palettes to the color table */
    PG_match_rgb_colors(dev, pal);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_RGB_INDEX - replace the contents of the given RGB color cell with
 *              - the nearest RGB values from the device color table
 *              - return the index from the device color table
 */

int PG_rgb_index(dev, cm)
   PG_device *dev;
   RGB_color_map *cm;
   {int i, ic, npc;
    double r, g, b, dc, dcmin, dr, dg, db;
    PG_palette *root;
    RGB_color_map *ct;

    r = cm->red;
    g = cm->green;
    b = cm->blue;

    root = dev->color_table;
    npc  = root->n_pal_colors;
    ct   = root->true_colormap;

    dcmin = HUGE;
    ic    = 0;
    for (i = 0; i < npc; i++)
        {dr = r - ct[i].red;
	 dg = g - ct[i].green;
	 db = b - ct[i].blue;
	 dc = dr*dr + dg*dg + db*db;
	 if (dc < dcmin)
	    {ic = i;
	     dcmin = dc;};};

    *cm = ct[ic];

    return(ic);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_MATCH_RGB_COLORS - map the RGB references in the palette to
 *                      - valid colors from the root color table generically
 */

void _PG_match_rgb_colors(dev, pal)
   PG_device *dev;
   PG_palette *pal;
   {int i, npc, ic;
    unsigned short *pv;
    RGB_color_map *true_cm;

    npc     = pal->n_pal_colors;
    true_cm = pal->true_colormap;
    pv      = pal->pixel_value;

/* add two for the black and white colors at the top */
    npc += 2;
    for (i = 0; i < npc; i++)
        {ic = PG_rgb_index(dev, true_cm++);
	 if (pv != NULL)
	    pv[i] = ic;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SETUP_STANDARD_PALETTES - setup the standard set of palettes for
 *                            - a PGS device
 *                            - NC is the number of colors in the spectrum
 *                            - from blue to blue thru green and red
 *                            - it sets the scale for the number of colors
 *                            - in the other palettes
 */

void PG_setup_standard_palettes(dev, nc, l1, l2, l3, l4, l5, l6)
   PG_device *dev;
   int nc, l1, l2, l3, l4, l5, l6;
   {int nbw, nsp, nrb, ntc;
    PG_palette *root_pal;

/* build up the device color table */
    nbw = 0.5*nc;
    nsp = nc;
    nrb = 1.3125*nc;
    ntc = nbw + nsp + nrb + 14;
    root_pal = PG_mk_palette(dev, "Root", ntc);

    _PG_bw_palette(dev, root_pal, 0, nbw);
    _PG_spectrum_palette(dev, root_pal, nbw, nsp);
    _PG_rainbow_palette(dev, root_pal, nbw+nsp, nrb);
    _PG_standard_palette(dev, root_pal, nbw+nsp+nrb, 14,
			 l1, l2, l3, l4, l5, l6);

    PG_map_to_color_table(dev, root_pal);

    dev->color_table = root_pal;

/* make and register the standard palettes */
    dev->current_palette = _PG_standard_palette(dev, (PG_palette *) NULL, 0,
                                                14, l1, l2, l3, l4, l5, l6);
    _PG_spectrum_palette(dev, (PG_palette *) NULL, 0, nsp);
    _PG_clr_contrast_palette(dev, (PG_palette *) NULL, 0, nsp);
    _PG_rainbow_palette(dev, (PG_palette *) NULL, 0, nrb);
    _PG_2drainbow_palette(dev, (PG_palette *) NULL, 0, nrb);
    _PG_2dctab1_palette(dev, (PG_palette *) NULL, 0);
    _PG_bw_palette(dev, (PG_palette *) NULL, 0, nbw);
    _PG_wb_palette(dev, (PG_palette *) NULL, nbw);
    _PG_rgb_palette(dev, (PG_palette *) NULL, nsp);
    _PG_cym_palette(dev, (PG_palette *) NULL, nsp);
    _PG_bgy_palette(dev, (PG_palette *) NULL, nsp);
    _PG_tri_palette(dev, (PG_palette *) NULL, nbw);
    _PG_iron_palette(dev, (PG_palette *) NULL, nbw);
    _PG_hc_palette(dev, (PG_palette *) NULL, nbw);
    _PG_thresh_palette(dev, (PG_palette *) NULL, nbw);
    _PG_rand_palette(dev, (PG_palette *) NULL);
    _PG_mono_palettes(dev, nrb);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SELECT_COLOR - given an array of N values return the color index
 *                 - from the N dimensional palette
 */

int PG_select_color(dev, n, a, extr)
   PG_device *dev;
   int n;
   REAL *a, *extr;
   {int i, color, prod, *shape;
    int **pd;
    REAL da, vl, av, an, ax, *xt, cf;
    PG_palette *pal;

    pal = dev->current_palette;
    pd  = pal->pal_dims;
    if ((pd == NULL) || (pal->max_pal_dims < n-1))
       {xt = extr;
	vl = 0.0;
        cf = ((double) pal->n_pal_colors - 1.0)/((double) n);
	for (i = 0; i < n; i++)
	    {av = a[i];
             an = xt[0];
             ax = xt[1];

             if (_PG_color_map_to_extrema)
                {if (av < an)
                    av = an;
                 else if (av > ax)
                    av = ax;}

	     if ((an <= av) && (av <= ax))
	        {da  = (av - an)/(ax - an + SMALL);
		 vl += da;}
	     else
	        return(dev->BLACK);

	     xt += 2;};

	color = vl*cf + 2;}
       
    else
       {shape = pal->pal_dims[n-1];
        color = 2;
	xt    = extr;
	prod  = 1;
	for (i = 0; i < n; i++)
	    {av = a[i];
             an = xt[0];
             ax = xt[1];

             if (_PG_color_map_to_extrema)
                {if (av < an)
                    av = an;
                 else if (av > ax)
                    av = ax;}

	     if ((an <= av) && (av <= ax))
	        {da     = (shape[i] - 1)/(ABS(ax - an) + SMALL);
		 color += (int)(ABS(av - an)*da + 0.49999) * prod;
		 prod  *= shape[i];}
	     else
	        return(dev->BLACK);

	     xt += 2;};};

    return(color);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_PALETTE - allocate and fill a PG_palette with the color
 *                 - palette of the given device
 */

PG_palette *PG_get_palette(dev, name)
   PG_device *dev;
   char *name;
   {PG_palette *pal;

    pal = dev->palettes;
    while (TRUE)
       {if (strcmp(name, pal->name) == 0)
           break;

        pal = pal->next;
        if (pal == dev->palettes)
           return(NULL);};

    return(pal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SET_PALETTE - set the color palette of the given device
 *                - return the named palette if successful
 *                - return NULL otherwise
 */

PG_palette *PG_set_palette(dev, name)
   PG_device *dev;
   char *name;
   {PG_palette *pal;

    pal = dev->palettes;
    while (TRUE)
       {if (strcmp(name, pal->name) == 0)
           {dev->current_palette = pal;
            break;};

        pal = pal->next;
        if (pal == dev->palettes)
	   {pal = NULL;
	    break;};};

    return(pal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_RGB_HSV - convert an RGB color-map to an HSV colormap */

static void PG_rgb_hsv(cm, nc, flg)
   RGB_color_map *cm;
   int nc, flg;
   {int i;
    double r, g, b;
    double h, s, v;
    double d, rc, gc, bc, norm, mlt;

    mlt = flg ? MAXPIX : 1.0;

    norm = 1.0/MAXPIX;
    for (i = 0; i < nc; i++)
        {r = norm*cm[i].red;
         g = norm*cm[i].green;
         b = norm*cm[i].blue;

         v = max(r, g);
         v = max(v, b);

         s = min(r, g);
         s = min(s, b);

         d = v - s;
         s = (v != 0) ? d/v : 0;

         if (s == 0.0)
            h = 0.0;
         else
            {d  = 1.0/d;
	     rc = (v - r)*d;
	     gc = (v - g)*d;
	     bc = (v - b)*d;
	     if (r == v)
	        h = 2.0 + bc - gc;
	     else if (g == v)
	        h = 4.0 + rc - bc;
	     else if (b == v)
	        h = 6.0 + gc - rc;};

         h /= 7.0;

         cm[i].red   = mlt*h;
         cm[i].green = mlt*s;
         cm[i].blue  = mlt*v;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_HSV_RGB - convert an HSV color-map to an RGB colormap */

static void PG_hsv_rgb(cm, nc, flg)
   RGB_color_map *cm;
   int nc, flg;
   {int i, j;
    double h, s, v;
    double f, p, q, t, norm, mlt;

    mlt  = flg ? 1.0 : MAXPIX;
    norm = flg ? 1.0/MAXPIX : 1.0;
    for (i = 0; i < nc; i++)
        {h = cm[i].red;
         s = norm*cm[i].green;
         v = mlt*cm[i].blue;

         h *= 7.0*norm;
         if (s == 0.0)
            {v = (h < 0.999999) ? v : 0.0;

	     cm[i].red   = v;
	     cm[i].green = v;
	     cm[i].blue  = v;}

	 else
	    {j = floor(h);
             f = h - j;
             p = v*(1.0 - s);
             q = v*(1.0 - s*f);
             t = v*(1.0 - s*(1.0 - f));
	     switch (j)

/* magenta */
	        {case 1 :
		      cm[i].red   = v;
		      cm[i].green = p;
		      cm[i].blue  = q;
		      break;

/* red */
	         case 2 :
		      cm[i].red   = v;
		      cm[i].green = t;
		      cm[i].blue  = p;
		      break;

/* yellow */
	         case 3 :
		      cm[i].red   = q;
		      cm[i].green = v;
		      cm[i].blue  = p;
		      break;

/* green */
	         case 4 :
		      cm[i].red   = p;
		      cm[i].green = v;
		      cm[i].blue  = t;
		      break;

/* cyan */
	         case 5 :
		      cm[i].red   = p;
		      cm[i].green = q;
		      cm[i].blue  = v;
		      break;

/* blue */
	         case 6 :
		      cm[i].red   = t;
		      cm[i].green = p;
		      cm[i].blue  = v;
		      break;};};};

    return;}

/*--------------------------------------------------------------------------*/

/*                        PALETTE DISPLAY ROUTINES                          */

/*--------------------------------------------------------------------------*/

/* PG_SHOW_COLORMAP - display the colormap for the given device */

void PG_show_colormap(type, wbck)
   char *type;
   int wbck;
   {int k, l, n_pal_colors, color, ns, mapped;
    REAL dx, dy, xmn, xmx, ymn, ymx, x[5], y[5];
    PG_device *dev;

    dev = PG_make_device(type, "COLOR", "Device Colormap");
    PG_white_background(dev, wbck);
    PG_open_device(dev, 0.05, 0.1, 0.4, 0.4);

    PG_set_viewport(dev, 0.0, 1.0, 0.0, 1.0);
    PG_set_window(dev, 0.0, 1.0, 0.0, 1.0);

    PG_make_device_current(dev);
        
    mapped = TRUE;
    n_pal_colors = dev->current_palette->n_dev_colors;

    ns = ceil(sqrt((double) n_pal_colors));

    PG_get_viewport_WC(dev, &xmn, &xmx, &ymn, &ymx);
    PG_draw_box(dev, xmn, xmx, ymn, ymx);

    dx = (xmx - xmn)/ns;
    dy = (ymx - ymn)/ns;

    color = 0;
    y[0]  = ymn;
    for (k = 0; (k < ns) && (color < n_pal_colors); k++)
        {x[0] = xmn;
         for (l = 0; (l < ns) && (color < n_pal_colors); l++)
             {x[1] = x[0] + dx;
              y[1] = y[0];
              x[2] = x[1];
              y[2] = y[1] + dy;
              x[3] = x[2] - dx;
              y[3] = y[2];
              x[4] = x[0];
              y[4] = y[0];

              PG_set_color_fill(dev, color++, mapped);
              PG_shade_poly(dev, x, y, 5);

              x[0] += dx;};

         y[0] += dy;};

    PG_set_color_line(dev, dev->WHITE, TRUE);

    ns++;
    PG_draw_multiple_line(dev, ns, xmn, ymn, xmn, ymx, xmx, ymn, xmx, ymx, NULL);
    PG_draw_multiple_line(dev, ns, xmn, ymn, xmx, ymn, xmn, ymx, xmx, ymx, NULL);

    PG_update_vs(dev);

    SC_pause();

    PG_close_device(dev);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DUMP_COLORMAP - dump the colormap for the given device */

void PG_dump_colormap(type, file)
   char *type;
   char *file;
   {int k, ntc, nc, nbw, nsp, nrb;
    REAL r, g, b;
    PG_palette *root_pal;
    RGB_color_map *cm;
    PG_device *dev;
    FILE *fp;

    fp = io_open(file, "w");

    dev = PG_make_device("PS", "COLOR", "dump-tmp");
    PG_open_device(dev, 0.0, 1.0, 1.0, 1.0);

    root_pal = dev->color_table;
    cm       = root_pal->true_colormap;
    ntc      = root_pal->n_pal_colors;
    nc       = (int) (((double) ntc - 14.0)/2.8125);

/* build up the device color table */
    nbw = 0.5*nc;
    nsp = nc;
    nrb = 1.3125*nc;

    PRINT(fp, "            Red     Green    Blue\n");
    for (k = 0; k < ntc; k++)
        {r = cm[k].red/MAXPIX;
	 g = cm[k].green/MAXPIX;
	 b = cm[k].blue/MAXPIX;
	 if (k == 2)
            PRINT(fp, "Black and White\n");
	 if (k == nbw+2)
            PRINT(fp, "Spectrum\n");
	 if (k == nsp+nbw+2)
            PRINT(fp, "Rainbow\n");
	 PRINT(fp, "          %5.4f   %5.4f   %5.4f\n", r, g, b);};

    PG_close_device(dev);

    io_close(fp);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_PAL_EV_HAND - handle the show palette buttons */

static void _PG_pal_ev_hand(d, ev)
   void *d;
   PG_event *ev;
   {

    if (ev != NULL)
       {PG_interface_object *iob;
        PG_text_box *b;

        _PG_input_bf[1] = '\0';

        iob  = (PG_interface_object *) d;
        if (strcmp(iob->type, PG_BUTTON_OBJECT_S) == 0)
           _PG_input_bf[0] = iob->action_name[0];

        else if (strcmp(iob->type, PG_TEXT_OBJECT_S) == 0)
           {b = (PG_text_box *) iob->obj;
	    _PG_input_bf[0] = b->text_buffer[0][0];};

        longjmp(io_avail, ERR_FREE);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_PAL_MOUSE_EV_HAND - handle mouse events for palette selection */

static void _PG_pal_mouse_ev_hand(dev, ev)
   PG_device *dev;
   PG_event *ev;
   {

#ifdef HAVE_WINDOW_DEVICE

    int wx, wy, btn, mod;
    REAL x1, y1;

    PG_MOUSE_EVENT_INFO(dev, ev, wx, wy, btn, mod);

    PtoS(dev, wx, wy, x1, y1);
    StoW(dev, x1, y1);

    _PG_input_bf[0] = 10.0*y1 - 0.5;

#else

    _PG_input_bf[0] = 0;

#endif

    longjmp(io_avail, ERR_FREE);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SHOW_PALETTES - display the palettes for the given device */

void PG_show_palettes(sdev, type, wbck)
   PG_device *sdev;
   char *type;
   int wbck;
   {int i, j, k, np, npx, page;
    REAL y, dy, z, old, wd;
    char bf[MAXLINE];
    PG_device *dev;
    PG_palette *std_pal, *pal, **pals;

    if (strcmp(type, "WINDOW") != 0)
       {dev = PG_make_device(type, "COLOR", "palettes");
	PG_white_background(dev, wbck);
	PG_open_device(dev, 0.01, 0.01, 0.98, 0.98);
	PG_set_viewport(dev, 0.0, 1.0, 0.0, 1.0);
	PG_set_window(dev, 0.0, 1.0, 0.0, 1.0);}

    else
       {dev = PG_make_device(type, "COLOR", "Color Palettes");
	PG_white_background(dev, wbck);
	PG_open_device(dev, 0.05, 0.1, 0.4, 0.4);

	PG_set_viewport(dev, 0.0, 1.0, 0.0, 1.0);
	PG_set_window(dev, 0.0, 1.0, 0.0, 1.0);

        PG_set_mouse_down_event_handler(dev, _PG_pal_mouse_ev_hand);

	PG_add_button(dev, 0.1, 0.24, 0.93, 0.97, "Forward", _PG_pal_ev_hand);
	PG_add_button(dev, 0.43, 0.57, 0.93, 0.97, "Backward", _PG_pal_ev_hand);
	PG_add_button(dev, 0.76, 0.9, 0.93, 0.97, "Close", _PG_pal_ev_hand);};

    if (sdev == NULL)
       sdev = dev;

    PG_make_device_current(dev);

    old = _PG_axis_label_y_standoff;
    _PG_axis_label_y_standoff = 0.5;

/* count the palettes and make an array of them */
    np   = 0;
    npx  = 0;
    pals = NULL;
    for (pal = sdev->palettes; TRUE; )
        {SC_REMEMBER(PG_palette *, pal, pals, np, npx, 10);
         pal = pal->next;
         if (pal == sdev->palettes)
            break;};

/* draw the palettes */
    dy = 0.1;
    wd = _PG_palette_width;

    std_pal = dev->current_palette;
    for (page = 0; page < np; )
        {PG_clear_window(dev);
	 if (!HARDCOPY_DEVICE(dev))
	    {dev->current_palette = std_pal;
	     PG_set_font(dev, "helvetica", "medium", 12);
	     PG_draw_interface_objects(dev);};

	 for (i = 0, y = dy; i < 8; i++, y += dy)
	     {j = i + page;
	      if (j >= np)
                 break;

	      pal = pals[j];
	      z   = pal->n_pal_colors;

	      dev->current_palette = std_pal;
	      PG_set_line_color(dev, dev->WHITE);
	      PG_set_font(dev, "helvetica", "medium", 12);
	      PG_write_WC(dev, 0.05, y, pal->name);

	      dev->current_palette = pal;
	      PG_set_font(dev, "helvetica", "medium", 8);
	      PG_draw_palette(dev, 0.2, y, 0.9, y, 0.0, z, wd);};

	 PG_finish_plot(dev);

	 if (!HARDCOPY_DEVICE(dev))
            {if (GETLN(bf, MAXLINE, stdin) == NULL)
                break;

	     k = bf[0];
	     switch (k)
	        {default :
		      if ((0 <= k) && (k < 8) && ((page + k) < np))
			 sdev->current_palette = pals[page + k];
		      else
			 break;

		 case 'c' :
		 case 'C' :
                      SFREE(pals);
		      PG_close_device(dev);
		      _PG_axis_label_y_standoff = old;
		      return;

	         case 'f' :
	         case 'F' :
                      page += 8;
                      page = (page < np) ? page : page - 8;
		      break;

	         case 'b' :
	         case 'B' :
                      page -= 8;
                      page = (page < 0) ? page + 8 : page;
		      break;};}

	 else
	    page += 8;};

    SFREE(pals);
    PG_close_device(dev);
    _PG_axis_label_y_standoff = old;

    return;}

/*--------------------------------------------------------------------------*/

/*                      PALETTE READ/WRITE ROUTINES                         */

/*--------------------------------------------------------------------------*/

/* PG_RD_PALETTE - read a palette from a file and install it in the
 *               - specified device
 */

PG_palette *PG_rd_palette(dev, fname)
   PG_device *dev;
   char *fname;
   {int i, ns, nc, ndims, dims[2];
    double mxr, mxg, mxb, r, g, b;
    char bf[MAXLINE], *name, *s, *nd;
    RGB_color_map *true_cm;
    PG_palette *pal;
    FILE *fp;

    fp = io_open(fname, "r");
    if (fp == NULL)
       return(NULL);

    if (GETLN(bf, MAXLINE, fp) == NULL)
       return(NULL);
    
    name  = SC_strtok(bf, " \t", s);
    nc    = SC_stoi(SC_strtok(NULL, " \t\r\n", s));
    ndims = 0;
    if ((nd   = SC_strtok(NULL," \t\r\n", s)) != NULL)
       {ndims   = SC_stoi(nd);
        dims[0] = SC_stoi(SC_strtok(NULL, " \t\r\n", s));
        dims[1] = SC_stoi(SC_strtok(NULL, " \t\r\n", s));} 
    
    pal  = PG_mk_palette(dev, name, nc);

    true_cm = pal->true_colormap + 2;

    ns  = MAXPIX;
    mxr = ns*dev->max_red_intensity;
    mxg = ns*dev->max_green_intensity;
    mxb = ns*dev->max_blue_intensity;

    for (i = 0; i < nc; i++)
	{if (GETLN(bf, MAXLINE, fp) == NULL)
	    {PG_rl_palette(pal);
             pal = NULL;
	     break;};

	 r = SC_stof(SC_strtok(bf, " \t\r\n", s));
	 g = SC_stof(SC_strtok(NULL, " \t\r\n", s));
	 b = SC_stof(SC_strtok(NULL, " \t\r\n", s));

         true_cm[i].red   = r*mxr;
         true_cm[i].green = g*mxg;
         true_cm[i].blue  = b*mxb;};

    io_close(fp);
    if (pal != NULL)
       {if (ndims != 0)
           _PG_attach_palette_dims(pal, nc, ndims, dims);
        _PG_register_palette(dev, pal, FALSE);}

    return(pal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_WR_PALETTE - write a palette to an ASCII file */

int PG_wr_palette(dev, pal, fname)
   PG_device *dev;
   PG_palette *pal;
   char *fname;
   {int i, ns, nc, ndims, *dims;
    double mxr, mxg, mxb, r, g, b;
    RGB_color_map *true_cm;
    FILE *fp;

    fp = io_open(fname, "w");
    if (fp == NULL)
       return(FALSE);

    nc      = pal->n_pal_colors;
    true_cm = pal->true_colormap + 2;
    ndims   = pal->max_pal_dims;

    ns  = MAXPIX;
    mxr = 1.0/(ns*dev->max_red_intensity);
    mxg = 1.0/(ns*dev->max_green_intensity);
    mxb = 1.0/(ns*dev->max_blue_intensity);

    if ((pal->pal_dims == NULL) || (ndims != 2))
       io_printf(fp, "%s %d\n", pal->name, nc);
    else
       {dims = pal->pal_dims[ndims-1];
        io_printf(fp, "%s %d %d %d %d\n", pal->name,
                  nc, ndims, dims[0], dims[1]);}

    for (i = 0; i < nc; i++)
	{r = mxr*true_cm[i].red;
	 g = mxg*true_cm[i].green;
	 b = mxb*true_cm[i].blue;

         io_printf(fp, "%.4f  %.4f  %.4f\n", r, g, b);}

    io_close(fp);

    return(TRUE);}

/*--------------------------------------------------------------------------*/

/*                      PALETTE GENERATION ROUTINES                         */

/*--------------------------------------------------------------------------*/

/* PG_MAKE_PALETTE - make a new 1-dimensional palette for the given
 *                   device graphically
 */

PG_palette *PG_make_palette(tdev, name, nclr, wbck)
   PG_device *tdev;
   char *name;
   int nclr, wbck;
   {return(PG_make_ndim_palette(tdev, name, 1, &nclr, wbck));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_ADD_SELECTED_COLOR - book keeping for one additional color
 *                        - return the index of the newly selected color
 */

static int _PG_add_selected_color(dev, h, v, cm, nc, ncx, ncy, ipc, npc,
				  xmn, xmx, ymn, ymx,
				  dxw, dyw, dx, dy)
   PG_device *dev;
   REAL h, v;
   RGB_color_map *cm;
   int nc, ncx, ncy, ipc, npc;
   double xmn, xmx, ymn, ymx, dxw, dyw, dx, dy;
   {int i, ic;
    double s, dh, ds, dv, dc, dcmin, x0, dx0, dybox, thrs;
    REAL x[5], y[5];

    thrs = 1.0/7.0 - 1.0e-6;

/* if a selection from the arrays of available colors has been made */
    h = (h - xmn)/dxw;
    if (h > thrs)
       {s = 1.0;
	v = 2.0*(v - ymn)/dyw;
	if (v > 1.0)
	   {s = 2.0 - v;
	    v = 1.0;};}

    else
      {h = 0.0;
       s = 0.0;
       v = (v - ymn)/(dyw - dy - 0.005);};

    dcmin = HUGE;
    ic    = 0;
    for (i = 0; i < nc; i++)
        {dh = h - cm[i].red;
	 ds = s - cm[i].green;
	 dv = v - cm[i].blue;
	 dc = dh*dh + ds*ds + dv*dv;
	 if (dc < dcmin)
	    {ic = i;
	     dcmin = dc;};};

    dx0 = 0.45*dxw/((double) ncx);
    x0  = xmn + 0.45*(dxw - ncx*dx0);
    dybox = (ncy > 4) ? (.98 - (ymx + dy))/ncy : dy;
    x[0] = x0 + (ipc % ncx) *dx0;
    y[0] = ymx + 0.03*dyw + (ipc/ncx) * dybox;
    x[1] = x[0] + dx0;
    y[1] = y[0];
    x[2] = x[1];
    y[2] = y[1] + dybox;
    x[3] = x[2] - dx0;
    y[3] = y[2];
    x[4] = x[0];
    y[4] = y[0];

    PG_set_color_fill(dev, ic, TRUE);
    PG_shade_poly(dev, x, y, 5);

    return(ic);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_SELECT_COLORS - pick colors with the mouse */

static void _PG_select_colors(dev, npal, nc, ncx, ncy,  cm)
   PG_device *dev;
   PG_palette *npal;
   int nc, ncx, ncy;
   RGB_color_map *cm;
   {int ix, iy, ndims, btn, mod, ipc, npc, ic;
    int reset, row, col;
    REAL h, v;
    REAL xmn, xmx, ymn, ymx, dx, dy, dxw, dyw;
    REAL dxbox, dybox, x0, x1, y0, y1;
    RGB_color_map *pcm;

    npc = npal->n_pal_colors;
    pcm = npal->true_colormap + 2;
    ipc = 0;

    ndims = (ncy > 1) ? 2 : 1;

    xmn = 0.1;
    xmx = 0.9;
    ymn = 0.1;
    ymx = 0.85;

    dxw   = (xmx - xmn);
    dyw   = (ymx - ymn);
    dx    = 0.03*dxw;
    dy    = 0.03*dyw;
    dxbox   = 0.45*dxw/((double) ncx);
    x0    = xmn + 0.45*(dxw - ncx*dxbox);
    y0    = ymx + 0.03*dyw;
    x1    = x0 + ncx*dxbox;
    dybox = (ncy > 4) ? (.98 - (ymx + dy))/ncy : dy;
    y1    = y0 + ncy*dybox;

    reset = TRUE;
    while (TRUE)
       {if ((ipc < 0) || (ipc >= npc))
	   ipc = 0;

	PG_query_pointer(dev, &ix, &iy, &btn, &mod);
        if (btn && reset)
	   {switch (btn)
	       {case MOUSE_LEFT :
		     PtoS(dev, ix, iy, h, v);
		     StoW(dev, h, v);

                     if ((ymn < v) && (v < ymx) &&
                         (xmn < h) && (h < xmx))
		        {ic = _PG_add_selected_color(dev, h, v, cm, nc,
						     ncx, ncy, ipc, npc,
						     xmn, xmx, ymn, ymx,
						     dxw, dyw, dx, dy);
			 pcm[ipc++] = cm[ic];
		         reset = FALSE;}

/* if a cell in the new palette has been indicated */
		     else if((y0 < v) && (v < y1) &&
                             (x0 < h) && (h < x1))
			{row = (int)((v - y0) / dybox);
                         col = (int)((h - x0) / dxbox);
                         ipc = row * ncx + col;}

		     break;

		case MOUSE_MIDDLE :
                     break;

		case MOUSE_RIGHT :
                     PG_hsv_rgb(pcm, npc, FALSE);
                     return;};}
	else
	   reset = !btn;};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_MAKE_NDIM_PALETTE - make a new n-dimensional palette for the given
 *                        device graphically 
 */

PG_palette *PG_make_ndim_palette(tdev, name, ndims, dims, wbck)
   PG_device *tdev;
   char *name;
   int ndims, *dims, wbck;
   {int i, k, l, nc, color, nclrx, nclry, nclr, shape[2];
    REAL h, s, v, thrs, x0, dx0, dybox;
    REAL dx, dy, dxw, dyw, xmn, xmx, ymn, ymx, x[5], y[5];
    RGB_color_map *cm, *pcm, *scm;
    PG_device *dev;
    PG_palette *root, *npal;

    if (name == NULL)
       name = SC_strsavef("new", "char*:PG_MAKE_PALETTE:name");

/* dimensionality greater than 2 not currently supported */
    if (ndims > 2)
       ndims = 2;

    if (ndims < 1)
       {ndims = 1;
        nclr  = 16;
        nclrx = 16;
        nclry = 1;}
    else
       {nclr = dims[0];
        for (i = 1; i < ndims; i++)
            nclr *= dims[i];
        nclrx = dims[0];
        nclry = (ndims > 1) ? dims[1] : 1;}
    
    dev = PG_make_device("WINDOW", "COLOR", "Color Selection");
    PG_white_background(dev, wbck);
    PG_open_device(dev, 0.05, 0.1, 0.4, 0.6);

    PG_set_viewport(dev, 0.0, 1.0, 0.0, 1.0);
    PG_set_window(dev, 0.0, 1.0, 0.0, 1.0);

/* make the new palette */
    npal = PG_mk_palette(dev, name, nclr);

/* copy the device color table and convert to HSV */
    root = dev->color_table;
    scm  = root->true_colormap;
    nc   = root->n_pal_colors + 2;

    cm  = FMAKE_N(RGB_color_map, nc, "PG_MAKE_PALETTE:cm");
    pcm = cm;
    for (l = 0; l < nc; l++)
        *pcm++ = *scm++;

    PG_rgb_hsv(cm, nc, FALSE);

/* draw the available colors in HSV space */
    PG_make_device_current(dev);

    PG_get_viewport_WC(dev, &xmn, &xmx, &ymn, &ymx);
    PG_draw_box(dev, xmn, xmx, ymn, ymx);

    xmn = 0.1;
    xmx = 0.9;
    ymn = 0.1;
    ymx = 0.85;

    dxw = (xmx - xmn);
    dyw = (ymx - ymn);
    dx  = 0.03*dxw;
    dy  = 0.03*dyw;

    thrs = 1.0/7.0 - 1.0e-6;

/* draw the axes and such */
    PG_set_line_color(dev, dev->WHITE);
    PG_draw_box(dev, xmn, xmx, ymn, ymx);

/* saturation axis */
    PG_draw_axis(dev, xmn, ymn+0.5*dyw, xmn, ymx, 0.0, -1.0, 1.0, 0.5,
		 1.0, "%3.1f", LEFT_OF_AXIS, ENDS,
		 FALSE, MAJOR, MINOR, 0);
    PG_write_WC(dev, xmn-0.1*dxw, ymx-0.5*dyw-0.02*dyw, "1.0");
    PG_write_WC(dev, xmn-0.1*dxw, ymx-0.02*dyw, "0.5");
    PG_write_WC(dev, xmn-0.08*dxw, ymn+0.73*dyw, "S");

/* value axis */
    PG_draw_axis(dev, xmn, ymn, xmn, ymn+0.5*dyw, 0.0, 1.0, 0.5, 1.0,
		 1.0, "%3.1f", LEFT_OF_AXIS, ENDS,
		 FALSE, MAJOR, MINOR, 0);
    PG_write_WC(dev, xmn-0.1*dxw, ymn+0.5*dyw-0.02*dyw, "1.0");
    PG_write_WC(dev, xmn-0.1*dxw, ymn-0.02*dyw, "0.5");
    PG_write_WC(dev, xmn-0.08*dxw, ymn+0.23*dyw, "V");

/* hue axis */
    PG_draw_axis(dev, xmn, ymn, xmx, ymn, 0.03, 0.97, 0.0, 7.0,
		 1.0, "%3.1f", RIGHT_OF_AXIS, RIGHT_OF_AXIS,
		 FALSE, MAJOR, MINOR, LABEL, 0);
    PG_write_WC(dev, xmn+0.49*dxw, ymn-0.06*dyw, "H");

/* draw the boxes for the new colors */
    dx0   = 0.45*dxw/((double) nclrx);
    x0    = xmn + 0.45*(dxw - nclrx*dx0);
    dybox = (nclry > 4) ? (.98 - (ymx + dy))/nclry : dy;
    for (k = 0; k < nclry; k++)
        for (l = 0; l < nclrx; l++)
            {x[0] = x0 + l*dx0;
	     y[0] = ymx + 0.03*dyw + k*dybox;
	     x[1] = x[0] + dx0;
	     y[1] = y[0] + dybox;
             PG_draw_box(dev, x[0], x[1], y[0], y[1]);}

/* draw the available colors */
    dev->current_palette = root;

    color = 0;
    for (l = 0; l < nc; l++)
        {h = cm[color].red;
	 s = cm[color].green;
	 v = cm[color].blue;

	 x[0] = xmn + 0.935*dxw*(h + 0.02);

	 if (h > thrs)
	    {if (s < 0.99)
	        y[0] = ymx - 0.5*dyw*s;
	     else
	        y[0] = ymn + dyw*v - 0.39;}
	 else
	    y[0] = ymn + (dyw - dy - 0.005)*v;

	 x[1] = x[0] + dx;
	 y[1] = y[0];
	 x[2] = x[1];
	 y[2] = y[1] + dy;
	 x[3] = x[2] - dx;
	 y[3] = y[2];
	 x[4] = x[0];
	 y[4] = y[0];

	 color++;
	 PG_set_color_fill(dev, l, TRUE);
	 PG_shade_poly(dev, x, y, 5);}

    PG_update_vs(dev);

    _PG_select_colors(dev, npal, nc, nclrx, nclry, cm);

    if ((npal != NULL) && (tdev != NULL))
       {if (ndims > 1)
           {shape[0] = nclrx;
            shape[1] = nclry;
            _PG_attach_palette_dims(npal, nclr, ndims, shape);}
        _PG_register_palette(tdev, npal, FALSE);}
    else
       PG_rl_palette(npal);

    PG_close_device(dev);

    SFREE(cm);

    return(npal);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

