/*
 *
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	NJN
 *
 *	$Id: sysvslop.c,v 6.1.1.1 96/12/13 15:06:18 nevin Exp $
 *
 *	Function:	- SYSV shared structure cleanup
 */

#include <lam_config.h>

#if __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/sem.h>
#include <unistd.h>

#include <typical.h>

/*
 * public functions
 */
#if __STDC__
void			_sysv_slop_add(int, ...);
#else
void			_sysv_slop_add();
#endif
void			_sysv_slop_del();
void			_sysv_slop_clean();

/*
 * external functions
 */
extern char		*killname();

/*
 * private functions
 */
static char		*sysv_slop_filename();
static void		clean_warning();
static void		del_warning();
static int		filelock();

/*
 * private variables
 */
static char		*slop_fname = 0;	/* registry filename */

#if HAVE_UNION_SEMUN
static	union semun	semctl_arg;
#else
static union {
	int		val;
	struct semid_ds	*buf;
	unsigned short	*array;
} semctl_arg;
#endif

/*
 *	sysv_slop_add
 *
 *	Function:	- add SYSV shared structures to registry
 *	Accepts:	- number of ids
 *			- variable length list of int pairs
 *			  <type> <id>, <type> <id>, ...
 *			  type is 'm' => shared memory, 's' => semaphore set
 */
#if __STDC__
void
_sysv_slop_add(int nids, ...)
#else
void
_sysv_slop_add(nids, va_alist)

int			nids;
va_dcl
#endif

{
	char		buf[32];		/* formatting buffer */
	char		type;			/* shared structure type */
	int		id;			/* shared structure id */
	int		fd;			/* registry file descriptor */
	int		i;
	va_list		arglist;

#if __STDC__
	va_start(arglist, nids);
#else
	va_start(arglist);
#endif

	if (slop_fname == 0) {
		slop_fname = sysv_slop_filename();
		if (slop_fname == 0) return;
	}
/*
 * Open file atomically.
 */
	fd = open(slop_fname, O_WRONLY | O_APPEND | O_CREAT | O_EXCL, 0600);

	if (fd == -1 && errno == EEXIST) {
		fd = open(slop_fname, O_WRONLY | O_APPEND, 0600);
	}

	if (fd == -1) return;
/*
 * Lock the file and add the shared structure identifiers.
 */
	filelock(fd, F_SETLKW, F_WRLCK, (off_t) 0, SEEK_SET, (off_t) 0);
	
	for (i = 0; i < nids; i++) {
		type = (char) va_arg(arglist, int);
		id = va_arg(arglist, int);

		sprintf(buf, "%c %d\n", type, id);
		write(fd, buf, strlen(buf));
	}

	close(fd);
	va_end(arglist);
}

/*
 *	sysv_slop_del
 *
 *	Function:	- delete SYSV shared structure id from registry file
 *			- does not delete the structure, this should be done
 *			  before calling this function
 *	Accepts:	- structure type
 *			- structure id
 */
void
_sysv_slop_del(deltype, delid)

int			deltype;
int			delid;

{
    struct stat		fstatus;		/* registry file status */
    FILE		*f;			/* registry file */
    int			fd;			/* registry file descriptor */
    int			*ids = 0;		/* array of identifiers */
    int			nids;			/* size of id array  */
    int			nwrite;			/* number of ids to write */
    long		wtoff;			/* file write offset */
    long		rdoff;			/* file read offset */
    char		type;			/* shared structure type */
    int			id;			/* shared structure id */
    int			deleted = 0;		/* id has been deleted? */
    int			*p;
    int			i;

    if (slop_fname == 0) {
	if ((slop_fname = sysv_slop_filename()) == 0) {
	    del_warning("sysv_slop_del (sysv_slop_filename)", (FILE *) 0);
	    return;
	}
    }
/*
 * Open the registry file and lock it.
 */
    if ((f = fopen(slop_fname, "r+")) == 0) {
	del_warning("sysv_slop_del (fopen)", (FILE *) 0);
	return;
    }

    fd = fileno(f);
    
    if (filelock(fd, F_SETLKW, F_WRLCK, (off_t) 0, SEEK_SET, (off_t) 0) == -1) {
	del_warning("sysv_slop_del (filelock)", f);
	return;
    }

    if (fstat(fd, &fstatus) == -1) {
	del_warning("sysv_slop_del (fstat)", f);
	return;
    }

    if (fstatus.st_size == 0) {
	return;
    }
/*
 * Estimate number of identifiers and allocate space.
 * Each id requires ~ 13 bytes (type + space + 10 digits + newline).
 */
    nids = (fstatus.st_size / 13) + 1;
    
    ids = (int *) malloc(2 * nids * sizeof(int));
    if (ids == 0) {
	del_warning("sysv_slop_del (malloc)", f);
	return;
    }

    p = ids;
    nwrite = 0;
    wtoff = 0;

    while (fscanf(f, "%c %d\n", &type, &id) == 2) {

	if (((char) deltype) == type && delid == id) {
	    deleted = 1;
	}
	else {
	    if (nwrite == nids) {
/*
 * No space left in array of ids to be written back to file.
 * Remember the current read position, flush array to file and reset
 * read position.
 */
		if ((rdoff = ftell(f)) == -1) {
		    del_warning("sysv_slop_del (ftell)", f);
		    return;
		}

		if (fseek(f, wtoff, SEEK_SET) == -1) {
		    del_warning("sysv_slop_del (fseek)", f);
		    return;
		}

		for (i = 0; i < 2 * nwrite; i += 2) {
		    fprintf(f, "%c %d\n", (char) ids[i], ids[i+1]);
		}

		fprintf(f, "%c %d\n", type, id);
		fflush(f);

		if ((wtoff = ftell(f)) == -1) {
		    del_warning("sysv_slop_del (ftell)", f);
		    return;
		}

		if (fseek(f, rdoff, SEEK_SET)) {
		    del_warning("sysv_slop_del (fseek)", f);
		    return;
		}

		p = ids;
		nwrite = 0;
	    }
	    else {
		*p++ = (int) type;
		*p++ = id;
		nwrite++;
	    }
	}
    }
/*
 * Flush any remaining ids to file and truncate.
 */
    if (nwrite > 0) {
	if (fseek(f, wtoff, SEEK_SET) == -1) {
	    del_warning("sysv_slop_del (fseek)", f);
	    return;
	}

	for (i = 0; i < 2 * nwrite; i += 2) {
	    fprintf(f, "%c %d\n", ids[i], ids[i+1]);
	}

	fflush(f);
	if ((wtoff = ftell(f)) == -1) {
	    del_warning("sysv_slop_del (ftell)", f);
	    return;
	}
    }

    if (deleted) {
	if (ftruncate(fd, wtoff)) {
	    del_warning("sysv_slop_del (ftruncate)", f);
	    return;
	}
    }

    fclose(f);
}

/*
 *	sysv_slop_clean
 *
 *	Function:	- clean all registered SYSV IPC structures
 *	Returns:	- 0 or LAMERROR
 */
void
_sysv_slop_clean()

{
    FILE		*f = 0;			/* registry file */
    int			fd;			/* registry file descriptor */
    char		type;			/* shared structure type */
    int			id;			/* shared structure id */
    char		errmsg[128];		/* error message */
    
    if (slop_fname == 0) {
	if ((slop_fname = sysv_slop_filename()) == 0) {
	    clean_warning("_sysv_slop_clean (sysv_slop_filename)", (FILE *) 0);
	    return;
	}
    }
    
    if ((f = fopen(slop_fname, "r+")) == 0) {
	if (errno != ENOENT) {
	    clean_warning("_sysv_slop_clean (fopen)", f);
	}
	return;
    }

    fd = fileno(f);
    if (filelock(fd, F_SETLKW, F_WRLCK, (off_t) 0, SEEK_SET, (off_t) 0) == -1) {
	clean_warning("_sysv_slop_clean (filelock)", f);
	return;
    }

    while (fscanf(f, "%c %d\n", &type, &id) == 2) {
/*
 * Don't check for errors here as structures may not exist.
 */
	if (type == 'm') {
	    shmctl(id, IPC_RMID, (struct shmid_ds *) 0);
	} else if (type == 's') {
	    semctl(id, 0, IPC_RMID, semctl_arg);
	}
    }

    fclose(f);

    if (unlink(slop_fname)) {
	lam_errorstr(errmsg, sizeof(errmsg));
	fprintf(stderr,
	    "Warning: %s: unlinking shared structure registry file \"%s\"",
	    slop_fname, errmsg);
	fprintf(stderr, "\tManual cleanup may be necessary.\n");
    } else {
	free(slop_fname);
	slop_fname = 0;
    }
}

/*
 *	sysv_slop_filename
 *
 *	Function:	- forms SYSV shared structure registry filename
 *			- allocates space
 *	Returns:	- registry filename
 */
static char *
sysv_slop_filename()

{
    	char		*fname;
	
	fname = killname();
	if (fname == 0) return(0);

	strcat(fname, "-sysvids");
	return(fname);
}

/*
 *	clean_warning
 *
 *	Function:	- print warning of error while cleaning and
 *			  close the registry file
 *			- errno contains the error code
 *	Accepts:	- message
 *			- registry file
 */
static void
clean_warning(msg, f)

char			*msg;
FILE			*f;

{
	char		errmsg[128];		/* error message */

	lam_errorstr(errmsg, sizeof(errmsg));
	fprintf(stderr,
		"%s: Warning: %s: deleting shared memory structures in \"%s\"",
		msg, slop_fname, errmsg);

	fprintf(stderr, "\tManual cleanup may be necessary. Try ipcrm.\n");

	if (f) {
		fclose(f);
	}
}

/*
 *	del_warning
 *
 *	Function:	- print warning of error while deleting an id from
 *			  the registry file and then close it
 *			- errno contains the error code
 *	Accepts:	- message
 *			- registry file
 */
static void
del_warning(msg, f)

char			*msg;
FILE			*f;

{
    char		errmsg[128];		/* error message */

    lam_errorstr(errmsg, sizeof(errmsg));
    if (errno) {
	fprintf(stderr,
	    "%s: Warning: %s: deleting shared memory id's from registry.\n",
	    msg, errmsg);
    } else {
	fprintf(stderr, "%s\n", msg);
    }

    if (f) {
	fclose(f);
    }
}

/*
 *	filelock
 *
 *	Function:	- lock file via fcntl
 *	Accepts:	- file descriptor to lock
 *			- fcntl locking cmd
 *			- type of lock
 *			- relative offset of lock
 *			- starting offset of lock
 *			- length of lock
 *	Returns:	- fcntl return value
 */
static int
filelock(fd, cmd, type, offset, whence, len)

int			fd;
int			cmd;
int			type;
off_t			offset;
int			whence;
off_t			len;

{
	struct flock	lock;

	lock.l_type = type;
	lock.l_start = offset;
	lock.l_whence = whence;
	lock.l_len = len;

	return(fcntl(fd, cmd, &lock));
}
