/*
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN
 *
 *	$Id: rpi.tcp.c,v 6.1.1.1 97/03/24 12:13:09 nevin Exp $
 *
 *	Function:	- TCP client-to-client interface
 */

#include <lam_config.h>
#include <sfh.h>

#include <debug.h>

#include <errno.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/time.h>				/* LINUX FD_SET etc. */
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <sys/uio.h>

#if NEED_SYS_SELECT_H
#include <sys/select.h>
#endif

#include <app_mgmt.h>
#include <blktype.h>
#include <dl_inet.h>
#include <mpi.h>
#include <mpisys.h>
#include <net.h>
#include <rpisys.h>
#include <terror.h>
#include <typical.h>
#include <t_types.h>

/*
 * public functions
 */ 
int			_rpi_c2c_init();
int			_rpi_c2c_addprocs();
int			_rpi_c2c_build();
int			_rpi_c2c_start();
int			_rpi_c2c_destroy();
int			_rpi_c2c_advance();
int			_rpi_c2c_iprobe();
int			_rpi_c2c_finalize();
int			_c2c_envl_cmp();
int			_c2c_comm_dead();
void			_c2c_fill_mpi_status();
void			_c2c_fill_wildcards();

/*
 * external functions
 */
extern int		sfh_sock_open_srv_inet_stm();
extern int		sfh_sock_open_clt_inet_stm();
extern int		sfh_sock_accept_tmout();
extern int		ldogetlinks();
extern int		_tcp_advmultiple();
extern int		_tcp_adv1();
extern int		_tcp_proc_read_env();
extern int		_tcp_req_recv();
extern int		_tcp_req_probe();
extern int		_tcp_req_send_long();
extern int		_tcp_req_send_short();
extern int		_tcp_req_send_synch();
extern int		_tcp_buffered_adv();
extern int		_tcp_buffer();
extern void		lam_commfault();

/*
 * private functions
 */
static void		add_read();
static void		add_write();
static void		add_read_any_src();
static int		send_to_self();
static int		send_to_self_match();
static int		connect_all();
static void		fill_sync();
static void		proc_init();
static int		finalize1();

/*
 * global variables
 */
fd_set			_tcp_disabled;		/* sockets disabled by GER */
int			_c2c_flblock;		/* blocking flag */
int			_c2c_gererr;		/* flags possible GER error */
int			_c2c_haveadv;		/* have advanced? */

/*
 * external variables
 */
extern int		_tcp_nio;		/* # processes doing tcp io */
extern int		_tcp_sockmax;		/* max. tcp io socket num. */
extern fd_set		_tcp_read;		/* read sockets */
extern fd_set		_tcp_write;		/* write sockets */
extern fd_set		_tcp_except;		/* exception sockets */
extern fd_set		_tcp_block;		/* blocked mode socket? */
extern fd_set		_tcp_eoferr;		/* eof on socket is error? */
extern MPI_Request	_tcp_lastreq;		/* last tcp request */
extern struct c2c_proc  *_tcp_smap[];		/* map socket to process */

/*
 *	_rpi_c2c_init
 *
 *	Function:	- primary initialiation of RPI subsystem
 *			- initialize buffering and socket connections
 *	Returns		- 0 or LAMERROR
 */
int
_rpi_c2c_init()

{
/*
 * Initialize unexpected message buffering and GER.
 */
    if (_cbuf_init()) return(LAMERROR);

    if (lam_ger) {
	FD_ZERO(&_tcp_disabled);
    }
/*
 * Set up all processes for client-to-client communication.
 */
    if (_rpi_c2c_addprocs()) return(LAMERROR);

    return(0);
}

/*
 *	_rpi_c2c_addprocs
 *
 *	Function:	- setup for new processes
 *			- makes socket connections
 *	Returns:	- 0 or LAMERROR
 */
int
_rpi_c2c_addprocs()

{
    return(connect_all());
}

/*
 *	_rpi_c2c_build
 *
 *	Function:	- builds RPI portion of a request from scratch
 *			- one-time cost separated from _rpi_c2c_start()
 *			  to optimize persistent requests
 *	Accepts:	- request
 *	Returns:	- 0 or LAMERROR
 */
int
_rpi_c2c_build(req)

MPI_Request		req;

{
    return(0);
}

/*
 *	_rpi_c2c_start
 *
 *	Function:	- initializes RPI dependent aspects of a request
 *			- cost per request start - separated from
 *			  _rpi_c2c_build() to optimize persistent requests
 *	Accepts:	- request list
 *			- request
 *	Returns:	- 0 or LAMERROR
 */
int
_rpi_c2c_start(req_top, req)

MPI_Request		req_top;
MPI_Request		req;

{
    struct cbuf_msg	*bmsg;			/* buffered message */

    if (commdead_m(req)) return(0);
/*
 * Set common synchronization and communication parameters. The peer in
 * a receive request may be a wildcard but will be set to the actual
 * peer upon matching with an incoming mesage.
 */
    req->rq_rpi.c2c.cq_peer = req->rq_rank;
    req->rq_rpi.c2c.cq_env.ce_flags = 0;
    req->rq_rpi.c2c.cq_env.ce_tag = req->rq_tag;
    req->rq_rpi.c2c.cq_env.ce_cid = req->rq_cid;
/*
 * receive specific initialization
 */
    if ((req->rq_type == LAM_RQIRECV) || (req->rq_type == LAM_RQIPROBE)) {
	req->rq_rpi.c2c.cq_state = C2CREAD;
	req->rq_rpi.c2c.cq_env.ce_rank = req->rq_rpi.c2c.cq_peer;
/*
 * Check for matching buffered envelope/message. If one is found fill
 * in any receive request wildcards and advance the request.  
 */
	if ((bmsg = _cbuf_find(&req->rq_rpi.c2c.cq_env))) {
	    _c2c_fill_wildcards(req, &bmsg->cm_env);
	    return(_tcp_buffered_adv(req, bmsg));
	}
/*
 * No match was found. Set the request protocol transition function.
 */
	req->rq_rpi.c2c.cq_adv =
	    (req->rq_type == LAM_RQIRECV) ? _tcp_req_recv : _tcp_req_probe;
    }
/*
 * send specific initialization
 */
    else {
	req->rq_rpi.c2c.cq_env.ce_rank = req->rq_comm->c_group->g_myrank;
	req->rq_rpi.c2c.cq_env.ce_seq = req->rq_seq;
	req->rq_rpi.c2c.cq_env.ce_len = req->rq_packsize;

	if (req->rq_proc == lam_myproc) {
/*
 * send to self
 */
	    return(send_to_self(req_top, req));
	}
	else {
/*
 * send to another process
 */
	    req->rq_rpi.c2c.cq_state = C2CWRITE;
	    req->rq_rpi.c2c.cq_msgbuf = req->rq_packbuf;

	    if (req->rq_packsize > TCPSHORTMSGLEN) {
/*
 * long message protocol
 */
		req->rq_rpi.c2c.cq_env.ce_flags |= C2CLONG;
		req->rq_rpi.c2c.cq_adv = _tcp_req_send_long;

	    } else {
/*
 * short message protocol
 */
		req->rq_rpi.c2c.cq_nmsgout = req->rq_packsize;

		if (req->rq_type == LAM_RQISSEND) {
		    req->rq_rpi.c2c.cq_env.ce_flags |= C2CSSEND;
		    req->rq_rpi.c2c.cq_adv = _tcp_req_send_synch;
		} else {
		    req->rq_rpi.c2c.cq_adv = _tcp_req_send_short;
		}
	    }
/*
 * prepare for writing of envelope
 */
	    tcp_set_out_envelope_m(req->rq_rpi.c2c);
	}
    }

    return(0);
}

/*
 *	_rpi_c2c_destroy
 *
 *	Function:	- destroys RPI portion of request
 *	Accepts:	- request
 *	Returns:	- 0 or LAMERROR
 */
int
_rpi_c2c_destroy(req)

MPI_Request		req;

{
    return(0);
}

/*
 *	_rpi_c2c_advance
 *
 *	Function:	- advances requests in c2c mode
 *			- we try to advance all requests as far as possible
 *			  as allowed by RPI
 *	Accepts:	- request list
 *			- block enable flag
 *	Returns:	- 1: state change, 0: no state change, LAMERROR: error
 */
int
_rpi_c2c_advance(req_top, fl_block)

MPI_Request		req_top;
int			fl_block;

{
	MPI_Request		req;		/* current request */
/*
 * Find which requests require IO.
 */
	FD_ZERO(&_tcp_read);
	FD_ZERO(&_tcp_write);
	FD_ZERO(&_tcp_except);
	FD_ZERO(&_tcp_eoferr);
	_c2c_flblock = fl_block;
	_c2c_gererr = 0;
	_c2c_haveadv = 0;
	_tcp_nio = 0;
	_tcp_sockmax = -1;

	for (req = req_top; req; req = req->rq_next) {
/*
 * If a blocking request is done we may no longer block.
 */
		if (req->rq_state == LAM_RQSDONE) {
			if (req->rq_flags & LAM_RQFBLOCK) {
				_c2c_flblock = 0;
			}
			continue;
		}

		if (commdead_m(req)) continue;

		if (req->rq_rpi.c2c.cq_state == C2CWRITE) {
			add_write(&req->rq_proc->p_rpi.c2c, req);
		}
		else if (req->rq_rpi.c2c.cq_state == C2CREAD) {
			if (req->rq_proc == 0) {
				add_read_any_src(req);
			} else {
				add_read(&req->rq_proc->p_rpi.c2c, req);
			}
		}
	}

	if (_c2c_gererr && _c2c_flblock) {
		errno = EGERFLOW;
		return(LAMERROR);
	}

	if (_tcp_nio >= 1) {
		do {
			if (_tcp_nio == 1) {
				if (_tcp_adv1()) return(LAMERROR);
			} else {
				if (_tcp_advmultiple()) return(LAMERROR);
			}
		} while (_c2c_flblock && !_c2c_haveadv);
	}

	return(_c2c_haveadv);
}

/*
 *      _rpi_c2c_iprobe
 *
 *      Function:       - non-blocking probe
 *                      - public interface for peculiar MPI_Iprobe() which
 *                        does not return a request to the user
 *      Accepts:        - request
 *      Returns:        - 0: no msg, 1: msg, LAMERROR: error
 */
int
_rpi_c2c_iprobe(req)

MPI_Request             req;

{
	int			err;		/* error code */
/*
 * Link the probe request and advance as far as possible.
 */
	_mpi_req_add(req);
	_mpi_req_blkclr();
	err = _mpi_req_advance();
	if (err != MPI_SUCCESS) return(LAMERROR);
/*
 * Unlink the request.
 */
	_mpi_req_rem(req);
/*
 * A message was found if the request is in the done state.
 */
	return((req->rq_state == LAM_RQSDONE) ? 1 : 0);
}

/*
 *	_rpi_c2c_finalize
 *
 *	Function:	- c2c cleanup
 *	Accepts:	- process to cleanup (0 => all processes)
 *	Returns:	- 0 or LAMERROR
 */
int
_rpi_c2c_finalize(p)

struct _proc		*p;

{

	if (p) {
		return(finalize1(p));
	} else {
/*
 * Clean up buffers.
 */
		_cbuf_end();
/*
 * Loop through all processes closing connections.
 */
		for (p = lam_topproc(); p; p = lam_nextproc()) {
			if (finalize1(p)) {
				return(LAMERROR);
			}
		}		
	}

	return(0);
}

/*
 *	send_to_self
 *
 *	Function:	- advance send to self
 *	Accepts:	- request list
 *			- send request
 */
static int
send_to_self(req_top, send)

MPI_Request		req_top;
MPI_Request		send;

{
    MPI_Request     	recv;			/* receive request */
    struct cbuf_msg 	msg;			/* buffer list entry */
/*
 * Look for inactive matching receive/probe and advance if found.
 */
    for (recv = req_top; recv; recv = recv->rq_next) {

	if ((recv->rq_state == LAM_RQSSTART)
		&& (recv->rq_rpi.c2c.cq_state == C2CREAD)
		&& (!_c2c_envl_cmp(
		    &send->rq_rpi.c2c.cq_env, &recv->rq_rpi.c2c.cq_env))) {

	    if (send_to_self_match(send, recv)) {
		return(0);
	    }
	}
    }
/*
 * No matching receive found, buffer the whole message and the send is
 * done unless its a synchronous send in which case we use the user
 * buffer and the send only completes once a matching receive is posted.  
 */
    msg.cm_env = send->rq_rpi.c2c.cq_env;
    msg.cm_proc = 0;
 
    if (send->rq_type == LAM_RQISSEND) {
	send->rq_rpi.c2c.cq_state = C2CSENDSELF;
	msg.cm_buf = send->rq_packbuf;
	msg.cm_req = send;
    }
    else {
	if (send->rq_packsize > 0) {
	    if ((msg.cm_buf = (char *) malloc(send->rq_packsize)) == 0) {
		return(LAMERROR);
	    }
	    memcpy(msg.cm_buf, send->rq_packbuf, send->rq_packsize);
	}
	msg.cm_req = 0;
   	send->rq_state = LAM_RQSDONE;
	send->rq_rpi.c2c.cq_state = C2CDONE;
    }

    return(_cbuf_append(&msg) ? 0 : LAMERROR);
}

/*
 *	send_to_self_match
 *
 *	Function:	- advance send and matching receive/probe
 *	Accepts:	- send request
 *			- receive/probe request
 *	Returns:	- 1: matched a receive, 0: matched a probe
 */
static int
send_to_self_match(send, recv)

MPI_Request		send;
MPI_Request		recv;

{
    int			len;			/* # bytes to transfer */

    recv->rq_seq = send->rq_seq;
    if (recv->rq_type == LAM_RQIPROBE) {
/*
 * The receive is actually a probe so the send is not complete.
 */
	_c2c_fill_mpi_status(recv, send->rq_rpi.c2c.cq_env.ce_rank,
		send->rq_rpi.c2c.cq_env.ce_tag, send->rq_rpi.c2c.cq_env.ce_len);
		
	recv->rq_state = LAM_RQSDONE;
	recv->rq_rpi.c2c.cq_state = C2CDONE;
	return(0);
    }
    else {
/*
 * It's really a receive. Do the data transfer.
 *
 * Check for mismatched message lengths.
 */
	if (send->rq_packsize > recv->rq_packsize) {
	    recv->rq_flags |= LAM_RQFTRUNC;
	    len = recv->rq_packsize;
	} else {
	    len = send->rq_packsize;
	}

	memcpy(recv->rq_packbuf, send->rq_packbuf, len);

	_c2c_fill_mpi_status(recv, send->rq_rpi.c2c.cq_env.ce_rank,
				send->rq_rpi.c2c.cq_env.ce_tag, len);

	send->rq_state = recv->rq_state = LAM_RQSDONE;
	send->rq_rpi.c2c.cq_state = recv->rq_rpi.c2c.cq_state = C2CDONE;
	return(1);
    }
}

/*
 *	add_write
 *
 *	Function:	- add process to write advance list
 *	Accepts:	- process
 *			- writing request
 */
static void
add_write(ps, req)

struct c2c_proc		*ps;
MPI_Request		req;

{
/*
 * Already added?
 */
	if (FD_ISSET(ps->cp_sock, &_tcp_write)) {
		return;
	}
/*
 * Associate request with process.
 */
	if (!ps->cp_wreq) {
		ps->cp_wreq = req;
	}

	_tcp_nio++;
	_tcp_lastreq = req;
	FD_SET(ps->cp_sock, &_tcp_write);
	FD_SET(ps->cp_sock, &_tcp_except);

	if (ps->cp_sock > _tcp_sockmax) {
		_tcp_sockmax = ps->cp_sock;
	}
}

/*
 *	add_read
 *
 *	Function:	- add process to read advance list
 *			- do not add in case process is self
 *	Accepts:	- process
 *			- request to start matching from
 */
static void
add_read(ps, req)

struct c2c_proc		*ps;
MPI_Request		req;

{
	if (ps->cp_sock >= 0) {
/*
 * Already added?
 */
		if (FD_ISSET(ps->cp_sock, &_tcp_read)) {
			return;
		}
/*
 * Check for GER error.
 */
		if (lam_ger && FD_ISSET(ps->cp_sock, &_tcp_disabled)) {
			_c2c_gererr = 1;
			return;
		}

		_tcp_nio++;
		_tcp_lastreq = req;
		ps->cp_mreq = req;
		FD_SET(ps->cp_sock, &_tcp_read);
		FD_SET(ps->cp_sock, &_tcp_except);
    
		if (ps->cp_sock > _tcp_sockmax) {
			_tcp_sockmax = ps->cp_sock;
		}
	}
}

/*
 *	add_read_any_src
 *
 *	Function:	- add to the read advance list all processes in
 *			  the peer group of a receive request on MPI_ANY_SOURCE
 *	Accepts:	- request
 */
static void
add_read_any_src(req)

MPI_Request		req;

{
	struct _group	*g;			/* peer group */
	struct _proc	**p;
	int		i;

	g = (LAM_IS_INTER(req->rq_comm))
		? req->rq_comm->c_rgroup : req->rq_comm->c_group;

	for (i = g->g_nprocs, p = g->g_procs; i > 0; i--, p++) {
		add_read(&(*p)->p_rpi.c2c, req);
	}
}

/*
 *	_c2c_fill_wildcards
 *
 *	Function:	- replace wildcards in request with matched values
 *			  and fill in the sequence number
 *	Accepts:	- request
 *			- matched envelope
 */
void
_c2c_fill_wildcards(req, env)

MPI_Request		req;
struct c2c_envl		*env;

{
	struct _group	*g;			/* peer group */
		
	req->rq_seq = env->ce_seq;
    
	if (req->rq_rpi.c2c.cq_env.ce_tag == MPI_ANY_TAG) {
		req->rq_rpi.c2c.cq_env.ce_tag = env->ce_tag;
	}

	if (req->rq_rpi.c2c.cq_peer == MPI_ANY_SOURCE) {
		req->rq_rpi.c2c.cq_peer = env->ce_rank;
		req->rq_rpi.c2c.cq_env.ce_rank = env->ce_rank;
	
		g = (LAM_IS_INTER(req->rq_comm))
			? req->rq_comm->c_rgroup : req->rq_comm->c_group;

		req->rq_proc = g->g_procs[req->rq_rpi.c2c.cq_peer];
	}
}

/*
 *	_c2c_fill_mpi_status
 *
 *	Function:	- fill in the MPI status object
 *	Accepts:	- request
 *			- rank
 *			- tag
 *			- message length
 */
void
_c2c_fill_mpi_status(req, rank, tag, length)

MPI_Request		req;
int			rank;
int			tag;
int			length;

{
	req->rq_status.MPI_SOURCE = rank;
	req->rq_status.MPI_TAG = tag;
	req->rq_status.st_length = length;
}

/*
 *	_c2c_envl_cmp
 *
 *	Function:	- check if envelopes match
 *			- second envelope may contain wildcards and first
 *			  may not
 *	Accepts:	- ptr to envelope
 *			- ptr to request envelope
 *	Returns:	- 0 if match, 1 if not
 */
int
_c2c_envl_cmp(pe, pq)

struct c2c_envl		*pe, *pq;

{
    if ((pe->ce_cid == pq->ce_cid)
	&& ((pe->ce_rank == pq->ce_rank) || (pq->ce_rank == MPI_ANY_SOURCE))
	&& ((pe->ce_tag == pq->ce_tag) || (pq->ce_tag == MPI_ANY_TAG))
	&& ((pe->ce_flags & C2CACK) == (pq->ce_flags & C2CACK))
	&& ((pe->ce_flags & C2C2ND) == (pq->ce_flags & C2C2ND))) {

	return(0);
    }

    return(1);
}

/*
 *      _c2c_comm_dead
 *
 *      Function:       - sets dead communicator error for request
 *      Accepts:        - request
 *      Returns:        - 1
 */
int
_c2c_comm_dead(req)

MPI_Request             req;

{
	req->rq_state = LAM_RQSDONE;
	req->rq_rpi.c2c.cq_state = C2CDONE;

	if (req->rq_comm->c_flags & LAM_CLDEAD) {
		req->rq_status.MPI_ERROR = lam_mkerr(MPI_ERR_LOCALDEAD, 0);
	} else {
		req->rq_status.MPI_ERROR = lam_mkerr(MPI_ERR_REMOTEDEAD, 0);
	}

	return(1);
}

/*
 *	connect_all
 *
 *	Function:	- make tcp connections to all other processes
 *	Returns:	- 0 or LAMERROR
 */
static int
connect_all()

{
    struct _proc	*p;
    struct _gps		*mygps;			/* my GPS */
    struct nmsg		inmsg;			/* incoming network msg hdr */
    struct nmsg		outmsg;			/* outgoing network msg hdr */
    struct dolink	*links;			/* links to neighbours */
    int4		nlinks;			/* number of links */
    int			sock;			/* socket descriptor */
    int			servsockd;		/* server socket descriptor */
    int			servport = 0;		/* server port number */
    int			rnode;			/* remote node */
    int			rport;			/* remote port */
    int 		flag;			/* for setting socket opts */
    int			bufsize;		/* c2c socket buffer size */
    unsigned char	*raddr;			/* remote host address */

    if (lam_nprocs() > 1) {
/*
 * Get links to neighbours, initialize server socket, message headers, etc.
 */
	if (ldogetlinks(&links, &nlinks)) return(LAMERROR);

	servsockd = sfh_sock_open_srv_inet_stm(&servport);
	if (servsockd < 0) return(LAMERROR);

	mygps = &lam_myproc->p_gps;

	inmsg.nh_length = 0;
	outmsg.nh_length = 0;
	outmsg.nh_data[0] = (int4) servport;

	bufsize = TCPSHORTMSGLEN + sizeof(struct c2c_envl);
    }
/*
 * Loop through all processes, initializing the process data and
 * connecting to those not already connected to.
 */
    for (p = lam_topproc(); p; p = lam_nextproc()) {

	if (p->p_mode & LAM_PRPIINIT) {
	    continue;
	}	
	proc_init(p);
	
	if (p != lam_myproc) {
		
	    if (LAM_GPSCMP(mygps, &p->p_gps) >= 0) {
/*
 * Act as a client.  
 */
		fill_sync(p, lam_myproc, &inmsg);
		if (nrecv(&inmsg)) return(LAMERROR);

		rport = (int) inmsg.nh_data[0];
		
		rnode = p->p_gps.gps_node;
				
		if (rnode > nlinks) return(LAMERROR);

		raddr = (unsigned char *)
			&links[rnode].dol_addr.sin_addr.s_addr;

		sock = sfh_sock_open_clt_inet_stm(raddr, rport);
		if (sock < 0) return(LAMERROR);

	    }
	    else {
/*
 * Act as a server.
 */
		fill_sync(lam_myproc, p, &outmsg);
		if (nsend(&outmsg)) return(LAMERROR);
				
		sock = sfh_sock_accept_tmout(servsockd, -1);
		if (sock < 0) return(LAMERROR);
	    }

	    p->p_rpi.c2c.cp_sock = sock;
	    _tcp_smap[sock] = &p->p_rpi.c2c;
/*
 * Set sockets in non-blocking mode and set the send and receive buffer sizes.
 */
	    flag = 1;

	    if (fcntl(sock, F_SETFL, O_NONBLOCK)) return(LAMERROR);
	    FD_CLR(sock, &_tcp_block);

	    if (setsockopt(sock, IPPROTO_TCP, TCP_NODELAY, 
				(char *) &flag, sizeof(flag))) {
		return(LAMERROR);
	    }

#if !BROKEN_SET_INET_SO_BUFSIZES
	    {
		int	defsize;		/* kernel default buf. size */
		int	optlen;			/* socket option length */

		optlen = sizeof(defsize);
		if (getsockopt(sock, SOL_SOCKET, SO_SNDBUF,
				(char *) &defsize, &optlen))
		    return(LAMERROR);

		if (defsize < bufsize) {
		    if (setsockopt(sock, SOL_SOCKET, SO_SNDBUF,
				    (char *) &bufsize, sizeof(bufsize)))
			return(LAMERROR);
		}
	    
		optlen = sizeof(defsize);
		if (getsockopt(sock, SOL_SOCKET, SO_RCVBUF,
				(char *) &defsize, &optlen))
		    return(LAMERROR);

		if (defsize < bufsize) {
		    if (setsockopt(sock, SOL_SOCKET, SO_RCVBUF,
		    		    (char *) &bufsize, sizeof(bufsize)))
			return(LAMERROR);
		}
	    }
#endif
	}
    }

    if (lam_nprocs() > 1) {
	close(servsockd);
    }

    return(0);
}

/*
 *	proc_init
 *
 *	Function:	- initialize c2c specific process data
 *	Accepts:	- process
 */
static void
proc_init(p)

struct _proc	*p;

{
	p->p_mode |= LAM_PRPIINIT;
	p->p_rpi.c2c.cp_sock = -1;
	p->p_rpi.c2c.cp_mreq = 0;
	p->p_rpi.c2c.cp_rreq = 0;
	p->p_rpi.c2c.cp_wreq = 0;
	p->p_rpi.c2c.cp_nbfde = 0;
	p->p_rpi.c2c.cp_extra = 0;
/*
 * Set up to read in an envelope.
 */
	p->p_rpi.c2c.cp_readfn = _tcp_proc_read_env;
	p->p_rpi.c2c.cp_envbuf = (char *) &p->p_rpi.c2c.cp_env;
	p->p_rpi.c2c.cp_nenvin = sizeof(struct c2c_envl);
}

/*
 *	fill_sync
 *
 *	Function:	- fill in network message sync for connecting
 *	Accepts:	- source process
 *			- destination process
 *			- network message header (filled)
 */
static void
fill_sync(src, dest, head)

struct _proc            *src;
struct _proc            *dest;
struct nmsg             *head;

{
/*
 * This uses in effect synchronization MPI_COMM_WORLD and tag 0.
 */
	head->nh_node = dest->p_gps.gps_node;
	head->nh_event = dest->p_gps.gps_idx & 0x7FFF;
	head->nh_type = (((src->p_gps.gps_idx & 0x1F) << 10) |
		(src->p_gps.gps_node & 0x1FF)) << 16;
/*
 * Add N2ND flag to fake out mpimsg.
 */
	head->nh_flags = KSYNCSQL | N2ND;
}

/*
 *	finalize1
 *
 *	Function:	- cleanup a process
 *	Accepts:	- process
 *	Returns:	- 0 or LAMERROR
 */
static int
finalize1(p)

struct _proc		*p;

{
	if (p->p_rpi.c2c.cp_sock >= 0) {
		shutdown(p->p_rpi.c2c.cp_sock, 2);
		close(p->p_rpi.c2c.cp_sock);
		p->p_rpi.c2c.cp_sock = -1;
	}

	return(0);
}
