/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	NJN/RBD
 *
 *	$Id: laminit.c,v 6.1 96/11/23 22:53:02 nevin Rel $
 *
 *	Function:	- LAM specific initialization for MPI
 */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <app_mgmt.h>
#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <mpitrace.h>
#include <net.h>
#include <preq.h>
#include <rpisys.h>
#include <terror.h>
#include <typical.h>
#include <t_types.h>

/*
 * public variables
 */
struct _comm		lam_mpi_comm_world;
struct _comm		lam_mpi_comm_self;
struct _comm		lam_mpi_comm_parent;

struct _group		lam_mpi_group_empty;
struct _op		lam_mpi_max, lam_mpi_min;
struct _op		lam_mpi_sum, lam_mpi_prod;
struct _op		lam_mpi_land, lam_mpi_band;
struct _op		lam_mpi_lor, lam_mpi_bor;
struct _op		lam_mpi_lxor, lam_mpi_bxor;
struct _op		lam_mpi_maxloc, lam_mpi_minloc;

struct _dtype		lam_mpi_char, lam_mpi_byte;
struct _dtype		lam_mpi_int, lam_mpi_logic;
struct _dtype		lam_mpi_short, lam_mpi_long;
struct _dtype		lam_mpi_float, lam_mpi_double;
struct _dtype		lam_mpi_long_double;
struct _dtype		lam_mpi_cplex, lam_mpi_packed;
struct _dtype		lam_mpi_unsigned_char;
struct _dtype		lam_mpi_unsigned_short;
struct _dtype		lam_mpi_unsigned;
struct _dtype		lam_mpi_unsigned_long;
struct _dtype		lam_mpi_ub, lam_mpi_lb;
struct _dtype		lam_mpi_float_int, lam_mpi_double_int;
struct _dtype		lam_mpi_long_int, lam_mpi_2int;
struct _dtype		lam_mpi_2float, lam_mpi_2double;
struct _dtype		lam_mpi_short_int, lam_mpi_dblcplex;
struct _dtype		lam_mpi_integer, lam_mpi_real;
struct _dtype		lam_mpi_dblprec, lam_mpi_character;
struct _dtype		lam_mpi_2real, lam_mpi_2dblprec;
struct _dtype		lam_mpi_2integer, lam_mpi_longdbl_int;

struct _errhdl		lam_mpi_errors_are_fatal;
struct _errhdl		lam_mpi_errors_return;

struct _proc		*lam_myproc;
LIST			*lam_comms = 0;
LIST			*lam_ports = 0;
float8			lam_clockskew = 0.0;
int			lam_f77init = 0;
int			lam_c2c = 0;
int			lam_ger = 0;
int			lam_homog = 0;
int			lam_jobid = 0;
int			lam_universe_size = -1;

/*
 * external functions
 */
extern int		lam_rtrstore();
extern int		lpattach();
extern void		_lam_atexit();

/*
 * private functions
 */
static int		get_singleton_idx();
static void		finalize();

/*
 *	lam_linit
 *
 *	Function:	- initialize the process
 *	Accepts:	- program name
 *			- # processes in local world (returned)
 *			- # processes in parent world (returned)
 *			- CID to use for parent intercomm (returned)
 *			- parent and new worlds process GPS array (returned)
 *			- root in parent comm if any (returned)
 *	Returns:	- 0 or LAMERROR
 */
int
lam_linit(name, world_n, parent_n, cid, worlds, root)

char			*name;
int			*world_n;
int			*parent_n;
int			*cid;
struct _gps		**worlds;
int			*root;

{
	char		*trworld;		/* world trace */
	int		trwlen;			/* length of world trace */
	char		*str;			/* favourite string */
	int		i;			/* favourite index */
	int		procs_n;		/* # processes in total */
	char		*env;			/* string in environment */
	char		workdir[PSMAXNAME];	/* modifiable workdir copy */
	struct _gps	*procs;			/* process GPS array */
	struct _gps	*p;			/* favourite pointer */
	struct nmsg	nhead;			/* network message header */
	struct _proc	*proc;			/* process entry */
/*
 * Attach process to the kernel and turn it into an MPI process.
 */
	if (kenter(name, 0)) return(LAMERROR);
	_kio.ki_rtf |= RTF_MPI;
	if (lpattach(name)) return(LAMERROR);
	lam_flinit = 1;
/*
 * Get the # of processes in the world.
 * Allocate the array of process GPS.
 */
	if ((_kio.ki_parent > 0) || (_kio.ki_rtf & RTF_MPIRUN)) {
		*world_n = _kio.ki_world;
	} else {
		*world_n = (env = getenv("LAMWORLDNODES")) ? getncomp() : 1;
	}

	procs_n = *world_n + _kio.ki_parent;

	procs = (struct _gps *)
			malloc((unsigned) procs_n * sizeof(struct _gps));

	if (procs == 0) return(LAMERROR);
/*
 * Set various runtime flags.
 */
	lam_c2c = _kio.ki_rtf & RTF_MPIC2C;
	lam_ger = _kio.ki_rtf & RTF_MPIGER;
	lam_homog = _kio.ki_rtf & RTF_HOMOG;
/*
 * Set the CWD from the environment if required (don't report chdir() errors).
 */
	if ((_kio.ki_rtf & RTF_APPWD) && (env = getenv("LAMWORKDIR"))) {

		strncpy(workdir, env, PSMAXNAME);
		str = (workdir[0]) ? strrchr(workdir + 1, STRDIR) : 0;
		
		if (str) {
			*str = '\0';
		}
		chdir(workdir);
	}
/*
 * If spawned or started by mpirun, receive the list of GPS.  Local
 * world GPS's are first followed by the parents (if any).  Otherwise if
 * the number of processes is one assume a singleton init, else assume one
 * process per node and pids are not needed.
 */
	if ((_kio.ki_parent > 0) || (_kio.ki_rtf & RTF_MPIRUN)) {

		nhead.nh_event = -getpid();
		nhead.nh_type = BLKMPIINIT;
		nhead.nh_flags = 0;
		nhead.nh_length = procs_n * sizeof(struct _gps);
		nhead.nh_msg = (char *) procs;

		if (nrecv(&nhead)) {
			free((char *) procs);
			return(LAMERROR);
		}
/*
 * If spawned save the intercommunicator context ID, spawning root and
 * universe size.
 */
		if (_kio.ki_parent > 0) {
			*cid = (int) nhead.nh_data[1];
			*root = (int) nhead.nh_data[2];
			lam_universe_size = (int) nhead.nh_data[3];
		}
	} else {
		if (procs_n == 1) {
			procs->gps_node = getnodeid();
			procs->gps_pid = getpid();
			procs->gps_idx = get_singleton_idx(procs->gps_pid);
			if (procs->gps_idx < 0)	return(LAMERROR);
		} else {
			for (i = 0, p = procs; i < procs_n; ++i, ++p) {
				p->gps_node = i;
				p->gps_pid = getpid();
				p->gps_idx = 0;
			}
		}
	}

	lam_jobid = (_kio.ki_jobid) ? _kio.ki_jobid
			: ((procs->gps_pid << 16) | procs->gps_node);
/*
 * Set the global ranks in the local world and add the local world
 * processes to the process list while finding out my identity.
 */
	for (i = 0, p = procs; i < *world_n; ++i, ++p) {
		p->gps_grank = i;
		proc = lam_procadd(p);
		if (proc == 0) return(LAMERROR);

		if ((p->gps_node == getnodeid()) && (p->gps_pid == getpid())) {
			lam_myproc = proc;
		}

		if (_kio.ki_parent > 0) {
			proc->p_mode |= LAM_PCLIENT;
		}
	}
/*
 * Add the parents (if any) to the process list.
 */
	for (i = 0; i < _kio.ki_parent; ++i, ++p) {
		proc = lam_procadd(p);
		if (proc == 0) {
			return(LAMERROR);
		}
		proc->p_mode |= LAM_PCLIENT;
	}
/*
 * interface specific initialization
 */
	if (RPI_SPLIT(_rpi_lamd_init, _rpi_c2c_init, ())) {
		return(LAMERROR);
	}
/*
 * Synchronize clocks across the nodes.
 */
	if (_kio.ki_rtf & RTF_TRACE) {
		if (lam_clocksync(procs_n, procs, &lam_clockskew))
			return(LAMERROR);
	}
/*
 * Global rank zero records the world trace.
 */
	if ((_kio.ki_rtf & RTF_TRACE)
			&& (lam_myproc->p_gps.gps_grank == 0)) {

		trwlen = 2 * sizeof(int4) + procs_n * sizeof(struct _gps);

		trworld = (char *) malloc((unsigned) trwlen);
		if (trworld == 0) return(LAMERROR);

		((int4 *) trworld)[0] = LAM_TRMAGIC;
		((int4 *) trworld)[1] = procs_n;
		memcpy(trworld + 2 * sizeof(int4), (char *) procs,
					procs_n * sizeof(struct _gps));

		mltoti4(trworld, trwlen / sizeof(int4));

		if (lam_rtrstore(LOCAL, TRWORLD, lam_myproc->p_gps.gps_pid,
				trworld, trwlen)) {
			free(trworld);
			return(LAMERROR);
		}

		free(trworld);
	}
/*
 * Make sure finalize will be called.
 */
	_lam_atexit(finalize);

	*parent_n = _kio.ki_parent;
	*worlds = procs;

	return(0);
}

/*
 *	get_singleton_idx
 *
 *	Function:	- get kenya index of singleton MPI process
 *	Accepts:	- process pid
 *	Returns:	- kenya index or LAMERROR
 */
static int
get_singleton_idx(pid)

int			pid;

{
	struct pstate	state;			/* state returned from kenyad */
	
	if (rpstate(LOCAL, SELECT_PID, pid, &state, 1) != 1) {
		return(LAMERROR);
	}

	return(state.ps_index);
}

/*
 *	finalize
 *
 *	Function:	- calls MPI_Finalize if process neglects to do so
 */
static void
finalize()

{
	if (lam_inited() && (_kio.ki_rtf & RTF_KENYA)
			&& (_kio.ki_rtf & RTF_MPI))  {
		MPI_Finalize();
	}
}
