/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD
 *
 *	$Id: icmerge.c,v 6.1 96/11/23 22:52:28 nevin Rel $
 *
 *	Function:	- merge inter-communicator into intra-communicator
 *	Accepts:	- inter-communicator
 *			- high/low flag
 *			- new intra-communicator (out)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <stdlib.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <terror.h>

int
MPI_Intercomm_merge(comm, high, newcomm)

MPI_Comm		comm;
int			high;
MPI_Comm		*newcomm;

{
	MPI_Group	newgroup;		/* new group */
	MPI_Group	g1, g2;			/* 2 groups to merge */
	MPI_Status	stat;			/* message status */
	int		mycid;			/* my context ID */
	int		cid;			/* global context ID */
	int		rhigh;			/* remote high value */
	int		err;			/* error code */
	int		inmsg[2];		/* input message buffer */
	int		outmsg[2];		/* output message buffer */
	int		i;			/* favourite index */

	lam_initerr();
	lam_setfunc(BLKMPIICOMMMERGE);
/*
 * Check the arguments.
 */
	if (comm == MPI_COMM_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPIICOMMMERGE, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (LAM_IS_INTRA(comm)) {
		return(lam_errfunc(comm,
			BLKMPIICOMMMERGE, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (newcomm == 0) {
		return(lam_errfunc(comm,
			BLKMPIICOMMMERGE, lam_mkerr(MPI_ERR_ARG, 0)));
	}

	LAM_TRACE(lam_tr_cffstart(BLKMPIICOMMMERGE));
/*
 * Create the new context ID: rank0 processes are the leaders of their
 * local group.  Each leader finds the max context ID of all remote group
 * processes (excluding their leader) and their "high" setting.
 * The leaders then swap the information and broadcast to the remote group.
 * Note: this is a criss-cross effect, processes talk to the remote leader.
 */
	mycid = lam_getcid();
	if (mycid < 0) {
		return(lam_errfunc(comm, BLKMPIICOMMMERGE,
				lam_mkerr(MPI_ERR_INTERN, EFULL)));
	}

	if (comm->c_group->g_myrank == 0) {
		cid = mycid;
/*
 * Find max context ID of remote non-leader processes.
 */
		for (i = 1; i < comm->c_rgroup->g_nprocs; ++i) {

			err = MPI_Recv(inmsg, 1, MPI_INT,
						i, MPI_MERGE_TAG, comm, &stat);
			if (err != MPI_SUCCESS) {
				return(lam_errfunc(comm,
						BLKMPIICOMMMERGE, err));
			}

			if (inmsg[0] > cid) cid = inmsg[0];
		}
/*
 * Swap context ID and high value with remote leader.
 */
		outmsg[0] = cid;
		outmsg[1] = high;

		err = MPI_Sendrecv(outmsg, 2, MPI_INT, 0, MPI_MERGE_TAG,
					inmsg, 2, MPI_INT, 0, MPI_MERGE_TAG,
					comm, &stat);
		if (err != MPI_SUCCESS) {
			return(lam_errfunc(comm, BLKMPIICOMMMERGE, err));
		}

		if (inmsg[0] > cid) cid = inmsg[0];

		rhigh = inmsg[1];
/*
 * Broadcast context ID and local high to remote non-leader processes.
 */
		outmsg[0] = cid;
		outmsg[1] = high;

		for (i = 1; i < comm->c_rgroup->g_nprocs; ++i) {

			err = MPI_Send(outmsg, 2, MPI_INT,
						i, MPI_MERGE_TAG, comm);
			if (err != MPI_SUCCESS) {
				return(lam_errfunc(comm,
						BLKMPIICOMMMERGE, err));
			}
		}
	}
/*
 * non-leader
 */
	else {
		err = MPI_Send(&mycid, 1, MPI_INT, 0, MPI_MERGE_TAG, comm);
		if (err != MPI_SUCCESS) {
			return(lam_errfunc(comm, BLKMPIICOMMMERGE, err));
		}

		err = MPI_Recv(inmsg, 2, MPI_INT, 0,
				MPI_MERGE_TAG, comm, &stat);
		if (err != MPI_SUCCESS) {
			return(lam_errfunc(comm, BLKMPIICOMMMERGE, err));
		}

		cid = inmsg[0];
		rhigh = inmsg[1];
	}
/*
 * All procs know the "high" for local and remote groups and the context ID.
 * Create the properly ordered union group.  In case of equal high values,
 * the group that has the leader with the "lowest process" goes first.
 */
	if (high && (!rhigh)) {
		g1 = comm->c_rgroup;
		g2 = comm->c_group;
	} else if ((!high) && rhigh) {
		g1 = comm->c_group;
		g2 = comm->c_rgroup;
	} else if (lam_proccmp(comm->c_group->g_procs[0],
				comm->c_rgroup->g_procs[0]) < 0) {
		g1 = comm->c_group;
		g2 = comm->c_rgroup;
	} else {
		g1 = comm->c_rgroup;
		g2 = comm->c_group;
	}

	err = MPI_Group_union(g1, g2, &newgroup);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIICOMMMERGE, err));
	}
/*
 * Create the new communicator.
 */
	*newcomm = 0;
	if (lam_comm_new(cid, newgroup, MPI_GROUP_NULL, 0, newcomm)) {
		return(lam_errfunc(comm, BLKMPIICOMMMERGE,
				lam_mkerr(MPI_ERR_OTHER, errno)));
	}

	(*newcomm)->c_errhdl = comm->c_errhdl;
	comm->c_errhdl->eh_refcount++;

	if (!al_insert(lam_comms, newcomm)) {
		return(lam_errfunc(comm, BLKMPIICOMMMERGE,
				lam_mkerr(MPI_ERR_INTERN, errno)));
	}

	if (lam_tr_comm(*newcomm)) {
		return(lam_errfunc(comm, BLKMPIICOMMMERGE,
				lam_mkerr(MPI_ERR_INTERN, errno)));
	}

	lam_setcid(cid);

	LAM_TRACE(lam_tr_cffend(BLKMPIICOMMMERGE, -1, comm, 0, 0));

	lam_resetfunc(BLKMPIICOMMMERGE);
	return(MPI_SUCCESS);
}
