/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD
 *
 *	$Id: cartsub.c,v 6.1 96/11/23 22:51:18 nevin Rel $
 *
 *	Function:	- divide cartesian communicator into sub-grids
 *	Accepts:	- old communicator
 *			- array of kept dimensions
 *			- ptr new communicator
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <stdlib.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <terror.h>

int
MPI_Cart_sub(comm, remdims, pnewcomm)

MPI_Comm		comm;
int			*remdims;
MPI_Comm		*pnewcomm;

{
	MPI_Comm	newcomm;
	int		errcode;
	int		colour;
	int		key;
	int		colfactor;
	int		keyfactor;
	int		rank;
	int		ndim;
	int		dim;
	int		i;
	int		*d;
	int		*c;
	int		*r;
	int		*p;

	lam_initerr();
	lam_setfunc(BLKMPICARTSUB);
/*
 * Check the arguments.
 */
	if (comm == MPI_COMM_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPICARTSUB, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (LAM_IS_INTER(comm)) {
		return(lam_errfunc(comm,
			BLKMPICARTSUB, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (!LAM_IS_CART(comm)) {
		return(lam_errfunc(comm,
			BLKMPICARTSUB, lam_mkerr(MPI_ERR_TOPOLOGY, 0)));
	}

	if ((remdims == 0) || (pnewcomm == 0)) {
		return(lam_errfunc(comm,
			BLKMPICARTSUB, lam_mkerr(MPI_ERR_ARG, 0)));
	}

	LAM_TRACE(lam_tr_cffstart(BLKMPICARTSUB));
/*
 * Compute colour and key used in splitting the communicator.
 */
	colour = key = 0;
	colfactor = keyfactor = 1;
	ndim = 0;

	i = comm->c_topo_ndims - 1;
	d = comm->c_topo_dims + i;
	c = comm->c_topo_coords + i;
	r = remdims + i;

	for (; i >= 0; --i, --d, --c, --r) {
		dim = (*d > 0) ? *d : -(*d);

		if (*r == 0) {
			colour += colfactor * (*c);
			colfactor *= dim;
		} else {
			++ndim;
			key += keyfactor * (*c);
			keyfactor *= dim;
		}
	}
/*
 * Split the communicator.
 */
	errcode = MPI_Comm_split(comm, colour, key, pnewcomm);
	if (errcode != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPICARTSUB, errcode));
	}
/*
 * Fill the communicator with topology information. 
 */
	newcomm = *pnewcomm;
	if (newcomm != MPI_COMM_NULL) {
		newcomm->c_topo_type = MPI_CART;
		newcomm->c_topo_nprocs = keyfactor;
		newcomm->c_topo_ndims = ndim;

		newcomm->c_topo_dims = (int *)
				malloc((unsigned) 2 * ndim * sizeof(int));
		if (newcomm->c_topo_dims == 0) {
			return(lam_errfunc(comm, BLKMPICARTSUB,
					lam_mkerr(MPI_ERR_OTHER, errno)));
		}
		newcomm->c_topo_coords = newcomm->c_topo_dims + ndim;

		p = newcomm->c_topo_dims;
		d = comm->c_topo_dims;
		r = remdims;
		for (i = 0; i < comm->c_topo_ndims; ++i, ++d, ++r) {
			if (*r) {
				*p++ = *d;
			}
		}
/*
 * Compute the caller's coordinates.
 */
		errcode = MPI_Comm_rank(newcomm, &rank);
		if (errcode != MPI_SUCCESS) {
			return(lam_errfunc(comm, BLKMPICARTSUB, errcode));
		}

		errcode = MPI_Cart_coords(newcomm, rank,
					ndim, newcomm->c_topo_coords);
		if (errcode != MPI_SUCCESS) {
			return(lam_errfunc(comm, BLKMPICARTSUB, errcode));
		}
	}

	LAM_TRACE(lam_tr_cffend(BLKMPICARTSUB, -1, comm, 0, 0));

	lam_resetfunc(BLKMPICARTSUB);
	return(MPI_SUCCESS);
}
