/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	NJN
 *
 *	$Id: accept.c,v 6.1 96/11/22 13:33:49 nevin Rel $
 *
 *	Function:	- accept a connection on an MPI port
 *	Accepts:	- port name
 *			- info
 *			- server root process rank
 *			- server communicator
 *			- intercomm between server and client (returned)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <lam_config.h>
#include <sfh.h>

#include <errno.h>
#include <stdlib.h>
#include <string.h>

#include <app_mgmt.h>
#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <net.h>
#include <portable.h>
#include <rpisys.h>
#include <terror.h>
#include <typical.h>

/*
 * private functions
 */
static int		accept_on_port();
static void		fill_fyi();


int
MPI_Accept(port_name, info, root, comm, newcomm)

char			*port_name;
MPI_Info		info;
int			root;
MPI_Comm		comm;
MPI_Comm		*newcomm;

{
	struct _proc	**p;
	struct _gps	*g;
	struct _gps	*cltgps;		/* array of client GPS */
	MPI_Group	cltgrp;			/* client group */
	int		cltsize;		/* client communicator size */
	int		cltcid;			/* client global context ID */
	int		rank;			/* my local rank */
	int		svrsize;		/* server communicator size */
	int		mycid;			/* my context ID */
	int		cid;			/* new context ID */
	int		err;			/* error code */
	int		msg[2];			/* message buffer */
	int		i;

	lam_initerr();
	lam_setfunc(BLKMPIACCEPT);
/*
 * Check the arguments.
 */
	if ((comm == MPI_COMM_NULL) || LAM_IS_INTER(comm)) {
		return(lam_errfunc(comm, BLKMPIACCEPT,
					lam_mkerr(MPI_ERR_COMM, 0)));
	}

	MPI_Comm_size(comm, &svrsize);

	if ((root >= svrsize) || (root < 0)) {
		return(lam_errfunc(comm, BLKMPIACCEPT,
					lam_mkerr(MPI_ERR_ROOT, 0)));
	}

	if (newcomm == 0) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIACCEPT,
					lam_mkerr(MPI_ERR_ARG, 0)));
	}

	MPI_Comm_rank(comm, &rank);

	if ((rank == root) && (strlen(port_name) > LAM_PORTLEN)) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIACCEPT,
					lam_mkerr(MPI_ERR_ARG, 0)));
	}

	LAM_TRACE(lam_tr_cffstart(BLKMPIACCEPT));
/*
 * Set debugging parameters.
 */
	g = &(comm->c_group->g_procs[root]->p_gps);

	lam_setparam(BLKMPIACCEPT, root | (g->gps_grank << 16),
				(g->gps_node << 16) | g->gps_idx);
/*
 * Synchronize all members of the server communicator and get the
 * server global context ID.
 */
	mycid = lam_getcid();

	if (mycid < 0) {
		return(lam_errfunc(comm, BLKMPIACCEPT,
				lam_mkerr(MPI_ERR_INTERN, EFULL)));
	}

	err = MPI_Reduce(&mycid, &cid, 1, MPI_INT, MPI_MAX, root, comm);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIACCEPT, err));
	}

	if (rank == root) {
/*
 * Accept a connection from a client.
 */
		err = accept_on_port(port_name, info,
					comm, cid, &cltcid, &cltsize, &cltgps);
		if (err != MPI_SUCCESS) {
			msg[0] = -1;
			msg[1] = err;
			MPI_Bcast(msg, 2, MPI_INT, root, comm);
			return(lam_errfunc(comm, BLKMPIACCEPT, err));
		}
/*
 * The intercomm context ID is the maximium of the server and client
 * global context IDs.
 */
		cid = max(cid, cltcid);
		msg[0] = cid;
		msg[1] = cltsize;
	}
/*
 * Broadcast the intercomm context ID, client communicator size and
 * array of client GPS to the server communicator.
 */
	err = MPI_Bcast(msg, 2, MPI_INT, root, comm);
	if (err != MPI_SUCCESS) {
		if (rank == root) free((char *) cltgps);
		return(lam_errfunc(comm, BLKMPIACCEPT, err));
	}

        if (rank != root) {
		if (msg[0] < 0) {
			return(lam_errfunc(comm, BLKMPIACCEPT, msg[1]));
		}
		
		cid = msg[0];
		cltsize = msg[1];

		cltgps = (struct _gps *) malloc(cltsize * sizeof(struct _gps));
		if (cltgps == 0) {
			return(lam_errfunc(comm, BLKMPIACCEPT,
					lam_mkerr(MPI_ERR_OTHER, errno)));
		}
	}

	err = MPI_Bcast(cltgps, cltsize * sizeof(struct _gps) / sizeof(int),
			MPI_INT, root, comm);
	if (err != MPI_SUCCESS) {
		free((char *) cltgps);
		return(lam_errfunc(comm, BLKMPIACCEPT, err));
	}
/*
 * Create the remote (client) group.
 */
	cltgrp = (MPI_Group) malloc((unsigned) (sizeof(struct _group) +
					(cltsize * sizeof(struct _proc *))));
	if (cltgrp == 0) {
		free((char *) cltgps);
		return(lam_errfunc(comm, BLKMPIACCEPT,
					lam_mkerr(MPI_ERR_OTHER, errno)));
	}
	cltgrp->g_nprocs = cltsize;
	cltgrp->g_myrank = MPI_UNDEFINED;
	cltgrp->g_refcount = 1;
	cltgrp->g_procs = (struct _proc **)
				((char *) cltgrp + sizeof(struct _group));

	g = cltgps;
	p = cltgrp->g_procs;

	for (i = 0; i < cltsize; ++i, ++p, ++g) {

		*p = lam_procadd(g);
		if (*p == 0) {
			free((char *) cltgps);
			free((char *) cltgrp);
			return(lam_errfunc(comm, BLKMPIACCEPT,
					lam_mkerr(MPI_ERR_OTHER, errno)));
		}
		if (!((*p)->p_mode & LAM_PRPIINIT)) {
			(*p)->p_mode |= LAM_PCLIENT;
		}
		(*p)->p_refcount++;
	}
/*
 * Create the server-client intercommunicator.
 */
	*newcomm = 0;
	if (lam_comm_new(cid, comm->c_group, cltgrp, LAM_CINTER, newcomm)) {
		free((char *) cltgps);
		free((char *) cltgrp);
		return(lam_errfunc(comm, BLKMPIACCEPT,
				lam_mkerr(MPI_ERR_OTHER, errno)));
	}

	comm->c_group->g_refcount++;
	(*newcomm)->c_errhdl = comm->c_errhdl;
	comm->c_errhdl->eh_refcount++;

	if (!al_insert(lam_comms, newcomm)) {
		return(lam_errfunc(comm, BLKMPIACCEPT,
				lam_mkerr(MPI_ERR_INTERN, errno)));
	}

	if (lam_tr_comm(*newcomm)) {
		return(lam_errfunc(comm, BLKMPIACCEPT,
				lam_mkerr(MPI_ERR_INTERN, errno)));
	}

	lam_setcid(cid);
/*
 * setup new processes
 */
	if (RPI_SPLIT(_rpi_lamd_addprocs, _rpi_c2c_addprocs, ())) {
		return(lam_errfunc(comm, BLKMPIACCEPT,
				lam_mkerr(MPI_ERR_OTHER, errno)));
	}

	LAM_TRACE(lam_tr_cffend(BLKMPIACCEPT, root, comm, 0, 0));

        lam_resetfunc(BLKMPIACCEPT);
	return(MPI_SUCCESS);
}

/*
 *	accept_on_port
 *
 *	Function:	- accept a connection on a port
 *			- returns client side information
 *			- storage is allocated for the array of client GPS
 *	Accepts:	- port name
 *			- info
 *			- server side communicator
 *			- server side global context ID
 *			- client side global context ID (returned)
 *			- client communicator size (returned)
 *			- array of client GPS (returned)
 *	Returns:	- MPI_SUCCESS or error code
 */
static int
accept_on_port(port_name, info, comm, cid, cltcid, cltsize, cltgps)

char 			*port_name;
MPI_Info		info;
MPI_Comm		comm;
int			cid;
int			*cltcid;
int			*cltsize;
struct _gps		**cltgps;

{
	struct nmsg	inmsg;			/* incoming message header */
	struct nmsg	outmsg;			/* outgoing message header */
	struct _proc	**p;
	struct _gps	*g;
	struct _gps	*svrgps;		/* array of server GPS */
	struct _port	*port;			/* port */
	struct _port	search;			/* search key for port */
	int		size;			/* local group size */
	int		svrgpslen;		/* #bytes server GPS array */
	int		cltgpslen;		/* #bytes client GPS array */
	int4		mismatch;		/* run-time flags mismatch? */
	int		i;
/*
 * Find the port.
 */
	strncpy(search.prt_name, port_name, LAM_PORTLEN);

	port = al_find(lam_ports, &search);
	if (port == 0) {
		return(lam_mkerr(MPI_ERR_OTHER, ENOENT));
	}
/*
 * Block waiting for connection.
 * Synchronization used is, cid = -1, src = any, and tag = port number.
 */
	_m2l_fill(0, lam_myproc, port->prt_num, -1, &inmsg);
	inmsg.nh_flags = KSYNCSQL | N2ND;
	inmsg.nh_length = 0;

	fill_fyi(comm);
	if (nrecv(&inmsg)) {
		return(lam_mkerr(MPI_ERR_OTHER, errno));
	}

	*cltcid = (int) inmsg.nh_data[2];
	*cltsize = (int) inmsg.nh_data[3];
	mismatch = (inmsg.nh_data[4] ^ _kio.ki_rtf)
			& (RTF_MPIGER | RTF_MPIC2C);
/*
 * Reply to the client with synchonization for the return message, local
 * context ID and communicator size.  Use the synchronization passed by
 * the client with the connect message.
 */
	MPI_Comm_size(comm, &size);
	outmsg.nh_node = inmsg.nh_node;
	outmsg.nh_event = ((-1 & 0x1FFF) << 18)
				| (inmsg.nh_data[1] & 0x7FFF);
	outmsg.nh_type = ((lam_myproc->p_gps.gps_idx << 10)
				| (lam_myproc->p_gps.gps_node & 0x3FF)) << 16;
	outmsg.nh_type |= (port->prt_num & 0x7FFF);
	outmsg.nh_flags = KSYNCSQL | N2ND;
	outmsg.nh_length = 0;
	outmsg.nh_data[1] = (int4) cid;
	outmsg.nh_data[2] = (int4) size;
	outmsg.nh_data[3] = (mismatch) ? MPI_ERR_FLAGS : 0;

	if (nsend(&outmsg)) {
		return(lam_mkerr(MPI_ERR_OTHER, errno));
	}

	if (mismatch) {
		return(lam_mkerr(MPI_ERR_FLAGS, 0));
	}
/*
 * Allocate storage for the client GPS array and receive it.
 */
	cltgpslen = *cltsize * sizeof(struct _gps);
	*cltgps = (struct _gps *) malloc((unsigned) cltgpslen);
	if (*cltgps == 0) {
		return(lam_mkerr(MPI_ERR_OTHER, errno));
	}

	inmsg.nh_type |= (((inmsg.nh_data[1] & 0x1F) << 10)
				| (inmsg.nh_node & 0x3FF)) << 16;
	inmsg.nh_flags = KSYNCSQL | N2ND | DINT4MSG;
	inmsg.nh_msg = (char *) *cltgps;
	inmsg.nh_length = (int4) cltgpslen;

	if (nrecv(&inmsg)) {
		free((char *) *cltgps);
		return(lam_mkerr(MPI_ERR_OTHER, errno));
	}
/*
 * Prepare information (proposed context ID, local communicator size and
 * GPS array) for passing to client.
 */
	svrgpslen = size * sizeof(struct _gps);
	svrgps = (struct _gps *) malloc((unsigned) svrgpslen);
	if (svrgps == 0) return(lam_mkerr(MPI_ERR_OTHER, errno));

	p = comm->c_group->g_procs;
	g = svrgps;
	for (i = size; i > 0; i--, p++, g++) {
		*g = (*p)->p_gps;
	}
/*
 * Send server information to the client.
 */
	outmsg.nh_flags = KSYNCSQL | N2ND | DINT4MSG;
	outmsg.nh_msg = (char *) svrgps;
	outmsg.nh_length = (int4) svrgpslen;

	if (nsend(&outmsg)) {
		free((char *) svrgps);
		free((char *) *cltgps);
		return(lam_mkerr(MPI_ERR_OTHER, errno));
	}

	free((char *) svrgps);
	return(MPI_SUCCESS);
}

/*
 *	fill_fyi
 *
 *	Function:	- fill in process fyi
 *	Accepts:	- communicator
 */
static void
fill_fyi(comm)

MPI_Comm		comm;

{
	struct _fyiproc *fyi;                   /* FYI space in KIO */

        fyi = (struct _fyiproc *) &_kio.ki_fyi[0];
        lam_getparam(&(fyi->fyp_root), &(fyi->fyp_rootgps));
        fyi->fyp_func = lam_getfunc();
        fyi->fyp_me = (lam_myproc->p_gps.gps_grank << 16) |
	                        comm->c_group->g_myrank;
        fyi->fyp_cidtag = (lam_pt2coll(comm->c_contextid) << 16);
}
