/*
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD/GDB
 *
 *	$Id: asc_parse.c,v 6.1.1.1 97/01/27 17:10:36 nevin Exp $
 *
 *	Function:	- application schema parsing
 */

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "all_list.h"
#include "args.h"
#include "app_schema.h"
#include "lam.h"
#include "ndi.h"

/*
 * global functions
 */
LIST *			asc_bufparse();
LIST *			asc_parse();
void			asc_free();

/*
 * local functions
 */
static int		parseline();

/*
 *	asc_parse
 *
 *	Function:	- parse an application schema
 *			- returns line # in case of parsing error
 *	Accepts:	- app. schema file 
 *			- ptr line # (returned value)
 *	Returns:	- app. schema descriptor or NULL
 */
LIST *
asc_parse(appfile, plineno)

CONST char		*appfile;
int			*plineno;

{
	int		fd;			/* file desc. */
	int		nbytes;			/* # bytes read */
	int		bufsize;		/* buffer size */
	char		*filebuf;		/* file buffer */
	struct stat	fileinfo;		/* file information */
	LIST		*appd;			/* app. schema desc. */
/*
 * Get file info.
 */
	if (stat(appfile, &fileinfo)) return((LIST *) 0);

	bufsize = (int) fileinfo.st_size;
	if (bufsize <= 0) return((LIST *) 0);
/*
 * Allocate a file buffer.
 */
	filebuf = malloc((unsigned) bufsize);
	if (filebuf == 0) return((LIST *) 0);
/*
 * Open and read the file.
 */
	fd = open(appfile, O_RDONLY, 0);
	if (fd < 0) {
		free(filebuf);
		return((LIST *) 0);
	}

	nbytes = read(fd, filebuf, bufsize);
	close(fd);

	if (nbytes < bufsize) {
		if (nbytes >= 0) errno = EIO;
		free(filebuf);
		return((LIST *) 0);
	}
/*
 * Parse the app. schema buffer.
 */
	appd = asc_bufparse(filebuf, bufsize, plineno);

	free(filebuf);
	return(appd);
}

/*
 *	asc_bufparse
 *
 *	Function:	- parse an application schema from a buffer
 *			- returns line # in case of parsing error
 *	Accepts:	- buffer containing the app. schema
 *			- buffer size
 *			- ptr line # (returned value)
 *	Returns:	- app. schema descriptor or NULL
 */
LIST *
asc_bufparse(appbuf, bufsize, pline)

char			*appbuf;
int			bufsize;
int			*pline;

{
	int		linesize;		/* size of parsed line */
	int		fl_comment;		/* comment mode flag */
	int		i;			/* favourite index */
	char		*p;			/* favourite pointer */
	LIST		*appd;			/* app. schema desc. */
/*
 * File must be pure ASCII.
 */
	*pline = 0;

	for (i = 0, p = appbuf; i < bufsize; ++i, ++p) {

		if ( ! isascii(*p)) {
			errno = ENOEXEC;
			return(0);
		}
	}

	appd = al_init(sizeof(struct aschema), (int (*)()) 0);
	if (appd == 0) return(0);
/*
 * Loop over the app. schema lines.
 */
	*pline = 1;
	fl_comment = 0;

	for ( ; bufsize > 0; ++appbuf, --bufsize) {
/*
 * Handle newline characters.
 */
		if (*appbuf == '\n') {
			++(*pline);
			fl_comment = 0;
		}
/*
 * Skip leading spaces and comments.
 */
		else if (isspace(*appbuf) || fl_comment) {
		}
/*
 * Skip comments.
 */
		else if (*appbuf == '#') {
			fl_comment = 1;
		}
/*
 * Parse the line.
 */
		else {
			linesize = parseline(appd, appbuf, bufsize);

			if (linesize < 0) {
				asc_free(appd);
				appd = 0;
				bufsize = 0;
			} else if (--linesize >= 0) {
				bufsize -= linesize;
				appbuf += linesize;
			}
		}
	}

	return(appd);
}

/*
 *	parseline
 *
 *	Function:	- parse a single app. schema line
 *			- update the app. schema desc.
 *	Accepts:	- ptr app. schema desc.
 *			- line buffer
 *			- buffer size
 *	Returns:	- parsed line size or LAMERROR
 */
static int
parseline(appd, buf, bufsize)

LIST *			appd;
char *			buf;
int			bufsize;

{
	int		argtailc;		/* args after "--" */
	int		lsize;			/* line size */
	int		othersc;		/* line w/o options */
	char **		argtailv;		/* args after "--" */
	char *		cmdline;		/* command line buffer */
	char **		cmdv;			/* command line argv */
	char **		othersv;		/* line w/o options */
	char *		p;
	char **		progv;			/* line w/o nodes */
	char **		pv1;			/* current arg vector */
	char *		p2;			/* after parsed integer */
	struct apparg * procargv;		/* process argv */
	struct aschema	proc;			/* process list entry */
/*
 * Copy the line to a temporary buffer.
 */
	p = strchr(buf, '\n');
	lsize = (p == 0) ? bufsize : (int) (p - buf);
/*
 * Allocate one extra char for terminator and 2 extra so we can put a fake
 * argv[0] at the start and then use the usual parsing routines.
 */
	cmdline = malloc((unsigned) lsize + 1 + 2);
	if (cmdline == 0) return(LAMERROR);
/*
 * Insert the fake argv[0] and then copy the line.
 */
	strcpy(cmdline, "x ");
	memcpy(cmdline + 2, buf, lsize);
	cmdline[lsize + 2] = '\0';
/*
 * Split it into a command line argv.
 */
	cmdv = argvbreak(cmdline, ' ');
	free(cmdline);

	if (cmdv == 0) return(LAMERROR);
/*
 * Parse the -c and -s options and establish argtail.
 * We could use all_opt but we do not want to pull it in.
 */
	othersc = 0;
	othersv = 0;

	if (argvadd(&othersc, &othersv, cmdv[0])) {
		argvfree(cmdv);
		return(LAMERROR);
	}

	proc.asc_errno = 0;
	proc.asc_proc_cnt = -1;
	proc.asc_srcnode = -1;
	pv1 = cmdv + 1;

	while (*pv1 && strcmp(*pv1, "--")) {

		if (! strcmp(*pv1, "-c")) {
			pv1++;
			proc.asc_proc_cnt = strtol(*pv1, &p2, 0);

			if (*pv1 == p2) {
				argvfree(cmdv);
				argvfree(othersv);
				return(LAMERROR);
			}
		} else if (! strcmp(*pv1, "-s")) {
			pv1++;
			proc.asc_srcnode = ndi_parse1(*pv1);

			if ((proc.asc_srcnode < 0)
					&& (proc.asc_srcnode != LOCAL)) {
				argvfree(cmdv);
				argvfree(othersv);
				return(LAMERROR);
			}
		} else {
			if (argvadd(&othersc, &othersv, *pv1)) {
				argvfree(cmdv);
				argvfree(othersv);
				return(LAMERROR);
			}
		}

		pv1++;
	}

	if (*pv1) {
		argtailv = pv1 + 1;
		argtailc = argvcount(argtailv);
	} else {
		argtailv = 0;
		argtailc = 0;
	}
/*
 * Parse the nodeids.
 */
	proc.asc_nodelist = ndi_parse(othersc, othersv, &progv);
	argvfree(othersv);

	if (proc.asc_nodelist == 0) {
		argvfree(cmdv);
		return(LAMERROR);
	}
/*
 * We should be left with the fake command name and the single program name.
 */
	if (argvcount(progv) != 2) {
		argvfree(cmdv);
		argvfree(progv);
		errno = EUSAGE;
		return(LAMERROR);
	}
/*
 * Build the process's argv structure.
 */
	procargv = (struct apparg *) malloc(sizeof(struct apparg));

	if (procargv == 0) {
		argvfree(cmdv);
		argvfree(progv);
		return(LAMERROR);
	}

	procargv->apa_refcount = 1;
	procargv->apa_argc = argtailc + 1;

	if (argtailc > 0) {
		p = argtailv[-1];
		argtailv[-1] = progv[1];
		procargv->apa_argv = argvdup(argtailv - 1);
		argtailv[-1] = p;
	} else {
		procargv->apa_argv = argvdup(progv + 1);
	}

	argvfree(cmdv);
	argvfree(progv);

	if (procargv->apa_argv == 0) {
		free((char *) procargv);
		return(LAMERROR);
	}

	proc.asc_args = procargv;

	if (al_append(appd, &proc) == 0) {
		argvfree(procargv->apa_argv);
		free((char *) procargv);
		return(LAMERROR);
	}

	return(lsize);
}

/*
 *	asc_free
 *
 *	Function:	- free an application schema descriptor
 *	Accepts:	- app. schema desc.
 */
void
asc_free(appd)

LIST			*appd;

{
	struct aschema	*pp;			/* ptr process entry */
	struct apparg	*pa;			/* ptr process args */

	if (appd == 0) return;

	for (pp = (struct aschema *) al_top(appd); pp;
			pp = (struct aschema *) al_next(appd, pp)) {

		if (pp->asc_nodelist) {
			al_free(pp->asc_nodelist);
		}

		pa = pp->asc_args;

		if (--pa->apa_refcount == 0) {
			argvfree(pa->apa_argv);
			free((char *) pa);
		}
	}

	al_free(appd);
}
