/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	GDB/NJN
 *
 *	$Id: pqcreate.c,v 6.1 96/11/23 18:43:13 nevin Rel $
 *
 *	Function:	- OTB kenyad
 *			- creates a process
 */

#include <lam_config.h>
#include <sfh.h>

#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include <args.h>
#include <kio.h>
#include <kreq.h>
#include <net.h>
#include <portable.h>
#include <preq.h>
#include <t_types.h>
#include <terror.h>
#include <typical.h>

/*
 * global functions
 */
int			pcreate();		/* create process */
char			*argsread();		/* read argv from file */

/*
 * private functions
 */
static void		exit_protocol();
static void		close_stdio_fds();
static int		recv_stdio_fds();

/*
 * external variables
 */
extern struct pprocess	ptable[];		/* kenya process table */
extern struct preq	pq;			/* kenya request */
extern struct preply	pr;			/* kenya reply */
extern char		**pathv;		/* argv of search paths */

/*
 * external functions
 */
extern void		psendr();
extern int		putenv();
extern int		pfind();
extern int		lam_mktmpid();
extern int		sfh_sock_open_clt_unix_stm();
extern int		sfh_recv_fd();
extern char		*_path_env_find();
extern char		*getworkdir();
extern char		*rforget();		/* tag -> filename */

/*
 * local variables
 */
static char		jobbuf[32];		/* TROLLIUSJOB env space */
static char		fdbuf[32];		/* TROLLIUSFD env space */
static char		rtfbuf[32];		/* TROLLIUSRTF env space */
static char		worldbuf[32];		/* LAMWORLD env space */
static char		parentbuf[32];		/* LAMPARENT env space */
static char		dirbuf[PSMAXNAME + 11];	/* LAMWORKDIR env space */

void
pqcreate()

{
	char		*args;			/* glued argv */
	char		**argv;			/* new process args */
	char		*argvname;		/* argv filename */
	char		*loadname;		/* load module filename */
	int		argc;			/* # args */

	pr.pr_reply = 0;
	loadname = 0;
	argvname = 0;
/*
 * Find the filename of the load module.
 */
	if (pq.pq_rtf & RTF_FLAT) {
		loadname = rforget(LOCAL, pq.pq_loadpt);
	} else {
		loadname = _path_env_find(pq.pq_name, R_OK | X_OK);
	}

	if (loadname == 0) {
		pr.pr_reply = ltot(errno);
	}
/*
 * Convert argv from tag to filename.
 */
	if (pq.pq_argv) {
		argvname = rforget(LOCAL, pq.pq_argv);

		if (argvname == 0) {
			pr.pr_reply = ltot(errno);
		}
/*
 * Read in argv structure.
 */
		else {
			args = argsread(argvname);

			if (args == 0) {
				pr.pr_reply = ltot(errno);
			}
/*
 * Reconstruct the argv structure.
 */
			else {
				argv = argvbreak(args, 0xa);
				if (argv == 0) lampanic("kenyad (argvbreak)");

				free(args);
				argc = argvcount(argv);
			}

			unlink(argvname);
			free(argvname);
		}
	}
/*
 * Construct an argv from the executable name.
 */
	else if (loadname) {
		argc = 1;
		argv = argvbreak(loadname, 0xa);
		if (argv == 0) lampanic("kenyad (argvbreak)");
	}

	else {
		argv = 0;
	}
/*
 * Clean up.
 */
	if (pr.pr_reply) {

		if (loadname) {

			if (pq.pq_rtf & RTF_FLAT) {
				unlink(loadname);
			}

			free(loadname);
		}

		if (argv) {
			free((char *) argv);
		}
/*
 * The file descriptor passer (filed client) is blocked on an accept.  We must
 * exit and let it continue.  This is done by opening and immediately closing
 * a connection to it.  
 */
		if (pq.pq_rtf & RTF_PFDIO) {
			exit_protocol();
		}
	}
/*
 * Create the process.
 */
	else {
		if (pcreate(loadname, argc, argv)) {
			pr.pr_reply = ltot(errno);
		}
	}
/*
 * Reply to client.
 */
	psendr((char *) &pr, sizeof(pr), 0);
}

/*
 *	pcreate
 *
 *	Function:	- creates a new process for a given load module
 *	Accepts:	- load module filename
 *			- command line args
 *	Returns:	- 0 or ERROR
 */
int
pcreate(loadname, argc, argv)

char			*loadname;
int4			argc;
char			**argv;

{
	struct pprocess	*pnew;			/* newborn process pointer */
	int		inew;			/* index of newborn process */
	sigset_t	newset;			/* new signal set */
	sigset_t	oldset;			/* old signal set */
	int		fds[3];			/* stdio file descriptors */
	char		*cwd;			/* current working directory */
/*
 * Find a slot in the process table.
 */
	inew = pfind((int4) NOTPROCESS);

	if (inew < 0) {
		if (pq.pq_rtf & RTF_PFDIO) {
			exit_protocol();
		}

		errno = ENOPDESCRIPTORS;
		return(LAMERROR);
	}

	pnew = ptable + inew;
	pnew->p_loadpt = loadname;
	pnew->p_rtf = pq.pq_rtf | RTF_KENYA;
	pnew->p_argc = argc;
	pnew->p_argv = argv;
	pnew->p_jobid = pq.pq_jobid;
	pnew->p_nodeid = pq.pq_src_node;
	pnew->p_event = pq.pq_src_event;
/*
 * Receive stdio file descriptors. The process about to be created
 * inherits them.
 */
	if (pq.pq_rtf & RTF_PFDIO) {
		if (recv_stdio_fds(fds)) {
			return(LAMERROR);
		}
	}
/*
 * Pass Trillium runtime variables through environment.
 */
	sprintf(fdbuf, "TROLLIUSFD=%d:%d:%d:%d", pq.pq_jobid & 0xFFFF,
			pq.pq_stdin, pq.pq_stdout, pq.pq_stderr);

	if (putenv(fdbuf)) lampanic("kenyad (putenv)");

	sprintf(rtfbuf, "TROLLIUSRTF=%d", pnew->p_rtf);
	if (putenv(rtfbuf)) lampanic("kenyad (putenv)");

	sprintf(jobbuf, "LAMJOBID=%d", pq.pq_jobid);
	if (putenv(jobbuf)) lampanic("kenyad (putenv)");

	sprintf(worldbuf, "LAMWORLD=%d", pq.pq_world);
	if (putenv(worldbuf)) lampanic("kenyad (putenv)");

	sprintf(parentbuf, "LAMPARENT=%d", pq.pq_parent);
	if (putenv(parentbuf)) lampanic("kenyad (putenv)");

	if (pq.pq_rtf & RTF_APPWD) {
		sprintf(dirbuf, "LAMWORKDIR=%s", loadname);
		if (putenv(dirbuf)) lampanic("kenyad (putenv)");
	}
/*
 * Change working directory.
 */
	if (pq.pq_rtf & RTF_CWD) {
		if ((cwd = getworkdir()) == 0) {
			if (pq.pq_rtf & RTF_PFDIO) {
				close_stdio_fds(fds);
			}
			return(LAMERROR);
		}

		if (chdir(pq.pq_wrkdir)) {
			if (pq.pq_rtf & RTF_PFDIO) {
				close_stdio_fds(fds);
			}
			free(cwd);
			return(LAMERROR);
		}
	}
/*
 * Create the process.
 */
	sigemptyset(&newset);
	sigaddset(&newset, SIGCHLD);
	sigprocmask(SIG_BLOCK, &newset, &oldset);


	pnew->p_pid = kcreate(loadname, argv,
				(pq.pq_rtf & RTF_PFDIO) ? fds : 0);

	sigprocmask(SIG_SETMASK, &oldset, (sigset_t *) 0);
/*
 * Close the passed file descriptors.  Kenyad does not use them.
 */
	if (pq.pq_rtf & RTF_PFDIO) {
		close_stdio_fds(fds);
	}
/*
 * Reset working directory.
 */
	if (pq.pq_rtf & RTF_CWD) {
		if (chdir(cwd)) {
			lampanic("kenyad (chdir)");
		}
		free(cwd);
	}

	if (pnew->p_pid < 0) {
		return(LAMERROR);
	}

	pr.pr_pid = ltot(pnew->p_pid);
	pr.pr_index = ltot(inew + 1);
	return(0);
}

/*
 *	argsread
 *
 *	Function:	- reads in glued argv structure from a file
 *	Accepts:	- filename
 *	Returns:	- argv ptr or null ptr
 */
char *
argsread(argvname)

char			*argvname;

{
	char		*args;			/* glued argv buffer */
	int		avfd;			/* argv file descriptor */
	struct stat	avstat;			/* argv file status */

	if (stat(argvname, &avstat)) return(0);

	args = malloc((unsigned int) avstat.st_size);
	if (args == 0) lampanic("kenyad (malloc)");

	avfd = open(argvname, O_RDONLY, 0);

	if (avfd < 0) {
		free(args);
		return(0);
	}

	if (read(avfd, args, (int) avstat.st_size) < 0) {
		free(args);
		return(0);
	}

	close(avfd);
	return(args);
}

/*
 *	recv_stdio_fds
 *
 *	Function:	- read file descriptors from server (filed client)
 *	Accepts:	- file descriptor array (out)
 *	Returns:	- 0 or LAMERROR
 */
static int
recv_stdio_fds(fds)

int			*fds;

{
	int		pass_fd;		/* stream to receive fds over */
	char		server[LAM_PATH_MAX];	/* fd server socket name */
/*
 * Open connection to server.
 */
	if (lam_mktmpid((int) -pq.pq_src_event, server, sizeof(server))) {
		return(LAMERROR);
	}
	
        if ((pass_fd = sfh_sock_open_clt_unix_stm(server)) < 0) {
		return(LAMERROR);
	}
/*
 * Receive the file descriptors.
 */
	if ((fds[0] = sfh_recv_fd(pass_fd)) < 0) {
		close(pass_fd);
		return(LAMERROR);
	}
	
	if ((fds[1] = sfh_recv_fd(pass_fd)) < 0) {
		close(fds[0]);
		close(pass_fd);
		return(LAMERROR);
	}

	if ((fds[2] = sfh_recv_fd(pass_fd)) < 0) {
		close(fds[0]);
		close(fds[1]);
		close(pass_fd);
		return(LAMERROR);
	}
/*
 * Close connection to server.
 */
	close(pass_fd);
	return(0);
}

/*
 *	exit_protocol
 *
 *	Function:	- exit the file descriptor passing protocol
 *			  unblocking the server side
 *			- connects to server and immediately closes
 *			  the connection
 */
static void
exit_protocol()

{
	int		fd;
	char		server[LAM_PATH_MAX];	/* fd server socket name */

	if (lam_mktmpid((int) -pq.pq_src_event, server, sizeof(server)) == 0) {
		if ((fd = sfh_sock_open_clt_unix_stm(server)) >= 0) {
			close(fd);
		}
	}
}

/*
 *	close_stdio_fds
 *
 *	Function:	- close passed stdio file descriptors
 *	Accepts:	- file descriptor array 
 */
static void
close_stdio_fds(fds)

int			*fds;

{
	close(fds[0]);
	close(fds[1]);
	close(fds[2]);
}
