/*
 *	Ohio Trollius
 *	Copyright 1995 The Ohio State University
 *	GDB/RBD
 *
 *	$Log:	couter.c,v $
 * Revision 6.1  96/11/23  19:42:18  nevin
 * Ohio Release
 * 
 * Revision 6.0  96/02/29  13:11:17  gdburns
 * Ohio Release
 * 
 * Revision 5.2.1.3  96/02/28  17:46:27  gdburns
 * Use unblock instead of restore on preemption signal.
 * 
 * Revision 5.2.1.2  96/01/13  15:12:50  gdburns
 * Revise for TCP kernel.
 * 
 * Revision 5.2.1.1  94/10/20  12:13:24  gdburns
 * Split ksr into front/back.
 * 
 * Revision 5.2  94/08/22  13:36:05  gdburns
 * Ohio Release
 * 
 * Revision 5.1.1.1  94/08/18  09:35:50  gdburns
 * overhaul for new kernel
 * 
 * Revision 5.1  94/05/18  10:36:02  gdburns
 * Ohio Release
 * 
 * Revision 2.3  94/04/22  12:21:04  gdburns
 * Ohio Release
 *
 *	Function:	- client protocols
 *			- OTB version
 */

#include <errno.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>

#include <kio.h>
#include <kreq.h>
#include <terror.h>
#include <typical.h>

/*
 * local functions
 */
static void		cipc_preempt();

/*
 * external functions
 */
extern int		_cio_kreq();		/* kernel request/reply */
extern int		_cio_kreqfront();	/* kernel request */
extern int		_cio_kreqback();	/* kernel reply */
extern int		_cio_init();		/* init kernel interface */
extern int		_cio_send();		/* transfer kernel msg */
extern int		_cio_recv();		/* transfer kernel msg */
extern void		_cio_cleanup();		/* frees resources */
extern void		_ksig_follow();		/* check signals */
extern void		(*_lam_signal())();	/* portable signal() */

/*
 * external variables
 */
extern struct kio_t	_kio;			/* Kernel I/O */

/*
 *	_cipc_init
 *
 *	Function:	- initializes OTB client interface
 *	Returns:	- 0 or ERROR
 */
int
_cipc_init()

{
	if (_cio_init()) return(LAMERROR);
/*
 * Vector SIGUSR1 to preemption routine.
 */
	if (_lam_signal(SIGUSR1, cipc_preempt) == SIG_ERR) return(LAMERROR);

	return(0);
}

/*
 *	_cipc_ksend
 *
 *	Function:	- OTB kernel message send
 *			- makes KQSEND request to kernel
 *	Accepts:	- kernel request ptr
 *			- kernel reply ptr
 *	Returns:	- 0 or ERROR
 */
int
_cipc_ksend(pkq, pkr)

struct kreq		*pkq;
struct kreply		*pkr;

{
	int		r;
	sigset_t	sigs_preempt;

	if (_kio.ki_pid != getpid()) {
		errno = ENOTATTACHED;
		return(LAMERROR);
	}

	memcpy((char *) pkq->kq_fyi, (char *) _kio.ki_fyi,
			sizeof(_kio.ki_fyi));

	sigemptyset(&sigs_preempt);
	sigaddset(&sigs_preempt, SIGUSR1);
	sigprocmask(SIG_BLOCK, &sigs_preempt, (sigset_t *) 0);

	if (_cio_kreq(pkq, pkr)) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(LAMERROR);
	}

	if (pkr->kr_reply) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(0);
	}

	pkq->kq_msg.k_length = pkr->kr_length;
	r = _cio_send(&pkq->kq_msg);
	sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);

	return(r);
}

/*
 *	_cipc_krecvfront
 *
 *	Function:	- OTB kernel message receive
 *			- makes KQRECV request to kernel
 *	Accepts:	- kernel request ptr
 *	Returns:	- client/kernel comm. handle or ERROR
 */
int
_cipc_krecvfront(pkq)

struct kreq		*pkq;

{
	sigset_t	sigs_preempt;
	int		r;

	if (_kio.ki_pid != getpid()) {
		errno = ENOTATTACHED;
		return(LAMERROR);
	}

	memcpy((char *) pkq->kq_fyi, (char *) _kio.ki_fyi,
			sizeof(_kio.ki_fyi));

	sigemptyset(&sigs_preempt);
	sigaddset(&sigs_preempt, SIGUSR1);
	sigprocmask(SIG_BLOCK, &sigs_preempt, (sigset_t *) 0);

	if ((r = _cio_kreqfront(pkq)) < 0) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(LAMERROR);
	}

	return(r);
}

/*
 *	_cipc_krecvback
 *
 *	Function:	- OTB kernel message receive
 *			- receives reply for KQRECV from kernel
 *			- transfers message
 *	Accepts:	- kernel request ptr
 *			- kernel reply ptr
 *	Returns:	- 0 or ERROR
 */
int
_cipc_krecvback(pkq, pkr)

struct kreq		*pkq;
struct kreply		*pkr;

{
	int		r;
	sigset_t	sigs_preempt;

	sigemptyset(&sigs_preempt);
	sigaddset(&sigs_preempt, SIGUSR1);

	if (_cio_kreqback(pkr)) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(LAMERROR);
	}

	if (pkr->kr_reply) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(0);
	}

	pkq->kq_msg.k_length = pkr->kr_length;
	r = _cio_recv(&pkq->kq_msg);
	sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);

	return(r);
}

/*
 *	_cipc_ksrfront
 *
 *	Function:	- OTB kernel message send/receive front end
 *			- makes KQSEND followed by KQRECV request to kernel
 *	Accepts:	- kernel request ptr
 *			- kernel reply ptr
 *	Returns:	- 0 or client/kernel socket
 */
int
_cipc_ksrfront(pkq, pkr)

struct kreq		*pkq;
struct kreply		*pkr;

{
	sigset_t	sigs_preempt;
	int		r;

	if (_kio.ki_pid != getpid()) {
		errno = ENOTATTACHED;
		return(LAMERROR);
	}

	memcpy((char *) pkq->kq_fyi, (char *) _kio.ki_fyi,
			sizeof(_kio.ki_fyi));

	sigemptyset(&sigs_preempt);
	sigaddset(&sigs_preempt, SIGUSR1);
	sigprocmask(SIG_BLOCK, &sigs_preempt, (sigset_t *) 0);

	if ((r = _cio_kreqfront(pkq)) < 0) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(LAMERROR);
	}

	if (_cio_kreqback(pkr) < 0) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(LAMERROR);
	}

	if (pkr->kr_reply) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(0);
	}

	pkq->kq_msg.k_length = pkr->kr_length;

	if (_cio_send(&pkq->kq_msg)) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(LAMERROR);
	}

	return(r);
}

/*
 *	_cipc_ksrback
 *
 *	Function:	- OTB kernel message send/receive back end
 */
int
_cipc_ksrback(pkq, pkr)

struct kreq		*pkq;
struct kreply		*pkr;

{
	int		r;
	sigset_t	sigs_preempt;

	sigemptyset(&sigs_preempt);
	sigaddset(&sigs_preempt, SIGUSR1);

	if (_cio_kreqback(pkr)) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(LAMERROR);
	}

	if (pkr->kr_reply) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(0);
	}

	pkq->kq_msg2.k_length = pkr->kr_length;

	r = _cio_recv(&pkq->kq_msg2);
	sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);

	return(r);
}

/*
 *	_cipc_kreq
 *
 *	Function:	- OTB kernel request
 *			- makes request to kernel
 *	Accepts:	- kernel request ptr
 *			- kernel reply ptr
 *	Returns:	- 0 or ERROR
 */
int
_cipc_kreq(pkq, pkr)

struct kreq		*pkq;
struct kreply		*pkr;

{
	sigset_t	sigs_preempt;

	if (_kio.ki_pid != getpid()) {
		errno = ENOTATTACHED;
		return(LAMERROR);
	}

	memcpy((char *) pkq->kq_fyi, (char *) _kio.ki_fyi,
			sizeof(_kio.ki_fyi));

	do {
		sigemptyset(&sigs_preempt);
		sigaddset(&sigs_preempt, SIGUSR1);
		sigprocmask(SIG_BLOCK, &sigs_preempt, (sigset_t *) 0);

		if (_cio_kreq(pkq, pkr)) {
			sigprocmask(SIG_UNBLOCK, &sigs_preempt, 
					(sigset_t *) 0);
			return(LAMERROR);
		}

		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
/*
 * Do we need to follow a signal?
 */
		if (pkr->kr_signal) {
			_kio.ki_signal |= pkr->kr_signal;
			_ksig_follow();
		}
	} while (pkr->kr_reply == EINTR);

	return(0);
}

/*
 *	_cipc_cleanup
 *
 *	Function:	- frees up client resources
 */
void
_cipc_cleanup()

{
	_cio_cleanup();
}

/*
 *	cipc_preempt
 *
 *	Function:	- preempts myself
 *			- signal handler for SIGUSR1
 */
static void
cipc_preempt()

{
	struct kreq	req;			/* kernel request */
	struct kreply	reply;			/* kernel reply */

	if (_kio.ki_pid != getpid()) return;

	req.kq_req = KQSURRENDER;
	req.kq_index = _kio.ki_index;

	if (_cio_kreq(&req, &reply)) return;

	if (reply.kr_signal) {
		_kio.ki_signal |= reply.kr_signal;
		_ksig_follow();
	}
/*
 * Vector SIGUSR1 to preemption routine.
 */
	if (_lam_signal(SIGUSR1, cipc_preempt) == SIG_ERR) return;
}
